/*
								+--------------------------------+
								|                                |
								|    ***   Sprite utils   ***    |
								|                                |
								|  Copyright  -tHE SWINe- 2012  |
								|                                |
								|         SpriteUtils.h          |
								|                                |
								+--------------------------------+
*/

#ifndef __SPRITE_GRAPHICS_UTILITIES_INCLUDED
#define __SPRITE_GRAPHICS_UTILITIES_INCLUDED

/**
 *	@file SpriteUtils.h
 *	@author -tHE SWINe-
 *	@brief utilities for sprite images (2D images displayed with alpha blending)
 */

#include "../UberLame_src/Bitmap.h"

class CSpriteUtils {
public:
	static void TransparentColor_to_Alpha(TBmp *p_sprite, bool b_force_alpha_recalc = false)
	{
		if(b_force_alpha_recalc || !p_sprite->b_alpha) {
			uint32_t n_transparent_color = p_sprite->p_buffer[0] & 0xffffff;
			// get transparent color from lower left corner

			for(int i = 0, n = p_sprite->n_width * p_sprite->n_height; i < n; ++ i) {
				uint32_t n_color = p_sprite->p_buffer[i];
				if(n_color == n_transparent_color)
					;//p_sprite->p_buffer[i] = n_color; // do nothing, color is transparent and alpha is zero
				else if((n_color & 0xffffff) == n_transparent_color)
					p_sprite->p_buffer[i] = n_color & 0xffffff; // clear alpha
				else
					p_sprite->p_buffer[i] = n_color | 0xff000000U; // set alpha
			}
			// calculate alpha based on transparent color (binary only)

			p_sprite->b_alpha = true;
		}
		// build alpha channel using "transparent color"
	}

	static bool OpaqueArea_Erosion(TBmp *p_sprite,
		int n_erode_step_num = 1, int n_erode_thresh = 6)
	{
		if(n_erode_step_num > 0) {
			uint32_t n_transparent_color = p_sprite->p_buffer[0] & 0xffffff;
			// get transparent color from lower left corner

			TBmp *p_clone;
			if(!(p_clone = p_sprite->p_Clone()))
				return false;
			// clone the bitmap

			uint32_t *p_buffer = p_sprite->p_buffer;
			uint32_t *p_buffer_pong = p_clone->p_buffer;
			for(int i = 0; i < n_erode_step_num; ++ i) {
				bool b_change = false;
				for(int y = 0, w = p_sprite->n_width, h = p_sprite->n_height; y < h; ++ y) {
					for(int x = 0; x < w; ++ x) {
						if(p_buffer[x + w * y] != n_transparent_color) {
							int n_neigh_num = 0;

							for(int sy = max(1, y) - 1, ey = min(y + 1, h - 1); sy <= ey; ++ sy) {
								for(int sx = max(1, x) - 1, ex = min(x + 1, w - 1); sx <= ex; ++ sx) {
									if(sx == x && sy == y)
										continue;
									uint32_t n_neigh = p_buffer[sx + w * sy];
									if(n_neigh != n_transparent_color)
										++ n_neigh_num;
								}
							}
							// gather neighbour colors

							if(n_neigh_num < n_erode_thresh) {
								p_buffer_pong[x + w * y] = n_transparent_color;
								b_change = true;
							}
							// erode
						} else
							p_buffer_pong[x + w * y] = p_buffer[x + w * y]; // just copy
					}
				}
				// erode 1px into opaque color

				if(b_change || p_buffer != p_sprite->p_buffer)
					std::swap(p_buffer, p_buffer_pong);
				// swap the buffers ...

				if(!b_change)
					break;
			}

			if(p_buffer != p_sprite->p_buffer) {
				memcpy(p_sprite->p_buffer, p_buffer,
					p_sprite->n_width * p_sprite->n_height * sizeof(uint32_t));
			}
			// in case the last result is not in

			p_clone->Delete();
			// cleanup
		}
		// apply erosion

		return true;
	}

	static bool SpriteEdge_MedianFilter(TBmp *p_sprite,
		bool b_prefer_darker = true, bool b_5x5_median = true)
	{
		{
			uint32_t n_transparent_color = p_sprite->p_buffer[0] & 0xffffff;
			// get transparent color from lower left corner

			TBmp *p_clone;
			if(!(p_clone = p_sprite->p_Clone()))
				return false;
			// clone the bitmap

			uint32_t *p_buffer = p_sprite->p_buffer;
			uint32_t *p_buffer_pong = p_clone->p_buffer;
			{
				const int n_off = (b_5x5_median)? 2 : 1;
				const int n_thresh = (b_5x5_median)? 25 : 9;

				bool b_change = false;
				for(int y = 0, w = p_sprite->n_width, h = p_sprite->n_height; y < h; ++ y) {
					for(int x = 0; x < w; ++ x) {
						if(p_buffer[x + w * y] != n_transparent_color) {
							uint32_t p_neigh_color[25];
							int n_neigh_num = 0;

							for(int sy = max(n_off, y) - n_off,
							   ey = min(y + n_off, h - 1); sy <= ey; ++ sy) {
								for(int sx = max(n_off, x) - n_off,
								   ex = min(x + n_off, w - 1); sx <= ex; ++ sx) {
									uint32_t n_neigh = p_buffer[sx + w * sy];
									if(n_neigh != n_transparent_color) {
										p_neigh_color[n_neigh_num] = n_neigh;
										++ n_neigh_num;
									}
								}
							}
							// gather neighbour colors (including self)

							if(n_neigh_num < n_thresh) { // if the pixel is on the edge ...
								uint32_t r[25], g[25], b[25];
								for(int i = 0; i < n_neigh_num; ++ i) {
									r[i] = p_neigh_color[i] & 0xff0000;
									g[i] = p_neigh_color[i] & 0xff00;
									b[i] = p_neigh_color[i] & 0xff;
								}
								std::sort(r, r + n_neigh_num);
								std::sort(g, g + n_neigh_num);
								std::sort(b, b + n_neigh_num);
								// calculate median neighbor color

								uint32_t n_self = p_buffer[x + w * y];
								int mr, mg, mb;
								if(b_prefer_darker) {
									mr = min(r[n_neigh_num / 2], n_self & 0xff0000);
									mg = min(g[n_neigh_num / 2], n_self & 0xff00);
									mb = min(b[n_neigh_num / 2], n_self & 0xff);
								} else {
									mr = r[n_neigh_num / 2];
									mg = g[n_neigh_num / 2];
									mb = b[n_neigh_num / 2];
								}
								int a = n_self & 0xff000000U;

								p_buffer_pong[x + w * y] = mr | mg | mb | a;
								b_change = true;
							}
						} else
							p_buffer_pong[x + w * y] = p_buffer[x + w * y]; // just copy
					}
				}
				// grow 1px into transparent color

				if(b_change || p_buffer != p_sprite->p_buffer)
					std::swap(p_buffer, p_buffer_pong);
				// swap the buffers ...
			}

			if(p_buffer != p_sprite->p_buffer) {
				memcpy(p_sprite->p_buffer, p_buffer,
					p_sprite->n_width * p_sprite->n_height * sizeof(uint32_t));
			}
			// in case the last result is not in

			p_clone->Delete();
			// cleanup
		}

		return true;
	}

	static bool Sprite_FloodEdgeColor(TBmp *p_sprite, int n_max_grow_step_num = 0)
	{
		{
			uint32_t n_transparent_color = p_sprite->p_buffer[0] & 0xffffff;
			// get transparent color from lower left corner

			TBmp *p_clone;
			if(!(p_clone = p_sprite->p_Clone()))
				return false;
			// clone the bitmap

			uint32_t *p_buffer = p_sprite->p_buffer;
			uint32_t *p_buffer_pong = p_clone->p_buffer;
			for(int i = 0; !n_max_grow_step_num || i < n_max_grow_step_num; ++ i) {
				bool b_change = false;
				for(int y = 0, w = p_sprite->n_width, h = p_sprite->n_height; y < h; ++ y) {
					for(int x = 0; x < w; ++ x) {
						if(p_buffer[x + w * y] == n_transparent_color) {
							int n_neigh_rb = 0, n_neigh_g = 0;
							int n_neigh_num = 0;

							for(int sy = max(1, y) - 1, ey = min(y + 1, h - 1); sy <= ey; ++ sy) {
								for(int sx = max(1, x) - 1, ex = min(x + 1, w - 1); sx <= ex; ++ sx) {
									if(sx == x && sy == y)
										continue; // skip self (it's transparent anyway)
									uint32_t n_neigh = p_buffer[sx + w * sy];
									if(n_neigh != n_transparent_color) {
										n_neigh_rb += n_neigh & 0xff00ff;
										n_neigh_g += n_neigh & 0xff00;
										++ n_neigh_num;
									}
								}
							}
							// gather neighbour colors

							if(n_neigh_num > 2) {
								int r = (n_neigh_rb & 0xffff0000) / n_neigh_num;
								int g = n_neigh_g / n_neigh_num;
								int b = (n_neigh_rb & 0xffff) / n_neigh_num;
								uint32_t n_color = (0xff0000 & min(0xff0000, r)) |
									(0xff00 & min(0xff00, g)) | (0xff & min(0xff, b));
								// calculate average neighbor color

								p_buffer_pong[x + w * y] = n_color;
								b_change = true;
							}
						} else
							p_buffer_pong[x + w * y] = p_buffer[x + w * y]; // just copy
					}
				}
				// grow 1px into transparent color

				if(b_change || p_buffer != p_sprite->p_buffer)
					std::swap(p_buffer, p_buffer_pong);
				// swap the buffers ...

				if(!b_change)
					break;
			}

			if(p_buffer != p_sprite->p_buffer) {
				memcpy(p_sprite->p_buffer, p_buffer,
					p_sprite->n_width * p_sprite->n_height * sizeof(uint32_t));
			}
			// in case the last result is not in

			p_clone->Delete();
			// cleanup
		}
		// bleed colors on edge into the transparent space (to enable hifi blending)

		return true;
	}

	static bool Spritify(TBmp *p_sprite, bool b_force_alpha_recalc = false,
		int n_erode_step_num = 1, int n_max_grow_step_num = 0)
	{
		TransparentColor_to_Alpha(p_sprite, b_force_alpha_recalc);
		// build alpha channel using "transparent color"

		if(!OpaqueArea_Erosion(p_sprite, n_erode_step_num))
			return false;
		// apply erosion

		if(!SpriteEdge_MedianFilter(p_sprite, true, true))
			return false;
		// median filter on the edge pixels

		return Sprite_FloodEdgeColor(p_sprite, n_max_grow_step_num);
		// bleed colors on edge into the transparent space (to enable hifi blending)
	}
};

#endif // __SPRITE_GRAPHICS_UTILITIES_INCLUDED
