/* SkyDome.h - nice OpenGL animated sky-dome */

#include <math.h>
#include <vector>
#include "../UberLame_src/gles2/GLES20Emu.h"
#include "../UberLame_src/gles2/Shader.h"
#include "../UberLame_src/gles2/BufferObjects.h"
#include "../UberLame_src/gles2/Texture.h"
#include "../UberLame_src/gles2/TextureUtil.h"
#include "../UberLame_src/Vector.h"
#include "../UberLame_src/lml/PolyMesh.h"

class CGLSkyDome {
public:
	struct TGLSkyDomeShader : public CGLESShader {
		int n_mvp_uniform;
		int n_time_uniform;
		int n_uw_uniform;
		int n_texture_uniform;

		bool Compile()
		{
			const char *p_s_vertex_shader =
				"precision mediump float;\n"
				"\n"
				"attribute vec3 v_pos;\n"
				"attribute vec3 v_col;\n"
				"attribute vec2 v_tex;\n"
				"\n"
				"uniform mat4 t_mvp;\n"
				"\n"
				"varying vec2 v_texcoord;\n"
				"varying vec3 v_color;\n"
				"varying float y;\n"
				"\n"
				"void main()\n"
				"{\n"
				"	gl_Position = t_mvp * vec4(v_pos, 1.0);\n"
				"	v_texcoord = v_tex;\n"
				"	y = v_pos.y;\n"
				"	v_color = v_col;\n"
				"}\n";

			const char *p_s_fragment_shader =
				"precision mediump float;\n"
				"\n"
				"varying vec2 v_texcoord;\n"
				"varying vec3 v_color;\n"
				"varying float y;\n"
				"\n"
				"uniform float f_time, f_uw;\n"
				"uniform sampler2D n_tex;\n"
				"\n"
				"void main()\n"
				"{\n"
				"    vec3 v_skycolor = v_color;\n"
				"    vec3 v_texsmp2 = texture2D(n_tex, v_texcoord + vec2(.05, .033) * f_time).xyz * v_color;\n"
				"    v_skycolor = v_skycolor * (1.0 - v_texsmp2) + v_texsmp2;\n"
				"    vec3 v_texsmp3 = texture2D(n_tex, v_texcoord + vec2(.092, .07) * f_time).xyz * v_color;\n"
				"    v_skycolor = v_skycolor * (1.0 - v_texsmp3) + v_texsmp3;\n"
				"    gl_FragColor = mix(vec4(v_skycolor, 1.0), vec4(.12, .2, .5, 1.0), f_uw * step(y, 1.0));\n"
				"}\n";

			const char *p_s_config =
				"vertex {\n"
				"	v_pos: 0;\n"
				"	v_col: 1;\n"
				"	v_tex: 2;\n"
				"}\n";

			std::string p_log[3];
			if(!CGLESShader::CompileConfigureLink(p_s_vertex_shader, p_s_fragment_shader, p_s_config,
			   p_log[0], p_log[1], p_log[2], true))
				return false;

			n_mvp_uniform = n_Get_Uniform_Location("t_mvp");
			n_texture_uniform = n_Get_Uniform_Location("n_tex");
			n_time_uniform = n_Get_Uniform_Location("f_time");
			n_uw_uniform = n_Get_Uniform_Location("f_uw");

			CGLESShader::Bind();
			Uniform1i(n_texture_uniform, 0);

			return true;
		}

		void Bind(const Matrix4f &r_t_mvp, float f_time, float f_underwater)
		{
			CGLESShader::Bind();
			UniformMatrix4fv(n_mvp_uniform, 1, false, &r_t_mvp[0][0]);
			Uniform1f(n_time_uniform, f_time);
			Uniform1f(n_uw_uniform, f_underwater);
		}
	};

protected:
	CGLArrayBufferObject m_vertex_buffer;
	CGLElementArrayBufferObject m_index_buffer;
	CGLVertexArrayObject m_vao;
	size_t m_n_index_num;
	TGLSkyDomeShader m_shader;
	CGLTexture_2D_Loadable<false> m_sky_texture;

public:
	CGLSkyDome(Vector3f v_color1 = Vector3f(.3f, .45f, .9f),
		Vector3f v_color2 = Vector3f(.3f, 1, .9f), int n_radial_tess = 20,
		int n_axial_tess = 10, float f_height = 350, float f_radius = 1500)
		:m_n_index_num(0), m_sky_texture("clouds.png")
	{
		CPolyMesh mesh;
		if(!Create_SkyDome_Mesh(mesh, v_color1, v_color2, n_radial_tess, n_axial_tess, f_height, f_radius))
			return;
		std::vector<uint32_t> index_buffer;
		std::vector<float> vertex_buffer;
		if(!mesh.Build_DrawBuffers(index_buffer, vertex_buffer, false, true, 2))
			return;

		m_vao.Bind();
		{
			m_index_buffer.Bind();
			m_index_buffer.BufferData(index_buffer.size() * sizeof(uint32_t), &index_buffer[0]);
			m_vertex_buffer.Bind();
			m_vertex_buffer.BufferData(vertex_buffer.size() * sizeof(float), &vertex_buffer[0]);

			glEnableVertexAttribArray(0);
			glEnableVertexAttribArray(1);
			glEnableVertexAttribArray(2);
			glVertexAttribPointer(0, 3, GL_FLOAT, false, 9 * sizeof(float),
				m_vertex_buffer.p_OffsetPointer(0)); // pos
			glVertexAttribPointer(1, 3, GL_FLOAT, false, 9 * sizeof(float),
				m_vertex_buffer.p_OffsetPointer(3 * sizeof(float))); // color
			glVertexAttribPointer(2, 2, GL_FLOAT, false, 9 * sizeof(float),
				m_vertex_buffer.p_OffsetPointer(7 * sizeof(float))); // texcoord
		}
		CGLVertexArrayObject::Release();

		if(!m_shader.Compile())
			return;

		m_n_index_num = index_buffer.size();
	}

	bool b_Status() const
	{
		return m_vao.b_Status() && m_vertex_buffer.b_Status() &&
			m_index_buffer.b_Status() && m_n_index_num;
	}

	void Draw(const Matrix4f &r_t_mvp, float f_time)
	{
		m_sky_texture.Bind();
		m_shader.Bind(r_t_mvp, f_time * .2f, 0.0);
		m_vao.Bind();
		glDrawElements(GL_TRIANGLES, m_n_index_num, GL_UNSIGNED_INT, m_index_buffer.p_OffsetPointer(0));
		CGLVertexArrayObject::Release();
	}

	void Draw_UW(const Matrix4f &r_t_mvp, float f_time)
	{
		m_sky_texture.Bind();
		m_shader.Bind(r_t_mvp, f_time * .2f, 1.0);
		m_vao.Bind();
		glDrawElements(GL_TRIANGLES, m_n_index_num, GL_UNSIGNED_INT, m_index_buffer.p_OffsetPointer(0));
		CGLVertexArrayObject::Release();
	}

	static bool Create_SkyDome_Mesh(CPolyMesh &r_mesh, Vector3f v_color1 = Vector3f(.3f, .45f, .9f),
		Vector3f v_color2 = Vector3f(.3f, 1, .9f), int n_radial_tess = 20, int n_axial_tess = 10,
		float f_height = 350, float f_radius = 1500)
	{
		size_t n_vertex_num = 1 + n_radial_tess * (n_axial_tess + 1);
		size_t n_index_num = n_radial_tess * 3 + 3 * 2 * n_radial_tess * n_axial_tess;
		// calculate new primitive counts

		if(!r_mesh.Alloc(n_vertex_num, n_index_num / 3))
			return false;
		// alloc

		{
			CPolyMesh::_TyVertexPool::CIterator p_vertex_it = r_mesh.r_Vertex_Pool().p_Begin_it();
			p_vertex_it->v_position = Vector3f(0, f_height, 0);
			p_vertex_it->v_color = Vector4f(v_color1, 1.0);
			p_vertex_it->p_texture_coord[0] = Vector4f(0, 0, 0, 1);
			++ p_vertex_it;
			for(int i = 0; i <= n_axial_tess; ++ i) {
				float f_axial_angle = float(.1 + f_pi * .6 * i) / n_axial_tess;
				for(int j = 0; j < n_radial_tess; ++ j, ++ p_vertex_it) {
					float f_radial_angle = float(2 * f_pi * j) / n_radial_tess;
					Vector3f v_position = Vector3f((float)sin(f_radial_angle),
						0, (float)cos(f_radial_angle)) * ((float)sin(f_axial_angle) * f_radius);
					v_position.y = (float)cos(f_axial_angle) * f_height;
					p_vertex_it->v_position = v_position;
					p_vertex_it->v_color.x = v_color2.x + (v_color1.x - v_color2.x) *
						((float)exp(v_position.y / f_height) / 2.8f);
					p_vertex_it->v_color.y = v_color2.y + (v_color1.y - v_color2.y) *
						((float)exp(v_position.y / f_height) / 2.8f);
					p_vertex_it->v_color.z = v_color2.z + (v_color1.z - v_color2.z) *
						((float)exp(v_position.y / f_height) / 2.8f);
					p_vertex_it->v_color.w = 1;
					p_vertex_it->p_texture_coord[0] = Vector4f(v_position.x * (1.0f / f_radius),
						v_position.z * (1.0f / f_radius), 0, 1);
				}
			}
			_ASSERTE(p_vertex_it == r_mesh.r_Vertex_Pool().p_End_it());
		}
		// generate vertices

		{
			CPolyMesh::_TyPolygonPool::CIterator p_poly_it = r_mesh.r_Polygon_Pool().p_Begin_it();
			for(int i = 0; i < n_radial_tess; ++ i, ++ p_poly_it) {
				p_poly_it->Insert_Vertex(r_mesh.t_RefVertex(0));
				p_poly_it->Insert_Vertex(r_mesh.t_RefVertex(1 + (i + 1) % n_radial_tess));
				p_poly_it->Insert_Vertex(r_mesh.t_RefVertex(1 + i));
			}
			// cap winging

			for(int k = 0; k < n_axial_tess; ++ k) {
				for(int j = 0; j < n_radial_tess; ++ j, ++ p_poly_it) {
					p_poly_it->Insert_Vertex(r_mesh.t_RefVertex(1 + j + k * n_radial_tess));
					p_poly_it->Insert_Vertex(r_mesh.t_RefVertex(1 + (j + 1) % n_radial_tess + k * n_radial_tess));
					p_poly_it->Insert_Vertex(r_mesh.t_RefVertex(1 + (j + 1) % n_radial_tess + (k + 1) * n_radial_tess));
					++ p_poly_it;
					p_poly_it->Insert_Vertex(r_mesh.t_RefVertex(1 + j + k * n_radial_tess));
					p_poly_it->Insert_Vertex(r_mesh.t_RefVertex(1 + (j + 1) % n_radial_tess + (k + 1) * n_radial_tess));
					p_poly_it->Insert_Vertex(r_mesh.t_RefVertex(1 + j + (k + 1) * n_radial_tess));
				}
			}
			// coat winding

			_ASSERTE(p_poly_it == r_mesh.r_Polygon_Pool().p_End_it());
		}
		// generate winding


		return true;
	}
};
