
struct CSimpleShader : public CGLESShader {
	int n_mvp_uniform;
	int n_texture_uniform;

	bool Compile()
	{
		const char *p_s_vertex_shader =
			"precision mediump float;\n"
			"\n"
			"attribute vec3 v_pos;\n"
			"attribute vec2 v_tex;\n"
			"\n"
			"uniform mat4 t_mvp;\n"
			"\n"
			"varying vec2 v_texcoord;\n"
			"\n"
			"void main()\n"
			"{\n"
			"	gl_Position = t_mvp * vec4(v_pos, 1.0);\n"
			"	v_texcoord = v_tex;\n"
			"}\n";

		const char *p_s_fragment_shader =
			"precision mediump float;\n"
			"\n"
			"varying vec2 v_texcoord;\n"
			"\n"
			"uniform sampler2D n_tex;\n"
			"\n"
			"void main()\n"
			"{\n"
			"    gl_FragColor = texture2D(n_tex, v_texcoord);\n"
			"}\n";

		const char *p_s_config =
			"vertex {\n"
			"	v_pos: 0;\n"
			"	v_tex: 1;\n"
			"}\n";

		std::string p_log[3];
		if(!CGLESShader::CompileConfigureLink(p_s_vertex_shader, p_s_fragment_shader, p_s_config,
		   p_log[0], p_log[1], p_log[2], true))
			return false;

		n_mvp_uniform = n_Get_Uniform_Location("t_mvp");
		n_texture_uniform = n_Get_Uniform_Location("n_tex");

		CGLESShader::Bind();
		Uniform1i(n_texture_uniform, 0);

		return true;
	}

	void Bind(const Matrix4f &r_t_mvp) const
	{
		CGLESShader::Bind();
		UniformMatrix4fv(n_mvp_uniform, 1, false, &r_t_mvp[0][0]);
	}
};

struct CColorShader : public CGLESShader {
	int n_mvp_uniform;
	int n_color_uniform;

	bool Compile()
	{
		const char *p_s_vertex_shader =
			"precision mediump float;\n"
			"\n"
			"attribute vec3 v_pos;\n"
			"\n"
			"uniform mat4 t_mvp;\n"
			"\n"
			"void main()\n"
			"{\n"
			"	gl_Position = t_mvp * vec4(v_pos, 1.0);\n"
			"}\n";

		const char *p_s_fragment_shader =
			"precision mediump float;\n"
			"\n"
			"varying vec2 v_texcoord;\n"
			"\n"
			"uniform vec4 v_color;\n"
			"\n"
			"void main()\n"
			"{\n"
			"    gl_FragColor = v_color;\n"
			"}\n";

		const char *p_s_config =
			"vertex {\n"
			"	v_pos: 0;\n"
			"}\n";

		std::string p_log[3];
		if(!CGLESShader::CompileConfigureLink(p_s_vertex_shader, p_s_fragment_shader, p_s_config,
		   p_log[0], p_log[1], p_log[2], true))
			return false;

		n_mvp_uniform = n_Get_Uniform_Location("t_mvp");
		n_color_uniform = n_Get_Uniform_Location("v_color");

		return true;
	}

	void Bind(const Matrix4f &r_t_mvp, Vector4f v_color) const
	{
		CGLESShader::Bind();
		UniformMatrix4fv(n_mvp_uniform, 1, false, &r_t_mvp[0][0]);
		Uniform4f(n_color_uniform, v_color.x, v_color.y, v_color.z, v_color.w);
	}
};

struct CPointShader : public CGLESShader {
	int n_mvp_uniform;
	int n_color_uniform;
	int n_size_uniform;

	bool Compile()
	{
		const char *p_s_vertex_shader =
			"precision mediump float;\n"
			"\n"
			"attribute vec3 v_pos;\n"
			"\n"
			"uniform mat4 t_mvp;\n"
			"uniform float f_point_size;\n"
			"\n"
			"void main()\n"
			"{\n"
			"	gl_Position = t_mvp * vec4(v_pos, 1.0);\n"
			"   gl_PointSize = f_point_size;\n"
			"}\n";

		const char *p_s_fragment_shader =
			"precision mediump float;\n"
			"\n"
			"varying vec2 v_texcoord;\n"
			"\n"
			"uniform vec4 v_color;\n"
			"\n"
			"void main()\n"
			"{\n"
			"    vec2 v_tex = gl_PointCoord.xy - vec2(.5, .5);\n"
			"    float r = length(v_tex);\n"
			"    float a = 1.0 - smoothstep(.5 - fwidth(r), .5, r);\n"
			"    if(a < 1.0 / 256)\n"
			"        discard;\n" // so we can render without alphatest
			"    gl_FragColor = vec4(v_color.xyz, v_color.w * a);\n"
			"}\n";

		const char *p_s_config =
			"vertex {\n"
			"	v_pos: 0;\n"
			"}\n";

		std::string p_log[3];
		if(!CGLESShader::CompileConfigureLink(p_s_vertex_shader, p_s_fragment_shader, p_s_config,
		   p_log[0], p_log[1], p_log[2], true))
			return false;

		n_mvp_uniform = n_Get_Uniform_Location("t_mvp");
		n_color_uniform = n_Get_Uniform_Location("v_color");
		n_size_uniform = n_Get_Uniform_Location("f_point_size");

		return true;
	}

	void Bind(const Matrix4f &r_t_mvp, Vector4f v_color, float f_size) const
	{
		CGLESShader::Bind();
		UniformMatrix4fv(n_mvp_uniform, 1, false, &r_t_mvp[0][0]);
		Uniform4f(n_color_uniform, v_color.x, v_color.y, v_color.z, v_color.w);
		Uniform1f(n_size_uniform, f_size);
	}
};

struct CRefractionShader : public CGLESShader {
	int n_mvp_uniform, n_mv_uniform;
	int n_texture_uniform;
	int n_time_uniform;

	bool Compile()
	{
		const char *p_s_vertex_shader =
			"precision mediump float;\n"
			"\n"
			"attribute vec4 v_pos;\n"
			"\n"
			"uniform mat4 t_mvp, t_mv;\n"
			"\n"
			"varying vec3 v_texcoord;\n"
			"varying vec4 v_view_, v_world_;\n"
			"\n"
			"void main()\n"
			"{\n"
			"    vec4 v_position = t_mvp * (v_world_ = v_pos);\n"
			"    gl_Position = v_position;\n"
			"    v_texcoord = vec3(v_position.xy * .5 + .5 * v_position.w, v_position.w);\n" // seems to work
			"    v_view_ = -(t_mv * v_pos);\n" // eyespace view vector (camera is at origin)
			"}\n";

		const char *p_s_fragment_shader =
			"precision mediump float;\n"
			"//\n"
			"// Description : Array and textureless GLSL 2D/3D/4D simplex \n"
			"//               noise functions.\n"
			"//      Author : Ian McEwan, Ashima Arts.\n"
			"//  Maintainer : ijm\n"
			"//     Lastmod : 20110822 (ijm)\n"
			"//     License : Copyright (C) 2011 Ashima Arts. All rights reserved.\n"
			"//               Distributed under the MIT License. See LICENSE file.\n"
			"//               https://github.com/ashima/webgl-noise\n"
			"// \n"
			"\n"
			"vec3 mod289(vec3 x) {\n"
			"    return x - floor(x * (1.0 / 289.0)) * 289.0;\n"
			"}\n"
			"\n"
			"vec4 mod289(vec4 x) {\n"
			"    return x - floor(x * (1.0 / 289.0)) * 289.0;\n"
			"}\n"
			"\n"
			"vec4 permute(vec4 x) {\n"
			"    return mod289(((x*34.0)+1.0)*x);\n"
			"}\n"
			"\n"
			"vec4 taylorInvSqrt(vec4 r)\n"
			"{\n"
			"    return 1.79284291400159 - 0.85373472095314 * r;\n"
			"}\n"
			"\n"
			"float snoise(vec3 v)\n"
			"{ \n"
			"    const vec2  C = vec2(1.0/6.0, 1.0/3.0) ;\n"
			"    const vec4  D = vec4(0.0, 0.5, 1.0, 2.0);\n"
			"\n"
			"    // First corner\n"
			"    vec3 i  = floor(v + dot(v, C.yyy) );\n"
			"    vec3 x0 =   v - i + dot(i, C.xxx) ;\n"
			"\n"
			"    // Other corners\n"
			"    vec3 g = step(x0.yzx, x0.xyz);\n"
			"    vec3 l = 1.0 - g;\n"
			"    vec3 i1 = min( g.xyz, l.zxy );\n"
			"    vec3 i2 = max( g.xyz, l.zxy );\n"
			"\n"
			"    //   x0 = x0 - 0.0 + 0.0 * C.xxx;\n"
			"    //   x1 = x0 - i1  + 1.0 * C.xxx;\n"
			"    //   x2 = x0 - i2  + 2.0 * C.xxx;\n"
			"    //   x3 = x0 - 1.0 + 3.0 * C.xxx;\n"
			"    vec3 x1 = x0 - i1 + C.xxx;\n"
			"    vec3 x2 = x0 - i2 + C.yyy; // 2.0*C.x = 1/3 = C.y\n"
			"    vec3 x3 = x0 - D.yyy;      // -1.0+3.0*C.x = -0.5 = -D.y\n"
			"\n"
			"    // Permutations\n"
			"    i = mod289(i); \n"
			"    vec4 p = permute( permute( permute( \n"
			"        i.z + vec4(0.0, i1.z, i2.z, 1.0 ))\n"
			"        + i.y + vec4(0.0, i1.y, i2.y, 1.0 )) \n"
			"        + i.x + vec4(0.0, i1.x, i2.x, 1.0 ));\n"
			"\n"
			"    // Gradients: 7x7 points over a square, mapped onto an octahedron.\n"
			"    // The ring size 17*17 = 289 is close to a multiple of 49 (49*6 = 294)\n"
			"    float n_ = 0.142857142857; // 1.0/7.0\n"
			"    vec3  ns = n_ * D.wyz - D.xzx;\n"
			"\n"
			"    vec4 j = p - 49.0 * floor(p * ns.z * ns.z);  //  mod(p,7*7)\n"
			"\n"
			"    vec4 x_ = floor(j * ns.z);\n"
			"    vec4 y_ = floor(j - 7.0 * x_ );    // mod(j,N)\n"
			"\n"
			"    vec4 x = x_ *ns.x + ns.yyyy;\n"
			"    vec4 y = y_ *ns.x + ns.yyyy;\n"
			"    vec4 h = 1.0 - abs(x) - abs(y);\n"
			"\n"
			"    vec4 b0 = vec4( x.xy, y.xy );\n"
			"    vec4 b1 = vec4( x.zw, y.zw );\n"
			"\n"
			"    //vec4 s0 = vec4(lessThan(b0,0.0))*2.0 - 1.0;\n"
			"    //vec4 s1 = vec4(lessThan(b1,0.0))*2.0 - 1.0;\n"
			"    vec4 s0 = floor(b0)*2.0 + 1.0;\n"
			"    vec4 s1 = floor(b1)*2.0 + 1.0;\n"
			"    vec4 sh = -step(h, vec4(0.0));\n"
			"\n"
			"    vec4 a0 = b0.xzyw + s0.xzyw*sh.xxyy ;\n"
			"    vec4 a1 = b1.xzyw + s1.xzyw*sh.zzww ;\n"
			"\n"
			"    vec3 p0 = vec3(a0.xy,h.x);\n"
			"    vec3 p1 = vec3(a0.zw,h.y);\n"
			"    vec3 p2 = vec3(a1.xy,h.z);\n"
			"    vec3 p3 = vec3(a1.zw,h.w);\n"
			"\n"
			"    //Normalise gradients\n"
			"    vec4 norm = taylorInvSqrt(vec4(dot(p0,p0), dot(p1,p1), dot(p2, p2), dot(p3,p3)));\n"
			"    p0 *= norm.x;\n"
			"    p1 *= norm.y;\n"
			"    p2 *= norm.z;\n"
			"    p3 *= norm.w;\n"
			"\n"
			"    // Mix final noise value\n"
			"    vec4 m = max(0.6 - vec4(dot(x0,x0), dot(x1,x1), dot(x2,x2), dot(x3,x3)), 0.0);\n"
			"    m = m * m;\n"
			"    return 42.0 * dot( m*m, vec4( dot(p0,x0), dot(p1,x1), \n"
			"        dot(p2,x2), dot(p3,x3) ) );\n"
			"}\n"
			"\n"
			"varying vec3 v_texcoord;\n"
			"varying vec4 v_view_, v_world_;\n"
			"\n"
			"uniform float f_time;\n"
			"uniform sampler2D n_tex;\n"
			"\n"
			"void main()\n"
			"{\n"
			"    vec3 v_world = v_world_.xyz / v_world_.w;\n"
			"    vec3 v_view = v_view_.xyz / v_view_.w;\n"
			"    vec2 v_tex = v_texcoord.xy / v_texcoord.z;\n"
			// perspective division for texcoord

			"    vec2 v_noise = vec2(snoise(vec3(v_world.xz, f_time) * .7),\n"
			"        snoise(vec3(v_world.zx, 3.0 - f_time) * .7)) * .5;\n" // or could calculate more expensive finite difference
			"    vec2 v_off = v_noise / max(20.0, length(v_view) * 3.0);\n" // roloff displacement at distance
			"    v_tex += v_off * 2e-1;\n"
			// create simple waves, based on fragment's worldspace position and time

			"    gl_FragColor = mix(texture2D(n_tex, v_tex), vec4(.12, .2, .5, 1.0),\n"
			"        smoothstep(.1, 30.0, length(v_view)));\n"
			"}\n";

		const char *p_s_config =
			"vertex {\n"
			"	v_pos: 0;\n"
			"}\n";

		std::string p_log[3];
		if(!CGLESShader::CompileConfigureLink(p_s_vertex_shader, p_s_fragment_shader, p_s_config,
		   p_log[0], p_log[1], p_log[2], true))
			return false;

		n_mvp_uniform = n_Get_Uniform_Location("t_mvp");
		n_mv_uniform = n_Get_Uniform_Location("t_mv");
		n_texture_uniform = n_Get_Uniform_Location("n_tex");
		n_time_uniform = n_Get_Uniform_Location("f_time");

		CGLESShader::Bind();
		Uniform1i(n_texture_uniform, 0);

		return true;
	}

	void Bind(const Matrix4f &r_t_mvp, const Matrix4f &r_t_mv, float f_time) const
	{
		CGLESShader::Bind();
		UniformMatrix4fv(n_mvp_uniform, 1, false, &r_t_mvp[0][0]);
		UniformMatrix4fv(n_mv_uniform, 1, false, &r_t_mv[0][0]);
		Uniform1f(n_time_uniform, f_time);
	}
};

struct CWaterShader : public CGLESShader {
	int n_mv_uniform, n_mvp_uniform;
	int n_texture_uniform;
	int n_time_uniform;

	bool Compile()
	{
		const char *p_s_vertex_shader =
			"precision mediump float;\n"
			"\n"
			"attribute vec4 v_pos;\n"
			"\n"
			"uniform mat4 t_mvp, t_mv;\n"
			"\n"
			"varying vec3 v_texcoord, v_normal_, v_to_light_;\n"
			"varying vec4 v_view_, v_world_;\n"
			"\n"
			"void main()\n"
			"{\n"
			"    vec4 v_position = t_mvp * (v_world_ = v_pos);\n"
			"    gl_Position = v_position;\n"
			"    v_texcoord = vec3(v_position.xy * .5 + .5 * v_position.w, v_position.w);\n" // seems to work
			"    v_view_ = -(t_mv * v_pos);\n" // eyespace view vector (camera is at origin)
			"    v_normal_ = -transpose(inverse(mat3(t_mv))) * vec3(.0, 1.0, .0);\n" // eyespace normal vector (should be constant over the whole plane)
			"    v_to_light_ = -transpose(inverse(mat3(t_mv))) * vec3(-82.657, 62.696, 225.598);\n"
			"}\n";

		const char *p_s_fragment_shader =
			"precision mediump float;\n"
			"//\n"
			"// Description : Array and textureless GLSL 2D/3D/4D simplex \n"
			"//               noise functions.\n"
			"//      Author : Ian McEwan, Ashima Arts.\n"
			"//  Maintainer : ijm\n"
			"//     Lastmod : 20110822 (ijm)\n"
			"//     License : Copyright (C) 2011 Ashima Arts. All rights reserved.\n"
			"//               Distributed under the MIT License. See LICENSE file.\n"
			"//               https://github.com/ashima/webgl-noise\n"
			"// \n"
			"\n"
			"vec3 mod289(vec3 x) {\n"
			"    return x - floor(x * (1.0 / 289.0)) * 289.0;\n"
			"}\n"
			"\n"
			"vec4 mod289(vec4 x) {\n"
			"    return x - floor(x * (1.0 / 289.0)) * 289.0;\n"
			"}\n"
			"\n"
			"vec4 permute(vec4 x) {\n"
			"    return mod289(((x*34.0)+1.0)*x);\n"
			"}\n"
			"\n"
			"vec4 taylorInvSqrt(vec4 r)\n"
			"{\n"
			"    return 1.79284291400159 - 0.85373472095314 * r;\n"
			"}\n"
			"\n"
			"float snoise(vec3 v)\n"
			"{ \n"
			"    const vec2  C = vec2(1.0/6.0, 1.0/3.0) ;\n"
			"    const vec4  D = vec4(0.0, 0.5, 1.0, 2.0);\n"
			"\n"
			"    // First corner\n"
			"    vec3 i  = floor(v + dot(v, C.yyy) );\n"
			"    vec3 x0 =   v - i + dot(i, C.xxx) ;\n"
			"\n"
			"    // Other corners\n"
			"    vec3 g = step(x0.yzx, x0.xyz);\n"
			"    vec3 l = 1.0 - g;\n"
			"    vec3 i1 = min( g.xyz, l.zxy );\n"
			"    vec3 i2 = max( g.xyz, l.zxy );\n"
			"\n"
			"    //   x0 = x0 - 0.0 + 0.0 * C.xxx;\n"
			"    //   x1 = x0 - i1  + 1.0 * C.xxx;\n"
			"    //   x2 = x0 - i2  + 2.0 * C.xxx;\n"
			"    //   x3 = x0 - 1.0 + 3.0 * C.xxx;\n"
			"    vec3 x1 = x0 - i1 + C.xxx;\n"
			"    vec3 x2 = x0 - i2 + C.yyy; // 2.0*C.x = 1/3 = C.y\n"
			"    vec3 x3 = x0 - D.yyy;      // -1.0+3.0*C.x = -0.5 = -D.y\n"
			"\n"
			"    // Permutations\n"
			"    i = mod289(i); \n"
			"    vec4 p = permute( permute( permute( \n"
			"        i.z + vec4(0.0, i1.z, i2.z, 1.0 ))\n"
			"        + i.y + vec4(0.0, i1.y, i2.y, 1.0 )) \n"
			"        + i.x + vec4(0.0, i1.x, i2.x, 1.0 ));\n"
			"\n"
			"    // Gradients: 7x7 points over a square, mapped onto an octahedron.\n"
			"    // The ring size 17*17 = 289 is close to a multiple of 49 (49*6 = 294)\n"
			"    float n_ = 0.142857142857; // 1.0/7.0\n"
			"    vec3  ns = n_ * D.wyz - D.xzx;\n"
			"\n"
			"    vec4 j = p - 49.0 * floor(p * ns.z * ns.z);  //  mod(p,7*7)\n"
			"\n"
			"    vec4 x_ = floor(j * ns.z);\n"
			"    vec4 y_ = floor(j - 7.0 * x_ );    // mod(j,N)\n"
			"\n"
			"    vec4 x = x_ *ns.x + ns.yyyy;\n"
			"    vec4 y = y_ *ns.x + ns.yyyy;\n"
			"    vec4 h = 1.0 - abs(x) - abs(y);\n"
			"\n"
			"    vec4 b0 = vec4( x.xy, y.xy );\n"
			"    vec4 b1 = vec4( x.zw, y.zw );\n"
			"\n"
			"    //vec4 s0 = vec4(lessThan(b0,0.0))*2.0 - 1.0;\n"
			"    //vec4 s1 = vec4(lessThan(b1,0.0))*2.0 - 1.0;\n"
			"    vec4 s0 = floor(b0)*2.0 + 1.0;\n"
			"    vec4 s1 = floor(b1)*2.0 + 1.0;\n"
			"    vec4 sh = -step(h, vec4(0.0));\n"
			"\n"
			"    vec4 a0 = b0.xzyw + s0.xzyw*sh.xxyy ;\n"
			"    vec4 a1 = b1.xzyw + s1.xzyw*sh.zzww ;\n"
			"\n"
			"    vec3 p0 = vec3(a0.xy,h.x);\n"
			"    vec3 p1 = vec3(a0.zw,h.y);\n"
			"    vec3 p2 = vec3(a1.xy,h.z);\n"
			"    vec3 p3 = vec3(a1.zw,h.w);\n"
			"\n"
			"    //Normalise gradients\n"
			"    vec4 norm = taylorInvSqrt(vec4(dot(p0,p0), dot(p1,p1), dot(p2, p2), dot(p3,p3)));\n"
			"    p0 *= norm.x;\n"
			"    p1 *= norm.y;\n"
			"    p2 *= norm.z;\n"
			"    p3 *= norm.w;\n"
			"\n"
			"    // Mix final noise value\n"
			"    vec4 m = max(0.6 - vec4(dot(x0,x0), dot(x1,x1), dot(x2,x2), dot(x3,x3)), 0.0);\n"
			"    m = m * m;\n"
			"    return 42.0 * dot( m*m, vec4( dot(p0,x0), dot(p1,x1), \n"
			"        dot(p2,x2), dot(p3,x3) ) );\n"
			"}\n"
			"\n"
			"varying vec3 v_texcoord, v_normal_, v_to_light_;\n"
			"varying vec4 v_view_, v_world_;\n"
			"\n"
			"uniform float f_time;\n"
			"uniform sampler2D n_tex;\n"
			"\n"
			"void main()\n"
			"{\n"
			"    vec3 v_world = v_world_.xyz / v_world_.w;\n"
			"    vec3 v_view = v_view_.xyz / v_view_.w;\n"
			"    vec2 v_tex = v_texcoord.xy / v_texcoord.z;\n"
			// perspective division for texcoord

			"    vec2 v_noise = vec2(snoise(vec3(v_world.xz, f_time) * .7),\n"
			"        snoise(vec3(v_world.zx, 3.0 - f_time) * .7)) * .5;\n" // or could calculate more expensive finite difference
			"    vec2 v_off = v_noise / max(20.0, length(v_view) * 3.0);\n" // roloff displacement at distance
			"    v_tex += v_off * 2e-1;\n"
			// create simple waves, based on fragment's worldspace position and time

			"    vec3 v_to_light = v_to_light_ + v_view;\n" // light vector (view is negative pos)
			"    v_view = normalize(v_view);\n"
			"    vec3 v_light = normalize(v_to_light);\n"
			"    vec3 v_normal = normalize(v_normal_);\n"
			"    v_normal.xz += 10.0 * v_off;\n"
			"    v_normal = normalize(v_normal);\n" // deflect normal as well to get fake bump-mapping
			"    float f_diffuse = max(.0, dot(v_light, v_normal)) * .2;\n"
			"    float f_specular = pow(max(.0, dot(v_view, -reflect(v_light, v_normal))), 10.0) * .6;\n"
			// lighting calculations

			"    gl_FragColor.a = 1.0 - pow(max(.0, dot(v_normal, v_view)), 3.0) * .75;\n"
			"    gl_FragColor.xyz = texture2D(n_tex, v_tex).xyz + f_diffuse + f_specular;\n"
			"}\n";

		const char *p_s_config =
			"vertex {\n"
			"	v_pos: 0;\n"
			"}\n";

		std::string p_log[3];
		if(!CGLESShader::CompileConfigureLink(p_s_vertex_shader, p_s_fragment_shader, p_s_config,
		   p_log[0], p_log[1], p_log[2], true))
			return false;

		n_mvp_uniform = n_Get_Uniform_Location("t_mvp");
		n_mv_uniform = n_Get_Uniform_Location("t_mv");
		n_texture_uniform = n_Get_Uniform_Location("n_tex");
		n_time_uniform = n_Get_Uniform_Location("f_time");

		CGLESShader::Bind();
		Uniform1i(n_texture_uniform, 0);

		return true;
	}

	void Bind(const Matrix4f &r_t_mvp, const Matrix4f &r_t_mv, float f_time) const
	{
		CGLESShader::Bind();
		UniformMatrix4fv(n_mvp_uniform, 1, false, &r_t_mvp[0][0]);
		UniformMatrix4fv(n_mv_uniform, 1, false, &r_t_mv[0][0]);
		Uniform1f(n_time_uniform, f_time);
	}
};

struct CBuoyShader : public CGLESShader {
	int n_mv_uniform, n_mvp_uniform;
	int n_scale_uniform;
	int n_time_uniform;

	bool Compile()
	{
		const char *p_s_vertex_shader =
			"precision mediump float;\n"
			"\n"
			"attribute vec4 v_pos;\n"
			"\n"
			"uniform mat4 t_mv;\n"
			"uniform mat4 t_mvp;\n"
			"uniform float f_point_scale;\n"
			"\n"
			"varying vec3 v_eye, v_right, v_up, v_center;\n"
			"varying float f_type;\n"
			"\n"
			"void main()\n"
			"{\n"
			"	gl_Position = t_mvp * vec4(v_pos.xyz, 1.0);\n"
			"   const float f_point_size = 1.0;\n" 
			"	gl_PointSize = f_point_scale * f_point_size / gl_Position.w;\n"
			"   v_eye = inverse(t_mv)[3].xyz;\n" // get worldspace eye position
			"   v_right = vec3(transpose(t_mv)[0]) * f_point_size;\n" // get right vector (do not scale by w, in object space all the particles are f_point_size big)
			"   v_up = -vec3(transpose(t_mv)[1]) * f_point_size;\n" // get up vector (flip it, the y texcoord on point sprites is upside down)
			"   v_center = v_pos.xyz;\n"
			"   f_type = v_pos.w;\n" // particle type
			"}\n";

		const char *p_s_fragment_shader =
			"precision mediump float;\n"
			"\n"
			"uniform float f_time;\n"
			"uniform mat4 t_mv;\n"
			"\n"
			"varying vec3 v_eye, v_right, v_up, v_center;\n"
			"varying float f_type;\n"
			"\n"
			"void main()\n"
			"{\n"
			"    vec2 v_tex = gl_PointCoord.xy - vec2(.5, .5);\n"
			"    vec3 v_pos = v_center + v_right * v_tex.x * 2.0 + v_up * v_tex.y * 2.0;\n" // worldspace position of a pixel
			"    vec3 v_to_eye = v_eye - v_pos;\n" // vector to eye (must be normalized later)

			"    const float f_r = .3, f_r2 = f_r * f_r;\n" // radius, radius squared

			"    float f_to_eye_invlen2 = 1.0 / dot(v_to_eye, v_to_eye);\n"
			"    vec3 v_center_proj = v_eye - v_to_eye * (dot(v_to_eye, v_eye - v_center) * f_to_eye_invlen2);\n" // project sphere center onto view ray
			"    float f_d2 = dot(v_center_proj - v_center, v_center_proj - v_center);\n" // squared distance from the center
			"    float f_cor = sqrt(max(.0, f_r2 - f_d2) * f_to_eye_invlen2);\n" // intersection correction (normalize v_to_eye inside this sqrt)
			"    vec3 v_isect = v_center_proj + v_to_eye * f_cor;\n" // v_to_eye is going in the direction of the correction - towards the eye!
#if 0
			"    vec3 n; n.xy = -v_tex * 2.0 / .3; n.z = sqrt(max(.0, 1.0 - dot(n.xy, n.xy)));\n"
			"    n = transpose(mat3(t_mv)) * n;\n" // calculate normal in object-space (no scales so it is orthogonal)
			// approximate normal, does not require the intersection
#else
			"    vec3 n = (v_isect - v_center) / f_r;\n" // calculate normal in object-space
#endif
			// calculate intersection of the view ray and a sphere (about the same speed as if using the quadratic equation)

			"    float r = 2.0 * length(v_tex);\n" // radius of the pixel
			"    float c = 1.0 - smoothstep(f_r - fwidth(r), f_r, r);\n" // core alpha
			"    float tc = atan(n.y / length(n.xz));\n" // texcoord (spherical coordinates)
			"    if(f_type < .5) {\n"
			"        float s = smoothstep(.1 - fwidth(tc), .1, abs(mod(tc, .4) - .2));\n" // stripes
			"        float f = max(.0, 1.0 - r);\n"
			"        float w = f * (.5 + .5 * sin(5.0 * f_time + 8.0 * 3.14159265358979 * f));\n" // waves
			"        float l = max(.25, dot(n, normalize(vec3(.2, 1.0, .3))));\n" // lighting (light at infinity)
			"        gl_FragColor = /*vec4(n * .5 + .5, 1.0);*/mix(vec4(.2, .2, 1.0, w), vec4(vec3(1.0, s, s) * l, 1.0), c);\n" // final color
			"    } else {\n"
			"        float s = 1.0 - smoothstep(.625 - fwidth(tc), .625, abs(mod(tc, 2.5) - 1.25));\n" // stripe
			"        float l = max(.25, dot(n, normalize(vec3(.2, 1.0, .3))));\n" // lighting (light at infinity)
			"        gl_FragColor = vec4(vec3(s, 1.0, s) * l, c);\n" // final color
			"    }\n"
			"}\n";

		const char *p_s_config =
			"vertex {\n"
			"	v_pos: 0;\n"
			"}\n";

		std::string p_log[3];
		if(!CGLESShader::CompileConfigureLink(p_s_vertex_shader, p_s_fragment_shader, p_s_config,
		   p_log[0], p_log[1], p_log[2], true))
			return false;

		n_mvp_uniform = n_Get_Uniform_Location("t_mvp");
		n_mv_uniform = n_Get_Uniform_Location("t_mv");
		n_time_uniform = n_Get_Uniform_Location("f_time");
		n_scale_uniform = n_Get_Uniform_Location("f_point_scale");

		return true;
	}

	void Bind(const Matrix4f &r_t_mvp, const Matrix4f &r_t_modelview, float f_time) const
	{
		CGLESShader::Bind();
		UniformMatrix4fv(n_mvp_uniform, 1, false, &r_t_mvp[0][0]);
		UniformMatrix4fv(n_mv_uniform, 1, false, &r_t_modelview[0][0]);
		Uniform1f(n_time_uniform, f_time);
		Uniform1f(n_scale_uniform, n_height);
	}
};

struct TLabelShader : public CGLESShader {
	GLint n_modelview_matrix_uniform;
	GLint n_texture_unit_uniform;
	GLint n_color_uniform;

	bool Compile()
	{
		const char *p_s_vertex_shader =
			"precision mediump float;\n"
			"attribute vec2 v_tex;\n"
			"attribute vec2 v_pos;\n"
			"\n"
			"uniform mat4 t_modelview_projection_matrix;\n"
			"\n"
			"varying vec2 v_texcoord;\n"
			"\n"
			"void main()\n"
			"{\n"
			"    gl_Position = t_modelview_projection_matrix * vec4(v_pos.xy, .0, 1.0);\n"
			"    v_texcoord = v_tex;\n"
			"}\n";
		const char *p_s_fragment_shader =
			"precision mediump float;\n"
			"varying vec2 v_texcoord;\n"
			"\n"
			"uniform vec3 v_color;\n"
			"uniform sampler2D n_texture_unit;\n"
			"\n"
			"void main()\n"
			"{\n"
			"    gl_FragColor = vec4(v_color, texture2D(n_texture_unit, v_texcoord).x);\n"
			"}\n";
		// vertex / fragment shader source code

		const char *p_s_config =
			"vertex {\n"
			"	v_tex: 0;\n"
			"	v_pos: 1;\n"
			"}\n";
		// shader configuration (doesn't actually need the newlines)
		// note this can't override layout qualifiers in the shader code

		std::string compile_log, config_log, link_log;
		if(!CompileConfigureLink(p_s_vertex_shader, p_s_fragment_shader,
		   p_s_config, compile_log, config_log, link_log, true))
			return false;
		// use the comfy function

		n_texture_unit_uniform = n_Get_Uniform_Location("n_texture_unit");
		n_modelview_matrix_uniform = n_Get_Uniform_Location("t_modelview_projection_matrix");
		n_color_uniform = n_Get_Uniform_Location("v_color");
		// get addresses of the uniforms

		CGLESShader::Bind();
		// bind the shader

		glUniform1i(n_texture_unit_uniform, 0);
		// always assume the texture in unit 0

		return true;
	}

	void Bind(const Matrix4f &r_t_mvp, Vector3f v_color = Vector3f(0, 0, 0)) const
	{
		CGLESShader::Bind();
		// bind the shader

		UniformMatrix4fv(n_modelview_matrix_uniform, 1, false, &r_t_mvp[0][0]);
		glUniform3f(n_color_uniform, v_color.x, v_color.y, v_color.z);
	}
};
