
class CBuoyObject {
public:
	struct CBuoyShader : public CGLESShader {
		int n_mv_uniform, n_mvp_uniform;
		int n_pos_scale_uniform;

		bool Compile()
		{
			const char *p_s_vertex_shader =
				"precision mediump float;\n"
				"\n"
				"attribute vec3 v_pos;\n"
				"attribute vec3 v_nrm;\n"
				"attribute vec2 v_tex;\n"
				"\n"
				"uniform vec4 v_pos_scl;\n"
				"uniform mat4 t_mvp, t_mv;\n"
				"\n"
				"varying vec2 v_texcoord;\n"
				"varying vec3 v_view_, v_normal_, v_to_light_;\n"
				"\n"
				"void main()\n"
				"{\n"
				"    gl_Position = t_mvp * vec4((v_pos * v_pos_scl.w + v_pos_scl.xyz), 1.0);\n"
				"    v_texcoord = v_tex;\n"
				"    v_view_ = -vec3(t_mv * vec4(v_pos * v_pos_scl.w + v_pos_scl.xyz, 1.0));\n" // eyespace view vector (camera is at origin)
				"    v_normal_ = transpose(inverse(mat3(t_mv))) * v_nrm;\n"
				"    v_to_light_ = transpose(inverse(mat3(t_mv))) * vec3(-82.657, 62.696, 225.598);\n"
				"}\n";

			const char *p_s_fragment_shader =
				"precision mediump float;\n"
				"\n"
				"varying vec2 v_texcoord;\n"
				"varying vec3 v_view_, v_normal_, v_to_light_;\n"
				"\n"
				"void main()\n"
				"{\n"
				"    vec3 v_view = normalize(v_view_);\n"
				"    vec3 v_light = normalize(v_to_light_ + v_view_);\n" // light vector (view is negative pos)
				"    vec3 v_normal = normalize(v_normal_);\n"
				"    float f_ambient = .25;\n"
				"    float f_diffuse = max(.0, dot(v_light, v_normal)) * .6;\n"
				"    float f_specular = pow(max(.0, dot(v_view, -reflect(v_light, v_normal))), 3.0) * .6;\n"
				"    float t = v_texcoord.y * 3.0;\n"
				"    float s = smoothstep(.1 - fwidth(t), .1, abs(mod(t, .4) - .2));\n" // stripes
				"    gl_FragColor = vec4(vec3(vec3(1.0, s, s) * (f_diffuse + f_ambient) + f_specular), 1.0);\n"
				"}\n";

			const char *p_s_config =
				"vertex {\n"
				"	v_pos: 0;\n"
				"	v_nrm: 1;\n"
				"	v_tex: 2;\n"
				"}\n";

			std::string p_log[3];
			if(!CGLESShader::CompileConfigureLink(p_s_vertex_shader, p_s_fragment_shader, p_s_config,
			   p_log[0], p_log[1], p_log[2], true))
				return false;

			n_mvp_uniform = n_Get_Uniform_Location("t_mvp");
			n_mv_uniform = n_Get_Uniform_Location("t_mv");
			n_pos_scale_uniform = n_Get_Uniform_Location("v_pos_scl");

			return true;
		}

		void Bind(const Matrix4f &r_t_mvp, const Matrix4f &r_t_mv, Vector3f v_position, float f_scale = 1)
		{
			CGLESShader::Bind();
			UniformMatrix4fv(n_mvp_uniform, 1, false, &r_t_mvp[0][0]);
			UniformMatrix4fv(n_mv_uniform, 1, false, &r_t_mv[0][0]);
			Uniform4f(n_pos_scale_uniform, v_position.x, v_position.y, v_position.z, f_scale);
		}
	};

	struct CBuoyHaloShader : public CGLESShader {
		int n_mvp_uniform;
		int n_scale_uniform;
		int n_time_uniform;

		bool Compile()
		{
			const char *p_s_vertex_shader =
				"precision mediump float;\n"
				"\n"
				"attribute vec4 v_pos;\n"
				"\n"
				"uniform mat4 t_mvp;\n"
				"uniform float f_point_scale;\n"
				"\n"
				"void main()\n"
				"{\n"
				"	gl_Position = t_mvp * vec4(v_pos.xyz, 1.0);\n"
				"   const float f_point_size = 1.0;\n" 
				"	gl_PointSize = f_point_scale * f_point_size / gl_Position.w;\n"
				"}\n";

			const char *p_s_fragment_shader =
				"precision mediump float;\n"
				"\n"
				"uniform float f_time;\n"
				"\n"
				"void main()\n"
				"{\n"
				"    vec2 v_tex = gl_PointCoord.xy - vec2(.5, .5);\n"
				"    float r = 2.0 * length(v_tex);\n" // radius of the pixel
				"    float f = max(.0, 1.0 - r);\n"
				"    float w = f * (.5 + .5 * sin(5.0 * f_time + 8.0 * 3.14159265358979 * f));\n" // waves
				"    gl_FragColor = vec4(.2, .2, 1.0, w);\n" // final color
				"}\n";

			const char *p_s_config =
				"vertex {\n"
				"	v_pos: 0;\n"
				"}\n";

			std::string p_log[3];
			if(!CGLESShader::CompileConfigureLink(p_s_vertex_shader, p_s_fragment_shader, p_s_config,
			   p_log[0], p_log[1], p_log[2], true))
				return false;

			n_mvp_uniform = n_Get_Uniform_Location("t_mvp");
			n_time_uniform = n_Get_Uniform_Location("f_time");
			n_scale_uniform = n_Get_Uniform_Location("f_point_scale");

			return true;
		}

		void Bind(const Matrix4f &r_t_mvp, float f_time)
		{
			CGLESShader::Bind();
			UniformMatrix4fv(n_mvp_uniform, 1, false, &r_t_mvp[0][0]);
			Uniform1f(n_time_uniform, f_time);
			Uniform1f(n_scale_uniform, n_height);
		}
	};

protected:
	CGLArrayBufferObject m_vertex_buffer, m_halo_vbo;
	CGLElementArrayBufferObject m_index_buffer;
	CGLVertexArrayObject m_vao, m_vao_halo;
	size_t m_n_index_num;
	CBuoyShader m_shader;
	CBuoyHaloShader m_halo_shader;

public:
	CBuoyObject()
		:m_n_index_num(0)
	{
		if(!m_shader.Compile() || !m_halo_shader.Compile())
			return;

		CPolyMesh mesh;
		CMakeGeoSphere::MakeGeoSphere(mesh, 2);
		Matrix4f t_gizmo = CTexCoordGen::t_Gizmo(mesh);
		{
			CPolyMesh mesh2;
			CPlatonicPrimitives::MakeCube(mesh2, 1, 1, 1);
			Matrix4f t_transform;
			t_transform.Translation(0, -1, 0);
			t_transform.Scale(.2f, .5f, .2f);
			mesh.Merge(mesh2, t_transform);
		}
		CTexCoordGen::CalcTexcoords(mesh, t_gizmo, CTexCoordGen::map_Sphere, 0);
		mesh.MakeTris();
		mesh.OptimizeVertices(true, false, 1);

		std::vector<uint32_t> index_buffer;
		std::vector<float> vertex_buffer;
		if(!mesh.Build_DrawBuffers(index_buffer, vertex_buffer, true, false, 2))
			return;

		m_vao.Bind();
		{
			m_index_buffer.Bind();
			m_index_buffer.BufferData(index_buffer.size() * sizeof(uint32_t), &index_buffer[0]);
			m_vertex_buffer.Bind();
			m_vertex_buffer.BufferData(vertex_buffer.size() * sizeof(float), &vertex_buffer[0]);

			glEnableVertexAttribArray(0);
			glEnableVertexAttribArray(1);
			glEnableVertexAttribArray(2);
			glVertexAttribPointer(0, 3, GL_FLOAT, false, 8 * sizeof(float),
				m_vertex_buffer.p_OffsetPointer(0)); // pos
			glVertexAttribPointer(1, 3, GL_FLOAT, false, 8 * sizeof(float),
				m_vertex_buffer.p_OffsetPointer(3 * sizeof(float))); // normal
			glVertexAttribPointer(2, 2, GL_FLOAT, false, 8 * sizeof(float),
				m_vertex_buffer.p_OffsetPointer(6 * sizeof(float))); // texcoord
		}
		m_vao_halo.Bind();
		{
			m_halo_vbo.Bind();
			m_halo_vbo.BufferData(sizeof(Vector3f));
			glEnableVertexAttribArray(0);
			glVertexAttribPointer(0, 3, GL_FLOAT, false, 0, m_halo_vbo.p_OffsetPointer(0)); 
		}
		CGLVertexArrayObject::Release();

		m_n_index_num = index_buffer.size();
	}

	void Draw(const Matrix4f &r_t_mvp, const Matrix4f &r_t_mv, Vector3f v_position, float f_scale = .2f)
	{
		m_shader.Bind(r_t_mvp, r_t_mv, v_position, f_scale);
		m_vao.Bind();
		glDrawElements(GL_TRIANGLES, m_n_index_num, GL_UNSIGNED_INT, m_index_buffer.p_OffsetPointer(0));
		CGLVertexArrayObject::Release();
	}

	void Draw_Halo(const Matrix4f &r_t_mvp, Vector3f v_position, float f_time)
	{
		m_halo_shader.Bind(r_t_mvp, f_time);
		m_vao_halo.Bind();
		{
			m_halo_vbo.Bind();
			m_halo_vbo.BufferSubData(0, sizeof(Vector3f), &v_position);
			glDrawArrays(GL_POINTS, 0, 1);
		}
		CGLVertexArrayObject::Release();
	}
};

class CSubObject {
public:
	struct CSubShader : public CGLESShader {
		int n_mv_uniform, n_mvp_uniform;
		int n_texture_uniform;

		bool Compile()
		{
			const char *p_s_vertex_shader =
				"precision mediump float;\n"
				"\n"
				"attribute vec3 v_pos;\n"
				"attribute vec3 v_nrm;\n"
				"attribute vec2 v_tex;\n"
				"\n"
				"uniform mat4 t_mvp, t_mv;\n"
				"\n"
				"varying vec2 v_texcoord;\n"
				"varying vec3 v_view_, v_normal_, v_to_light_;\n"
				"\n"
				"void main()\n"
				"{\n"
				"    gl_Position = t_mvp * vec4(v_pos, 1.0);\n"
				"    v_texcoord = v_tex;\n"
				"    v_view_ = -vec3(t_mv * vec4(v_pos, 1.0));\n" // eyespace view vector (camera is at origin)
				"    v_normal_ = transpose(inverse(mat3(t_mv))) * v_nrm;\n"
				"    v_to_light_ = transpose(inverse(mat3(t_mv))) * vec3(-82.657, 62.696, 225.598);\n"
				"}\n";

			const char *p_s_fragment_shader =
				"precision mediump float;\n"
				"\n"
				"varying vec2 v_texcoord;\n"
				"varying vec3 v_view_, v_normal_, v_to_light_;\n"
				"\n"
				"uniform sampler2D n_tex;\n"
				"\n"
				"void main()\n"
				"{\n"
				"    vec3 v_view = normalize(v_view_);\n"
				"    vec3 v_light = normalize(v_to_light_ + v_view_);\n" // light vector (view is negative pos)
				"    vec3 v_normal = normalize(v_normal_);\n"
				"    float f_ambient = .25;\n"
				"    float f_diffuse = max(.0, dot(v_light, v_normal)) * .6;\n"
				"    float f_specular = pow(max(.0, dot(v_view, -reflect(v_light, v_normal))), 3.0) * .4;\n"

				//"    vec2 t = v_texcoord * 3.0;\n"
				//"    vec2 s = smoothstep(.1 - fwidth(t), vec2(.1), abs(mod(t, .4) - .2));\n" // stripes
				//"    gl_FragColor = vec4(vec3(vec3(.8, abs(s.x - s.y), s.x * (1.0 - s.y)) * (f_diffuse + f_ambient) + f_specular), 1.0);\n" // navy gray
				"    gl_FragColor = vec4(texture2D(n_tex, v_texcoord) * (f_diffuse + f_ambient) + f_specular);\n" // navy gray
				"}\n";

			const char *p_s_config =
				"vertex {\n"
				"	v_pos: 0;\n"
				"	v_nrm: 1;\n"
				"	v_tex: 2;\n"
				"}\n";

			std::string p_log[3];
			if(!CGLESShader::CompileConfigureLink(p_s_vertex_shader, p_s_fragment_shader, p_s_config,
			   p_log[0], p_log[1], p_log[2], true))
				return false;

			n_mvp_uniform = n_Get_Uniform_Location("t_mvp");
			n_mv_uniform = n_Get_Uniform_Location("t_mv");
			n_texture_uniform = n_Get_Uniform_Location("n_tex");

			CGLESShader::Bind();
			glUniform1i(n_texture_uniform, 0);

			return true;
		}

		void Bind(const Matrix4f &r_t_mvp, const Matrix4f &r_t_mv)
		{
			CGLESShader::Bind();
			UniformMatrix4fv(n_mvp_uniform, 1, false, &r_t_mvp[0][0]);
			UniformMatrix4fv(n_mv_uniform, 1, false, &r_t_mv[0][0]);
		}
	};

protected:
	CGLArrayBufferObject m_vertex_buffer, m_halo_vbo;
	CGLElementArrayBufferObject m_index_buffer;
	CGLVertexArrayObject m_vao, m_vao_halo;
	size_t m_n_index_num;
	CSubShader m_shader;
	CGLTexture_2D_Loadable<> m_texture;

public:
	CSubObject()
		:m_n_index_num(0), m_texture("sub.png")
	{
		if(!m_shader.Compile())
			return;

		CPolyMesh mesh, mesh2;
		CPlatonicPrimitives::MakeCylinder(mesh, 1, 24, 1);
		{
			CPlatonicPrimitives::MakeSphere(mesh2, 12, 24);
			Matrix4f t_transform;
			t_transform.Translation(0, 1, 0);
			mesh.Merge(mesh2, t_transform);
			t_transform.Translation(0, -1, 0);
			mesh.Merge(mesh2, t_transform); // front and back hull
		}
		{
			CPlatonicPrimitives::MakeTube(mesh2, 1, 12, .8f);
			Matrix4f t_transform;
			t_transform.Translation(0, -2.3f, 0);
			t_transform.Scale(.5f, .25f, .5f);
			mesh.Merge(mesh2, t_transform); // propeller
		}
		Matrix4f t_texmap_gizmo = CTexCoordGen::t_Gizmo(mesh); // without the asymetric stuff
		{
			CPlatonicPrimitives::MakeCylinder(mesh2, 1, 12, 1);
			Matrix4f t_transform;
			t_transform.Translation(0, 0, 1);
			t_transform.RotateX(f_pi / 2);
			t_transform.Scale(.5f, .25f, .5f);
			mesh.Merge(mesh2, t_transform); // entrance
		}
		{
			CPlatonicPrimitives::MakeCylinder(mesh2, 1, 3, 1);
			Matrix4f t_transform;
			t_transform.Translation(.2f, .1f, 1.5f);
			t_transform.RotateX(f_pi / 2);
			t_transform.Scale(.1f, .5f, .1f);
			mesh.Merge(mesh2, t_transform); // periscope / antenna bank
		}
		{
			CPlatonicPrimitives::MakeCube(mesh2, 1, 1, 1);
			Matrix4f t_transform;
			t_transform.Translation(0, .3f, 0);
			t_transform.Scale(1.5f, .5f, .075f);
			mesh.Merge(mesh2, t_transform); // level fin
		}
		{
			CPlatonicPrimitives::MakeCube(mesh2, 1, 1, 1);
			Matrix4f t_transform, t_transform2;
			t_transform.Translation(.6f, -2.3f, 0);
			t_transform.Scale(.15f, .25f, .05f);
			mesh.Merge(mesh2, t_transform);
			t_transform2.RotationY(f_pi / 2);
			mesh.Merge(mesh2, t_transform2 * t_transform);
			t_transform2.RotateY(f_pi / 2);
			mesh.Merge(mesh2, t_transform2 * t_transform);
			t_transform2.RotateY(f_pi / 2);
			mesh.Merge(mesh2, t_transform2 * t_transform); // propeller fins
		}
		CTexCoordGen::CalcTexcoords(mesh, t_texmap_gizmo, CTexCoordGen::map_Cylinder, 0);
		{
			Matrix4f t_transform;
			t_transform.RotationX(f_pi / 2);
			mesh.Transform(t_transform);
		}
#ifndef _DEBUG
		mesh.CalcFaceNormals(); // !!
		CBooleanOps::Remove_HiddenSurfaces(mesh, true);
#endif // !_DEBUG
		mesh.MakeTris();
		mesh.OptimizeVertices(true, false, 1);

		std::vector<uint32_t> index_buffer;
		std::vector<float> vertex_buffer;
		if(!mesh.Build_DrawBuffers(index_buffer, vertex_buffer, true, false, 2))
			return;

		m_vao.Bind();
		{
			m_index_buffer.Bind();
			m_index_buffer.BufferData(index_buffer.size() * sizeof(uint32_t), &index_buffer[0]);
			m_vertex_buffer.Bind();
			m_vertex_buffer.BufferData(vertex_buffer.size() * sizeof(float), &vertex_buffer[0]);

			glEnableVertexAttribArray(0);
			glEnableVertexAttribArray(1);
			glEnableVertexAttribArray(2);
			glVertexAttribPointer(0, 3, GL_FLOAT, false, 8 * sizeof(float),
				m_vertex_buffer.p_OffsetPointer(0)); // pos
			glVertexAttribPointer(1, 3, GL_FLOAT, false, 8 * sizeof(float),
				m_vertex_buffer.p_OffsetPointer(3 * sizeof(float))); // normal
			glVertexAttribPointer(2, 2, GL_FLOAT, false, 8 * sizeof(float),
				m_vertex_buffer.p_OffsetPointer(6 * sizeof(float))); // texcoord
		}
		m_vao_halo.Bind();
		{
			m_halo_vbo.Bind();
			m_halo_vbo.BufferData(sizeof(Vector3f));
			glEnableVertexAttribArray(0);
			glVertexAttribPointer(0, 3, GL_FLOAT, false, 0, m_halo_vbo.p_OffsetPointer(0)); 
		}
		CGLVertexArrayObject::Release();

		m_n_index_num = index_buffer.size();
	}

	void Draw(Matrix4f t_mvp, Matrix4f t_mv, Vector3f v_position,
		Vector3f v_direction = Vector3f(1, 0, 0), float f_scale = .1f) // matrices are copied
	{
		m_texture.Bind();
		Matrix4f t_lookat;
		CGLTransform::LookAt(t_lookat, v_position, v_position + v_direction, Vector3f(0, 1, 0));
		t_lookat.Invert_Inplace(); // model position to camera position (horrible)
		Matrix4f t_lookat0 = t_lookat;
		t_lookat0.Scale(f_scale);
		t_lookat.Scale(-f_scale); // to flip triangle sides back (also horrible)
		v_position = Vector3f(0, 0, 0);
		t_mv *= t_lookat0;
		t_mvp *= t_lookat;
		m_shader.Bind(t_mvp, t_mv);
		m_vao.Bind();
		glDrawElements(GL_TRIANGLES, m_n_index_num, GL_UNSIGNED_INT, m_index_buffer.p_OffsetPointer(0));
		CGLVertexArrayObject::Release();
	}
};

class CRecordedTrajectory {
protected:
	double m_f_time, m_f_dt;
	std::vector<Vector3f> buoy_ground_truth;
	std::vector<Vector3f> diver_ground_truth;
	std::vector<Vector3f> buoy;
	std::vector<Vector3f> diver, diver_velocity;

	CGLArrayBufferObject ground_truth_trajcetory;
	CGLArrayBufferObject recovered_trajcetory;
	CGLVertexArrayObject gt_vao;
	CGLVertexArrayObject rec_vao;
	CGLArrayBufferObject temp_trajectory;
	CGLVertexArrayObject end_vao;

public:
	CRecordedTrajectory(const char *p_s_folder)
	{
		std::string s_manifest, s_solution;
		stl_ut::Format(s_manifest, "%s/manifest.txt", p_s_folder);
		stl_ut::Format(s_solution, "%s/solution.txt", p_s_folder);
		{
			FILE *p_fr;
#if defined(_MSC_VER) && !defined(__MWERKS__) && _MSC_VER >= 1400
			if(fopen_s(&p_fr, s_solution.c_str(), "r"))
#else //_MSC_VER && !__MWERKS__ && _MSC_VER >= 1400
			if(!(p_fr = fopen(s_solution.c_str(), "r")))
#endif //_MSC_VER && !__MWERKS__ && _MSC_VER >= 1400
				throw std::runtime_error("failed to open solution");
			std::string s_line;
			std::vector<std::string> values;
			while(!feof(p_fr)) {
				if(!stl_ut::ReadLine(s_line, p_fr))
					throw std::runtime_error("failed to read solution");
				stl_ut::TrimSpace(s_line);
				if(s_line.empty())
					continue;
				if(!stl_ut::Split(values, s_line, " ", 0))
					throw std::bad_alloc(); // rethrow
				if(values.size() == 3) { // a buoy
					Vector3f b;
					for(int j = 0; j < 3; ++ j) {
						if(sscanf(values[j].c_str(), "%f", &b[j]) != 1)
							throw std::runtime_error("failed to read solution");
					}
					buoy.push_back(b);
				} else if(values.size() == 6) { // a pose
					Vector3f d;
					for(int j = 0; j < 3; ++ j) {
						if(sscanf(values[j].c_str(), "%f", &d[j]) != 1)
							throw std::runtime_error("failed to read solution");
					}
					diver.push_back(d);
					for(int j = 0; j < 3; ++ j) {
						if(sscanf(values[j + 3].c_str(), "%f", &d[j]) != 1)
							throw std::runtime_error("failed to read solution");
					}
					diver_velocity.push_back(d);
				}
			}
			fclose(p_fr);
		}
		{
			FILE *p_fr;
#if defined(_MSC_VER) && !defined(__MWERKS__) && _MSC_VER >= 1400
			if(fopen_s(&p_fr, s_manifest.c_str(), "r"))
#else //_MSC_VER && !__MWERKS__ && _MSC_VER >= 1400
			if(!(p_fr = fopen(s_manifest.c_str(), "r")))
#endif //_MSC_VER && !__MWERKS__ && _MSC_VER >= 1400
				throw std::runtime_error("failed to open manifest");

			std::string s_diver;
			int n_buoy_num;
			if(fscanf(p_fr, "time=%lf\n", &m_f_time) != 1 ||
			   fscanf(p_fr, "dt=%lf\n", &m_f_dt) != 1 ||
			   !stl_ut::ReadLine(s_diver, p_fr) || s_diver.find("diver=(") != 0 ||
			   s_diver.rfind(")") != s_diver.length() - 1 ||
			   fscanf(p_fr, "buoy_num=%d\n", &n_buoy_num) != 1)
				throw std::runtime_error("failed to read manifest");
			// parse header

			s_diver.erase(0, s_diver.find('(') + 1); // erase the first open parenthesis
			s_diver.erase(s_diver.rfind(')')); // erase the first close parenthesis
			CDoubleExpression p_diver_eqn[3];
			std::vector<std::string> diver_eqns;
			if(!stl_ut::Split(diver_eqns, s_diver, ",") || diver_eqns.size() != 3 || // split to equations
			   !p_diver_eqn[0].Parse(diver_eqns[0].c_str()) || p_diver_eqn[0].n_Variable_num() > 1 ||
			   !p_diver_eqn[1].Parse(diver_eqns[1].c_str()) || p_diver_eqn[1].n_Variable_num() > 1 ||
			   !p_diver_eqn[2].Parse(diver_eqns[2].c_str()) || p_diver_eqn[2].n_Variable_num() > 1)
				throw std::runtime_error("failed to interpret diver motion");
			// parse diver equations

			buoy_ground_truth.resize(n_buoy_num);
			for(int i = 0; i < n_buoy_num; ++ i) {
				char p_s_format[64];
				stl_ut::Format(p_s_format, sizeof(p_s_format), "buoy[%d]=(%%f, %%f, %%f)\n", i);
				if(fscanf(p_fr, p_s_format, &buoy_ground_truth[i].x,
				   &buoy_ground_truth[i].y, &buoy_ground_truth[i].z) != 3)
					throw std::runtime_error("failed to read manifest");
			}
			// parse buoy ground truth

			int n_swap_yz;
			if(fscanf(p_fr, "swap_y_z=%d", &n_swap_yz) != 1)
				n_swap_yz = 0; // default
			// parse swap options

			if(n_swap_yz) {
				for(int i = 0; i < n_buoy_num; ++ i)
					std::swap(buoy_ground_truth[i].y, buoy_ground_truth[i].z);
			}
			// handle swaps on buoys

			for(double f_time = 1; f_time < m_f_time + m_f_dt / 2; f_time += m_f_dt) { // matlab starts time at 1 (!)
				Vector3f v_pose;
				for(int j = 0; j < 3; ++ j) {
					if(p_diver_eqn[j].b_HaveVariables()) {
						if(!p_diver_eqn[j].SetVariable("t", f_time))
							throw std::runtime_error("failed to calculate diver motion");
					}
					v_pose[j] = p_diver_eqn[j].t_Evaluate();
				}
				if(n_swap_yz)
					std::swap(v_pose.y, v_pose.z);
				diver_ground_truth.push_back(v_pose);
			}
			// calculate diver poses

			fclose(p_fr);

			if(n_swap_yz) {
				for(size_t i = 0, n = buoy.size(); i < n; ++ i)
					std::swap(buoy[i].y, buoy[i].z);
				for(size_t i = 0, n = diver.size(); i < n; ++ i)
					std::swap(diver[i].y, diver[i].z);
				for(size_t i = 0, n = diver_velocity.size(); i < n; ++ i)
					std::swap(diver_velocity[i].y, diver_velocity[i].z);
			}
			// handle swaps on recovered data
		}

		gt_vao.Bind();
		{
			ground_truth_trajcetory.Bind();
			ground_truth_trajcetory.BufferData(sizeof(Vector3f) * diver_ground_truth.size(), &diver_ground_truth[0]);
			glEnableVertexAttribArray(0);
			glVertexAttribPointer(0, 3, GL_FLOAT, false, 0, 0);
		}
		rec_vao.Bind();
		{
			recovered_trajcetory.Bind();
			recovered_trajcetory.BufferData(sizeof(Vector3f) * diver.size(), &diver[0]);
			glEnableVertexAttribArray(0);
			glVertexAttribPointer(0, 3, GL_FLOAT, false, 0, 0);
		}
		end_vao.Bind();
		{
			temp_trajectory.Bind();
			temp_trajectory.BufferData(sizeof(Vector3f) * 2);
			glEnableVertexAttribArray(0);
			glVertexAttribPointer(0, 3, GL_FLOAT, false, 0, 0);
		}
		CGLVertexArrayObject::Release();
	}

	size_t n_GT_Buoy_Num() const
	{
		return buoy_ground_truth.size();
	}

	Vector3f v_GT_Buoy(size_t n_index)
	{
		return buoy_ground_truth[n_index];
	}

	size_t n_Buoy_Num() const
	{
		return buoy.size();
	}

	Vector3f v_Buoy(size_t n_index)
	{
		return buoy[n_index];
	}

	Vector3f v_Diver(double f_time)
	{
		if(diver.empty())
			return Vector3f(0, 0, 0);

		f_time = fmod(f_time, m_f_time - 1); // matlab starts time at 1 (!)
		// calculate modulo time

		size_t n_pose = size_t(f_time / m_f_dt);
		n_pose %= diver.size();
		// calculate pose

		double f_lerp = (f_time - n_pose * m_f_dt) / m_f_dt;
		// calculate sub-pose

		return diver[n_pose] + (diver[min(n_pose + 1, diver.size() - 1)] - diver[n_pose]) * f_lerp;
	}

	Vector3f v_GT_Diver(double f_time)
	{
		if(diver.empty())
			return Vector3f(0, 0, 0);

		f_time = fmod(f_time, m_f_time - 1); // matlab starts time at 1 (!)
		// calculate modulo time

		size_t n_pose = size_t(f_time / m_f_dt);
		n_pose %= diver_ground_truth.size();
		// calculate pose

		double f_lerp = (f_time - n_pose * m_f_dt) / m_f_dt;
		// calculate sub-pose

		return diver_ground_truth[n_pose] + (diver_ground_truth[min(n_pose + 1,
			diver_ground_truth.size() - 1)] - diver_ground_truth[n_pose]) * f_lerp;
	}

	Vector3f v_Diver_Direction(double f_time)
	{
		if(diver.empty())
			return Vector3f(0, 0, 0);

		f_time = fmod(f_time, m_f_time - 1); // matlab starts time at 1 (!)
		// calculate modulo time

		size_t n_pose = size_t(f_time / m_f_dt);
		n_pose %= diver.size();
		// calculate pose

		double f_lerp = (f_time - n_pose * m_f_dt) / m_f_dt;
		// calculate sub-pose

		Vector3f v_direction, v_direction1;
#if 0
		if(n_pose + 2 < diver.size()) {
			v_direction = diver[min(n_pose + 1, diver.size() - 1)] - diver[n_pose];
			v_direction1 = diver[min(n_pose + 2, diver.size() - 1)] - diver[min(n_pose + 1, diver.size() - 1)];
		} else {
			v_direction = diver[n_pose] - diver[max(n_pose, size_t(1)) - 1];
			v_direction1 = diver[min(n_pose + 1, diver.size() - 1)] - diver[n_pose];
		}
#else
		v_direction = diver_velocity[n_pose];
		v_direction1 = diver_velocity[min(n_pose + 1, diver.size() - 1)];
#endif
		v_direction.Normalize();
		v_direction1.Normalize();

		return (v_direction + (v_direction1 - v_direction) * f_lerp).v_Normalized();
	}

	void Draw_Trajectory(bool b_ground_truth)
	{
		((b_ground_truth)? gt_vao : rec_vao).Bind();
		glDrawArrays(GL_LINE_STRIP, 0, ((b_ground_truth)? diver_ground_truth : diver).size());
		CGLVertexArrayObject::Release();
	}

	void Draw_Trajectory(bool b_ground_truth, double f_time, GLenum n_mode = GL_LINE_STRIP)
	{
		Vector3f v_endpoint = (b_ground_truth)? v_GT_Diver(f_time) : v_Diver(f_time);
		// precise endpoint

		f_time = fmod(f_time, m_f_time - 1); // matlab starts time at 1 (!)
		// calculate modulo time

		size_t n_pose = size_t(f_time / m_f_dt);
		n_pose %= ((b_ground_truth)? diver_ground_truth : diver).size();
		// calculate pose

		Vector3f v_last_whole = ((b_ground_truth)? diver_ground_truth : diver)[n_pose];
		// last integer pose

		((b_ground_truth)? gt_vao : rec_vao).Bind();
		glDrawArrays(n_mode, 0, n_pose + 1);
		end_vao.Bind();
		{
			temp_trajectory.Bind();
			Vector3f p_data[] = {v_last_whole, v_endpoint};
			temp_trajectory.BufferSubData(0, 2 * sizeof(Vector3f), &p_data);
			glDrawArrays(n_mode, (n_mode == GL_POINTS)? 1 : 0, (n_mode == GL_POINTS)? 1 : 2);
		}
		CGLVertexArrayObject::Release();
	}
};

#include "SpriteUtils.h"
#include "../UberLame_src/FontTexture2.h"
#include "../UberLame_src/BitmapFont2.h"

class CGUIWidget {
protected:
	static const float m_p_vertex_array[4 * 5];
	static const int m_p_index_array[6];
	Vector2f m_v_org;
	float m_f_size;
	CGLElementArraySetup m_geometry;
	CGLTexture_2D_Loadable<> m_texture;
	CSimpleShader m_shader;
	TLabelShader m_text_shader;
	CGLElementArraySetup *m_p_title_geom;
	CGLBitmapFont2 *m_p_bitmap_font[2];
	Vector2f m_v_text_size;
	bool m_b_slanted;

public:
	CGUIWidget(Vector2f v_org, float f_size = 1)
		:m_v_org(v_org), m_f_size(f_size), m_geometry(m_p_vertex_array, sizeof(m_p_vertex_array),
		5 * sizeof(float), GL_FLOAT, 3, 0, GL_FLOAT, 2, 3 * sizeof(float), m_p_index_array, 6,
		GL_UNSIGNED_INT, GL_TRIANGLES), m_texture("gui_bk_alpha.png", 0), m_p_title_geom(0)
	{
		m_p_bitmap_font[0] = 0;
		m_p_bitmap_font[1] = 0;
		/*TBmp *p_image = CPngCodec::p_Load_PNG("gui_bk.png");
		CSpriteUtils::Spritify(p_image);
		CPngCodec::Save_PNG("gui_bk_alpha.png", *p_image, true);
		p_image->Delete();*/
		// prepare the image

		{
			if(!TFileInfo("data/fonts/cmunrm00.tga").b_exists ||
			   !TFileInfo("data/fonts/cmunsl00.tga").b_exists) {
				CBitmapFont font;
				font.Request_UnicodeBlock("Basic Latin"); // need those
				font.Request_UnicodeBlock("Latin-1 Supplement"); // degree sign
				font.Request_UnicodeBlock("Greek and Coptic"); // need greek letters
				font.Request_UnicodeBlock("Mathematical Operators "); // delta
				if(!TFileInfo("data/fonts/cmunrm00.tga").b_exists) {
					CBitmapFont::CBitmapWriter write_pages("data/fonts/cmunrm%02" _PRIsize ".tga");
					font.Create("data/fonts/cmunrm.otf", 48, write_pages, false, true);
				}
				if(!TFileInfo("data/fonts/cmunsl00.tga").b_exists) {
					CBitmapFont::CBitmapWriter write_pages("data/fonts/cmunsl%02" _PRIsize ".tga");
					font.Create("data/fonts/cmunsl.otf", 48, write_pages, false, true);
				}
			}
			// build the fonts, if not there yet

			m_p_bitmap_font[0] = new CGLBitmapFont2();
			m_p_bitmap_font[0]->Load_2D("data/fonts/cmunrm%02" _PRIsize ".tga", true,
				CGLTextureLoader::n_Pixel_InternalFormat(1, 8, false), false, 0x00ffffffU);
			m_p_bitmap_font[0]->Set_Mode(GL_TRIANGLES, false, -1); // this is default anyway
			m_p_bitmap_font[1] = new CGLBitmapFont2();
			m_p_bitmap_font[1]->Load_2D("data/fonts/cmunsl%02" _PRIsize ".tga", true,
				CGLTextureLoader::n_Pixel_InternalFormat(1, 8, false), false, 0x00ffffffU);
			m_p_bitmap_font[1]->Set_Mode(GL_TRIANGLES, false, -1); // this is default anyway
			// load, with mipmaps
		}

		m_shader.Compile();
		m_text_shader.Compile();
	}

	~CGUIWidget()
	{
		if(m_p_title_geom)
			delete m_p_title_geom;
		if(m_p_bitmap_font[0])
			delete m_p_bitmap_font[0];
		if(m_p_bitmap_font[1])
			delete m_p_bitmap_font[1];
		// dispose of the old title
	}

	void Set_Title(const char *p_s_title, float f_title_height = .25f, bool b_slanted = false)
	{
		if(m_p_title_geom) {
			delete m_p_title_geom;
			m_p_title_geom = 0;
		}
		// dispose of the old title

		if(p_s_title && *p_s_title != 0) {
			m_b_slanted = b_slanted;
			int n_idx = (b_slanted)? 1 : 0;
			size_t n_printable_num = m_p_bitmap_font[n_idx]->n_Printable_Char_Num(p_s_title);
			if(n_printable_num) {
				const int n_latex_padding_px = 5; // convert padds the image a few pixels, this needs to be compensated
				const float f_zoom = f_title_height / (2 * n_latex_padding_px +
					m_p_bitmap_font[n_idx]->t_Font_Info().n_newline_height); // right?
				std::vector<float> vertex_list;
				std::vector<uint32_t> index_list;
				m_p_bitmap_font[n_idx]->Generate_Geometry(vertex_list, 0, index_list,
					p_s_title, Vector2f(0, 0), f_zoom);
				m_v_text_size = m_p_bitmap_font[n_idx]->v_Text_Size(p_s_title, f_zoom).v_zy();
				m_v_text_size.y -= (m_p_bitmap_font[n_idx]->t_Font_Info().n_newline_height -
					m_p_bitmap_font[n_idx]->t_Font_Info().n_descent) * f_zoom; // for the last line that does not end in newline
				// t_odo - render text using built-in font engine
				// todo - test this with a bunch of fonts, develop a centering formula that actually makes sense
				// this now makes sense - each line is counted as its height, and the descent is deducted
				// to shift the origin under the characters (otherwise the origin is on the baseline)

				for(size_t i = 4, n = vertex_list.size(); i < n; i += 5) {
					vertex_list[i] = f_title_height - vertex_list[i] +
						(m_p_bitmap_font[n_idx]->t_Font_Info().n_descent - n_latex_padding_px) * f_zoom;
				}
				// the letters are somehow upside-down

				m_p_title_geom = new CGLElementArraySetup(&vertex_list[0],
					vertex_list.size() * sizeof(float), 5 * sizeof(float),
					GL_FLOAT, 2, 3 * sizeof(float), GL_FLOAT, 3, 0,
					&index_list[0], index_list.size(), GL_UNSIGNED_INT, GL_TRIANGLES);
				// create indexed geometry (font renderer also uses indexed)
			} else
				m_p_title_geom = 0; // nothing to draw
		}
		// t_odo - make this a SetTitle function, try to cache the bitmaps for smooth animations
	}

	Vector2f v_Min() const
	{
		if(m_f_size < 0) {
			const float f_ratio = float(m_texture.n_Width()) / m_texture.n_Height() * n_height / float(n_width);
			return m_v_org + Vector2f(f_ratio * m_f_size, m_f_size);
		}
		return m_v_org;
	}

	Vector2f v_Max() const
	{
		if(m_f_size < 0)
			return m_v_org;
		const float f_ratio = float(m_texture.n_Width()) / m_texture.n_Height() * n_height / float(n_width);
		return m_v_org + Vector2f(f_ratio * m_f_size, m_f_size);
	}

	void Draw() const
	{
		glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
		glEnable(GL_BLEND);

		Matrix4f t_mvp;
		t_mvp.Identity();
		t_mvp.Translate(m_v_org.x, m_v_org.y, 0);
		const float f_ratio = float(m_texture.n_Width()) / m_texture.n_Height() * n_height / float(n_width);
		t_mvp.Scale(f_ratio * m_f_size, m_f_size, 1);

		m_texture.Bind();
		m_shader.Bind(t_mvp);
		m_geometry.Draw();

		if(m_p_title_geom) {
			glDisable(GL_CULL_FACE);
			int n_idx = (m_b_slanted)? 1 : 0;
			m_p_bitmap_font[n_idx]->p_Texture_2D()->Bind(); // loaded as 2D font
			const float f_box_width = 1 * float(m_texture.n_Width()) /
				m_texture.n_Height(), f_box_height = 1;
			t_mvp.Identity();
			const float f_ratio = n_height / float(n_width);
			Vector2f v_org = m_v_org;
			v_org.x += (f_box_width - m_v_text_size.x) / 2 * m_f_size * f_ratio;
			v_org.y += (f_box_height - m_v_text_size.y) / 2 * m_f_size;
			t_mvp.Translate(v_org.x, v_org.y, 0);
			t_mvp.Scale(f_ratio * m_f_size, -m_f_size, 1);
			m_text_shader.Bind(t_mvp, Vector3f(.3f, 1, .4f));
			m_p_title_geom->Draw();
		}

		glDisable(GL_BLEND);
	}
};

const float CGUIWidget::m_p_vertex_array[4 * 5] = {
	1, 0, 0, 1, 1,
	0, 0, 0, 0, 1,
	0, 1, 0, 0, 0,
	1, 1, 0, 1, 0
};

const int CGUIWidget::m_p_index_array[6] = {
	0, 1, 2, 0, 2, 3
};
