#pragma once
#ifndef __SIRF_BINARY_PROTOCOL_UITILITIES_INCLUDED
#define __SIRF_BINARY_PROTOCOL_UITILITIES_INCLUDED

const uint8_t *p_Find_SIRF_MessageStartMarker(const uint8_t *p_buffer, const uint8_t *p_end);

uint16_t n_SIRF_GetWord(const uint8_t *p_buffer, const uint8_t *p_end);

bool SIRF_MessageChecksum(uint16_t &r_n_checksum, const uint8_t *p_buffer, const uint8_t *p_end);

const uint8_t *p_Find_SIRF_MessageChecksum(const uint8_t *p_buffer, const uint8_t *p_end);

bool b_Validate_SIRF_Message(uint16_t &r_n_length, const uint8_t *p_buffer, const uint8_t *p_end);

void Calculate_SIRF_MessageChecksum(uint8_t *p_buffer, const uint8_t *p_end);

void  SIRF_Finalize_Message(uint8_t *p_buffer, const uint8_t *p_end);

enum EResetFlags {
	reset_DataValid = 1, // Use data in ECEF X, Y, Z, Clock Offset, Time of Week and Week number to initialize the receiver
	reset_ClearEphemeris = 2, // Clear ephemeris from memory: blocks Snap or Hot Start from occurring
	reset_ClearHistory = 4, // Clear all history (except clock drift) frommemory: blocks Snap, Hot, and Warm Starts
	reset_FactoryReset = 8, // Factory Reset: clears all GPS memory including clock drift. Also clears almanac stored in flash memory
	reset_EnableNavLib = 16, // Enable Nav Lib data
	reset_EnableDebug = 32, // Enable debug data
	reset_RTC_Imprecise = 64, // Indicate that Real-Time Clock (RTC) is not precise: blocks Snap Start
	reset_ClearUserData = 128 // SiRFstarII = clear user data in memory; SiRFstarIII = perform full system reset
};

bool SIRF_InitializeDataSource(FT_HANDLE h_device, int n_reset_flags,
	int32_t n_X_meters = 0, int32_t n_Y_meters = 0, int32_t n_Z_meters = 0,
	int32_t n_clock_drift_Hz = 75000, uint32_t n_time_of_week_100sec = 0,
	uint16_t n_week_number = 0, uint8_t n_channel_num = 12);

bool SIRF_ConfigureUART0(FT_HANDLE h_device, int n_baud_rate, int n_bit_num = 8,
	int n_stop_bit_num = 1, int n_parity = FT_PARITY_NONE);

bool SIRF_SetMessageRate(FT_HANDLE h_device, int n_message_id, int n_rate_sec = 1, int n_mode = 0);

#define SIRF_Finalize_StaticArrayMessage(p_msg) \
	SIRF_Finalize_Message((uint8_t*)p_msg, (uint8_t*)(p_msg + sizeof(p_msg) / sizeof(p_msg[0])))

#define n_SIRF_GetShort(p_msg) ((((uint8_t*)(p_msg))[0] << 8) | (((uint8_t*)(p_msg))[1] << 0))

#define n_SIRF_GetInt(p_msg) ((((uint8_t*)(p_msg))[0] << 24) | \
	(((uint8_t*)(p_msg))[1] << 16) | (((uint8_t*)(p_msg))[2] << 8) | (((uint8_t*)(p_msg))[3] << 0))

#define n_SIRF_GetLong(p_msg) ((uint64_t(n_SIRF_GetInt(p_msg)) << 32) | n_SIRF_GetInt(((uint8_t*)(p_msg)) + 4))

inline float f_SIRF_GetFloat(const uint8_t *p_msg)
{
	union {
		uint32_t n;
		float f;
	} u;
	u.n = n_SIRF_GetInt(p_msg);
	return u.f;
}

inline double f_SIRF_GetDouble(const uint8_t *p_msg)
{
	union {
		uint64_t n;
		double f;
	} u;
	u.n = n_SIRF_GetLong(p_msg); // B7, B6, B5, B4, B3, B2, B1, B0
	u.n = (u.n >> 32) | (u.n << 32); // word swap (B3, B2, B1, B0, B7, B6, B5, B4)
	return u.f;
}

#include "NMEA.h"

/**
 *	@brief SIRF binary protocol sanity tester (for baudrate initializazion)
 */
class CSIRFProtocolSanitizer {
public:
	/**
	 *	@brief tests whether the device communicates at the given baud rate
	 *
	 *	@param[in] h_device is device handle, with serial port already configured
	 *		to the corresponding speed / features (no need to e.g. call FT_SetBaudRate())
	 *	@param[in] n_baud_rate is baud rate (e.g. 38400)
	 *	@param[in] n_bit_num is number of message bits (e.g. 8)
	 *	@param[in] n_stop_bit_num is number of stop bits (e.g. 1)
	 *	@param[in] n_parity is parity (one of FT_PARITY_*)
	 *
	 *	@return Returns true if the device communicates
	 *		at the given speed, otherwise returns false.
	 */
	static bool b_Validate(FT_HANDLE h_device, int n_baud_rate,
		int n_bit_num, int n_stop_bit_num, int n_parity);
};

#endif // __SIRF_BINARY_PROTOCOL_UITILITIES_INCLUDED
