#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <stdio.h>
#include <assert.h>
#define _ASSERTE(expr) assert(expr)

#include "NMEA.h"
#include "SIRF_Binary.h"

/*
 *								=== globals ===
 */

bool NMEA_MessageChecksum(uint8_t &r_n_checksum, const char *p_s_message)
{
	_ASSERTE(p_s_message); // not null

	r_n_checksum = 0;
	// no checksum yet

	if(p_s_message[0] != '$')
		return false; // not a valid NMEA message
	_ASSERTE(p_s_message[0] == '$'); // make sure there is dollar sign at the beginning
	const char *p_begin = p_s_message + 1;
	// assert begin marker

	const char *p_end = strchr(p_begin, '*');
	if(p_end) {
		if(strlen(p_end) < 5/*strlen("*__\r\n")*/)
			return false; // not a whole message
		if(p_end[3] != '\r' || p_end[4] != '\n')
			return false; // not a valid message
	}
	_ASSERTE(!p_end || (p_end[3] == '\r' && p_end[4] == '\n'));
	const char *p_end2 = strstr(p_begin, "\r\n");
	if((p_end2 && !p_end) || (p_end2 && p_end && p_end2 < p_end))
		p_end = p_end2; // in case CRLF is closer than asterisk
	if(!p_end || p_end <= p_begin) // make sure there is the asterisk
		return false; // no valid end marker
	const char *p_next = strchr(p_begin, '$');
	if(p_end && p_next && p_next < p_end)
		return false; // bad start of the message
	// find end marker

	uint8_t n_parity = 0;
	for(; p_begin != p_end; ++ p_begin) {
		uint8_t n_char = *p_begin;
		n_parity ^= n_char;
	}
	r_n_checksum = n_parity;
	// calculate the checksum

	return true;
}

const char *p_s_Find_NMEA_MessageChecksum(const char *p_s_message)
{
	_ASSERTE(p_s_message); // not null

	if(p_s_message[0] != '$')
		return 0; // not a valid NMEA message
	_ASSERTE(p_s_message[0] == '$'); // make sure there is dollar sign at the beginning
	const char *p_begin = p_s_message + 1;
	// assert begin marker

	const char *p_end = strchr(p_begin, '*');
	const char *p_next = strchr(p_begin, '$');
	const char *p_end2 = strstr(p_begin, "\r\n"); // in case more messages are in the buffer
	if(!p_end || (p_end2 && p_end2 < p_end))
		return 0; // no checksum there
	if(p_end && p_next && p_next < p_end)
		return 0; // bad start of the message
	if(strlen(p_end) < 5/*strlen("*__\r\n")*/)
		return 0; // not a whole message
	if(p_end[3] != '\r' || p_end[4] != '\n')
		return 0; // not a valid message
	_ASSERTE(!p_end || (p_end[3] == '\r' && p_end[4] == '\n'));
	// find checksum marker

	return p_end + 1;
}

void Calculate_NMEA_MessageChecksum(char *p_s_message)
{
	uint8_t n_checksum;
	bool b_checksum = NMEA_MessageChecksum(n_checksum, p_s_message);
	_ASSERTE(n_checksum);
	// calculate checksum

	char *p_s_checksum = (char*)p_s_Find_NMEA_MessageChecksum(p_s_message);
	_ASSERTE(p_s_checksum);
	// get place for checksum

	char p_s_parity[3];
	_ASSERTE(int(n_checksum) >= 0 && int(n_checksum) < 0xff); // make sure it fits in two places
	sprintf(p_s_parity, "%02X", int(n_checksum)); // todo - use handmade conversion instead to stop the compiller fussing about it
	p_s_checksum[0] = p_s_parity[0];
	p_s_checksum[1] = p_s_parity[1];
	// format and write the parity
}

/*
 *								=== ~globals ===
 */

/*
 *								=== CNMEAProtocolSanitizer ===
 */

bool CNMEAProtocolSanitizer::b_Validate(FT_HANDLE h_device,
	int n_baud_rate, int n_bit_num, int n_stop_bit_num, int n_parity)
{
	char p_s_init[256];
	sprintf(p_s_init, "$PSRF100,1,%d,%d,%d,%d*__\r\n", n_baud_rate, n_bit_num,
		n_stop_bit_num, (n_parity == FT_PARITY_NONE)? 0 : (n_parity == FT_PARITY_ODD)? 1 : 2);
	Calculate_NMEA_MessageChecksum(p_s_init); // !!
	// set to NMEA ascii protocol

	uint8_t p_init2[] = {0xa0, 0xa2, // start marker
		0x00, 0x00, // space for message length
		0x81, // Message ID
		0x02, // Do not change last-set value for NMEA debug messages
		0x01, // GGA Message
		0x01, // Checksum
		0x00, // GLL Message
		0x01, // Checksum
		0x01, // GSA Message
		0x01, // Checksum
		0x05, // GSV Message
		0x01, // Checksum
		0x01, // RMC Message
		0x01, // Checksum
		0x00, // VTG Message
		0x01, // Checksum
		0x00, // MSS Message
		0x01, // Checksum
		0x00, // EPE Message
		0x00, // Checksum
		0x00, // ZDA Message
		0x01, // Checksum
		0x00, // Unused Field
		0x00, // Unused Field
		uint8_t(n_baud_rate >> 8),
		uint8_t(n_baud_rate), // Bit Rate (2B unsigned)
		0x00, 0x00, // space for checksum
		0xb0, 0xb3}; // end marker
	SIRF_Finalize_StaticArrayMessage(p_init2);
	// set to NMEA ascii protocol (from SIRF binary)

	char p_s_buffer[1024];
	int n_switch_num = 0;
	for(int i = 0; i < 4; ++ i) { // try a few times
		DWORD n_written;
		FT_STATUS n_result;
		n_result = FT_Write(h_device, p_s_init, DWORD(strlen(p_s_init) * sizeof(char)), &n_written); // in NMEA
		// write the request

		DWORD n_read;
		n_result = FT_Read(h_device, p_s_buffer, sizeof(p_s_buffer) - sizeof(char), &n_read);
		_ASSERTE(n_read < sizeof(p_s_buffer) / sizeof(p_s_buffer[0]));
		p_s_buffer[n_read] = 0; // zero terminate
		// read a message

		const uint8_t *p_begin = (const uint8_t*)p_s_buffer, *p_end = (const uint8_t*)(p_s_buffer + 1024);
		while((p_begin = p_Find_SIRF_MessageStartMarker(p_begin, p_end)) != 0) {
			uint16_t n_len;
			if(b_Validate_SIRF_Message(n_len, p_begin, p_end)) {
				//printf("GPS is communicating in binary, switching to NMEA\n");
				n_result = FT_Write(h_device, p_init2, sizeof(p_init2), &n_written); // in SIRF binary
				if(++ n_switch_num < 5) // avoid infinite looping
					-- i; // will have to do this possibly couple of times before it empties the buffers
				break;
			}
			++ p_begin;
		}
		// see if it is in binary mode

		char *p_s_message = p_s_buffer;
		while(p_s_message = strchr(p_s_message, '$')) {
			/*char p_s_part[64];
			strncpy(p_s_part, p_s_message, 63);
			p_s_part[63] = 0;
			printf("debug: \'%s ...\'\r", p_s_part);*/
			uint8_t n_checksum;
			if(!NMEA_MessageChecksum(n_checksum, p_s_message)) {
				++ p_s_message;
				//printf("nochecksum\r");
				continue;
			}
			// see if the message has a terminator and we can get a checksum

			const char *p_s_checksum;
			if(!(p_s_checksum = p_s_Find_NMEA_MessageChecksum(p_s_message))) {
				++ p_s_message;
				//printf("noembed\r");
				continue;
			}
			// try to find checksum chars

			//printf("checksum is %c%c, calculated %02x ", *p_s_checksum, p_s_checksum[1], n_checksum);
			int n_recv_checksum;
			sscanf(p_s_checksum, "%x\r\n", &n_recv_checksum);
			if(n_checksum != n_recv_checksum) {
				++ p_s_message;
				//printf("mismatch\r");
				continue;
			}
			// compare checksum

			return true;
			// we received a valid message with matching checksum
		}
	}

	return false;
	// despite many tries, we are only receiving gibberish
}

/*
 *								=== ~CNMEAProtocolSanitizer ===
 */

/*
 *		end-of-file
 */
