#pragma once
#ifndef __NMEA_PROTOCOL_UITILITIES_INCLUDED
#define __NMEA_PROTOCOL_UITILITIES_INCLUDED

#include <stdint.h>
#if defined(_WIN32) || defined(_WIN64)
#include "ftd2xx.h"
#else // _WIN32 || _WIN64
#include "ftd2xx_linux.h"
#endif // _WIN32 || _WIN64

/**
 *	@brief calculates checksum of a NMEA message
 *
 *	@param[out] r_n_checksum is the calculated checksum
 *	@param[in] p_s_message is NMEA message, starting with '$' and ending
 *		either with "\r\n" or "*..\r\n" (where '.' is any character)
 *
 *	@return Returns	true on success (the checksum is valid on return),
 *		or false on failure (message is malformed, checksum not calculated).
 */
bool NMEA_MessageChecksum(uint8_t &r_n_checksum, const char *p_s_message);

/**
 *	@brief finds position of the first checksum character in a message
 *	@param[in] p_s_message is a NMEA message, starting with '$'
 *	@return Returns pointer to the first checksum character
 *		or 0 on failure (checksum not present / malformed message).
 */
const char *p_s_Find_NMEA_MessageChecksum(const char *p_s_message);

/**
 *	@brief calculates NMEA message checksum and inserts it in the string
 *	@param[in,out] p_s_message is NMEA message, starting with '$' and ending
 *		with "*__\r\n" (the underscores will be replaced by the checksum)
 */
void Calculate_NMEA_MessageChecksum(char *p_s_message);

/**
 *	@brief NMEA ascii protocol sanity tester (for baudrate initializazion)
 */
class CNMEAProtocolSanitizer {
public:
	/**
	 *	@brief tests whether the device communicates at the given baud rate
	 *
	 *	@param[in] h_device is device handle, with serial port already configured
	 *		to the corresponding speed / features (no need to e.g. call FT_SetBaudRate())
	 *	@param[in] n_baud_rate is baud rate (e.g. 38400)
	 *	@param[in] n_bit_num is number of message bits (e.g. 8)
	 *	@param[in] n_stop_bit_num is number of stop bits (e.g. 1)
	 *	@param[in] n_parity is parity (one of FT_PARITY_*)
	 *
	 *	@return Returns true if the device communicates
	 *		at the given speed, otherwise returns false.
	 */
	static bool b_Validate(FT_HANDLE h_device, int n_baud_rate,
		int n_bit_num, int n_stop_bit_num, int n_parity);
};

#endif // __NMEA_PROTOCOL_UITILITIES_INCLUDED
