--!
--! open_loop.vhd: Open-loop solution single signal synchronizer.
--! Copyright (C) 2014 CESNET
--! Author(s): Jakub Cabal <jakubcabal@gmail.com>
--!
--! Redistribution and use in source and binary forms, with or without
--! modification, are permitted provided that the following conditions
--! are met:
--! 1. Redistributions of source code must retain the above copyright
--!    notice, this list of conditions and the following disclaimer.
--! 2. Redistributions in binary form must reproduce the above copyright
--!    notice, this list of conditions and the following disclaimer in
--!    the documentation and/or other materials provided with the
--!    distribution.
--! 3. Neither the name of the Company nor the names of its contributors
--!    may be used to endorse or promote products derived from this
--!    software without specific prior written permission.
--!
--! This software is provided ``as is'', and any express or implied
--! warranties, including, but not limited to, the implied warranties of
--! merchantability and fitness for a particular purpose are disclaimed.
--! In no event shall the company or contributors be liable for any
--! direct, indirect, incidental, special, exemplary, or consequential
--! damages (including, but not limited to, procurement of substitute
--! goods or services; loss of use, data, or profits; or business
--! interruption) however caused and on any theory of liability, whether
--! in contract, strict liability, or tort (including negligence or
--! otherwise) arising in any way out of the use of this software, even
--! if advised of the possibility of such damage.
--!
--! $Id$
--!

library IEEE;	
use IEEE.std_logic_1164.all;	
use IEEE.std_logic_unsigned.all;	
use IEEE.std_logic_arith.all;

--! pragma translate_off
library UNISIM;
use UNISIM.vcomponents.all;
--! pragma translate_on

   --! -------------------------------------------------------------------------
   --!                      Readme!!!
   --! -------------------------------------------------------------------------

   --! When you sync from slow to fast clock domain,
   --! you must hold this condition:
   --!                              frequency_fast >= 1.5x frequency_slow
     
   --! When you sync from fast to slow clock domain,
   --! you must hold this condition:   
   --!                              input_pulse_length >= 1.5x period_slow

   --! -------------------------------------------------------------------------
   --!                      Entity declaration
   --! -------------------------------------------------------------------------

entity ASYNC_OPEN_LOOP_TOP is
   Generic (
      IN_REG   : BOOLEAN := false; --! For one register on input = true
      TWO_REG  : BOOLEAN := false  --! For two reg = true, for three reg = false
   );    
   Port (
      --! A clock domain 
      ACLK     : in  STD_LOGIC;    --! Source clock
      ARST     : in  STD_LOGIC;    --! Source reset     
      ADATAIN  : in  STD_LOGIC;    --! Data input
      
      --! B clock domain   
      BCLK     : in  STD_LOGIC;    --! Target clock
      BRST     : in  STD_LOGIC;    --! Target reset  
      BDATAOUT : out STD_LOGIC     --! Data output
   );
end ASYNC_OPEN_LOOP_TOP;

   --! -------------------------------------------------------------------------
   --!                      Architecture declaration
   --! -------------------------------------------------------------------------

architecture FULL of ASYNC_OPEN_LOOP_TOP is

   --! -------------------------------------------------------------------------
   --!                      SIGNALS   
   --! -------------------------------------------------------------------------   
   
   signal signal_D1   : std_logic := '0';
   signal signal_Q1   : std_logic := '0';
   signal signal_Q2   : std_logic := '0';
   
   --! Attributes for signal_Q1_reg and signal_Q2_reg
   attribute shreg_extract                : string;
   attribute async_reg                    : string;

   attribute shreg_extract of signal_Q1   : signal is "no";	
   attribute async_reg of signal_Q1       : signal is "true";
   	
   attribute shreg_extract of signal_Q2   : signal is "no";	
   attribute async_reg of signal_Q2       : signal is "true";
	
   --! -------------------------------------------------------------------------

begin
     
   --! Generics input register
   input_reg : if IN_REG generate
	
      process(ACLK)
      begin
         if (rising_edge(ACLK)) then
            if (ARST = '1') then
               signal_D1 <= '0';
            else
               signal_D1 <= ADATAIN;
            end if;      
         end if;
      end process;
      	
   end generate;
  
   --! -------------------------------------------------------------------------
  
   --! Generics input register off
   not_in_reg : if NOT IN_REG generate	
      
      signal_D1 <= ADATAIN;
   	
   end generate;

   --! -------------------------------------------------------------------------
   
   --! Two synchronization registers
   process(BCLK)
   begin
      if (rising_edge(BCLK)) then
         if (BRST = '1') then
            signal_Q1 <= '0';
            signal_Q2 <= '0';
         else
            signal_Q1 <= signal_D1;
            signal_Q2 <= signal_Q1;
         end if;      
      end if;
   end process;

   --! -------------------------------------------------------------------------
   
   --! Generics two synchronization registers
   two_reg_sync : if TWO_REG generate

      BDATAOUT <= signal_Q2;
  
   end generate;  
  
   --! -------------------------------------------------------------------------  
  
   --! Generics three synchronization registers
   three_reg_sync : if NOT TWO_REG generate
      
      --! Signals
      signal signal_Q3                       : std_logic := '0';   
      
      --! Attributes	for signal_Q3_reg
      attribute shreg_extract of signal_Q3   : signal is "no";	
      attribute async_reg of signal_Q3       : signal is "true";	
   
      begin
   
      process(BCLK)
      begin
         if (rising_edge(BCLK)) then
            if (BRST = '1') then
               signal_Q3 <= '0';
            else
               signal_Q3 <= signal_Q2;
            end if;      
         end if;
      end process;
 
      BDATAOUT <= signal_Q3;
 
   end generate; 

   --! -------------------------------------------------------------------------

end architecture FULL;
