--
-- nfifo.vhd - top entity, emulates N fifos wrapped into a component
-- Copyright (C) 2009 CESNET
-- Author(s): Vozenilek Jan <xvozen00@stud.fit.vutbr.cz>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--
--
-- TODO:
--

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_unsigned.all;
use IEEE.std_logic_arith.all;

-- library containing log2 function
use work.math_pack.all;

-- ----------------------------------------------------------------------
--                            Entity declaration
-- ----------------------------------------------------------------------
entity NFIFO is
  generic(
    DATA_WIDTH : integer := 64;
    FLOWS      : integer := 2;
    -- Number of unused flows with odd number.
    -- Starts unconnect from higher numbers. So if FLOWS=32 and UNUSED_ODD=5
    -- then flows with number 31,29,27,25,23 will be unconnected
    UNUSED_ODD : integer := 0;
    -- Number of unused flows with even number.
    -- Starts unconnect from higher numbers. So if FLOWS=32 and UNUSED_EVEN=5
    -- then flows with number 30,28,26,24,22 will be unconnected
    UNUSED_EVEN: integer := 0;
    BLOCK_SIZE : integer := 512;
    LUT_MEMORY : boolean := false;
    -- use OUTPUT_REG carefully - combined with PIPE_EN not in '1' can cause
    -- reading data from different block than given by RD_BLK_ADDR
    OUTPUT_REG : boolean := false
  );

  port(
    CLK         : in  std_logic;
    RESET       : in  std_logic;
    INIT        : in  std_logic_vector(FLOWS-1 downto 0);

    -- Write interface
    DATA_IN     : in  std_logic_vector(DATA_WIDTH-1 downto 0);
    WR_BLK_ADDR : in  std_logic_vector(abs(log2(FLOWS)-1) downto 0);
    WRITE       : in  std_logic;
    FULL        : out std_logic_vector(FLOWS-1 downto 0);

    -- Read interface
    DATA_OUT    : out std_logic_vector(DATA_WIDTH-1 downto 0);
    DATA_VLD    : out std_logic;
    RD_BLK_ADDR : in  std_logic_vector(abs(log2(FLOWS)-1) downto 0);
    READ        : in  std_logic;
    PIPE_EN     : in  std_logic;
    EMPTY       : out std_logic_vector(FLOWS-1 downto 0);

    STATUS      : out std_logic_vector(log2(BLOCK_SIZE+1)*FLOWS-1 downto 0)
  );
end entity;

-- ----------------------------------------------------------------------
--                      Architecture declaration
-- ----------------------------------------------------------------------
architecture behavioral of NFIFO is

  -- used if BLOCK_SIZE is not a power of 2
  function ADDR_FIX_CONST (FLOW_ID: integer)
    return integer is
  begin
    return FLOW_ID*(2**log2(BLOCK_SIZE) - BLOCK_SIZE);
  end function;

  subtype ADDR_RANGE is natural range log2(BLOCK_SIZE)-1 downto 0;

  type t_addr_low_cnt is array(FLOWS-1 downto 0) of std_logic_vector(log2(BLOCK_SIZE)-1 downto 0);
  type t_addr_cnt     is array(FLOWS-1 downto 0) of std_logic_vector(log2(BLOCK_SIZE) downto 0);
  type t_status       is array(FLOWS-1 downto 0) of std_logic_vector(log2(BLOCK_SIZE+1)-1 downto 0);
  type t_fix          is array(FLOWS-1 downto 0) of integer;

  signal blk_write_allow     : std_logic_vector(FLOWS-1 downto 0);
  signal cnt_write_addr_rst  : std_logic_vector(FLOWS-1 downto 0);
  signal cnt_write_addr_low  : t_addr_low_cnt;
  signal cnt_write_addr_high : std_logic_vector(FLOWS-1 downto 0);
  signal cnt_write_addr      : t_addr_cnt;
  signal reg_write_addr      : t_addr_cnt;
  signal blk_read_allow      : std_logic_vector(FLOWS-1 downto 0);
  signal cnt_read_addr_rst   : std_logic_vector(FLOWS-1 downto 0);
  signal cnt_read_addr_low   : t_addr_low_cnt;
  signal cnt_read_addr_high  : std_logic_vector(FLOWS-1 downto 0);
  signal cnt_read_addr       : t_addr_cnt;
  signal reg_read_addr       : t_addr_cnt;

  signal sig_init            : std_logic_vector(FLOWS-1 downto 0);
  signal blk_full            : std_logic_vector(FLOWS-1 downto 0);
  signal blk_empty           : std_logic_vector(FLOWS-1 downto 0);
  signal blk_status          : t_status;
  signal sig_status          : std_logic_vector(log2(BLOCK_SIZE+1)*FLOWS-1 downto 0);

  signal write_allow         : std_logic;
  signal read_allow          : std_logic;
  signal write_addr          : std_logic_vector(log2(BLOCK_SIZE*FLOWS)-1 downto 0);
  signal read_addr           : std_logic_vector(log2(BLOCK_SIZE*FLOWS)-1 downto 0);
  signal addr_fix            : t_fix;

-- ----------------------------------------------------------------------------
begin

GEN_ONE_FLOW: if (FLOWS = 1) generate

  blk_write_allow(0) <= WRITE and (not blk_full(0));
  blk_read_allow(0)  <= READ and PIPE_EN and (not blk_empty(0));

  write_allow <= blk_write_allow(0);
  read_allow  <= blk_read_allow(0);

  write_addr <= reg_write_addr(0)(ADDR_RANGE);
  read_addr  <= reg_read_addr(0)(ADDR_RANGE);

end generate;

GEN_MORE_FLOWS: if (FLOWS > 1) generate

  write_allow <= blk_write_allow(conv_integer(WR_BLK_ADDR));
  read_allow  <= blk_read_allow(conv_integer(RD_BLK_ADDR));

  write_addr <= (WR_BLK_ADDR & reg_write_addr(conv_integer(WR_BLK_ADDR))(ADDR_RANGE)) - addr_fix(conv_integer(WR_BLK_ADDR));
  read_addr  <= (RD_BLK_ADDR & reg_read_addr(conv_integer(RD_BLK_ADDR))(ADDR_RANGE)) - addr_fix(conv_integer(RD_BLK_ADDR));

end generate;

GEN_BLOCKS: for j in 0 to FLOWS-1 generate

  GEN_MORE_FLOWS_BLOCKS: if (FLOWS > 1) generate

    -- block write allow
    process(WR_BLK_ADDR, WRITE, blk_full)
    begin
      if (WR_BLK_ADDR = conv_std_logic_vector(j, WR_BLK_ADDR'length)) then
        blk_write_allow(j) <= WRITE and (not blk_full(j));
      else
        blk_write_allow(j) <= '0';
      end if;
    end process;

    -- block read allow
    process(RD_BLK_ADDR, READ, PIPE_EN, blk_empty)
    begin
      if (RD_BLK_ADDR = conv_std_logic_vector(j, RD_BLK_ADDR'length)) then
        blk_read_allow(j) <= READ and PIPE_EN and (not blk_empty(j));
      else
        blk_read_allow(j) <= '0';
      end if;
    end process;

  end generate;

  -- init signals
  sig_init(j) <= RESET or INIT(j);

  -- write address counter ----------------------------------------------------
  process(cnt_write_addr_low)
  begin
    if (cnt_write_addr_low(j) = conv_std_logic_vector(BLOCK_SIZE-1, log2(BLOCK_SIZE))) then
      cnt_write_addr_rst(j) <= '1';
    else
      cnt_write_addr_rst(j) <= '0';
    end if;
  end process;

  process(CLK)
  begin
    if ((CLK'event) and (CLK = '1')) then
      if (sig_init(j) = '1') then
        cnt_write_addr_low(j) <= conv_std_logic_vector(1, cnt_write_addr_low(j)'length);
      else
        if (blk_write_allow(j) = '1') then
          if (cnt_write_addr_rst(j) = '1') then
            cnt_write_addr_low(j) <= (others => '0');
          else
            cnt_write_addr_low(j) <= cnt_write_addr_low(j) + 1;
          end if;
        end if;
      end if;
    end if;
  end process;

  process(CLK)
  begin
    if ((CLK'event) and (CLK = '1')) then
      if (sig_init(j) = '1') then
        cnt_write_addr_high(j) <= '0';
      else
        if (blk_write_allow(j) = '1') then
          if (cnt_write_addr_rst(j) = '1') then
            cnt_write_addr_high(j) <= not cnt_write_addr_high(j);
          end if;
        end if;
      end if;
    end if;
  end process;

  cnt_write_addr(j) <= cnt_write_addr_high(j) & cnt_write_addr_low(j);

  -- write address register ---------------------------------------------------
  process(CLK)
  begin
    if ((CLK'event) and (CLK = '1')) then
      if (sig_init(j) = '1') then
        reg_write_addr(j) <= (others => '0');
      else
        if (blk_write_allow(j) = '1') then
          reg_write_addr(j) <= cnt_write_addr(j);
        end if;
      end if;
    end if;
  end process;

  -- read address counter -----------------------------------------------------
  process(cnt_read_addr_low)
  begin
    if (cnt_read_addr_low(j) = conv_std_logic_vector(BLOCK_SIZE-1, log2(BLOCK_SIZE))) then
      cnt_read_addr_rst(j) <= '1';
    else
      cnt_read_addr_rst(j) <= '0';
    end if;
  end process;

  process(CLK)
  begin
    if ((CLK'event) and (CLK = '1')) then
      if (sig_init(j) = '1') then
        cnt_read_addr_low(j) <= conv_std_logic_vector(1, cnt_read_addr_low(j)'length);
      else
        if (blk_read_allow(j) = '1') then
          if (cnt_read_addr_rst(j) = '1') then
            cnt_read_addr_low(j) <= (others => '0');
          else
            cnt_read_addr_low(j) <= cnt_read_addr_low(j) + 1;
          end if;
        end if;
      end if;
    end if;
  end process;

  process(CLK)
  begin
    if ((CLK'event) and (CLK = '1')) then
      if (sig_init(j) = '1') then
        cnt_read_addr_high(j) <= '0';
      else
        if (blk_read_allow(j) = '1') then
          if (cnt_read_addr_rst(j) = '1') then
            cnt_read_addr_high(j) <= not cnt_read_addr_high(j);
          end if;
        end if;
      end if;
    end if;
  end process;

  cnt_read_addr(j) <= cnt_read_addr_high(j) & cnt_read_addr_low(j);

  -- read address register ----------------------------------------------------
  process(CLK)
  begin
    if ((CLK'event) and (CLK = '1')) then
      if (sig_init(j) = '1') then
        reg_read_addr(j) <= (others => '0');
      else
        if (blk_read_allow(j) = '1') then
          reg_read_addr(j) <= cnt_read_addr(j);
        end if;
      end if;
    end if;
  end process;
   
  gen_status_yes : if ((((FLOWS-UNUSED_EVEN*2) > j) and (j mod 2 = 0)) or
		      (((FLOWS-UNUSED_ODD*2) > j) and (j mod 2 = 1))) generate
  blk_status_sigs : entity work.BUF_STATUS
    generic map (
      ITEMS       => BLOCK_SIZE,
      MULTI_WRITE => false,
      MULTI_READ  => false
    ) 
    port map (
      CLK      => CLK,
      RESET    => sig_init(j),

      WRITE_EN => blk_write_allow(j),
      READ_EN  => blk_read_allow(j),
      WR_CNT   => cnt_write_addr(j),
      WR_REG   => reg_write_addr(j),
      RD_CNT   => cnt_read_addr(j),
      RD_REG   => reg_read_addr(j),

      EMPTY    => blk_empty(j),
      FULL     => blk_full(j),
      STATUS   => blk_status(j)
    );
   end generate;

  gen_status_no : if not ((((FLOWS-UNUSED_EVEN*2) > j) and (j mod 2 = 0)) or
		      (((FLOWS-UNUSED_ODD*2) > j) and (j mod 2 = 1)))  generate
      blk_empty(j) <= '1';
      blk_full(j) <= '1';
      blk_status(j) <= (others => '0');
      
  end generate;



  sig_status((j+1)*log2(BLOCK_SIZE+1)-1 downto j*log2(BLOCK_SIZE+1)) <= blk_status(j);
  addr_fix(j) <= ADDR_FIX_CONST(j);

end generate;

buf_mem_i : entity work.BUF_MEM
  generic map (
    LUT_MEMORY => LUT_MEMORY,
    DATA_WIDTH => DATA_WIDTH,
    ITEMS      => BLOCK_SIZE*FLOWS,
    OUTPUT_REG => OUTPUT_REG
  ) 
  port map (
    CLK      => CLK,
    RESET    => RESET,

    -- Write interface
    WR_ADDR  => write_addr,
    DATA_IN  => DATA_IN,
    WRITE    => write_allow,

    -- Read interface
    RD_ADDR  => read_addr,
    DATA_OUT => DATA_OUT,
    DATA_VLD => DATA_VLD,
    READ     => read_allow,
    PIPE_EN  => PIPE_EN
  );

-- ----------------------------------------------------------------------------
-- interface mapping
FULL   <= blk_full;
EMPTY  <= blk_empty;
STATUS <= sig_status;

end architecture;
-- ----------------------------------------------------------------------------
