-- flu_prfifo.vhd: Asynchronous packet buffer
-- from BlockRAM memories with RX discarding.
-- Both sides addressed as FIFO, supports unaligned SOPs.
-- Copyright (C) 2012 CESNET
-- Author(s): Viktor Puš <pus@cesnet.cz>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--
-- $Id$

library ieee;
use ieee.std_logic_1164.all;
use ieee.std_logic_unsigned.all;
use IEEE.std_logic_unsigned.all;
use IEEE.std_logic_arith.all;

-- library containing log2 function
use work.math_pack.all;

-- auxilarity function needed to compute address width
use WORK.bmem_func.all;

-- pragma translate_off
library UNISIM;
use UNISIM.vcomponents.all;
-- pragma translate_on
--
-- ----------------------------------------------------------------------------
--                      Architecture declaration
-- ----------------------------------------------------------------------------
architecture full of flu_prfifo is

   --* Number of word address bits
   constant FADDR             : integer := LOG2(ITEMS);
   --* Width of EOP_POS (bits)
   constant EOP_POS_W         : integer := log2(DATA_WIDTH/8);
   --* Width of the main memory (bits)
   constant MEMW              : integer := DATA_WIDTH+EOP_POS_W+SOP_POS_W+2;

   signal sig_di              : std_logic_vector(MEMW-1 downto 0);
   signal sig_do              : std_logic_vector(MEMW-1 downto 0);
   signal sig_rd              : std_logic;
   signal sig_wr              : std_logic;
   signal sig_release         : std_logic;
   signal sig_do_dv           : std_logic;
   signal sig_full            : std_logic;
   signal sig_empty           : std_logic;
   signal sig_mark            : std_logic;
   signal rx_eop_wr           : std_logic;
   signal rx_sop_wr           : std_logic;

   signal sig_do_sop          : std_logic;
   signal sig_do_eop          : std_logic;
   signal sig_do_sop_pos      : std_logic_vector(SOP_POS_W-1 downto 0);
   signal sig_do_eop_pos      : std_logic_vector(EOP_POS_W-1 downto 0);
   signal sig_do_sop_pos_exp  : std_logic_vector(EOP_POS_W-1 downto 0);
   signal sig_do_shared       : std_logic;

   signal writing_eop         : std_logic;
   signal writing_sop         : std_logic;
   signal writing_split       : std_logic;
   signal writing_split_true  : std_logic;
   signal sop_pos_exp         : std_logic_vector(EOP_POS_W-1 downto 0);
   signal reg_in_packet       : std_logic;

   signal sig_rx_dst_rdy      : std_logic;
   signal sig_tx_src_rdy      : std_logic;

   signal reg_delay_mark      : std_logic;
   signal reg_delay_mark_set  : std_logic;
   signal reg_delay_mark_clr  : std_logic;

   signal reg_wr_helper       : std_logic;
   signal reg_wr_helper_set   : std_logic;
   signal reg_wr_helper_clr   : std_logic;

   signal need_helper         : std_logic;

   signal sig_hfifo_full      : std_logic;
   signal sig_hfifo_wr        : std_logic;
   signal sig_hfifo_di        : std_logic_vector(0 downto 0);
   signal sig_hfifo_do        : std_logic_vector(0 downto 0);
   signal sig_hfifo_rd        : std_logic;
   signal sig_hfifo_empty     : std_logic;

   signal gnd                 : std_logic;
   signal gndvec              : std_logic_vector(DATA_WIDTH-1 downto 0);
   signal pwr                 : std_logic;

   --* Stupid type conversion, please ignore
   function bool_to_toutput_reg(use_outreg : boolean) return toutput_reg is
   begin
      if use_outreg = true then return true;
      else return false;
      end if;
   end bool_to_toutput_reg;

begin

   -- Checking assertions
   assert ((DATA_WIDTH mod (2**SOP_POS_W)) = 0)
   report "Packet buffer: Invalid generics."
   severity error;

   assert (((DATA_WIDTH/(2**SOP_POS_W)) mod 8) = 0)
   report "Packet buffer: Invalid generics."
   severity error;

   gnd      <= '0';
   gndvec   <= (others => '0');
   pwr      <= '1';

   writing_eop <= RX_SRC_RDY and sig_rx_dst_rdy and RX_EOP;
   writing_sop <= RX_SRC_RDY and sig_rx_dst_rdy and RX_SOP;

   sop_pos_exp <= RX_SOP_POS & gndvec(EOP_POS_W-1 downto SOP_POS_W) ;
   writing_split <= '1' when sop_pos_exp > RX_EOP_POS else '0';
   writing_split_true <= writing_split and writing_sop and writing_eop;

   --* Keep state of the input protocol.
   reg_in_packet_p : process(RX_CLK)
   begin
      if RX_CLK'event and RX_CLK = '1' then
         if RX_RESET = '1' then
            reg_in_packet <= '0';
         else
            if writing_sop='1' or writing_eop='1' or RX_RELEASE='1' then
               reg_in_packet <= writing_sop and 
                                (writing_split or not writing_eop);
            end if;
         end if;
      end if;
   end process;

   --* Set to 1 if the mark signal has to be delayed by one word
   reg_delay_mark_p : process(RX_CLK)
   begin
      if RX_CLK'event and RX_CLK = '1' then
         if RX_RESET = '1' then
            reg_delay_mark <= '0';
         else
            if reg_delay_mark_set = '1' then
               reg_delay_mark <= '1';
            elsif reg_delay_mark_clr = '1' then
               reg_delay_mark <= '0';
            end if;
         end if;
      end if;
   end process;

   reg_delay_mark_set <= writing_split_true and not RX_RELEASE;
   reg_delay_mark_clr <= reg_delay_mark and RX_SRC_RDY and sig_rx_dst_rdy;

   --* Stores data + sop/eop(_pos)
   data_fifo : entity work.asfifo_bram_release_prfifo
   generic map(
      ITEMS        => ITEMS,
      DATA_WIDTH   => MEMW,
      STATUS_WIDTH => STATUS_WIDTH,
      AUTO_PIPELINE=> true)
   port map(
      CLK_WR      => RX_CLK,
      RESET_WR    => RX_RESET,
      WR          => sig_wr,
      DI          => sig_di,
      FULL        => sig_full,
      STATUS      => STATUS,
      MARK        => sig_mark,
      RELEASE     => sig_release,
      RELEASE_WR  => writing_split_true,

      CLK_RD      => TX_CLK,
      RESET_RD    => TX_RESET,
      RD          => sig_rd,
      DO          => sig_do,
      DO_DV       => sig_do_dv,
      EMPTY       => sig_empty
   );

   -- Write if RX_SRC_RDY, but not when
   -- writing single-word transaction or sop-only word while being discarded
   -- and also not when writing the discarded second word while the first was 
   -- shared
   sig_wr <= RX_SRC_RDY and sig_rx_dst_rdy 
             and (reg_in_packet or writing_sop)
             and not(RX_RELEASE and RX_sop and 
               ((RX_eop and not writing_split) or not RX_eop))
             and not (reg_delay_mark and RX_RELEASE and not rx_sop);

   -- Release if RX_RELEASE, but not when
   -- having single-word transaction or sop-only word or 
   -- second word while the first was shared,
   -- because it is not written instead
   sig_release <= RX_RELEASE 
                  and (reg_in_packet or writing_sop)
                  and not (RX_SRC_RDY and sig_rx_dst_rdy and RX_sop and
               ((RX_eop and not writing_split) or not RX_eop))
                  and not reg_delay_mark;

   sig_di <= RX_SOP_POS & RX_EOP_POS & rx_sop_wr & rx_eop_wr & RX_DATA;
   rx_sop_wr <= RX_SOP;-- when RX_RELEASE = '0' else '0';
   rx_eop_wr <= RX_EOP when RX_RELEASE = '0' else '0';

   sig_mark <= writing_sop or reg_delay_mark_clr
               or not reg_in_packet;

   RX_DST_RDY <= sig_rx_dst_rdy;
   sig_rx_dst_rdy <= (not sig_full) and (not sig_hfifo_full);

   sig_rd <= sig_tx_src_rdy and TX_DST_RDY;
   --sig_rd <= TX_DST_RDY;

   -- We need help when we read shared word from fifo
--   need_helper <= TX_DST_RDY and sig_do_dv and 
--                  sig_do_sop and sig_do_eop and sig_do_shared;

   need_helper <= sig_do_dv and 
                  sig_do_sop and sig_do_eop and sig_do_shared;

   TX_DATA     <= sig_do(DATA_WIDTH-1 downto 0);
   TX_EOP      <= sig_do_eop;
   TX_EOP_POS  <= sig_do_eop_pos;
   TX_SOP_POS  <= sig_do_sop_pos;

   sig_do_eop     <= sig_do(DATA_WIDTH);
   sig_do_eop_pos <= sig_do(DATA_WIDTH+1+EOP_POS_W downto DATA_WIDTH+2);
   sig_do_sop_pos <= sig_do(MEMW-1 downto MEMW-SOP_POS_W);
   sig_do_sop     <= sig_do(DATA_WIDTH+1);
   sig_do_sop_pos_exp <= sig_do_sop_pos & gndvec(EOP_POS_W-1 downto SOP_POS_W);
   sig_do_shared  <= '1' when sig_do_sop='1' and sig_do_eop='1' and
                              sig_do_sop_pos_exp > sig_do_eop_pos
                     else '0';

   -- SOP is sent only if this is not shared word or it is shared
   -- and the packet was not discarded at the input.
   TX_SOP      <= sig_do(DATA_WIDTH+1) when
                     need_helper = '0' or 
                     (need_helper = '1' and sig_hfifo_do(0) = '1')
                  else '0';

   sig_tx_src_rdy  <= sig_do_dv when
                     need_helper = '0' or
                     (need_helper = '1' and sig_hfifo_empty = '0')
                  else '0';

   TX_SRC_RDY <= sig_tx_src_rdy;
   
   --* Stores 1 at the successful end of a packet that shared its first word
   --* with the previous one. Stores 0 when such packet is discarded.
   helper_fifo : entity work.asfifo
   generic map(
      DATA_WIDTH  => 1,
      ITEMS       => HELPER_FIFO_ITEMS,
      STATUS_WIDTH=> 1)
   port map(
      CLK_WR      => RX_CLK,
      RST_WR      => RX_RESET,
      DI          => sig_hfifo_di,
      WR          => sig_hfifo_wr,
      FULL        => sig_hfifo_full,
      STATUS      => open,

      CLK_RD      => TX_CLK,
      RST_RD      => TX_RESET,
      DO          => sig_hfifo_do,
      RD          => sig_hfifo_rd,
      EMPTY       => sig_hfifo_empty
   );

   sig_hfifo_wr <= reg_wr_helper and (RX_RELEASE or writing_eop);
   sig_hfifo_di(0) <= not RX_RELEASE;
   sig_hfifo_rd <= need_helper and TX_DST_RDY;

   --* Set if we will need to store value into helper fifo,
   --* clear when we do.
   reg_wr_helper_p : process(RX_CLK)
   begin
      if RX_CLK'event and RX_CLK = '1' then
         if RX_RESET = '1' then
            reg_wr_helper <= '0';
         else
            if reg_wr_helper_set = '1' then
               reg_wr_helper <= '1';
            elsif reg_wr_helper_clr = '1' then
               reg_wr_helper <= '0';
            end if;
         end if;
      end if;
   end process;

   reg_wr_helper_set <= writing_split_true and rx_eop_wr and sig_wr;
   reg_wr_helper_clr <= sig_hfifo_wr;

end full;
