-- xlgmii_dec_fsm.vhd: FSM for CGMII_dec unit of CGMII Input buffer
-- Copyright (C) 2012 CESNET
-- Author(s): Jan Kucera <xkucer73@stud.fit.vutbr.cz>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--
-- $Id$
--
-- TODO:
--

library IEEE;
use IEEE.std_logic_1164.all;

-- ----------------------------------------------------------------------------
--                            Entity declaration
-- ----------------------------------------------------------------------------

entity cgmii_dec_fsm is
   generic(
      -- Synthesise design without resets, default values of registers are set
      -- by initialization of signals
      RESET_BY_INIT  : boolean := false
   );
   port(
      -- Common signals
      CLK            : in std_logic;
      RESET          : in std_logic;

      -- Inputs
      WP_IN          : in std_logic;
      SOP_IN         : in std_logic;
      EOP_IN         : in std_logic;
      SOP_ERR_IN     : in std_logic;
      EOP_ERR_IN     : in std_logic;
      CTRL_ERR_IN    : in std_logic;
      DATA_ERR_IN    : in std_logic;

      -- Outputs
      SOP_OUT        : out std_logic;
      EOP_OUT        : out std_logic;
      ERR_OUT        : out std_logic
   );
end entity cgmii_dec_fsm;


-- ----------------------------------------------------------------------------
--                               Architecture
-- ----------------------------------------------------------------------------

architecture cgmii_dec_fsm_arch of cgmii_dec_fsm is

   -- Type definition ---------------------------------------------------------
   type fsm_states is (st_ready, st_packet, st_error);

   -- Signals declaration -----------------------------------------------------
   signal curr_state             : fsm_states := st_ready;
   signal next_state             : fsm_states;

   signal err                    : std_logic;

begin

   err <= SOP_ERR_IN or EOP_ERR_IN or CTRL_ERR_IN or WP_IN or DATA_ERR_IN;

   -- -------------------------------------------------------------------------

   current_state_logic : process(CLK)
   begin
      if (CLK'event and CLK = '1') then
         if (not RESET_BY_INIT and RESET = '1') then
            curr_state <= st_ready;
         else
            curr_state <= next_state;
         end if;
      end if;
   end process current_state_logic;

   -- -------------------------------------------------------------------------

   next_state_logic : process(curr_state, SOP_IN, EOP_IN, err, SOP_ERR_IN, WP_IN, DATA_ERR_IN, EOP_ERR_IN, CTRL_ERR_IN)
   begin
      case (curr_state) is

         -- st_ready
         when st_ready =>
            if (SOP_IN = '1' and err = '0') then
               next_state <= st_packet;
            elsif (SOP_IN = '1' and EOP_IN = '0' and err = '1') then
               if (DATA_ERR_IN = '1') then
                  next_state <= st_ready;
               else
                  next_state <= st_error;
               end if;
            elsif (SOP_IN = '1' and EOP_IN = '1' and err = '1') then
               if (WP_IN = '1' or DATA_ERR_IN = '1') then
                  next_state <= st_ready;
               elsif (WP_IN = '0' and SOP_ERR_IN = '1') then
                  next_state <= st_error;
               else
                  next_state <= st_packet;
               end if;
            else
               next_state <= st_ready;
            end if;

         -- st_packet
         when st_packet =>
            if (SOP_IN = '0' and EOP_IN = '1') then
               next_state <= st_ready;
            elsif ((SOP_IN = '1' and EOP_IN = '0') or (SOP_IN = '0' and EOP_IN = '0' and err = '1')) then
               next_state <= st_error;
            elsif (SOP_IN = '1' and EOP_IN = '1' and err = '1') then
               if (WP_IN = '1' or DATA_ERR_IN = '1') then
                  next_state <= st_ready;
               elsif (WP_IN = '0' and SOP_ERR_IN = '1') then
                  next_state <= st_error;
               else
                  next_state <= st_packet;
               end if;
            else
               next_state <= st_packet;
            end if;

         -- st_error
         when st_error =>
            if (SOP_IN = '0' and EOP_IN = '1') then
               next_state <= st_ready;
            elsif (SOP_IN = '1' and EOP_IN = '1' and err = '0') then
               next_state <= st_packet;
            elsif (SOP_IN = '1' and EOP_IN = '1' and err = '1') then
               if (WP_IN = '1' or DATA_ERR_IN = '1') then
                  next_state <= st_ready;
               elsif (WP_IN = '0' and SOP_ERR_IN = '1') then
                  next_state <= st_error;
               else
                  next_state <= st_packet;
               end if;
            else
               next_state <= st_error;
            end if;

         when others =>
            next_state <= st_ready;

      end case;
   end process next_state_logic;

   -- -------------------------------------------------------------------------

   output_logic: process(curr_state, SOP_IN, EOP_IN, err, SOP_ERR_IN, EOP_ERR_IN, WP_IN, DATA_ERR_IN, CTRL_ERR_IN)
   begin
      SOP_OUT <= '0';
      EOP_OUT <= '0';
      ERR_OUT <= '0';

      case (curr_state) is

         -- st_ready
         when st_ready =>
            if (SOP_IN = '1' and err = '0') then
               SOP_OUT <= '1';
            elsif (SOP_IN = '1' and EOP_IN = '0' and err = '1') then
               if (DATA_ERR_IN = '1') then
                  SOP_OUT <= '0';
               else
                  SOP_OUT <= '1';
               end if;
            elsif (SOP_IN = '1' and EOP_IN = '1' and err = '1') then
               if (DATA_ERR_IN = '1') then
                  SOP_OUT <= '0';
                  EOP_OUT <= '0';
               elsif (WP_IN = '1') then
                  if (SOP_ERR_IN = '1' or EOP_ERR_IN = '1' or CTRL_ERR_IN = '1') then
                     SOP_OUT <= '1';
                     EOP_OUT <= '1';
                     ERR_OUT <= '1';
                  else
                     SOP_OUT <= '1';
                     EOP_OUT <= '1';
                     ERR_OUT <= '0';
                  end if;
               elsif (WP_IN = '0' and WP_IN = '1') then
                  SOP_OUT <= '1';
               else
                  SOP_OUT <= '1';
               end if;
            end if;

         -- st_packet
         when st_packet =>
            if (SOP_IN = '0' and EOP_IN = '1') then
               EOP_OUT <= '1';
               ERR_OUT <= err;
            elsif (SOP_IN = '1' and EOP_IN = '0') then
               SOP_OUT <= '0';
            elsif (SOP_IN = '1' and EOP_IN = '1' and err = '1') then
               if (WP_IN = '1' or DATA_ERR_IN = '1') then
                  EOP_OUT <= '1';
                  ERR_OUT <= '1';
               elsif (WP_IN = '0' and SOP_ERR_IN = '1') then
                  SOP_OUT <= '1';
                  EOP_OUT <= '1';
                  ERR_OUT <= EOP_ERR_IN;
               else
                  SOP_OUT <= '1';
                  EOP_OUT <= '1';
                  ERR_OUT <= '1';
               end if;
            else
               SOP_OUT <= SOP_IN;
               EOP_OUT <= EOP_IN;
            end if;

         -- st_error
         when st_error =>
            if (SOP_IN = '0' and EOP_IN = '1') then
               EOP_OUT <= '1';
               ERR_OUT <= '1';
            elsif (SOP_IN = '1' and EOP_IN = '1' and err = '0') then
               SOP_OUT <= '1';
               EOP_OUT <= '1';
               ERR_OUT <= '1';
            elsif (SOP_IN = '1' and EOP_IN = '1' and err = '1') then
               if (WP_IN = '1' or DATA_ERR_IN = '1') then
                  EOP_OUT <= '1';
                  ERR_OUT <= '1';
               else
                  SOP_OUT <= '1';
                  EOP_OUT <= '1';
                  ERR_OUT <= '1';
               end if;
            end if;

         -- other states
         when others =>

      end case;
   end process output_logic;

end architecture cgmii_dec_fsm_arch;
