-- cgmii_dec.vhd: CGMII_dec unit of CGMII Input buffer
-- Copyright (C) 2012 CESNET
-- Author(s): Jan Kucera <xkucer73@stud.fit.vutbr.cz>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--
-- $Id$
--
-- TODO:
--
--

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_unsigned.all;
use work.cgmii_pkg.all;

-- ----------------------------------------------------------------------------
--                        Architecture declaration
-- ----------------------------------------------------------------------------

architecture cgmii_dec_arch of cgmii_dec is


   -- Signals declaration -----------------------------------------------------
   
   -- Input signals
   signal rxd_in                   : std_logic_vector(511 downto 0);
   signal rxc_in                   : std_logic_vector(63 downto 0) := (others => '0');

   -- Control characters and sequences positions
   signal error_pos                : std_logic_vector(63 downto 0); 
   signal start_pos                : std_logic_vector(63 downto 0);    
   signal terminate_pos            : std_logic_vector(63 downto 0);
   signal preamble_pos             : std_logic_vector(7 downto 0);
   signal locfault_pos             : std_logic_vector(7 downto 0);

   -- Presence detection signals
   signal error_occured            : std_logic;
   signal terminate_occured        : std_logic;
   signal preamble_occured         : std_logic;
   signal locfault_occured         : std_logic;

   -- First pipeline stage registers
   signal reg1_rxd                 : std_logic_vector(511 downto 0);
   signal reg1_rxc                 : std_logic_vector(63 downto 0) := (others => '0');
   signal reg1_error_pos           : std_logic_vector(63 downto 0) := (others => '0'); 
   signal reg1_start_pos           : std_logic_vector(63 downto 0) := (others => '0');    
   signal reg1_terminate_pos       : std_logic_vector(63 downto 0) := (others => '0');
   signal reg1_preamble_pos        : std_logic_vector(7 downto 0) := (others => '0');
   signal reg1_locfault_pos        : std_logic_vector(7 downto 0) := (others => '0');
   signal reg1_error_occured       : std_logic := '0';
   signal reg1_terminate_occured   : std_logic := '0';
   signal reg1_preamble_occured    : std_logic := '0';
   signal reg1_locfault_occured    : std_logic := '0';
 
   -- Terminate detection signals  
   signal first_terminate_pos        : std_logic_vector(63 downto 0);
   signal first_terminate_before_pos : std_logic_vector(63 downto 0);
   signal last_terminate_after_pos   : std_logic_vector(63 downto 0);

   -- Second pipeline stage registers
   signal reg2_rxd                         : std_logic_vector(511 downto 0);
   signal reg2_rxc                         : std_logic_vector(63 downto 0) := (others => '0');
   signal reg2_preamble_occured            : std_logic := '0';
   signal reg2_terminate_occured           : std_logic := '0';
   signal reg2_start_pos                   : std_logic_vector(63 downto 0) := (others => '0');  
   signal reg2_first_terminate_pos         : std_logic_vector(63 downto 0) := (others => '0');
   signal reg2_first_terminate_before_pos  : std_logic_vector(63 downto 0) := (others => '0');
   signal reg2_last_terminate_after_pos    : std_logic_vector(63 downto 0) := (others => '0');

   -- Start detection signals  
   signal first_start_pos       : std_logic_vector(63 downto 0);
   signal first_start_after_pos : std_logic_vector(63 downto 0);
   signal first_start_candidate_pos : std_logic_vector(63 downto 0);
   signal first_start_candidate_occured : std_logic;

   -- Third pipeline stage registers
   signal reg3_rxd                        : std_logic_vector(511 downto 0);
   signal reg3_rxc                        : std_logic_vector(63 downto 0) := (others => '0');
   signal reg3_preamble_occured           : std_logic := '0';
   signal reg3_terminate_occured          : std_logic := '0';
   signal reg3_first_terminate_pos        : std_logic_vector(63 downto 0) := (others => '0');
   signal reg3_first_terminate_before_pos : std_logic_vector(63 downto 0) := (others => '0');
   signal reg3_first_start_candidate_occured : std_logic := '0';
   signal reg3_first_start_pos            : std_logic_vector(63 downto 0) := (others => '0');
   signal reg3_first_start_after_pos      : std_logic_vector(63 downto 0) := (others => '0');

   -- SOP, EOP deriving related signals
   signal sop                      : std_logic;
   signal eop                      : std_logic;
   signal sop_next                 : std_logic;
   signal eop_prev                 : std_logic;
   signal ctrl                     : std_logic;
   signal ctrl_after_start         : std_logic;
   signal ctrl_before_terminate    : std_logic;
   signal rol_first_start_pos      : std_logic_vector(63 downto 0);
   signal ror_first_terminate_pos   : std_logic_vector(63 downto 0);
   signal sop_pos_64               : std_logic_vector(5 downto 0);
   signal sop_pos                  : std_logic_vector(2 downto 0);
   signal eop_pos                  : std_logic_vector(5 downto 0);
   signal sop_ok                   : std_logic;
   signal eop_ok                   : std_logic;

   -- Fourth pipeline stage registers
   signal reg4_rxd                 : std_logic_vector(511 downto 0);
   signal reg4_sop                 : std_logic := '0';
   signal reg4_sop_ok              : std_logic;
   signal reg4_sop_pos             : std_logic_vector(2 downto 0);
   signal reg4_sop_next            : std_logic := '0';
   signal reg4_eop                 : std_logic := '0';
   signal reg4_eop_ok              : std_logic;
   signal reg4_eop_pos             : std_logic_vector(5 downto 0);
   signal reg4_eop_prev            : std_logic := '0';
   signal reg4_ctrl                : std_logic := '0';

   -- Extra registers for SOP-related signals
   signal reg_sop                  : std_logic := '0';
   signal reg_sop_ok               : std_logic;
   signal reg_sop_pos              : std_logic_vector(2 downto 0);
   signal reg_sop_next             : std_logic := '0';

   -- SOP-related multiplexers
   signal mx_sop                   : std_logic;
   signal mx_sop_ok                : std_logic;
   signal mx_sop_pos               : std_logic_vector(2 downto 0);

   -- EOP-related multiplexers
   signal mx_eop                   : std_logic;
   signal mx_eop_ok                : std_logic;
   signal mx_eop_pos               : std_logic_vector(5 downto 0);

   -- FSM input related signals
   signal sop_err                  : std_logic;
   signal eop_err                  : std_logic;
   signal ctrl_err                 : std_logic;
   signal data_err                 : std_logic;
   signal sop_before_eop           : std_logic;
   signal wp                       : std_logic;

   -- Fifth pipeline stage registers
   signal reg5_rxd                 : std_logic_vector(511 downto 0);
   signal reg5_sop_err             : std_logic := '0';
   signal reg5_eop_err             : std_logic := '0';
   signal reg5_ctrl_err            : std_logic := '0';
   signal reg5_data_err            : std_logic := '0';
   signal reg5_sop_before_eop      : std_logic := '0';
   signal reg5_wp                  : std_logic := '0';
   signal reg5_sop                 : std_logic := '0';
   signal reg5_eop                 : std_logic := '0';
   signal reg5_sop_pos             : std_logic_vector(2 downto 0);
   signal reg5_eop_pos             : std_logic_vector(5 downto 0);

   -- FSM outputs
   signal fsm_sop                  : std_logic;
   signal fsm_eop                  : std_logic;
   signal fsm_err                  : std_logic;
            
   -- Output pipeline stage registers
   signal reg6_rxd                 : std_logic_vector(511 downto 0);
   signal reg6_sop                 : std_logic := '0';
   signal reg6_sop_pos             : std_logic_vector(2 downto 0);
   signal reg6_eop                 : std_logic := '0';
   signal reg6_eop_pos             : std_logic_vector(5 downto 0);
   signal reg6_err                 : std_logic := '0';

   -- Link state determination signals
   signal link_error             : std_logic;
   signal cnt_link_error_timeout : std_logic_vector(CNT_ERROR_LENGTH - 1 downto 0) := (others => '0');
   signal reg_link_up            : std_logic := '0';
   signal reg_incoming_packet    : std_logic := '0';

begin


   -- -------------------------------------------------------------------------
   --                        Assignment of input ports
   -- -------------------------------------------------------------------------

   rxd_in <= RXD;
   rxc_in <= RXC;


   -- -------------------------------------------------------------------------
   --     Control characters and sequences position and presence detection
   -- -------------------------------------------------------------------------

   -- Decoder of controls and sequences positions
   control_decoder_i : entity work.cgmii_dec_ctrl_dec
   port map(
      -- Input
      RXD         => rxd_in,
      RXC         => rxc_in,
      -- Outputs
      ERROR_POS     => error_pos,
      START_POS     => start_pos,
      TERMINATE_POS => terminate_pos,
      PREAMBLE_POS  => preamble_pos,
      LOCFAULT_POS  => locfault_pos
   );

   -- Detection of the presence
   error_occured <= '1' when (error_pos /= 0) else '0';
   terminate_occured <= '1' when (terminate_pos /= 0) else '0';
   preamble_occured <= '1' when (preamble_pos /= 0) else '0';
   locfault_occured <= '1' when (locfault_pos /= 0) else '0';


   -- -------------------------------------------------------------------------
   --                           First pipeline stage
   -- -------------------------------------------------------------------------

   -- 1st pipeline stage for inputs
   regs1_input_p : process(CLK)
   begin
      if (CLK'event and CLK = '1') then
         if (not RESET_BY_INIT and RESET = '1') then
            reg1_rxc <= (others => '0');
         else
            reg1_rxd <= rxd_in;
            reg1_rxc <= rxc_in;
         end if;
      end if;
   end process regs1_input_p;

   -- 1st for signals related to control positions
   regs1_pos_p : process(CLK)
   begin
      if (CLK'event and CLK = '1') then
         if (not RESET_BY_INIT and RESET = '1') then
            reg1_error_occured      <= '0';
            reg1_terminate_occured  <= '0';
            reg1_preamble_occured   <= '0';
            reg1_locfault_occured   <= '0';
            reg1_error_pos          <= (others => '0');
            reg1_start_pos          <= (others => '0');
            reg1_terminate_pos      <= (others => '0');
            reg1_preamble_pos       <= (others => '0');
            reg1_locfault_pos       <= (others => '0');
         else
            reg1_error_occured      <= error_occured;
            reg1_terminate_occured  <= terminate_occured;
            reg1_preamble_occured   <= preamble_occured;
            reg1_locfault_occured   <= locfault_occured;
            reg1_error_pos          <= error_pos;
            reg1_start_pos          <= start_pos;
            reg1_terminate_pos      <= terminate_pos;
            reg1_preamble_pos       <= preamble_pos;
            reg1_locfault_pos       <= locfault_pos;
         end if;
      end if;
   end process regs1_pos_p;


   -- -------------------------------------------------------------------------
   --                       Decoding Terminate position
   -- -------------------------------------------------------------------------

   -- Decoder of Terminate position
   terminate_priority_decoder_p : entity work.cgmii_dec_pdec
   generic map(
      DATA_WIDTH => 64
   )
   port map(
      -- Input
      DI             => reg1_terminate_pos,
      -- Outputs
      FIRST_POS      => first_terminate_pos,
      FIRST_NEXT_POS	=> open,
      FIRST_PREV_POS => first_terminate_before_pos,
      LAST_POS       => open,
      LAST_NEXT_POS  => last_terminate_after_pos,
      LAST_PREV_POS	=> open
   );


   -- -------------------------------------------------------------------------
   --                          Second pipeline stage
   -- -------------------------------------------------------------------------

   -- 2nd pipeline stage for inputs
   regs2_input_p : process(CLK)
   begin
      if (CLK'event and CLK = '1') then
         if (not RESET_BY_INIT and RESET = '1') then
            reg2_rxc <= (others => '0');
         else
            reg2_rxd <= reg1_rxd;
            reg2_rxc <= reg1_rxc;
         end if;
      end if;
   end process regs2_input_p;

   -- 2nd pipeline stage for signals related to control positions
   regs2_pos_p : process(CLK)
   begin
      if (CLK'event and CLK = '1') then
         if (not RESET_BY_INIT and RESET = '1') then
            reg2_preamble_occured            <= '0';
            reg2_terminate_occured           <= '0';
            reg2_start_pos                   <= (others => '0');
            reg2_first_terminate_pos         <= (others => '0');
            reg2_first_terminate_before_pos  <= (others => '0');
            reg2_last_terminate_after_pos    <= (others => '0');
         else
            reg2_preamble_occured            <= reg1_preamble_occured;
            reg2_terminate_occured           <= reg1_terminate_occured;
            reg2_start_pos                   <= reg1_start_pos;
            reg2_first_terminate_pos         <= first_terminate_pos;
            reg2_first_terminate_before_pos  <= first_terminate_before_pos;
            reg2_last_terminate_after_pos    <= last_terminate_after_pos;
         end if;
      end if;
   end process regs2_pos_p;


   -- -------------------------------------------------------------------------
   --                          Decoding Start position
   -- -------------------------------------------------------------------------

   -- First start candidates positions (positions after last terminate)
   first_start_candidate_pos <= reg2_start_pos when (reg2_terminate_occured = '0') else
      reg2_start_pos and reg2_last_terminate_after_pos;

   -- First start candidate presence
   first_start_candidate_occured <= '1' when (first_start_candidate_pos /= 0) else '0';

   -- Decoder of Start position
   start_priority_decoder_p : entity work.cgmii_dec_pdec
   generic map(
      DATA_WIDTH => 64
   )
   port map(
      -- Input
      DI             => first_start_candidate_pos,
      -- Outputs
      FIRST_POS      => first_start_pos,
      FIRST_NEXT_POS	=> first_start_after_pos,
      FIRST_PREV_POS => open,
      LAST_POS       => open,
      LAST_NEXT_POS  => open,
      LAST_PREV_POS	=> open
   );


   -- -------------------------------------------------------------------------
   --                           3rd pipeline stage
   -- -------------------------------------------------------------------------

   -- 3rd pipeline stage for inputs
   regs3_input_p : process(CLK)
   begin
      if (CLK'event and CLK = '1') then
         if (not RESET_BY_INIT and RESET = '1') then
            reg3_rxc <= (others => '0');
         else
            reg3_rxd <= reg2_rxd;
            reg3_rxc <= reg2_rxc;
         end if;
      end if;
   end process regs3_input_p;

   -- 3rd pipeline stage for signals related to control position
   regs3_control_pos_p : process(CLK)
   begin
      if (CLK'event and CLK = '1') then
         if (not RESET_BY_INIT and RESET = '1') then
            reg3_preamble_occured           <= '0';
            reg3_terminate_occured          <= '0';
            reg3_first_terminate_pos        <= (others => '0');
            reg3_first_terminate_before_pos <= (others => '0');
            reg3_first_start_candidate_occured <= '0';
            reg3_first_start_pos            <= (others => '0');
            reg3_first_start_after_pos      <= (others => '0');
         else
            reg3_preamble_occured           <= reg2_preamble_occured;
            reg3_terminate_occured          <= reg2_terminate_occured;
            reg3_first_terminate_pos        <= reg2_first_terminate_pos;
            reg3_first_terminate_before_pos <= reg2_first_terminate_before_pos;
            reg3_first_start_candidate_occured <= first_start_candidate_occured;
            reg3_first_start_pos            <= first_start_pos;
            reg3_first_start_after_pos      <= first_start_after_pos;
         end if;
      end if;
   end process regs3_control_pos_p;


   -- -------------------------------------------------------------------------
   --         Deriving SOP, SOP_POS, EOP, EOP_POS and related signals
   -- -------------------------------------------------------------------------

   -- Active when at least one Start control character occurs
   sop <= '1' when (reg3_first_start_candidate_occured = '1') else '0';

   -- Active when at least one Terminate control character occurs
   eop <= '1' when (reg3_terminate_occured = '1') else '0';

   -- Indication that SOP will be asserted in the next clock cycle
   sop_next <= reg3_first_start_pos(56);
  
   -- Indication that EOP will be asserted in the previous clock cycle
   eop_prev <= reg3_first_terminate_pos(0);

   -- Active when at least one control character occurs
   ctrl <= '1' when (reg3_rxc /= 0) else '0';

   -- Detection whether there are some control characters after Start
   ctrl_after_start <= '1' when ((reg3_first_start_after_pos and reg3_rxc) /= 0) else '0';

   -- Detection whether there are some control characters before Terminate
   ctrl_before_terminate <= '1' when ((reg3_first_terminate_before_pos and reg3_rxc) /= 0) else '0';

   -- Rotate Start position to the left by 8 (jump over the Preamble)
   rol_first_start_pos <= reg3_first_start_pos(55 downto 0) & 
                          reg3_first_start_pos(63 downto 56);   

   -- Rotate Terminate position to the right by 1 (truncate the Terminate)
   ror_first_terminate_pos <= reg3_first_terminate_pos(0) & 
                              reg3_first_terminate_pos(63 downto 1);

   -- SOP position encoder (64 possible values)
   sop_pos_enc_i : entity work.gen_enc
   generic map (
      ITEMS => 64
   )
   port map (
      DI    => rol_first_start_pos,
      ADDR  => sop_pos_64
   );

   -- Transformation of sop_pos_64 to 3-bit value (8 possible values)
   sop_pos <= sop_pos_64(5 downto 3);

   -- EOP position encoder (64 possible values)
   eop_pos_enc_i : entity work.gen_enc
   generic map (
      ITEMS => 64
   )
   port map (
      DI    => ror_first_terminate_pos,
      ADDR  => eop_pos
   );

   -- Active when there are no control characters in the range given by
   -- Start and Terminate or Start and the end of the word and least one
   -- Preamble pattern occurs
   sop_ok <= '1' when (ctrl_after_start = '0' and reg3_preamble_occured = '1') else '0';

   -- Active when there are no control characters in the range given by
   -- Start and Terminate or Start of the word and Terminate
   eop_ok <= '1' when ctrl_before_terminate = '0' else '0';
 

   -- -------------------------------------------------------------------------
   --                          4th pipeline stage
   -- -------------------------------------------------------------------------

   regs4_p : process(CLK)
   begin
      if (CLK'event and CLK = '1') then
         if (not RESET_BY_INIT and RESET = '1') then 
            reg4_sop                 <= '0';
            reg4_eop                 <= '0';
            reg4_sop_next            <= '0';
            reg4_eop_prev            <= '0';
            reg4_ctrl                <= '0';
         else
            reg4_rxd                 <= reg3_rxd;
            reg4_sop                 <= sop;
            reg4_eop                 <= eop;
            reg4_sop_next            <= sop_next;
            reg4_eop_prev            <= eop_prev;
            reg4_sop_pos             <= sop_pos;
            reg4_eop_pos             <= eop_pos;
            reg4_sop_ok              <= sop_ok;
            reg4_eop_ok              <= eop_ok;
            reg4_ctrl                <= ctrl;
         end if;
      end if;
   end process regs4_p;


   -- -------------------------------------------------------------------------
   --           Additional pipeline stage for SOP-related signals
   -- -------------------------------------------------------------------------

   regs_sop_p : process(CLK)
   begin
      if (CLK'event and CLK = '1') then
         if (not RESET_BY_INIT and RESET = '1') then
            reg_sop                 <= '0';
            reg_sop_next            <= '0';
         else
            reg_sop                 <= reg4_sop;
            reg_sop_ok              <= reg4_sop_ok;
            reg_sop_pos             <= reg4_sop_pos;
            reg_sop_next            <= reg4_sop_next;
         end if;
      end if;
   end process regs_sop_p;


   -- -------------------------------------------------------------------------
   --           Multiplexers for SOP- and EOP-related values
   -- -------------------------------------------------------------------------
 
   -- SOP-related multiplexers ------------------------------------------------
   mx_sop <= '0' when (reg4_sop_next = '1') else
             reg_sop when (reg_sop_next = '1' and reg4_sop = '0') else
             reg4_sop when (eop_prev = '0' or reg4_eop = '0') else
             '0';

   mx_sop_ok <= '0' when (reg4_sop_next = '1') else
                reg_sop_ok when (reg_sop_next = '1' and reg4_sop = '0') else
                reg4_sop_ok;

   mx_sop_pos <= (others => '0') when (reg4_sop_next = '1') else
                 reg_sop_pos when (reg_sop_next = '1' and reg4_sop = '0') else
                 reg4_sop_pos;

   -- EOP-related multiplexers ------------------------------------------------
   mx_eop <= '0' when (reg4_eop_prev = '1') else
             eop when (eop_prev = '1' and reg4_eop = '0') else
             reg4_eop;

   mx_eop_ok <= '0' when (reg4_eop_prev = '1') else
                eop_ok when (eop_prev = '1' and reg4_eop = '0') else
                reg4_eop_ok;

   mx_eop_pos <= (others => '0') when (reg4_eop_prev = '1') else
                 eop_pos when (eop_prev = '1' and reg4_eop = '0') else
                 reg4_eop_pos;


   -- -------------------------------------------------------------------------
   --                      Error signals derivation
   -- -------------------------------------------------------------------------

   -- Erroneous SOP
   sop_err <= mx_sop and (not mx_sop_ok);

   -- Erroneous EOP
   eop_err <= mx_eop and (not mx_eop_ok);

   -- Control characters other than Start or Terminate occured
   ctrl_err <= (not reg4_sop) and (not reg4_eop) and (reg4_ctrl);

   -- Active when there are no data between Start and Terminate
   data_err <= (reg_sop_next and reg4_eop_prev) or (reg4_sop_next and eop_prev);  

   -- SOP is before EOP, position of SOP is smaller than of EOP
   sop_before_eop <= '1' when ((mx_sop_pos & "000") < mx_eop_pos) else '0';

   -- Active when there is a whole packet in one word
   wp <= mx_sop and mx_eop and sop_before_eop;


   -- -------------------------------------------------------------------------
   --                          5th pipeline stage
   -- -------------------------------------------------------------------------

   regs5_p : process(CLK)
   begin
      if (CLK'event and CLK = '1') then
         if (not RESET_BY_INIT and RESET = '1') then 
            reg5_wp                  <= '0';
            reg5_sop                 <= '0';
            reg5_eop                 <= '0';
            reg5_sop_err             <= '0';
            reg5_eop_err             <= '0';
            reg5_ctrl_err            <= '0';
            reg5_data_err            <= '0';
         else
            reg5_rxd                 <= reg4_rxd;
            reg5_wp                  <= wp;
            reg5_sop                 <= mx_sop;
            reg5_eop                 <= mx_eop;
            reg5_sop_pos             <= mx_sop_pos;
            reg5_eop_pos             <= mx_eop_pos;
            reg5_sop_err             <= sop_err;
            reg5_eop_err             <= eop_err;
            reg5_ctrl_err            <= ctrl_err;
            reg5_data_err            <= data_err;
         end if;
      end if;
   end process regs5_p;


   -- -------------------------------------------------------------------------
   --                                    FSM
   -- -------------------------------------------------------------------------
   
   -- FSM instantiation
   cgmii_dec_fsm_i : entity work.cgmii_dec_fsm
   generic map(
      RESET_BY_INIT  => RESET_BY_INIT
   )
   port map(
      CLK            => CLK,
      RESET          => RESET,

      -- Inputs
      WP_IN          => reg5_wp,
      SOP_IN         => reg5_sop,
      EOP_IN         => reg5_eop,  

      SOP_ERR_IN     => reg5_sop_err,
      EOP_ERR_IN     => reg5_eop_err,
      CTRL_ERR_IN    => reg5_ctrl_err,
      DATA_ERR_IN    => reg5_data_err,
    
      -- Outputs
      SOP_OUT        => fsm_sop,
      EOP_OUT        => fsm_eop,
      ERR_OUT        => fsm_err
   );   


   -- -------------------------------------------------------------------------
   --                            6th pipeline stage
   -- -------------------------------------------------------------------------
   
   regs6_p : process(CLK)
   begin
      if (CLK'event and CLK = '1') then
         if (not RESET_BY_INIT and RESET = '1') then
            reg6_sop            <= '0';
            reg6_eop            <= '0';
            reg6_err            <= '0';
         else
            reg6_rxd            <= reg5_rxd;
            reg6_sop            <= fsm_sop;
            reg6_sop_pos        <= reg5_sop_pos;
            reg6_eop            <= fsm_eop;
            reg6_eop_pos        <= reg5_eop_pos;
            reg6_err            <= fsm_err;
         end if;
      end if;
   end process regs6_p;


   -- -------------------------------------------------------------------------
   --                        Link state determination
   -- -------------------------------------------------------------------------

   link_error <= error_occured or locfault_occured;

   -- link error timeout counter
   cnt_link_error_timeout_p : process(CLK)
   begin
      if  (CLK'event and CLK = '1') then
         if (not RESET_BY_INIT and RESET = '1') then
            cnt_link_error_timeout <= (others => '0');
         elsif (link_error = '1') then
            cnt_link_error_timeout <= (others => '1');
         elsif (terminate_occured = '1') then
            cnt_link_error_timeout <= (others => '0');
         elsif (cnt_link_error_timeout /= 0) then
            cnt_link_error_timeout <= cnt_link_error_timeout - 1;
         end if;
      end if;
   end process cnt_link_error_timeout_p;
   
   -- link_up register
   reg_link_up_p : process(CLK)
   begin
      if (CLK'event and CLK = '1') then
         if (not RESET_BY_INIT and RESET = '1') then
            reg_link_up <= '0';
         elsif (cnt_link_error_timeout /= 0) then
            reg_link_up <= '0';
         else
            reg_link_up <= '1';
         end if;
      end if;
   end process reg_link_up_p;

   -- reg_incoming_packet register
   reg_incoming_packet_p : process(CLK)
   begin
      if (CLK'event and CLK = '1') then
         if (not RESET_BY_INIT and RESET = '1') then
            reg_incoming_packet <= '0';
         elsif (fsm_sop = '1') then
            reg_incoming_packet <= '1';
         elsif (reg6_eop = '1') then
            reg_incoming_packet <= '0';
         end if;
      end if;
   end process reg_incoming_packet_p;


   -- -------------------------------------------------------------------------
   --                        Dealing with output ports
   -- -------------------------------------------------------------------------

   -- State of the link -------------------------------------------------------
   -- Active when link is up
   LINK_UP           <= reg_link_up;
   -- Active when a packet is being received
   INCOMING_PACKET   <= reg_incoming_packet;

   -- FLU-like output ---------------------------------------------------------
   -- Packet data   
   TX_DATA           <= reg6_rxd;
   -- Start of the packet, active in '1'
   TX_SOP            <= reg6_sop;
   -- Position of the start of the packet, valid only if SOP is set to '1'.
   TX_SOP_POS        <= reg6_sop_pos;
   -- End of the packet, active in '1'.
   TX_EOP            <= reg6_eop;
   -- Position of the end of the packet, valid only if EOP is set to '1'.
   TX_EOP_POS        <= reg6_eop_pos;
   -- Error inside the packet was detected, active in '1'.
   TX_ERR            <= reg6_err;

   
end architecture cgmii_dec_arch;
