-- tree.vhd: Binary search tree over prefixes 
-- Copyright (C) 2013 Brno University of Technology
-- Author(s): Lukas Kekely <ikekely@fit.vutbr.cz>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--
-- $Id: tree.vhd 4531 2013-12-20 16:29:12Z xkekel00 $
--

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_arith.all;
use IEEE.std_logic_unsigned.all;
use IEEE.numeric_std.all;
use WORK.math_pack.all;
use WORK.tree_func.all;

-- ----------------------------------------------------------------------------
--                      Architecture declaration
-- ----------------------------------------------------------------------------
architecture arch of tree is   
  type tree_stage_interconnect_t is record
    key   : std_logic_vector(KEY_WIDTH-1 downto 0);
    addr  : std_logic_vector(TREE_STAGES-1 downto 0);
    vld   : std_logic;
    foundable : std_logic;
    chen  : std_logic;
  end record;
  type tree_stage_interconnect_array_t is array (0 to TREE_STAGES) of tree_stage_interconnect_t;
  signal stage_connect : tree_stage_interconnect_array_t; 

  type tree_stage_config_t is record
    di    : std_logic_vector(1+KEY_WIDTH+1+log2(KEY_WIDTH)-1 downto 0);
    do    : std_logic_vector(1+KEY_WIDTH+1+log2(KEY_WIDTH)-1 downto 0);
    drdy  : std_logic;
    addr  : std_logic_vector(TREE_STAGES-1 downto 0);
    we    : std_logic;
    en    : std_logic;
  end record;
  type tree_stage_config_array_t is array (0 to TREE_STAGES-1) of tree_stage_config_t;
  signal stage_config : tree_stage_config_array_t;
  signal stage_config_di       : std_logic_vector(TREE_STAGES*(KEY_WIDTH+log2(KEY_WIDTH)+2)-1 downto 0);
  signal stage_config_do       : std_logic_vector(TREE_STAGES*(KEY_WIDTH+log2(KEY_WIDTH)+2)-1 downto 0);
  signal stage_config_drdy     : std_logic_vector(TREE_STAGES-1 downto 0);
  signal stage_config_addr     : std_logic_vector(TREE_STAGES*TREE_STAGES-1 downto 0);
  signal stage_config_we       : std_logic_vector(TREE_STAGES-1 downto 0); 
  signal stage_config_en       : std_logic_vector(TREE_STAGES-1 downto 0);
  
  type tree_leaf_config_t is record
    di    : std_logic_vector(DATA_WIDTH+log2(KEY_WIDTH) downto 0);
    do    : std_logic_vector(DATA_WIDTH+log2(KEY_WIDTH) downto 0);
    drdy  : std_logic;
    addr  : std_logic_vector(TREE_STAGES-1 downto 0);
    we    : std_logic;
    en    : std_logic;
  end record;
  signal leaf_config : tree_leaf_config_t;
     
  signal fix_key_start : std_logic_vector(KEY_WIDTH-1 downto 0); 
  signal fix_key_end   : std_logic_vector(KEY_WIDTH-1 downto 0);
  signal fix_len       : std_logic_vector(log2(KEY_WIDTH)-1 downto 0);
  signal fix_datalen   : std_logic_vector(log2(KEY_WIDTH)-1 downto 0);
  signal fix_data      : std_logic_vector(DATA_WIDTH-1 downto 0);
  signal fix_found     : std_logic;
  signal fix_active    : std_logic;
  signal fix_is_add    : std_logic;
  signal fix_activate  : std_logic;
  
begin
  -- connection of tree pipeline start
  stage_connect(0).key     <= IN_KEY;
  stage_connect(0).vld     <= IN_VLD;
  stage_connect(0).foundable <= IN_FOUNDABLE;
  stage_connect(0).addr(0) <= '0';
  stage_connect(0).chen    <= IN_CHANGE_EN;
  
  -- not-leaf stages of the search tree
  tree_stages_gen : for i in 0 to TREE_STAGES-1 generate
    tree_single_stage_gen : if USE_STAGES(i)=STAGE_TYPE_BMEM or USE_STAGES(i)=STAGE_TYPE_REGARRAY or USE_STAGES(i)=STAGE_TYPE_DISTMEM generate 
      single_stage : entity work.tree_stage
        generic map (
          TREE_LEVEL     => i,
          KEY_WIDTH      => KEY_WIDTH,
          MEM_TYPE       => STAGE2MEMORY(USE_STAGES(i)),
          INPUT_REG      => LOGIC2BOOLEAN(USE_REGS(i*3+0)),
          MIDDLE_REG     => LOGIC2BOOLEAN(USE_REGS(i*3+1)), 
          OUTPUT_REG     => LOGIC2BOOLEAN(USE_REGS(i*3+2))
        ) port map(
          CLK           => CLK,
          RESET         => RESET,
          IN_ADDR       => stage_connect(i).addr(max(0,i-1) downto 0),
          IN_KEY        => stage_connect(i).key,
          IN_FOUNDABLE  => stage_connect(i).foundable,
          IN_VLD        => stage_connect(i).vld,
          IN_CHANGE_EN  => stage_connect(i).chen,
          OUT_ADDR      => stage_connect(i+1).addr(i downto 0),
          OUT_KEY       => stage_connect(i+1).key,
          OUT_FOUNDABLE => stage_connect(i+1).foundable,
          OUT_VLD       => stage_connect(i+1).vld,
          OUT_CHANGE_EN => stage_connect(i+1).chen,
          CFG_DI        => stage_config(i).di,
          CFG_DO        => stage_config(i).do,
          CFG_DRDY      => stage_config(i).drdy,
          CFG_ADDR      => stage_config(i).addr(TREE_STAGES-1 downto TREE_STAGES-max(i,1)),
          CFG_WE        => stage_config(i).we,
          CFG_EN        => stage_config(i).en
        );
    end generate;
    tree_tripple_stage_gen : if USE_STAGES(i)=STAGE_TYPE_DIST3MEM generate
      tripple_stage : entity work.tree_3stages_distmem
        generic map (
          TREE_LEVEL     => i,
          KEY_WIDTH      => KEY_WIDTH,
          INPUT0_REG     => LOGIC2BOOLEAN(USE_REGS(i*3+0)),
          MIDDLE0_REG    => LOGIC2BOOLEAN(USE_REGS(i*3+1)), 
          OUTPUT0_REG    => LOGIC2BOOLEAN(USE_REGS(i*3+2)),
          INPUT1_REG     => LOGIC2BOOLEAN(USE_REGS(i*3+3)),
          MIDDLE1_REG    => LOGIC2BOOLEAN(USE_REGS(i*3+4)), 
          OUTPUT1_REG    => LOGIC2BOOLEAN(USE_REGS(i*3+5)),
          INPUT2_REG     => LOGIC2BOOLEAN(USE_REGS(i*3+6)),
          MIDDLE2_REG    => LOGIC2BOOLEAN(USE_REGS(i*3+7)), 
          OUTPUT2_REG    => LOGIC2BOOLEAN(USE_REGS(i*3+8))
        ) port map (
          CLK           => CLK,
          RESET         => RESET,
          IN_ADDR       => stage_connect(i).addr(max(0,i-1) downto 0),
          IN_KEY        => stage_connect(i).key,
          IN_VLD        => stage_connect(i).vld,
          IN_FOUNDABLE  => stage_connect(i).foundable,
          IN_CHANGE_EN  => stage_connect(i).chen,
          OUT_ADDR      => stage_connect(i+3).addr(i+2 downto 0),
          OUT_KEY       => stage_connect(i+3).key,
          OUT_VLD       => stage_connect(i+3).vld,
          OUT_FOUNDABLE => stage_connect(i+3).foundable,
          OUT_CHANGE_EN => stage_connect(i+3).chen,
          CFG0_DI       => stage_config(i).di,
          CFG0_DO       => stage_config(i).do,
          CFG0_DRDY     => stage_config(i).drdy,
          CFG0_ADDR     => stage_config(i).addr(TREE_STAGES-1 downto TREE_STAGES-max(i,1)),
          CFG0_WE       => stage_config(i).we,
          CFG0_EN       => stage_config(i).en,
          CFG1_DI       => stage_config(i+1).di,
          CFG1_DO       => stage_config(i+1).do,
          CFG1_DRDY     => stage_config(i+1).drdy,
          CFG1_ADDR     => stage_config(i+1).addr(TREE_STAGES-1 downto TREE_STAGES-(i+1)),
          CFG1_WE       => stage_config(i+1).we,
          CFG1_EN       => stage_config(i+1).en,
          CFG2_DI       => stage_config(i+2).di,
          CFG2_DO       => stage_config(i+2).do,
          CFG2_DRDY     => stage_config(i+2).drdy,
          CFG2_ADDR     => stage_config(i+2).addr(TREE_STAGES-1 downto TREE_STAGES-(i+2)),
          CFG2_WE       => stage_config(i+2).we,
          CFG2_EN       => stage_config(i+2).en
        );
    end generate;
  end generate;
  
  -- leaf stage of the search tree with rule adding/removing fixes
  tree_result_i : entity work.tree_final_stage_fixed
    generic map (
      TREE_STAGES    => TREE_STAGES,
      KEY_WIDTH      => KEY_WIDTH,
      DATA_WIDTH     => DATA_WIDTH,
      MEM_TYPE       => USE_STAGE_LEAF,
      OUTPUT_REG     => LOGIC2BOOLEAN(USE_REGS(TREE_STAGES*3+0))
    ) port map (
      CLK           => CLK,
      RESET         => RESET,
      IN_ADDR       => stage_connect(TREE_STAGES).addr,
      IN_KEY        => stage_connect(TREE_STAGES).key,
      IN_FOUNDABLE  => stage_connect(TREE_STAGES).foundable,
      IN_VLD        => stage_connect(TREE_STAGES).vld,
      IN_CHANGE_EN  => stage_connect(TREE_STAGES).chen,
      OUT_DATA      => OUT_DATA,
      OUT_LENGTH    => OUT_LENGTH,
      OUT_FOUND     => OUT_FOUND,
      OUT_VLD       => OUT_VLD,
      CFG_DI        => leaf_config.di,
      CFG_DO        => leaf_config.do,
      CFG_DRDY      => leaf_config.drdy,
      CFG_ADDR      => leaf_config.addr,
      CFG_WE        => leaf_config.we,
      CFG_EN        => leaf_config.en,
      FIX_KEY_START => fix_key_start,
      FIX_KEY_END   => fix_key_end,
      FIX_LEN       => fix_len,
      FIX_DATALEN   => fix_datalen,
      FIX_DATA      => fix_data,
      FIX_FOUND     => fix_found,
      FIX_ACTIVE    => fix_active,
      FIX_IS_ADD    => fix_is_add,
      FIX_ACTIVATE  => fix_activate
    );
    
  -- configuration block
  tree_config_block : entity work.tree_config
    generic map(
      TREE_STAGES    => TREE_STAGES,
      KEY_WIDTH      => KEY_WIDTH,
      DATA_WIDTH     => DATA_WIDTH,
      USE_STAGES     => USE_STAGES,
      USE_STAGE_LEAF => USE_STAGE_LEAF,
      USE_REGS       => USE_REGS
    ) port map(
      CLK           => CLK,
      RESET         => RESET,
      NODE_DI       => stage_config_di,
      NODE_DO       => stage_config_do,
      NODE_DRDY     => stage_config_drdy,
      NODE_ADDR     => stage_config_addr,
      NODE_WE       => stage_config_we,
      NODE_EN       => stage_config_en,     
      LEAF_DI       => leaf_config.di,
      LEAF_DO       => leaf_config.do,
      LEAF_DRDY     => leaf_config.drdy,
      LEAF_ADDR     => leaf_config.addr,
      LEAF_WE       => leaf_config.we,
      LEAF_EN       => leaf_config.en,
      CFG_KEY       => CFG_KEY,
      CFG_KEYLEN    => CFG_KEYLEN,
      CFG_DATA      => CFG_DATA,
      CFG_ADD_VLD   => CFG_ADD_VLD,
      CFG_REM_VLD   => CFG_REM_VLD,
      CFG_CLR_VLD   => CFG_CLR_VLD,
      CFG_BUSY      => CFG_BUSY,
      CFG_FULL      => CFG_FULL,
      DIR_ADDR      => DIR_ADDR,
      DIR_RD        => DIR_RD,
      DIR_KEY       => DIR_KEY,
      DIR_KEYLEN    => DIR_KEYLEN,
      DIR_ISEND     => DIR_ISEND,
      DIR_KEYVLD    => DIR_KEYVLD,
      DIR_DATA      => DIR_DATA,
      DIR_DRDY      => DIR_DRDY,
      FIX_KEY_START => fix_key_start,
      FIX_KEY_END   => fix_key_end,
      FIX_LEN       => fix_len,
      FIX_DATALEN   => fix_datalen,
      FIX_DATA      => fix_data,
      FIX_FOUND     => fix_found,
      FIX_ACTIVE    => fix_active,
      FIX_IS_ADD    => fix_is_add,
      FIX_ACTIVATE  => fix_activate
    );
    
  config_agregation_gen : for i in 0 to TREE_STAGES-1 generate
    stage_config(i).di <= stage_config_di((stage_config(i).di'length)*(i+1)-1 downto (stage_config(i).di'length)*i);
    stage_config_do((stage_config(i).do'length)*(i+1)-1 downto (stage_config(i).do'length)*i) <= stage_config(i).do;
    stage_config_drdy(i) <= stage_config(i).drdy;
    stage_config(i).addr <= stage_config_addr((stage_config(i).addr'length)*(i+1)-1 downto (stage_config(i).addr'length)*i);
    stage_config(i).we <= stage_config_we(i);
    stage_config(i).en <= stage_config_en(i);
  end generate;
end architecture;