-- reg_cmp.vhd: Generic comparator with register in the middle 
-- Copyright (C) 2012 Brno University of Technology
-- Author(s): Lukas Kekely <xkekel00@stud.fit.vutbr.cz>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--
-- $Id: reg_cmp.vhd 4531 2013-12-20 16:29:12Z xkekel00 $
--
-- TODO:
--
--

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_arith.all;
use IEEE.std_logic_unsigned.all;
use IEEE.numeric_std.all;
use WORK.math_pack.all;
-- ----------------------------------------------------------------------------
--                        Entity declaration
-- ----------------------------------------------------------------------------
entity reg_cmp is
  generic(
    DATA_WIDTH : integer := 128
  );
  port (
    CLK     : in std_logic;
    RESET   : in std_logic;
    A       : in std_logic_vector(DATA_WIDTH-1 downto 0);
    B       : in std_logic_vector(DATA_WIDTH-1 downto 0);
    VLD     : in std_logic;
    RES     : out std_logic;
    RES_VLD : out std_logic
  );
end entity;

-- ----------------------------------------------------------------------------
--                      Architecture declaration
-- ----------------------------------------------------------------------------
architecture full of reg_cmp is
  constant S1_WIDTH      : integer := log2(DATA_WIDTH)/2;
   constant S2_WIDTH      : integer := log2(DATA_WIDTH)-S1_WIDTH;
   constant S1_ITEMS      : integer := 2**S1_WIDTH;
   constant S2_ITEMS      : integer := ((DATA_WIDTH-1)/S1_ITEMS)+1;
   signal s1_do           : std_logic_vector(S2_ITEMS-1 downto 0);
   signal s2_di           : std_logic_vector(S2_ITEMS-1 downto 0);
   signal cmp_vector      : std_logic_vector(DATA_WIDTH+S1_ITEMS-1 downto 0);
   signal sig_res         : std_logic;
begin
  cmp_vector(DATA_WIDTH-1 downto 0) <= A xor B;
  cmp_vector(DATA_WIDTH+S1_ITEMS-1 downto DATA_WIDTH) <= (others => '0');
  
  -- stage 1 ----------------------------------
  s1_gen : for i in 0 to S2_ITEMS-1 generate
    process(cmp_vector)
      variable or_int : std_logic;
    begin
      or_int := '0';
      for k in 0 to S1_ITEMS-1 loop
         or_int := or_int or cmp_vector(S1_ITEMS*i+k);
      end loop;
      s1_do(i) <= or_int;
    end process; 
  end generate;
   
  -- register between stages -----------------
  process(CLK,RESET)
  begin
    if CLK'event and CLK = '1' then
      if RESET = '1' then
        s2_di  <= (others => '0');
      else
        s2_di  <= s1_do;
      end if;
    end if;
  end process;
  
  -- stage 2 ----------------------------------
  process(s2_di)
    variable or_int : std_logic;
  begin
    or_int := '0';
    for k in 0 to S2_ITEMS-1 loop
      or_int := or_int or s2_di(k);
    end loop;
    sig_res <= or_int;
  end process; 
   
  -- register for valid signal
  process(CLK,RESET)
  begin
    if CLK'event and CLK = '1' then
      if RESET = '1' then
        RES_VLD <= '0';
      else
        RES_VLD <= VLD;
      end if;
    end if;
  end process;
  RES <= not sig_res;
end architecture;
 