-- full.vhd: HFE-M full architecture, employing all protocols we know.
-- Copyright (C) 2011 CESNET
-- Author(s): Viktor Puš <pus@cesnet.cz>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--
--
--

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_unsigned.all;
use IEEE.std_logic_arith.all;
use IEEE.numeric_std.all;
use work.math_pack.all;

-- ----------------------------------------------------------------------------
--                      Entity declaration
-- ----------------------------------------------------------------------------

entity hfe_m_full is
   generic (
      --* FrameLink width
      DATA_WIDTH           : integer := 512;
      --* Use input pipe
      IN_PIPE_EN           : boolean := true;
      --* Use output register of input pipe
      IN_PIPE_OUTREG       : boolean := true;
      --* Use middle pipe
      MID_PIPE_EN          : boolean := true;
      --* Use output register of middle pipe
      MID_PIPE_OUTREG      : boolean := true;
      --* Use output pipe
      OUT_PIPE_EN          : boolean := true;
      --* Use output register of output pipe
      OUT_PIPE_OUTREG      : boolean := true;
      --* Width of OFFSET_* signals (which are in Bytes).
      --* 8 means that headers should fit in first 256 B of packet
      --* (realistic guess). 14 Eth + 4*4 VLAN/MPLS + min. 40 IPv6 = min. 70
      --* Don't set less than 8 unless you know what you're doing.
      OFFSET_WIDTH         : integer := 8;
      --* Skip this amount of FL words (SZE header typically)
      HEADER_SKIP          : integer := 1;
      --* Length of IPv6 Hdr Ext Len field.
      --* Should be 8, lower it to save logic.
      IPV6_EXT_LEN_BITS    : integer := 3;
      --* Maximum supported IPv6 Extension headers, max 8
      IPV6_MAX_EXT_HEADERS : integer := 2
   );
   port (
      CLK         : in std_logic;
      RESET       : in std_logic;

      --+ FrameLink input interface
      RX_DATA     : in std_logic_vector(DATA_WIDTH-1 downto 0);
      RX_DREM     : in std_logic_vector(log2(DATA_WIDTH/8)-1 downto 0);
      RX_SRC_RDY_N: in std_logic;
      RX_DST_RDY_N: out std_logic;
      RX_SOF_N    : in std_logic;
      RX_EOF_N    : in std_logic;
      RX_SOP_N    : in std_logic;
      RX_EOP_N    : in std_logic;

      --+ FrameLink output interface
      TX_DATA     : out std_logic_vector(DATA_WIDTH-1 downto 0);
      TX_DREM     : out std_logic_vector(log2(DATA_WIDTH/8)-1 downto 0);
      TX_SRC_RDY_N: out std_logic;
      TX_DST_RDY_N: in  std_logic;
      TX_SOF_N    : out std_logic;
      TX_EOF_N    : out std_logic;
      TX_SOP_N    : out std_logic;
      TX_EOP_N    : out std_logic;

      --------------------------------------------------------------------
      -- Analysis results
      --------------------------------------------------------------------
      -- Link layer ------------------------------------------------------
      --* Offset of the link layer (Ethernet) header start from SOF (in Bytes)
      LINK_OFFSET       : out std_logic_vector(OFFSET_WIDTH-1 downto 0);
      --* Ethernet header present
      ACTIVE_ETHERNET   : out std_logic;
      --* Link layer analysis still running (above outputs are not valid)
      RUNNING_LINK      : out std_logic;

      -- VLAN and MPLS ---------------------------------------------------
      --* Offset of the first VLAN header start from SOF (in Bytes)
      VLAN0_OFFSET      : out std_logic_vector(OFFSET_WIDTH-1 downto 0);
      --* Offset of the second VLAN header start from SOF (in Bytes)
      VLAN1_OFFSET      : out std_logic_vector(OFFSET_WIDTH-1 downto 0);
      --* Offset of the first MPLS header start from SOF (in Bytes)
      MPLS0_OFFSET      : out std_logic_vector(OFFSET_WIDTH-1 downto 0);
      --* Offset of the second MPLS header start from SOF (in Bytes)
      MPLS1_OFFSET      : out std_logic_vector(OFFSET_WIDTH-1 downto 0);

      --* First VLAN header present
      ACTIVE_VLAN0      : out std_logic;
      --* Second VLAN header present
      ACTIVE_VLAN1      : out std_logic;
      --* First MPLS header present
      ACTIVE_MPLS0      : out std_logic;
      --* Second MPLS header present
      ACTIVE_MPLS1      : out std_logic;

      --* VLAN & MPLS layer analysis still running (above outputs are not valid)
      RUNNING_VLAN_MPLS : out std_logic;

      -- Internet layer ------------------------------------------------
      --* Offset of the Internet layer (IP) header start from SOF (in Bytes)
      INTERNET_OFFSET   : out std_logic_vector(OFFSET_WIDTH-1 downto 0);
      --* IPv4 header present
      ACTIVE_IPV4       : out std_logic;
      --* IPv6 header present
      ACTIVE_IPV6       : out std_logic;
      --* Is set to 1 when packet is NOT fragmented or has
      --* fragment offset = 0
      FRAGMENT_FIRST : out std_logic;
      --* Is set to 1 when packet is NOT fragmented or is
      --* the last fragment of the fragmented packet
      FRAGMENT_LAST  : out std_logic;
      --* Internet layer analysis still running (above outputs are not valid)
      RUNNING_INTERNET  : out std_logic;
      --* IPv4 header present but has errors (may appear after RUNNING_INTERNET)
      ERROR_IPV4        : out std_logic;
      --* IPv6 header present but has errors (may appear after RUNNING_INTERNET)
      ERROR_IPV6        : out std_logic;

      -- Transport layer -----------------------------------------------
      -- Something returned only if IP Fragment Offset == 0
      --* Offset of the Transport layer header start from SOF (in Bytes)
      TRANSPORT_OFFSET  : out std_logic_vector(OFFSET_WIDTH-1 downto 0);
      --* TCP header present
      ACTIVE_TCP        : out std_logic;
      --* UDP header present
      ACTIVE_UDP        : out std_logic;
      --* ICMP header present
      ACTIVE_ICMP       : out std_logic;
      --* IGMP header present
      ACTIVE_IGMP       : out std_logic;
      --* MPLS in IP header present
      ACTIVE_MPLS_TUN   : out std_logic;
      --* SCTP header present
      ACTIVE_SCTP       : out std_logic;
      --* IPv6 in IP header present
      ACTIVE_IPV6_TUN   : out std_logic;
      --* IPv4 in IP header present
      ACTIVE_IPV4_TUN   : out std_logic;
      --* IPv6 No Next Header identifier present
      ACTIVE_IPV6_NO_NEXT:out std_logic;
      --* Next header protocol number
      PROTOCOL          : out std_logic_vector(7 downto 0);
      --* Transport layer analysis still running (above outputs are not valid)
      RUNNING_TRANSPORT : out std_logic
   );
end entity;

architecture full of hfe_m_full is

   -- Must be able to count FL words that contain some header
   -- 128 Bytes should be enough with good reserve
   constant COUNTER_WIDTH  : integer := OFFSET_WIDTH - log2(DATA_WIDTH/8);

   -- Main word counter
   signal word_counter     : std_logic_vector(COUNTER_WIDTH-1 downto 0);
   -- Word counter after middle pipeline
   signal mid_word_counter : std_logic_vector(COUNTER_WIDTH-1 downto 0);

   -- Input pipeline
   signal in_pipe_rx_data     : std_logic_vector(DATA_WIDTH-1 downto 0);
   signal in_pipe_rx_drem     : std_logic_vector(log2(DATA_WIDTH/8)-1 downto 0);
   signal in_pipe_rx_src_rdy_n: std_logic;
   signal in_pipe_rx_dst_rdy_n: std_logic;
   signal in_pipe_rx_sof_n    : std_logic;
   signal in_pipe_rx_eof_n    : std_logic;
   signal in_pipe_rx_sop_n    : std_logic;
   signal in_pipe_rx_eop_n    : std_logic;

   -- Middle pipeline
   signal mid_pipe_rx_data    : std_logic_vector(DATA_WIDTH-1 downto 0);
   signal mid_pipe_rx_drem    : std_logic_vector(log2(DATA_WIDTH/8)-1 downto 0);
   signal mid_pipe_rx_src_rdy_n:std_logic;
   signal mid_pipe_rx_dst_rdy_n:std_logic;
   signal mid_pipe_rx_sof_n   : std_logic;
   signal mid_pipe_rx_eof_n   : std_logic;
   signal mid_pipe_rx_sop_n   : std_logic;
   signal mid_pipe_rx_eop_n   : std_logic;

   -- Ethernet layer inputs
   signal sig_link_offset  : std_logic_vector(OFFSET_WIDTH-1 downto 0);
   signal reg_active_ethernet:std_logic;
   signal sig_active_ethernet:std_logic;
   -- Ethernet analysis results
   signal eth_offset_out   : std_logic_vector(OFFSET_WIDTH-1 downto 0);
   signal eth_active_ipv4  : std_logic;
   signal eth_active_ipv6  : std_logic;
   signal eth_active_vlan  : std_logic;
   signal eth_active_mpls  : std_logic;
   signal eth_active_unknown:std_logic;
   signal eth_running      : std_logic;

   -- VLAN & MPLS layer inputs
   signal sig_vlan0_offset : std_logic_vector(OFFSET_WIDTH-1 downto 0);
   signal sig_vlan1_offset : std_logic_vector(OFFSET_WIDTH-1 downto 0);
   signal sig_mpls0_offset : std_logic_vector(OFFSET_WIDTH-1 downto 0);
   signal sig_mpls1_offset : std_logic_vector(OFFSET_WIDTH-1 downto 0);
   signal sig_active_vlan0 : std_logic;
   signal sig_active_vlan1 : std_logic;
   signal sig_active_mpls0 : std_logic;
   signal sig_active_mpls1 : std_logic;
   -- VLAN & MPLS analysis results
   signal vlan0_offset_out   : std_logic_vector(OFFSET_WIDTH-1 downto 0);
   signal vlan0_active_ipv4  : std_logic;
   signal vlan0_active_ipv6  : std_logic;
   signal vlan0_active_vlan  : std_logic;
   signal vlan0_active_mpls  : std_logic;
   signal vlan0_active_unknown:std_logic;
   signal vlan0_running      : std_logic;

   signal vlan1_offset_out   : std_logic_vector(OFFSET_WIDTH-1 downto 0);
   signal vlan1_active_ipv4  : std_logic;
   signal vlan1_active_ipv6  : std_logic;
   signal vlan1_active_vlan  : std_logic;
   signal vlan1_active_mpls  : std_logic;
   signal vlan1_active_unknown:std_logic;
   signal vlan1_running      : std_logic;

   signal mpls0_offset_out   : std_logic_vector(OFFSET_WIDTH-1 downto 0);
   signal mpls0_active_ipv4  : std_logic;
   signal mpls0_active_ipv6  : std_logic;
   signal mpls0_active_vlan  : std_logic;
   signal mpls0_active_mpls  : std_logic;
   signal mpls0_active_unknown:std_logic;
   signal mpls0_running      : std_logic;

   signal mpls1_offset_out   : std_logic_vector(OFFSET_WIDTH-1 downto 0);
   signal mpls1_active_ipv4  : std_logic;
   signal mpls1_active_ipv6  : std_logic;
   signal mpls1_active_vlan  : std_logic;
   signal mpls1_active_mpls  : std_logic;
   signal mpls1_active_unknown:std_logic;
   signal mpls1_running      : std_logic;

   signal sig_running_vlan_mpls:std_logic;

   -- Middle pipeline
   signal mid_pipe_din        : std_logic_vector(5*OFFSET_WIDTH+13 downto 0);
   signal mid_pipe_rx_src_rdy : std_logic;
   signal mid_pipe_dout       : std_logic_vector(5*OFFSET_WIDTH+13 downto 0);
   signal mid_pipe_tx_dst_rdy : std_logic;

   signal pipe_sig_active_ethernet  : std_logic;
   signal pipe_vlan0_running        : std_logic;
   signal pipe_mpls0_running        : std_logic;
   signal pipe_vlan1_running        : std_logic;
   signal pipe_mpls1_running        : std_logic;
   signal pipe_sig_running_vlan_mpls: std_logic;
   signal pipe_sig_vlan0_offset     : std_logic_vector(OFFSET_WIDTH-1 downto 0);
   signal pipe_sig_vlan1_offset     : std_logic_vector(OFFSET_WIDTH-1 downto 0);
   signal pipe_sig_mpls0_offset     : std_logic_vector(OFFSET_WIDTH-1 downto 0);
   signal pipe_sig_mpls1_offset     : std_logic_vector(OFFSET_WIDTH-1 downto 0);
   signal pipe_sig_active_vlan0     : std_logic;
   signal pipe_sig_active_vlan1     : std_logic;
   signal pipe_sig_active_mpls0     : std_logic;
   signal pipe_sig_active_mpls1     : std_logic;
   signal pipe_eth_running          : std_logic;

   -- Internet layer inputs
   signal sig_internet_offset:std_logic_vector(OFFSET_WIDTH-1 downto 0);
   signal sig_active_ipv4  : std_logic;
   signal sig_active_ipv6  : std_logic;
   signal sig_running_internet:std_logic;
   -- Pipelined
   signal pipe_internet_offset:std_logic_vector(OFFSET_WIDTH-1 downto 0);
   signal pipe_active_ipv4  : std_logic;
   signal pipe_active_ipv6  : std_logic;
   signal pipe_running_internet:std_logic;

   -- Internet layer analysis results
   signal ipv4_offset_out  : std_logic_vector(OFFSET_WIDTH-1 downto 0);
   signal ipv4_active_tcp  : std_logic;
   signal ipv4_active_udp  : std_logic;
   signal ipv4_active_icmp : std_logic;
   signal ipv4_active_igmp : std_logic;
   signal ipv4_active_mpls : std_logic;
   signal ipv4_active_sctp : std_logic;
   signal ipv4_active_ipv6 : std_logic;
   signal ipv4_active_ipv4 : std_logic;
   signal ipv4_active_unknown:std_logic;
   signal ipv4_protocol    : std_logic_vector(7 downto 0);
   signal ipv4_frag_first  : std_logic;
   signal ipv4_frag_last   : std_logic;
   signal ipv4_malformed   : std_logic;
   signal ipv4_running     : std_logic;
   signal sig_error_ipv4   : std_logic;

   signal ipv6_offset_out  : std_logic_vector(OFFSET_WIDTH-1 downto 0);
   signal ipv6_active_tcp  : std_logic;
   signal ipv6_active_udp  : std_logic;
   signal ipv6_active_icmp : std_logic;
   signal ipv6_active_igmp : std_logic;
   signal ipv6_active_mpls : std_logic;
   signal ipv6_active_sctp : std_logic;
   signal ipv6_active_ipv6 : std_logic;
   signal ipv6_active_ipv6_no_next:std_logic;
   signal ipv6_active_ipv4 : std_logic;
   signal ipv6_active_unknown:std_logic;
   signal ipv6_protocol    : std_logic_vector(7 downto 0);
   signal ipv6_frag_first  : std_logic;
   signal ipv6_frag_last   : std_logic;
   signal ipv6_malformed   : std_logic;
   signal ipv6_running     : std_logic;
   signal sig_error_ipv6   : std_logic;

   -- Transport layer inputs
   signal sig_active_tcp         : std_logic;
   signal sig_active_udp         : std_logic;
   signal sig_active_icmp        : std_logic;
   signal sig_active_igmp        : std_logic;
   signal sig_active_mpls        : std_logic;
   signal sig_active_sctp        : std_logic;
   signal sig_active_ipv6_in_ip  : std_logic;
   signal sig_active_ipv6_no_next:std_logic;
   signal sig_active_ipv4_in_ip  : std_logic;
   signal sig_protocol           : std_logic_vector(7 downto 0);
   signal sig_frag_first         : std_logic;
   signal sig_frag_last         : std_logic;
   signal sig_transport_offset   :std_logic_vector(OFFSET_WIDTH-1 downto 0);
   signal sig_running_transport  : std_logic;

   -- Output pipeline
   signal out_pipe_rx_src_rdy    : std_logic;
   signal out_pipe_rx_dst_rdy    : std_logic;
   signal out_pipe_tx_src_rdy    : std_logic;
   signal out_pipe_tx_dst_rdy    : std_logic;

   signal out_pipe_din           : std_logic_vector(DATA_WIDTH +
                                   log2(DATA_WIDTH/8) + 6*OFFSET_WIDTH + 35
                                   downto 0);
   signal out_pipe_dout          : std_logic_vector(DATA_WIDTH +
                                   log2(DATA_WIDTH/8) + 6*OFFSET_WIDTH + 35
                                   downto 0);

   -- Send zeros to output if no valid data in outfifo yet
   signal init_zeros       : std_logic;
   signal reg_init_zeros   : std_logic;

   signal zeros            : std_logic_vector(127 downto 0);
   signal ones             : std_logic_vector(127 downto 0);

begin

   zeros <= (others => '0');
   ones  <= (others => '1');

   ----------------------------------------------------------------------
   -- Input pipeline
   ----------------------------------------------------------------------
   use_inpipe_gen : if IN_PIPE_EN generate
      in_pipe_i : entity work.FL_PIPE
      generic map(
         DATA_WIDTH     => DATA_WIDTH,
         USE_OUTREG     => IN_PIPE_OUTREG
      )
      port map(
         CLK            => CLK,
         RESET          => RESET,
         -- Input interf
         RX_SOF_N       => RX_SOF_N,
         RX_SOP_N       => RX_SOP_N,
         RX_EOP_N       => RX_EOP_N,
         RX_EOF_N       => RX_EOF_N,
         RX_SRC_RDY_N   => RX_SRC_RDY_N,
         RX_DST_RDY_N   => RX_DST_RDY_N,
         RX_DATA        => RX_DATA,
         RX_REM         => RX_DREM,
         -- Output inter
         TX_SOF_N       => in_pipe_rx_sof_n,
         TX_SOP_N       => in_pipe_rx_sop_n,
         TX_EOP_N       => in_pipe_rx_eop_n,
         TX_EOF_N       => in_pipe_rx_eof_n,
         TX_SRC_RDY_N   => in_pipe_rx_src_rdy_n,
         TX_DST_RDY_N   => in_pipe_rx_dst_rdy_n,
         TX_DATA        => in_pipe_rx_data,
         TX_REM         => in_pipe_rx_drem
      );
   end generate;

   no_use_inpipe_gen : if not IN_PIPE_EN generate
      in_pipe_rx_data      <= RX_DATA;
      in_pipe_rx_drem      <= RX_DREM;
      in_pipe_rx_src_rdy_n <= RX_SRC_RDY_N;
      in_pipe_rx_sof_n     <= RX_SOF_N;
      in_pipe_rx_eof_n     <= RX_EOF_N;
      in_pipe_rx_sop_n     <= RX_SOP_N;
      in_pipe_rx_eop_n     <= RX_EOP_N;
      RX_DST_RDY_N      <= in_pipe_rx_dst_rdy_n;
   end generate;

   --* Count FL words, reset with SOF, saturate
   word_counter_p : process(CLK)
   begin
      if CLK'event and CLK = '1' then
         if RESET = '1' then
            word_counter <= (others => '0');
         else
            if in_pipe_rx_src_rdy_n = '0' and in_pipe_rx_dst_rdy_n = '0' then
               if in_pipe_rx_eof_n = '0' then
                  word_counter <= (others => '0');
               else
                  if word_counter /= ones(COUNTER_WIDTH-1 downto 0) then
                     word_counter <= word_counter + 1;
                  end if;
               end if;
            end if;
         end if;
      end if;
   end process;

   -- ----------------------------------------------------------------------
   -- Link layer -----------------------------------------------------------
   -- ----------------------------------------------------------------------

   --* Instead of parsing data in first HEADER_SKIP words
   reg_active_ethernet_p : process(CLK)
   begin
      if CLK'event and CLK = '1' then
         if RESET = '1' then
            reg_active_ethernet <= '0';
         else
            if in_pipe_rx_src_rdy_n = '0' and in_pipe_rx_dst_rdy_n = '0' then
               if word_counter = HEADER_SKIP then
                  reg_active_ethernet <= '1';
               end if;
               if in_pipe_rx_eof_n = '0' then
                  reg_active_ethernet <= '0';
               end if;
            end if;
         end if;
      end if;
   end process;

   sig_active_ethernet <= '1' when reg_active_ethernet = '1' or
               (in_pipe_rx_src_rdy_n = '0' and
                in_pipe_rx_dst_rdy_n = '0' and
                word_counter = HEADER_SKIP) else '0';

   sig_link_offset <= conv_std_logic_vector(HEADER_SKIP, COUNTER_WIDTH) &
                      zeros(log2(DATA_WIDTH/8)-1 downto 0);


   --* Ethernet analyzer
   eth_i : entity work.hfe_m_eth
   generic map (
      DATA_WIDTH     => DATA_WIDTH,
      COUNTER_WIDTH  => COUNTER_WIDTH)
   port map(
      CLK            => CLK,
      RESET          => RESET,

      FL_DATA        => in_pipe_rx_data,
      FL_DREM        => in_pipe_rx_drem,
      FL_SRC_RDY_N   => in_pipe_rx_src_rdy_n,
      FL_DST_RDY_N   => in_pipe_rx_dst_rdy_n,
      FL_SOF_N       => in_pipe_rx_sof_n,
      FL_EOF_N       => in_pipe_rx_eof_n,
      FL_SOP_N       => in_pipe_rx_sop_n,
      FL_EOP_N       => in_pipe_rx_eop_n,

      WORD_COUNTER   => word_counter,
      OFFSET_IN      => sig_link_offset,
      ACTIVE_IN      => sig_active_ethernet,

      OFFSET_OUT     => eth_offset_out,
      ACTIVE_IPV4    => eth_active_ipv4,
      ACTIVE_IPV6    => eth_active_ipv6,
      ACTIVE_MPLS    => eth_active_mpls,
      ACTIVE_VLAN    => eth_active_vlan,
      ACTIVE_UNKNOWN => eth_active_unknown,
      MALFORMED      => open,
      RUNNING        => eth_running
   );

   -- ----------------------------------------------------------------------
   -- VLAN & MPLS layer ----------------------------------------------------
   -- ----------------------------------------------------------------------

   -- Only Ethernet may precede first VLAN
   sig_vlan0_offset <= eth_offset_out;
   sig_active_vlan0 <= eth_active_vlan;

   --* VLAN analyzer 0
   vlan_i0 : entity work.hfe_m_vlan
   generic map (
      DATA_WIDTH     => DATA_WIDTH,
      COUNTER_WIDTH  => COUNTER_WIDTH)
   port map(
      CLK            => CLK,
      RESET          => RESET,

      FL_DATA        => in_pipe_rx_data,
      FL_DREM        => in_pipe_rx_drem,
      FL_SRC_RDY_N   => in_pipe_rx_src_rdy_n,
      FL_DST_RDY_N   => in_pipe_rx_dst_rdy_n,
      FL_SOF_N       => in_pipe_rx_sof_n,
      FL_EOF_N       => in_pipe_rx_eof_n,
      FL_SOP_N       => in_pipe_rx_sop_n,
      FL_EOP_N       => in_pipe_rx_eop_n,

      WORD_COUNTER   => word_counter,
      OFFSET_IN      => sig_vlan0_offset,
      ACTIVE_IN      => sig_active_vlan0,

      OFFSET_OUT     => vlan0_offset_out,
      ACTIVE_IPV4    => vlan0_active_ipv4,
      ACTIVE_IPV6    => vlan0_active_ipv6,
      ACTIVE_MPLS    => vlan0_active_mpls,
      ACTIVE_VLAN    => vlan0_active_vlan,
      ACTIVE_UNKNOWN => vlan0_active_unknown,
      MALFORMED      => open,
      RUNNING        => vlan0_running
   );

   -- Only first VLAN may precede second VLAN
   sig_vlan1_offset <= vlan0_offset_out;
   sig_active_vlan1 <= vlan0_active_vlan;

   --* VLAN analyzer 1
   vlan_i1 : entity work.hfe_m_vlan
   generic map (
      DATA_WIDTH     => DATA_WIDTH,
      COUNTER_WIDTH  => COUNTER_WIDTH)
   port map(
      CLK            => CLK,
      RESET          => RESET,

      FL_DATA        => in_pipe_rx_data,
      FL_DREM        => in_pipe_rx_drem,
      FL_SRC_RDY_N   => in_pipe_rx_src_rdy_n,
      FL_DST_RDY_N   => in_pipe_rx_dst_rdy_n,
      FL_SOF_N       => in_pipe_rx_sof_n,
      FL_EOF_N       => in_pipe_rx_eof_n,
      FL_SOP_N       => in_pipe_rx_sop_n,
      FL_EOP_N       => in_pipe_rx_eop_n,

      WORD_COUNTER   => word_counter,
      OFFSET_IN      => sig_vlan1_offset,
      ACTIVE_IN      => sig_active_vlan1,

      OFFSET_OUT     => vlan1_offset_out,
      ACTIVE_IPV4    => vlan1_active_ipv4,
      ACTIVE_IPV6    => vlan1_active_ipv6,
      ACTIVE_MPLS    => vlan1_active_mpls, -- Third level not supported
      ACTIVE_VLAN    => vlan1_active_vlan, -- Third level not supported
      ACTIVE_UNKNOWN => vlan1_active_unknown,
      MALFORMED      => open,
      RUNNING        => vlan1_running
   );

   -- Ethernet and both VLANs may precede first MPLS
   sig_mpls0_offset <= vlan1_offset_out when vlan1_active_mpls = '1' else
                       vlan0_offset_out when vlan0_active_mpls = '1' else
                       eth_offset_out;
   sig_active_mpls0 <= eth_active_mpls or vlan0_active_mpls
                       or vlan1_active_mpls;

   --* MPLS analyzer 0
   mpls_i0 : entity work.hfe_m_mpls
   generic map (
      DATA_WIDTH     => DATA_WIDTH,
      COUNTER_WIDTH  => COUNTER_WIDTH)
   port map(
      CLK            => CLK,
      RESET          => RESET,

      FL_DATA        => in_pipe_rx_data,
      FL_DREM        => in_pipe_rx_drem,
      FL_SRC_RDY_N   => in_pipe_rx_src_rdy_n,
      FL_DST_RDY_N   => in_pipe_rx_dst_rdy_n,
      FL_SOF_N       => in_pipe_rx_sof_n,
      FL_EOF_N       => in_pipe_rx_eof_n,
      FL_SOP_N       => in_pipe_rx_sop_n,
      FL_EOP_N       => in_pipe_rx_eop_n,

      WORD_COUNTER   => word_counter,
      OFFSET_IN      => sig_mpls0_offset,
      ACTIVE_IN      => sig_active_mpls0,

      OFFSET_OUT     => mpls0_offset_out,
      ACTIVE_IPV4    => mpls0_active_ipv4,
      ACTIVE_IPV6    => mpls0_active_ipv6,
      ACTIVE_MPLS    => mpls0_active_mpls,
      ACTIVE_UNKNOWN => mpls0_active_unknown,
      MALFORMED      => open,
      RUNNING        => mpls0_running
   );

   -- Only first MPLS may precede second MPLS
   sig_mpls1_offset <= mpls0_offset_out;
   sig_active_mpls1 <= mpls0_active_mpls;

   --* MPLS analyzer 1
   mpls_i1 : entity work.hfe_m_mpls
   generic map (
      DATA_WIDTH     => DATA_WIDTH,
      COUNTER_WIDTH  => COUNTER_WIDTH)
   port map(
      CLK            => CLK,
      RESET          => RESET,

      FL_DATA        => in_pipe_rx_data,
      FL_DREM        => in_pipe_rx_drem,
      FL_SRC_RDY_N   => in_pipe_rx_src_rdy_n,
      FL_DST_RDY_N   => in_pipe_rx_dst_rdy_n,
      FL_SOF_N       => in_pipe_rx_sof_n,
      FL_EOF_N       => in_pipe_rx_eof_n,
      FL_SOP_N       => in_pipe_rx_sop_n,
      FL_EOP_N       => in_pipe_rx_eop_n,

      WORD_COUNTER   => word_counter,
      OFFSET_IN      => sig_mpls1_offset,
      ACTIVE_IN      => sig_active_mpls1,

      OFFSET_OUT     => mpls1_offset_out,
      ACTIVE_IPV4    => mpls1_active_ipv4,
      ACTIVE_IPV6    => mpls1_active_ipv6,
      ACTIVE_MPLS    => mpls1_active_mpls,
      ACTIVE_UNKNOWN => mpls1_active_unknown,
      MALFORMED      => open,
      RUNNING        => mpls1_running
   );

   sig_running_vlan_mpls <= (not sig_active_ethernet) or vlan0_running or
                        mpls0_running or vlan1_running or mpls1_running;

   -- Prepare for Internet layer
   sig_internet_offset  <= mpls1_offset_out when
                          (mpls1_active_ipv4 = '1' or
                           mpls1_active_ipv6 = '1') else
                       mpls0_offset_out when
                          (mpls0_active_ipv4 = '1' or
                           mpls0_active_ipv6 = '1') else
                       vlan1_offset_out when
                          (vlan1_active_ipv4 = '1' or
                           vlan1_active_ipv6 = '1') else
                       vlan0_offset_out when
                          (vlan0_active_ipv4 = '1' or
                           vlan0_active_ipv6 = '1') else
                       eth_offset_out;

   sig_active_ipv4      <= eth_active_ipv4 or vlan0_active_ipv4 or
                       vlan1_active_ipv4 or mpls0_active_ipv4 or
                       mpls1_active_ipv4;
   sig_active_ipv6      <= eth_active_ipv6 or vlan0_active_ipv6 or
                       vlan1_active_ipv6 or mpls0_active_ipv6 or
                       mpls1_active_ipv6;

   sig_running_internet <= (not sig_active_ethernet) or
                           vlan0_running or mpls0_running or
                           vlan1_running or mpls1_running;
   -- ----------------------------------------------------------------------
   -- Middle pipeline
   -- ----------------------------------------------------------------------
   use_midpipe_gen : if MID_PIPE_EN generate
      --* FrameLink middle pipeline
      mid_pipe_i : entity work.FL_PIPE
      generic map(
         DATA_WIDTH     => DATA_WIDTH,
         USE_OUTREG     => MID_PIPE_OUTREG
      )
      port map(
         CLK            => CLK,
         RESET          => RESET,
         -- Input interf
         RX_SOF_N       => in_pipe_rx_sof_n,
         RX_SOP_N       => in_pipe_rx_sop_n,
         RX_EOP_N       => in_pipe_rx_eop_n,
         RX_EOF_N       => in_pipe_rx_eof_n,
         RX_SRC_RDY_N   => in_pipe_rx_src_rdy_n,
         RX_DST_RDY_N   => in_pipe_rx_dst_rdy_n,
         RX_DATA        => in_pipe_rx_data,
         RX_REM         => in_pipe_rx_drem,
         -- Output inter
         TX_SOF_N       => mid_pipe_rx_sof_n,
         TX_SOP_N       => mid_pipe_rx_sop_n,
         TX_EOP_N       => mid_pipe_rx_eop_n,
         TX_EOF_N       => mid_pipe_rx_eof_n,
         TX_SRC_RDY_N   => mid_pipe_rx_src_rdy_n,
         TX_DST_RDY_N   => mid_pipe_rx_dst_rdy_n,
         TX_DATA        => mid_pipe_rx_data,
         TX_REM         => mid_pipe_rx_drem
      );
   end generate;

   no_use_midpipe_gen : if not MID_PIPE_EN generate
      mid_pipe_rx_data      <= in_pipe_rx_data;
      mid_pipe_rx_drem      <= in_pipe_rx_drem;
      mid_pipe_rx_src_rdy_n <= in_pipe_rx_src_rdy_n;
      mid_pipe_rx_sof_n     <= in_pipe_rx_sof_n;
      mid_pipe_rx_eof_n     <= in_pipe_rx_eof_n;
      mid_pipe_rx_sop_n     <= in_pipe_rx_sop_n;
      mid_pipe_rx_eop_n     <= in_pipe_rx_eop_n;
      in_pipe_rx_dst_rdy_n  <= mid_pipe_rx_dst_rdy_n;
   end generate;

   mid_pipe_din(OFFSET_WIDTH-1 downto 0) <= sig_internet_offset;
   mid_pipe_din(OFFSET_WIDTH)   <= sig_active_ipv4;
   mid_pipe_din(OFFSET_WIDTH+1) <= sig_active_ipv6;
   mid_pipe_din(OFFSET_WIDTH+2) <= sig_running_internet;
   mid_pipe_din(OFFSET_WIDTH+3) <= sig_active_ethernet;
   mid_pipe_din(OFFSET_WIDTH+4) <= vlan0_running;
   mid_pipe_din(OFFSET_WIDTH+5) <= mpls0_running;
   mid_pipe_din(OFFSET_WIDTH+6) <= vlan1_running;
   mid_pipe_din(OFFSET_WIDTH+7) <= mpls1_running;
   mid_pipe_din(OFFSET_WIDTH+8) <= sig_running_vlan_mpls;
   mid_pipe_din(2*OFFSET_WIDTH+9-1 downto OFFSET_WIDTH+9)
      <= sig_vlan0_offset;
   mid_pipe_din(3*OFFSET_WIDTH+9-1 downto 2*OFFSET_WIDTH+9)
      <= sig_vlan1_offset;
   mid_pipe_din(4*OFFSET_WIDTH+9-1 downto 3*OFFSET_WIDTH+9)
      <= sig_mpls0_offset;
   mid_pipe_din(5*OFFSET_WIDTH+9-1 downto 4*OFFSET_WIDTH+9)
      <= sig_mpls1_offset;
   mid_pipe_din(5*OFFSET_WIDTH+9) <=  sig_active_vlan0;
   mid_pipe_din(5*OFFSET_WIDTH+10) <= sig_active_vlan1;
   mid_pipe_din(5*OFFSET_WIDTH+11) <= sig_active_mpls0;
   mid_pipe_din(5*OFFSET_WIDTH+12) <= sig_active_mpls1;
   mid_pipe_din(5*OFFSET_WIDTH+13) <= eth_running;

   -- Flow control same as FrameLink
   mid_pipe_rx_src_rdy <= not in_pipe_rx_src_rdy_n;
   mid_pipe_tx_dst_rdy <= not mid_pipe_rx_dst_rdy_n;

   --* Data middle pipeline
   mid_pipe_i : entity work.PIPE
   generic map(
      DATA_WIDTH     => 5*OFFSET_WIDTH + 14,
      USE_OUTREG     => MID_PIPE_OUTREG,
      FAKE_PIPE      => not MID_PIPE_EN
   )
   port map(
      CLK            => CLK,
      RESET          => RESET,
      -- Input interface ---------------------------------------------------
      IN_DATA        => mid_pipe_din,
      IN_SRC_RDY     => mid_pipe_rx_src_rdy,
      IN_DST_RDY     => open,
      -- Output interface --------------------------------------------------
      OUT_DATA       => mid_pipe_dout,
      OUT_SRC_RDY    => open,
      OUT_DST_RDY    => mid_pipe_tx_dst_rdy
   );

   pipe_internet_offset  <= mid_pipe_dout(OFFSET_WIDTH-1 downto 0);
   pipe_active_ipv4      <= mid_pipe_dout(OFFSET_WIDTH);
   pipe_active_ipv6      <= mid_pipe_dout(OFFSET_WIDTH+1);
   pipe_running_internet <= mid_pipe_dout(OFFSET_WIDTH+2);
   pipe_sig_active_ethernet <= mid_pipe_dout(OFFSET_WIDTH+3);
   pipe_vlan0_running       <= mid_pipe_dout(OFFSET_WIDTH+4);
   pipe_mpls0_running       <= mid_pipe_dout(OFFSET_WIDTH+5);
   pipe_vlan1_running       <= mid_pipe_dout(OFFSET_WIDTH+6);
   pipe_mpls1_running       <= mid_pipe_dout(OFFSET_WIDTH+7);
   pipe_sig_running_vlan_mpls<=mid_pipe_dout(OFFSET_WIDTH+8);
   pipe_sig_vlan0_offset <=
      mid_pipe_dout(2*OFFSET_WIDTH+9-1 downto OFFSET_WIDTH+9);
   pipe_sig_vlan1_offset <=
      mid_pipe_dout(3*OFFSET_WIDTH+9-1 downto 2*OFFSET_WIDTH+9);
   pipe_sig_mpls0_offset <=
      mid_pipe_dout(4*OFFSET_WIDTH+9-1 downto 3*OFFSET_WIDTH+9);
   pipe_sig_mpls1_offset <=
      mid_pipe_dout(5*OFFSET_WIDTH+9-1 downto 4*OFFSET_WIDTH+9);
   pipe_sig_active_vlan0 <= mid_pipe_dout(5*OFFSET_WIDTH+9);
   pipe_sig_active_vlan1 <= mid_pipe_dout(5*OFFSET_WIDTH+10);
   pipe_sig_active_mpls0 <= mid_pipe_dout(5*OFFSET_WIDTH+11);
   pipe_sig_active_mpls1 <= mid_pipe_dout(5*OFFSET_WIDTH+12);
   pipe_eth_running <= mid_pipe_dout(5*OFFSET_WIDTH+13);

   --* Count FL words again after pipelining
   mid_word_counter_p : process(CLK)
   begin
      if CLK'event and CLK = '1' then
         if RESET = '1' then
            mid_word_counter <= (others => '0');
         else
            if mid_pipe_rx_src_rdy_n = '0' and mid_pipe_rx_dst_rdy_n = '0' then
               if mid_pipe_rx_eof_n = '0' then
                  mid_word_counter <= (others => '0');
               else
                  if mid_word_counter /= ones(COUNTER_WIDTH-1 downto 0) then
                     mid_word_counter <= mid_word_counter + 1;
                  end if;
               end if;
            end if;
         end if;
      end if;
   end process;

   -- ----------------------------------------------------------------------
   -- Internet layer -------------------------------------------------------
   -- ----------------------------------------------------------------------

   --* IPv4 analyzer
   ipv4_i : entity work.hfe_m_ipv4
   generic map(
      DATA_WIDTH     => DATA_WIDTH,
      COUNTER_WIDTH  => COUNTER_WIDTH)
   port map(
      CLK            => CLK,
      RESET          => RESET,

      FL_DATA        => mid_pipe_rx_data,
      FL_DREM        => mid_pipe_rx_drem,
      FL_SRC_RDY_N   => mid_pipe_rx_src_rdy_n,
      FL_DST_RDY_N   => mid_pipe_rx_dst_rdy_n,
      FL_SOF_N       => mid_pipe_rx_sof_n,
      FL_EOF_N       => mid_pipe_rx_eof_n,
      FL_SOP_N       => mid_pipe_rx_sop_n,
      FL_EOP_N       => mid_pipe_rx_eop_n,

      WORD_COUNTER   => mid_word_counter,
      OFFSET_IN      => pipe_internet_offset,
      ACTIVE_IN      => pipe_active_ipv4,

      OFFSET_OUT     => ipv4_offset_out,
      ACTIVE_TCP     => ipv4_active_tcp,
      ACTIVE_UDP     => ipv4_active_udp,
      ACTIVE_ICMP    => ipv4_active_icmp,
      ACTIVE_IGMP    => ipv4_active_igmp,
      ACTIVE_MPLS    => ipv4_active_mpls,
      ACTIVE_SCTP    => ipv4_active_sctp,
      ACTIVE_IPV6    => ipv4_active_ipv6,
      ACTIVE_IPV4    => ipv4_active_ipv4,
      ACTIVE_UNKNOWN => ipv4_active_unknown,
      PROTOCOL       => ipv4_protocol,
      FRAGMENT_FIRST => ipv4_frag_first,
      FRAGMENT_LAST  => ipv4_frag_last,
      MALFORMED      => ipv4_malformed,
      RUNNING        => ipv4_running
   );

   sig_error_ipv4 <= pipe_active_ipv4 and ipv4_malformed;

   --* IPv6 analyzer
   ipv6_i : entity work.hfe_m_ipv6
   generic map(
      DATA_WIDTH     => DATA_WIDTH,
      COUNTER_WIDTH  => COUNTER_WIDTH,
      EXT_LEN_BITS   => IPV6_EXT_LEN_BITS,
      MAX_EXT_HEADERS=> IPV6_MAX_EXT_HEADERS)
   port map(
      CLK            => CLK,
      RESET          => RESET,

      FL_DATA        => mid_pipe_rx_data,
      FL_DREM        => mid_pipe_rx_drem,
      FL_SRC_RDY_N   => mid_pipe_rx_src_rdy_n,
      FL_DST_RDY_N   => mid_pipe_rx_dst_rdy_n,
      FL_SOF_N       => mid_pipe_rx_sof_n,
      FL_EOF_N       => mid_pipe_rx_eof_n,
      FL_SOP_N       => mid_pipe_rx_sop_n,
      FL_EOP_N       => mid_pipe_rx_eop_n,

      WORD_COUNTER   => mid_word_counter,
      OFFSET_IN      => pipe_internet_offset,
      ACTIVE_IN      => pipe_active_ipv6,

      OFFSET_OUT     => ipv6_offset_out,
      ACTIVE_TCP     => ipv6_active_tcp,
      ACTIVE_UDP     => ipv6_active_udp,
      ACTIVE_ICMP    => ipv6_active_icmp,
      ACTIVE_IGMP    => ipv6_active_igmp,
      ACTIVE_MPLS    => ipv6_active_mpls,
      ACTIVE_SCTP    => ipv6_active_sctp,
      ACTIVE_IPV6    => ipv6_active_ipv6,
      ACTIVE_IPV4    => ipv6_active_ipv4,
      ACTIVE_IPV6_NO_NEXT=>ipv6_active_ipv6_no_next,
      ACTIVE_UNKNOWN => ipv6_active_unknown,
      PROTOCOL       => ipv6_protocol,
      FRAGMENT_FIRST => ipv6_frag_first,
      FRAGMENT_LAST  => ipv6_frag_last,
      MALFORMED      => ipv6_malformed,
      RUNNING        => ipv6_running
   );

   sig_error_ipv6 <= pipe_active_ipv6 and ipv6_malformed;

   sig_running_transport <= (not pipe_sig_active_ethernet) or
                       pipe_vlan0_running or pipe_mpls0_running or
                       pipe_vlan1_running or pipe_mpls1_running or
                       ipv6_running or ipv4_running;

   -- ----------------------------------------------------------------------
   -- Transport layer ------------------------------------------------------
   -- ----------------------------------------------------------------------
   sig_transport_offset <= ipv4_offset_out when pipe_active_ipv4 = '1' else
                           ipv6_offset_out;

   sig_active_tcp          <= ipv4_active_tcp or ipv6_active_tcp;
   sig_active_udp          <= ipv4_active_udp or ipv6_active_udp;
   sig_active_icmp         <= ipv4_active_icmp or ipv6_active_icmp;
   sig_active_igmp         <= ipv4_active_igmp or ipv6_active_igmp;
   sig_active_mpls         <= ipv4_active_mpls or ipv6_active_mpls;
   sig_active_sctp         <= ipv4_active_sctp or ipv6_active_sctp;
   sig_active_ipv6_in_ip   <= ipv4_active_ipv6 or ipv6_active_ipv6;
   sig_active_ipv4_in_ip   <= ipv4_active_ipv4 or ipv6_active_ipv4;
   sig_active_ipv6_no_next <= ipv6_active_ipv6_no_next;
   sig_protocol <= ipv4_protocol when pipe_active_ipv4 = '1' else
                   ipv6_protocol;
   sig_frag_first <= ipv4_frag_first when pipe_active_ipv4 = '1' else
                     ipv6_frag_first when pipe_active_ipv6 = '1' else
                     '1';
   sig_frag_last  <= ipv4_frag_last when pipe_active_ipv4 = '1' else
                     ipv6_frag_last when pipe_active_ipv6 = '1' else
                     '1';

   -- ----------------------------------------------------------------------
   -- Output pipeline
   -- ----------------------------------------------------------------------
   out_pipe_din(0) <= sig_running_transport;
   out_pipe_din(1) <= sig_active_tcp;
   out_pipe_din(2) <= sig_active_udp;
   out_pipe_din(3) <= sig_active_icmp;
   out_pipe_din(4) <= sig_active_igmp;
   out_pipe_din(5) <= sig_active_mpls;
   out_pipe_din(6) <= sig_active_sctp;
   out_pipe_din(7) <= sig_active_ipv6_in_ip;
   out_pipe_din(8) <= sig_active_ipv4_in_ip;
   out_pipe_din(9) <= sig_active_ipv6_no_next;
   out_pipe_din(17 downto 10) <= sig_protocol;
   out_pipe_din(18+OFFSET_WIDTH-1 downto 18) <= sig_transport_offset;

   out_pipe_din(18+2*OFFSET_WIDTH-1 downto 18+OFFSET_WIDTH)
      <= pipe_internet_offset;
   out_pipe_din(18+2*OFFSET_WIDTH) <= pipe_active_ipv4;
   out_pipe_din(19+2*OFFSET_WIDTH) <= pipe_active_ipv6;
   out_pipe_din(20+2*OFFSET_WIDTH) <= pipe_running_internet;
   out_pipe_din(21+2*OFFSET_WIDTH) <= sig_error_ipv4;
   out_pipe_din(22+2*OFFSET_WIDTH) <= sig_error_ipv6;

   out_pipe_din(23+3*OFFSET_WIDTH-1 downto 23+2*OFFSET_WIDTH)
      <= pipe_sig_vlan0_offset;
   out_pipe_din(23+3*OFFSET_WIDTH) <= pipe_sig_active_vlan0;
   out_pipe_din(24+4*OFFSET_WIDTH-1 downto 24+3*OFFSET_WIDTH)
      <= pipe_sig_vlan1_offset;
   out_pipe_din(24+4*OFFSET_WIDTH) <= pipe_sig_active_vlan1;
   out_pipe_din(25+5*OFFSET_WIDTH-1 downto 25+4*OFFSET_WIDTH)
      <= pipe_sig_mpls0_offset;
   out_pipe_din(25+5*OFFSET_WIDTH) <= pipe_sig_active_mpls0;
   out_pipe_din(26+6*OFFSET_WIDTH-1 downto 26+5*OFFSET_WIDTH)
      <= pipe_sig_mpls1_offset;
   out_pipe_din(26+6*OFFSET_WIDTH) <= pipe_sig_active_mpls1;
   out_pipe_din(27+6*OFFSET_WIDTH) <= pipe_sig_running_vlan_mpls;

   out_pipe_din(28+6*OFFSET_WIDTH) <= pipe_sig_active_ethernet;

   out_pipe_din(29+6*OFFSET_WIDTH) <= mid_pipe_rx_sof_n;
   out_pipe_din(30+6*OFFSET_WIDTH) <= mid_pipe_rx_eof_n;
   out_pipe_din(31+6*OFFSET_WIDTH) <= mid_pipe_rx_sop_n;
   out_pipe_din(32+6*OFFSET_WIDTH) <= mid_pipe_rx_eop_n;
   out_pipe_din(33+6*OFFSET_WIDTH+DATA_WIDTH-1 downto 33+6*OFFSET_WIDTH)
      <= mid_pipe_rx_data;
   out_pipe_din(33+6*OFFSET_WIDTH+DATA_WIDTH+log2(DATA_WIDTH/8)-1 downto
                33+6*OFFSET_WIDTH+DATA_WIDTH)
      <= mid_pipe_rx_drem;
   out_pipe_din(33+6*OFFSET_WIDTH+DATA_WIDTH+log2(DATA_WIDTH/8))
      <= pipe_eth_running;
   out_pipe_din(34+6*OFFSET_WIDTH+DATA_WIDTH+log2(DATA_WIDTH/8))
      <= sig_frag_first;
   out_pipe_din(35+6*OFFSET_WIDTH+DATA_WIDTH+log2(DATA_WIDTH/8))
      <= sig_frag_last;


   out_pipe_rx_src_rdy <= not mid_pipe_rx_src_rdy_n;
   mid_pipe_rx_dst_rdy_n <= not out_pipe_rx_dst_rdy;

   --* Output pipe
   out_pipe_i : entity work.PIPE
   generic map(
      DATA_WIDTH     => DATA_WIDTH + log2(DATA_WIDTH/8) + 6*OFFSET_WIDTH + 36,
      USE_OUTREG     => OUT_PIPE_OUTREG,
      FAKE_PIPE      => not OUT_PIPE_EN
   )
   port map(
      CLK            => CLK,
      RESET          => RESET,
      -- Input interface ------------------------------------------------------
      IN_DATA        => out_pipe_din,
      IN_SRC_RDY     => out_pipe_rx_src_rdy,
      IN_DST_RDY     => out_pipe_rx_dst_rdy,
      -- Output interface -----------------------------------------------------
      OUT_DATA       => out_pipe_dout,
      OUT_SRC_RDY    => out_pipe_tx_src_rdy,
      OUT_DST_RDY    => out_pipe_tx_dst_rdy
   );

   RUNNING_TRANSPORT  <= out_pipe_dout(0)
                           when init_zeros = '0' else '0';
   ACTIVE_TCP         <= out_pipe_dout(1)
                           when init_zeros = '0' else '0';
   ACTIVE_UDP         <= out_pipe_dout(2)
                           when init_zeros = '0' else '0';
   ACTIVE_ICMP        <= out_pipe_dout(3)
                           when init_zeros = '0' else '0';
   ACTIVE_IGMP        <= out_pipe_dout(4)
                           when init_zeros = '0' else '0';
   ACTIVE_MPLS_TUN    <= out_pipe_dout(5)
                           when init_zeros = '0' else '0';
   ACTIVE_SCTP        <= out_pipe_dout(6)
                           when init_zeros = '0' else '0';
   ACTIVE_IPV6_TUN    <= out_pipe_dout(7)
                           when init_zeros = '0' else '0';
   ACTIVE_IPV4_TUN    <= out_pipe_dout(8)
                           when init_zeros = '0' else '0';
   ACTIVE_IPV6_NO_NEXT<= out_pipe_dout(9)
                           when init_zeros = '0' else '0';
   PROTOCOL           <= out_pipe_dout(17 downto 10);
   TRANSPORT_OFFSET   <= out_pipe_dout(18+OFFSET_WIDTH-1 downto 18);

   INTERNET_OFFSET
      <= out_pipe_dout(18+2*OFFSET_WIDTH-1 downto 18+OFFSET_WIDTH);
   ACTIVE_IPV4      <= out_pipe_dout(18+2*OFFSET_WIDTH)
                           when init_zeros = '0' else '0';
   ACTIVE_IPV6      <= out_pipe_dout(19+2*OFFSET_WIDTH)
                           when init_zeros = '0' else '0';
   RUNNING_INTERNET <= out_pipe_dout(20+2*OFFSET_WIDTH)
                           when init_zeros = '0' else '0';
   ERROR_IPV4       <= out_pipe_dout(21+2*OFFSET_WIDTH)
                           when init_zeros = '0' else '0';
   ERROR_IPV6       <= out_pipe_dout(22+2*OFFSET_WIDTH)
                           when init_zeros = '0' else '0';

   VLAN0_OFFSET
      <= out_pipe_dout(23+3*OFFSET_WIDTH-1 downto 23+2*OFFSET_WIDTH);
   ACTIVE_VLAN0 <= out_pipe_dout(23+3*OFFSET_WIDTH)
                           when init_zeros = '0' else '0';
   VLAN1_OFFSET
      <= out_pipe_dout(24+4*OFFSET_WIDTH-1 downto 24+3*OFFSET_WIDTH);
   ACTIVE_VLAN1 <= out_pipe_dout(24+4*OFFSET_WIDTH)
                           when init_zeros = '0' else '0';
   MPLS0_OFFSET
      <= out_pipe_dout(25+5*OFFSET_WIDTH-1 downto 25+4*OFFSET_WIDTH);
   ACTIVE_MPLS0 <= out_pipe_dout(25+5*OFFSET_WIDTH)
                           when init_zeros = '0' else '0';
   MPLS1_OFFSET
      <= out_pipe_dout(26+6*OFFSET_WIDTH-1 downto 26+5*OFFSET_WIDTH);
   ACTIVE_MPLS1 <= out_pipe_dout(26+6*OFFSET_WIDTH)
                           when init_zeros = '0' else '0';
   RUNNING_VLAN_MPLS <= out_pipe_dout(27+6*OFFSET_WIDTH)
                           when init_zeros = '0' else '0';

   ACTIVE_ETHERNET <= out_pipe_dout(28+6*OFFSET_WIDTH)
                           when init_zeros = '0' else '0';
   LINK_OFFSET <= sig_link_offset; -- constant
   RUNNING_LINK <=
      out_pipe_dout(33+6*OFFSET_WIDTH+DATA_WIDTH+log2(DATA_WIDTH/8))
                           when init_zeros = '0' else '0';
   FRAGMENT_FIRST <=
      out_pipe_dout(34+6*OFFSET_WIDTH+DATA_WIDTH+log2(DATA_WIDTH/8));
   FRAGMENT_LAST <=
      out_pipe_dout(35+6*OFFSET_WIDTH+DATA_WIDTH+log2(DATA_WIDTH/8));


   TX_SOF_N <= out_pipe_dout(29+6*OFFSET_WIDTH);
   TX_EOF_N <= out_pipe_dout(30+6*OFFSET_WIDTH);
   TX_SOP_N <= out_pipe_dout(31+6*OFFSET_WIDTH);
   TX_EOP_N <= out_pipe_dout(32+6*OFFSET_WIDTH);
   TX_DATA <=
   out_pipe_dout(33+6*OFFSET_WIDTH+DATA_WIDTH-1 downto 33+6*OFFSET_WIDTH);
   TX_DREM <=
   out_pipe_dout(33+6*OFFSET_WIDTH+DATA_WIDTH+log2(DATA_WIDTH/8)-1 downto
                33+6*OFFSET_WIDTH+DATA_WIDTH);

   TX_SRC_RDY_N <= not out_pipe_tx_src_rdy;
   out_pipe_tx_dst_rdy <= not TX_DST_RDY_N;

   reg_init_zeros_p : process(CLK)
   begin
      if CLK'event and CLK = '1' then
         if RESET = '1' then
            reg_init_zeros <= '1';
         else
            if out_pipe_tx_src_rdy = '1' then
               reg_init_zeros <= '0';
            end if;
         end if;
      end if;
   end process;

   init_zeros <= '1' when reg_init_zeros = '1' and out_pipe_tx_src_rdy = '0'
                 else '0';

end architecture full;
