-- host_int.vhd : IP core implementing host interface
--                (encapsulates clock generator, PCIe endpoint, MI32 switch,
--                 ID component, BootFPGA component, DMA bus and DMA module)
--!
--! \file
--! \brief IP core implementing host interface
--! \author Jiri Matousek <xmatou06@stud.fit.vutbr.cz>
--! \date 2014
--!
--! \section License
--!
--! Copyright (C) 2014 CESNET
--!
--! Redistribution and use in source and binary forms, with or without
--! modification, are permitted provided that the following conditions
--! are met:
--! 1. Redistributions of source code must retain the above copyright
--!    notice, this list of conditions and the following disclaimer.
--! 2. Redistributions in binary form must reproduce the above copyright
--!    notice, this list of conditions and the following disclaimer in
--!    the documentation and/or other materials provided with the
--!    distribution.
--! 3. Neither the name of the Company nor the names of its contributors
--!    may be used to endorse or promote products derived from this
--!    software without specific prior written permission.
--!
--! This software is provided ``as is'', and any express or implied
--! warranties, including, but not limited to, the implied warranties of
--! merchantability and fitness for a particular purpose are disclaimed.
--! In no event shall the company or contributors be liable for any
--! direct, indirect, incidental, special, exemplary, or consequential
--! damages (including, but not limited to, procurement of substitute
--! goods or services; loss of use, data, or profits; or business
--! interruption) however caused and on any theory of liability, whether
--! in contract, strict liability, or tort (including negligence or
--! otherwise) arising in any way out of the use of this software, even
--! if advised of the possibility of such damage.
--!


library ieee;
use ieee.std_logic_1164.all;
use ieee.std_logic_unsigned.all;
use ieee.std_logic_arith.all;

library unisim;
use unisim.vcomponents.all;

--! Package with log2 function
use work.math_pack.all;

--! \name Packages with constants declaration
-- ----------------------------------------------------------------------------
--! \brief NetCOPE constants
use work.fb100g1_const.all;
--! \brief Project constants
use work.host_int_const.all;


-- ----------------------------------------------------------------------------
--                            Entity declaration
-- ----------------------------------------------------------------------------

entity host_int is
generic (
   --! Divide factor for user clock (from 1400 MHz base frequency)
   USER_CLK_DIV          : integer := 14;
   --! Project text identifier
   PROJECT_TEXT          : std_logic_vector(255 downto 0) :=
      X"313030475F4E4943000000000000000000000000000000000000000000000000";
      --1 0 0 G _ N I C
   --! Is there separate FL interface for SZE headers from user application?
   SEPARATE_HEADER_IFC   : boolean := true;
   --! \brief Data width of IBUF's FL interface for SZE headers
   --! \details In bits, must be power of 2 and greater than 8.
   HDR_WIDTH             : integer := 128
);
port (
   --! \name PCIe reference clocks (100 MHz) and reset
   -- -------------------------------------------------------------------------
   --! \brief Reference clock from PCIe0 port, positive
   PCIE0_SYS_CLK_P       : in  std_logic;
   --! \brief Reference clock from PCIe0 port, negative
   PCIE0_SYS_CLK_N       : in  std_logic;
   --! \brief Reference clock from PCIe1 port, positive
   PCIE1_SYS_CLK_P       : in  std_logic;
   --! \brief Reference clock from PCIe1 port, negative
   PCIE1_SYS_CLK_N       : in  std_logic;
   --! \brief PCIe reset, active low
   PCIE_SYS_RESET_N      : in  std_logic;

   --! \name PCIe 0 Gen3 x8 interface
   -- -------------------------------------------------------------------------
   --! \brief Receive data, positive
   PCI0_EXP_RXP          : in  std_logic_vector(7 downto 0);
   --! \brief Receive data, negative
   PCI0_EXP_RXN          : in  std_logic_vector(7 downto 0);
   --! \brief Transmit data, positive
   PCI0_EXP_TXP          : out std_logic_vector(7 downto 0);
   --! \brief Transmit data, negative
   PCI0_EXP_TXN          : out std_logic_vector(7 downto 0);

   --! \name PCIe 1 Gen3 x8 interface
   -- -------------------------------------------------------------------------
   --! \brief Receive data, positive
   PCI1_EXP_RXP          : in  std_logic_vector(7 downto 0);
   --! \brief Receive data, negative
   PCI1_EXP_RXN          : in  std_logic_vector(7 downto 0);
   --! \brief Transmit data, positive
   PCI1_EXP_TXP          : out std_logic_vector(7 downto 0);
   --! \brief Transmit data, negative
   PCI1_EXP_TXN          : out std_logic_vector(7 downto 0);

   --! \name 125 MHz external clock
   -- -------------------------------------------------------------------------
   --! \brief 125 MHz clock, positive
   CLK125_P              : in  std_logic;
   --! \brief 125 MHz clock, negative
   CLK125_N              : in  std_logic;

   --! \name SPI interface to BootFPGA
   -- -------------------------------------------------------------------------
   --! \brief SPI clock
   BF_CLK                : out std_logic;
   --! \brief SPI not slave select 
   BF_NSS                : out std_logic;
   --! \brief SPI data from Master to Slave
   BF_MOSI               : out std_logic;
   --! \brief SPI data from Slave to Master
   BF_MISO               : in  std_logic;

   --! \name Clock and reset for 100G Ethernet interface
   -- -------------------------------------------------------------------------
   --! \brief Clock signal
   DRPCLK                : out std_logic;
   --! \brief Reset signal
   RESET                 : out std_logic;

   --! \name Clock and reset for user application
   -- -------------------------------------------------------------------------
   --! \brief Clock signal
   USER_CLK              : out std_logic;
   --! \brief Reset signal
   USER_RESET            : out std_logic;

   --! \name FrameLink Unaligned interface from user application (data payload)
   -- -------------------------------------------------------------------------
   --! \brief Payload data
   RX_DATA               : in  std_logic_vector(511 downto 0);
   --! \brief Position of the start of the payload
   --! \details Valid only if RX_SOP is set to '1'.
   RX_SOP_POS            : in  std_logic_vector(  2 downto 0);
   --! \brief Position of the end of the payload
   --! \details Valid only if RX_EOP is set to '1'.
   RX_EOP_POS            : in  std_logic_vector(  5 downto 0);
   --! \brief Start of the payload
   RX_SOP                : in  std_logic;
   --! \brief End of the payload
   RX_EOP                : in  std_logic;
   --! \brief Source is ready
   RX_SRC_RDY            : in  std_logic;
   --! \brief Destination is ready
   RX_DST_RDY            : out std_logic;

   --! \name FrameLink interface from user application (data header)
   -- -------------------------------------------------------------------------
   --! \brief Header data
   RX_HDATA              : in  std_logic_vector(HDR_WIDTH-1 downto 0);
   --! \brief Start of the header, active in '0'
   RX_HSOP_N             : in  std_logic;
   --! \brief End of the header, active in '0'
   RX_HEOP_N             : in  std_logic;
   --! \brief Position of the end of the header
   --! \details Valid only if RX_HEOP_N is set to '0'.
   RX_HREM               : in  std_logic_vector(log2(HDR_WIDTH/8)-1 downto 0);
   --! \brief Source is ready, active in '0'
   RX_HSRC_RDY_N         : in  std_logic;
   --! \brief Destination is ready, active in '0'
   RX_HDST_RDY_N         : out std_logic;

   --! \name FrameLink Unaligned interface to user application
   -- -------------------------------------------------------------------------
   --! \brief Packet data
   TX_DATA               : out std_logic_vector(511 downto 0);
   --! \brief Position of the start of the packet
   --! \details Valid only if TX_SOP is set to '1'.
   TX_SOP_POS            : out std_logic_vector(  2 downto 0);
   --! \brief Position of the end of the packet
   --! \details Valid only if TX_EOP is set to '1'.
   TX_EOP_POS            : out std_logic_vector(  5 downto 0);
   --! \brief Start of the packet
   TX_SOP                : out std_logic;
   --! \brief End of the packet
   TX_EOP                : out std_logic;
   --! \brief Source is ready
   TX_SRC_RDY            : out std_logic;
   --! \brief Destination is ready
   TX_DST_RDY            : in  std_logic;

   --! \name Clock and reset for MI32 interfaces
   -- -------------------------------------------------------------------------
   --! \brief Clock signal
   MI_CLK                : out std_logic;
   --! \brief Reset signal
   MI_RESET              : out std_logic;

   --! \name PCS/PMA layer MI32 interface
   -- -------------------------------------------------------------------------
   --! \brief Input Data
   ETH_MI_DWR            : out std_logic_vector(31 downto 0);
   --! \brief Address
   ETH_MI_ADDR           : out std_logic_vector(31 downto 0);
   --! \brief Read Request
   ETH_MI_RD             : out std_logic;
   --! \brief Write Request
   ETH_MI_WR             : out std_logic;
   --! \brief Byte Enable
   ETH_MI_BE             : out std_logic_vector( 3 downto 0);
   --! \brief Output Data
   ETH_MI_DRD            : in  std_logic_vector(31 downto 0);
   --! \brief Address Ready
   ETH_MI_ARDY           : in  std_logic;
   --! \brief Data Ready
   ETH_MI_DRDY           : in  std_logic;

   --! \name Network module MI32 interface
   -- -------------------------------------------------------------------------
   --! \brief Input Data
   NET_MI_DWR            : out std_logic_vector(31 downto 0);
   --! \brief Address
   NET_MI_ADDR           : out std_logic_vector(31 downto 0);
   --! \brief Read Request
   NET_MI_RD             : out std_logic;
   --! \brief Write Request
   NET_MI_WR             : out std_logic;
   --! \brief Byte Enable
   NET_MI_BE             : out std_logic_vector( 3 downto 0);
   --! \brief Output Data
   NET_MI_DRD            : in  std_logic_vector(31 downto 0);
   --! \brief Address Ready
   NET_MI_ARDY           : in  std_logic;
   --! \brief Data Ready
   NET_MI_DRDY           : in  std_logic;

   --! \name MI32 interface for free use
   -- -------------------------------------------------------------------------
   --! \brief Input Data
   USER_MI_DWR           : out std_logic_vector(31 downto 0);
   --! \brief Address
   USER_MI_ADDR          : out std_logic_vector(31 downto 0);
   --! \brief Read Request
   USER_MI_RD            : out std_logic;
   --! \brief Write Request
   USER_MI_WR            : out std_logic;
   --! \brief Byte Enable
   USER_MI_BE            : out std_logic_vector( 3 downto 0);
   --! \brief Output Data
   USER_MI_DRD           : in  std_logic_vector(31 downto 0);
   --! \brief Address Ready
   USER_MI_ARDY          : in  std_logic;
   --! \brief Data Ready
   USER_MI_DRDY          : in  std_logic
);
end entity host_int;


-- ----------------------------------------------------------------------------
--                         Architecture declaration
-- ----------------------------------------------------------------------------

architecture full of host_int is

   --! Signals declaration
   -- -------------------------------------------------------------------------

   --! PCIe reference clocks (100 MHz) and reset
   signal pcie0_sys_clk_p_ibuf_out : std_logic;
   signal pcie0_sys_clk_n_ibuf_out : std_logic;
   signal pcie0_sys_clk            : std_logic;
   signal pcie1_sys_clk_p_ibuf_out : std_logic;
   signal pcie1_sys_clk_n_ibuf_out : std_logic;
   signal pcie1_sys_clk            : std_logic;
   signal pcie_sys_reset_n_ibuf    : std_logic;
   signal pcie_sys_reset           : std_logic;

   --! PCIe data interface signals
   signal pci_exp_rxp_sig          : std_logic_vector(15 downto 0);
   signal pci_exp_rxn_sig          : std_logic_vector(15 downto 0);
   signal pci_exp_txp_sig          : std_logic_vector(15 downto 0);
   signal pci_exp_txn_sig          : std_logic_vector(15 downto 0);

   --! CLK125 clock and reset synchronized to this domain
   signal clk125_ibuf_out        : std_logic;
   signal clk125                 : std_logic;
   signal reset125               : std_logic;

   --! Clock and reset signals for MI32 bus
   signal mi_clk_int             : std_logic;
   signal mi_reset_int           : std_logic;

   --! Clkgen output clock and locked signal
   signal clk_user               : std_logic;
   signal clkgen_locked          : std_logic;

   --! Asynchronous reset synchronized to clkgen output clock
   signal clkgen_async_reset     : std_logic;
   signal reset_user             : std_logic;

   --! Clock and reset for DMA interface of PCIe core 0
   signal pcie_dma0_clk          : std_logic;
   signal pcie_dma0_reset        : std_logic;

   --! DMA UP interface of PCIe core 0 (clocked at pcie_dma0_clk)
   signal pcie_dma0_up_data      : std_logic_vector(255 downto 0);
   signal pcie_dma0_up_hdr       : std_logic_vector( 95 downto 0);
   signal pcie_dma0_up_sop       : std_logic;
   signal pcie_dma0_up_eop       : std_logic;
   signal pcie_dma0_up_src_rdy   : std_logic;
   signal pcie_dma0_up_dst_rdy   : std_logic;

   --! DMA DOWN interface of PCIe core 0 (clocked at pcie_dma0_clk)
   signal pcie_dma0_down_data    : std_logic_vector(255 downto 0);
   signal pcie_dma0_down_hdr     : std_logic_vector( 31 downto 0);
   signal pcie_dma0_down_sop     : std_logic;
   signal pcie_dma0_down_eop     : std_logic;
   signal pcie_dma0_down_src_rdy : std_logic;
   signal pcie_dma0_down_dst_rdy : std_logic;

   --! Clock and reset for DMA interface of PCIe core 1
   signal pcie_dma1_clk          : std_logic;
   signal pcie_dma1_reset        : std_logic;

   --! DMA UP interface of PCIe core 1 (clocked at pcie_dma1_clk)
   signal pcie_dma1_up_data      : std_logic_vector(255 downto 0);
   signal pcie_dma1_up_hdr       : std_logic_vector( 95 downto 0);
   signal pcie_dma1_up_sop       : std_logic;
   signal pcie_dma1_up_eop       : std_logic;
   signal pcie_dma1_up_src_rdy   : std_logic;
   signal pcie_dma1_up_dst_rdy   : std_logic;

   --! DMA DOWN interface of PCIe core 1 (clocked at pcie_dma1_clk)
   signal pcie_dma1_down_data    : std_logic_vector(255 downto 0);
   signal pcie_dma1_down_hdr     : std_logic_vector( 31 downto 0);
   signal pcie_dma1_down_sop     : std_logic;
   signal pcie_dma1_down_eop     : std_logic;
   signal pcie_dma1_down_src_rdy : std_logic;
   signal pcie_dma1_down_dst_rdy : std_logic;

   --! Configuration interrupt controller interface
   --! Internally connected to PCIe core 0.
   signal cfg_interrupt_msi_int  : std_logic_vector(31 downto 0);
   signal cfg_interrupt_msi_sent : std_logic;

   --! MI32 interface of PCIe core (clocked at mi_clk_int)
   --! Internally connected to PCIe core 0.
   signal mi_dwr                 : std_logic_vector(31 downto 0);
   signal mi_addr                : std_logic_vector(31 downto 0);
   signal mi_be                  : std_logic_vector( 3 downto 0);
   signal mi_rd                  : std_logic;
   signal mi_wr                  : std_logic;
   signal mi_drd                 : std_logic_vector(31 downto 0);
   signal mi_ardy                : std_logic;
   signal mi_drdy                : std_logic;

   --! MI32 interface to ID component (clocked at mi_clk_int)
   signal id_mi_dwr              : std_logic_vector(31 downto 0);
   signal id_mi_addr             : std_logic_vector(31 downto 0);
   signal id_mi_be               : std_logic_vector( 3 downto 0);
   signal id_mi_rd               : std_logic;
   signal id_mi_wr               : std_logic;
   signal id_mi_drd              : std_logic_vector(31 downto 0);
   signal id_mi_ardy             : std_logic;
   signal id_mi_drdy             : std_logic;

   --! MI32 interface to BootFPGA component (clocked at mi_clk_int)
   signal boot_mi_dwr            : std_logic_vector(31 downto 0);
   signal boot_mi_addr           : std_logic_vector(31 downto 0);
   signal boot_mi_be             : std_logic_vector( 3 downto 0);
   signal boot_mi_rd             : std_logic;
   signal boot_mi_wr             : std_logic;
   signal boot_mi_drd            : std_logic_vector(31 downto 0);
   signal boot_mi_ardy           : std_logic;
   signal boot_mi_drdy           : std_logic;

   --! MI32 interface to Timestamp unit (clocked at mi_clk_int)
   signal tsu_mi_rd              : std_logic;
   signal tsu_mi_wr              : std_logic;
   signal tsu_mi_ardy            : std_logic;
   signal tsu_mi_drdy            : std_logic;

   --! MI32 interface to DMA module (clocked at mi_clk_int)
   signal dma_mi_addr            : std_logic_vector(31 downto 0);
   signal dma_mi_ardy            : std_logic;
   signal dma_mi_wr              : std_logic;
   signal dma_mi_dwr             : std_logic_vector(31 downto 0);
   signal dma_mi_be              : std_logic_vector( 3 downto 0);
   signal dma_mi_rd              : std_logic;
   signal dma_mi_drdy            : std_logic;
   signal dma_mi_drd             : std_logic_vector(31 downto 0);

   --! ID module signals (clocked at mi_clk_int)
   signal id_interrupt_in        : std_logic_vector(31 downto 0);

   --! BootFPGA SPI interface signals
   signal bf_clk_sig             : std_logic;
   signal bf_nss_sig             : std_logic;
   signal bf_mosi_sig            : std_logic;
   signal bf_miso_sig            : std_logic;

   --! Outputs from CDC component on MI32 interface of DMA module
   signal dma_mi_async_dwr_out          : std_logic_vector(31 downto 0);
   signal dma_mi_async_addr_out         : std_logic_vector(31 downto 0);
   signal dma_mi_async_be_out           : std_logic_vector(3 downto 0);
   signal dma_mi_async_rd_out           : std_logic;
   signal dma_mi_async_wr_out           : std_logic;
   signal dma_mi_async_drd_out          : std_logic_vector(31 downto 0);
   signal dma_mi_async_ardy_out         : std_logic;
   signal dma_mi_async_drdy_out         : std_logic;

   --! Header inserter header interface signals
   signal hins_hdr_ready                : std_logic;
   signal hins_hdr_next                 : std_logic;

   --! Header inserter output interface signals
   signal hins_tx_data                  : std_logic_vector(511 downto 0);
   signal hins_tx_sop_pos               : std_logic_vector(2 downto 0);
   signal hins_tx_eop_pos               : std_logic_vector(5 downto 0);
   signal hins_tx_sop                   : std_logic;
   signal hins_tx_eop                   : std_logic;
   signal hins_tx_src_rdy               : std_logic;
   signal hins_tx_dst_rdy               : std_logic;

   --! HINS to DMA FLU pipe output interface signals
   signal hins_dma_flu_pipe_tx_data     : std_logic_vector(511 downto 0);
   signal hins_dma_flu_pipe_tx_sop_pos  : std_logic_vector(2 downto 0);
   signal hins_dma_flu_pipe_tx_eop_pos  : std_logic_vector(5 downto 0);
   signal hins_dma_flu_pipe_tx_sop      : std_logic;
   signal hins_dma_flu_pipe_tx_eop      : std_logic;
   signal hins_dma_flu_pipe_tx_src_rdy  : std_logic;
   signal hins_dma_flu_pipe_tx_dst_rdy  : std_logic;

   --! DMA UP interface of DMA module (clocked at clk_user)
   signal dma_up_data            : std_logic_vector(511 downto 0);
   signal dma_up_hdr             : std_logic_vector( 95 downto 0);
   signal dma_up_sop             : std_logic;
   signal dma_up_eop             : std_logic;
   signal dma_up_src_rdy         : std_logic;
   signal dma_up_dst_rdy         : std_logic;

   -- DMA DOWN interface of DMA module (clocked at clk_user)
   signal dma_down_data          : std_logic_vector(511 downto 0);
   signal dma_down_hdr           : std_logic_vector( 31 downto 0);
   signal dma_down_sop           : std_logic;
   signal dma_down_eop           : std_logic;
   signal dma_down_src_rdy       : std_logic;
   signal dma_down_dst_rdy       : std_logic;

   --! Interrup logic signals
   signal dma_rx_interrupt              : std_logic;
   signal dma_tx_interrupt              : std_logic;
   signal dma_interrupt                 : std_logic;
   signal dma_interrupt_reg             : std_logic;
   signal dma_interrupt_synchronized    : std_logic;
   signal interrupt_reg                 : std_logic;


-- ----------------------------------------------------------------------------
--                             Architecture body
-- ----------------------------------------------------------------------------

begin


   -- -------------------------------------------------------------------------
   --                         PCIe-related buffers
   -- -------------------------------------------------------------------------

   --! Input buffers related to PCIe 0 reference clocks
   -- -------------------------------------------------------------------------

   --! IBUF for PCIE0_SYS_CLK_P input port
   pcie0_sys_clk_p_ibuf_i : IBUF
   port map (
      O => pcie0_sys_clk_p_ibuf_out,
      I => PCIE0_SYS_CLK_P
   );

   --! IBUF for PCIE0_SYS_CLK_N input port
   pcie0_sys_clk_n_ibuf_i : IBUF
   port map (
      O => pcie0_sys_clk_n_ibuf_out,
      I => PCIE0_SYS_CLK_N
   );

   --! Input clock buffer for the PCIe 0 reference clock (100 MHz)
   pcie0_sys_clk_ibuf : IBUFDS_GTE2
   port map(
      I     => pcie0_sys_clk_p_ibuf_out,
      IB    => pcie0_sys_clk_n_ibuf_out,
      O     => pcie0_sys_clk,
      ODIV2 => open,
      CEB   => '0'
   );

   --! Input buffers related to PCIe 1 reference clocks
   -- -------------------------------------------------------------------------

   --! IBUF for PCIE1_SYS_CLK_P input port
   pcie1_sys_clk_p_ibuf_i : IBUF
   port map (
      O => pcie1_sys_clk_p_ibuf_out,
      I => PCIE1_SYS_CLK_P
   );

   --! IBUF for PCIE1_SYS_CLK_N input port
   pcie1_sys_clk_n_ibuf_i : IBUF
   port map (
      O => pcie1_sys_clk_n_ibuf_out,
      I => PCIE1_SYS_CLK_N
   );

   --! Input clock buffer for the PCIe 1 reference clock (100 MHz)
   pciei1_sys_clk_ibuf : IBUFDS_GTE2
   port map(
      I     => pcie1_sys_clk_p_ibuf_out,
      IB    => pcie1_sys_clk_n_ibuf_out,
      O     => pcie1_sys_clk,
      ODIV2 => open,
      CEB   => '0'
   );

   --! Logic related to PCIe reset
   -- -------------------------------------------------------------------------

   --! Input buffer for the PCIe reset
   pci_sys_reset_n_ibuf : IBUF
   port map (
      O => pcie_sys_reset_n_ibuf,
      I => PCIE_SYS_RESET_N
   );

   --! Switching PCIe reset to active high mode
   --! Can not be triple-registered because pcie#_sys_clk are from IBUFDS_GTE2.
   pcie_sys_reset <= NOT pcie_sys_reset_n_ibuf;


   -- -------------------------------------------------------------------------
   --            CLK125-related buffer and reset synchronization
   -- -------------------------------------------------------------------------

   --! 125 MHz input buffer
   clk125_ibuf : IBUFGDS
   port map (
      I  => CLK125_P,
      IB => CLK125_N,
      O  => clk125_ibuf_out
   );

   --! 125 MHz global clock buffer
   clk125_bufg : BUFG
   port map (
      I  => clk125_ibuf_out,
      O  => clk125
   );

   --! Triple-register the reset to synchronize it to clk125
   reset125_sync_i : entity work.ASYNC_RESET
   generic map(
      TWO_REG    => false
   )
   port map(
      CLK        => clk125,
      ASYNC_RST  => pcie_sys_reset,
      OUT_RST(0) => reset125
   );


   -- -------------------------------------------------------------------------
   --                           Clock generator
   -- -------------------------------------------------------------------------

   --! clkgen instantiation
   -- -------------------------------------------------------------------------

   clkgen_i : entity work.clkgen
   generic map (
      CLK0_DIV => real(USER_CLK_DIV)
   )
   port map (
      CLKIN    => clk125,
      RESET    => reset125,
      CLK200   => open,
      CLK0     => clk_user,
      CLK1     => open,
      CLK2     => open,
      CLK3     => open,
      CLK4     => open,
      CLK5     => open,
      LOCKED   => clkgen_locked
   );

   --! Reset synchronization to output clock
   -- -------------------------------------------------------------------------

   --! deriving asynchronous reset from clkgen output
   clkgen_async_reset <= NOT clkgen_locked;

   --! Triple-register the reset to synchronize it to clk_user
   reset_user_sync_i : entity work.ASYNC_RESET
   generic map(
      TWO_REG    => false
   )
   port map(
      CLK        => clk_user,
      ASYNC_RST  => clkgen_async_reset,
      OUT_RST(0) => reset_user
   );


   -- -------------------------------------------------------------------------
   --                            PCIe connection
   -- -------------------------------------------------------------------------

   --! PCIe wrapper instantiation
   -- -------------------------------------------------------------------------

   pcie_i : entity work.pcie
   generic map (
      --! Constants are defined in host_int_const
      VENDOR_ID              => PCIE_VENDOR_ID,
      DEVICE0_ID             => PCIE0_DEVICE_ID, 
      DEVICE1_ID             => PCIE1_DEVICE_ID,
      SUB_VENDOR_ID          => PCIE_SUB_VENDOR_ID,
      SUB_DEVICE0_ID         => PCIE0_SUB_DEVICE_ID,
      SUB_DEVICE1_ID         => PCIE1_SUB_DEVICE_ID,

      --! Constants are defined in fb100g1_const
      BAR0_BASE_ADDR         => BAR0_BASE_ADDR,
      BAR1_BASE_ADDR         => BAR1_BASE_ADDR,
      BAR2_BASE_ADDR         => BAR2_BASE_ADDR,
      BAR3_BASE_ADDR         => BAR3_BASE_ADDR,
      BAR4_BASE_ADDR         => BAR4_BASE_ADDR,
      BAR5_BASE_ADDR         => BAR5_BASE_ADDR,
      EXP_ROM_BASE_ADDR      => EXP_ROM_BASE_ADDR,

      --! Constant is defined in fb100g1_const
      AXI2MI_FIFO_SIZE       => AXI2MI_FIFO_SIZE,

      --! Constant is defined in host_int_const
      DOUBLE_PCIE            => DOUBLE_PCIE
   )
   port map (
      --! Input clock and reset (from PCIe ports, 100 MHz)
      PCIE0_SYS_CLK          => pcie0_sys_clk,
      PCIE1_SYS_CLK          => pcie1_sys_clk,
      PCIE_SYS_RESET         => pcie_sys_reset,

      --! PCIe interface
      PCI_EXP_RXP            => pci_exp_rxp_sig,
      PCI_EXP_RXN            => pci_exp_rxn_sig,
      PCI_EXP_TXP            => pci_exp_txp_sig,
      PCI_EXP_TXN            => pci_exp_txn_sig,

      --! Clock and reset for DMA bus from PCIe core 0 (250 MHz)
      --! Use this clock and reset only for DMA 0 bus!
      DMA0_CLK               => pcie_dma0_clk,
      DMA0_RESET             => pcie_dma0_reset,

      --! DMA UP interface (user application --> host system), PCIe core 0
      DMA0_UP_DATA           => pcie_dma0_up_data,
      DMA0_UP_HDR            => pcie_dma0_up_hdr,
      DMA0_UP_SOP            => pcie_dma0_up_sop,
      DMA0_UP_EOP            => pcie_dma0_up_eop,
      DMA0_UP_SRC_RDY        => pcie_dma0_up_src_rdy,
      DMA0_UP_DST_RDY        => pcie_dma0_up_dst_rdy,

      --! DMA DOWN interface (host system --> user application), PCIe core 0
      DMA0_DOWN_DATA         => pcie_dma0_down_data,
      DMA0_DOWN_HDR          => pcie_dma0_down_hdr,
      DMA0_DOWN_SOP          => pcie_dma0_down_sop,
      DMA0_DOWN_EOP          => pcie_dma0_down_eop,
      DMA0_DOWN_SRC_RDY      => pcie_dma0_down_src_rdy,
      DMA0_DOWN_DST_RDY      => pcie_dma0_down_dst_rdy,

      --! Clock and reset for DMA bus from PCIe core 1 (250 MHz)
      --! Use this clock and reset only for DMA 1 bus!
      DMA1_CLK               => pcie_dma1_clk,
      DMA1_RESET             => pcie_dma1_reset,

      --! DMA UP interface (user application --> host system), PCIe core 1
      DMA1_UP_DATA           => pcie_dma1_up_data,
      DMA1_UP_HDR            => pcie_dma1_up_hdr,
      DMA1_UP_SOP            => pcie_dma1_up_sop,
      DMA1_UP_EOP            => pcie_dma1_up_eop,
      DMA1_UP_SRC_RDY        => pcie_dma1_up_src_rdy,
      DMA1_UP_DST_RDY        => pcie_dma1_up_dst_rdy,

      --! DMA DOWN interface (host system --> user application), PCIe core 1
      DMA1_DOWN_DATA         => pcie_dma1_down_data,
      DMA1_DOWN_HDR          => pcie_dma1_down_hdr,
      DMA1_DOWN_SOP          => pcie_dma1_down_sop,
      DMA1_DOWN_EOP          => pcie_dma1_down_eop,
      DMA1_DOWN_SRC_RDY      => pcie_dma1_down_src_rdy,
      DMA1_DOWN_DST_RDY      => pcie_dma1_down_dst_rdy,

      --! Configuration status interface, both PCIe cores
      PCIE0_MAX_PAYLOAD      => open,
      PCIE0_MAX_READ_REQ     => open,
      PCIE1_MAX_PAYLOAD      => open,
      PCIE1_MAX_READ_REQ     => open,

      --! Configuration interrupt controller interface
      --! Ports are internally connected to PCIe core 0.
      CFG_INTERRUPT_MSI_INT  => cfg_interrupt_msi_int,
      CFG_INTERRUPT_MSI_SENT => cfg_interrupt_msi_sent,

      --! Output clock and reset (to user application, 250 MHz)
      --! Ports are internally connected to PCIe core 1.
      USER_CLK               => open,
      USER_RESET             => open,

      --! MI32 interface (root of the MI32 bus tree)
      --! Ports are internally connected to PCIe core 0.
      MI_CLK                 => mi_clk_int,
      MI_RESET               => mi_reset_int,
      MI_DWR                 => mi_dwr,
      MI_ADDR                => mi_addr,
      MI_BE                  => mi_be,
      MI_RD                  => mi_rd,
      MI_WR                  => mi_wr,
      MI_DRD                 => mi_drd,
      MI_ARDY                => mi_ardy,
      MI_DRDY                => mi_drdy
   );

   --! Distribution of single PCIe x16 interface to two PCIe x8 interfaces
   -- -------------------------------------------------------------------------

   --! PCIe 0 x8 interface
   pci_exp_rxp_sig(7 downto 0) <= PCI0_EXP_RXP;
   pci_exp_rxn_sig(7 downto 0) <= PCI0_EXP_RXN;
   PCI0_EXP_TXP <= pci_exp_txp_sig(7 downto 0);
   PCI0_EXP_TXN <= pci_exp_txn_sig(7 downto 0);

   --! PCIe 1 x8 interface
   pci_exp_rxp_sig(15 downto 8) <= PCI1_EXP_RXP;
   pci_exp_rxn_sig(15 downto 8) <= PCI1_EXP_RXN;
   PCI1_EXP_TXP <= pci_exp_txp_sig(15 downto 8);
   PCI1_EXP_TXN <= pci_exp_txn_sig(15 downto 8);


   -- -------------------------------------------------------------------------
   --                          NetCOPE MI32 switch
   -- -------------------------------------------------------------------------

   --! Definition of mi_clk_int and mi_reset_int (drived by clk125 and reset125)
   -- -------------------------------------------------------------------------

   mi_clk_int   <= clk125;
   mi_reset_int <= reset125;

   --! MI32 switch instantiation
   -- ------------------------------------------------------------------------- 

   netcope_adc_i : entity work.netcope_adc
   generic map (
      --! Constants are defined in host_int_const
      IN_PIPE      => ADC_IN_PIPE,
      ID_PIPE      => ADC_ID_PIPE,
      BOOT_PIPE    => ADC_BOOT_PIPE,
      NET_PIPE     => ADC_NET_PIPE,
      DMA_PIPE     => ADC_DMA_PIPE,
      ETH_PIPE     => ADC_ETH_PIPE,
      USER_PIPE    => ADC_USER_PIPE
   )
   port map (
      --! Clock and reset
      CLK          => mi_clk_int,
      RESET        => mi_reset_int,

      --! Input MI32 interface from PCIe
      IN_MI_DWR    => mi_dwr,
      IN_MI_ADDR   => mi_addr,
      IN_MI_RD     => mi_rd,
      IN_MI_WR     => mi_wr,
      IN_MI_BE     => mi_be,
      IN_MI_DRD    => mi_drd,
      IN_MI_ARDY   => mi_ardy,
      IN_MI_DRDY   => mi_drdy,

      --! Output MI32 interface to ID component
      --! (0x0000:0000-0x0000:1FFF)
      ID_MI_DWR    => id_mi_dwr,
      ID_MI_ADDR   => id_mi_addr,
      ID_MI_RD     => id_mi_rd,
      ID_MI_WR     => id_mi_wr,
      ID_MI_BE     => id_mi_be,
      ID_MI_DRD    => id_mi_drd,
      ID_MI_ARDY   => id_mi_ardy,
      ID_MI_DRDY   => id_mi_drdy,

      --! Output MI32 interface to BootFPGA component
      --! (0x0000:2000-0x0000:3FFF)
      BOOT_MI_DWR  => boot_mi_dwr,
      BOOT_MI_ADDR => boot_mi_addr,
      BOOT_MI_RD   => boot_mi_rd,
      BOOT_MI_WR   => boot_mi_wr,
      BOOT_MI_BE   => boot_mi_be,
      BOOT_MI_DRD  => boot_mi_drd,
      BOOT_MI_ARDY => boot_mi_ardy,
      BOOT_MI_DRDY => boot_mi_drdy,

      --! Output MI32 interface to Timestamp unit
      --! (0x0000:4000-0x0000:7FFF)
      TSU_MI_DWR   => open,
      TSU_MI_ADDR  => open,
      TSU_MI_RD    => tsu_mi_rd,
      TSU_MI_WR    => tsu_mi_wr,
      TSU_MI_BE    => open,
      TSU_MI_DRD   => (others => '0'),
      TSU_MI_ARDY  => tsu_mi_ardy,
      TSU_MI_DRDY  => tsu_mi_drdy,

      --! Output MI32 interface to Network module
      --! (0x0000:8000-0x0000:BFFF)
      NET_MI_DWR   => NET_MI_DWR,
      NET_MI_ADDR  => NET_MI_ADDR,
      NET_MI_RD    => NET_MI_RD,
      NET_MI_WR    => NET_MI_WR,
      NET_MI_BE    => NET_MI_BE,
      NET_MI_DRD   => NET_MI_DRD,
      NET_MI_ARDY  => NET_MI_ARDY,
      NET_MI_DRDY  => NET_MI_DRDY,

      --! Output MI32 interface to DMA module
      --! (0x0000:C000-0x0000:FFFF)
      DMA_MI_DWR   => dma_mi_dwr,
      DMA_MI_ADDR  => dma_mi_addr,
      DMA_MI_RD    => dma_mi_rd,
      DMA_MI_WR    => dma_mi_wr,
      DMA_MI_BE    => dma_mi_be,
      DMA_MI_DRD   => dma_mi_drd,
      DMA_MI_ARDY  => dma_mi_ardy,
      DMA_MI_DRDY  => dma_mi_drdy,

      --! Unused output MI32 interface
      --! (0x0001:0000-0x03FF:FFFF)

      --! Output MI32 interface to Ethernet physical layer
      --! (0x0400:0000-0x07FF:FFFF)
      ETH_MI_DWR   => ETH_MI_DWR,
      ETH_MI_ADDR  => ETH_MI_ADDR,
      ETH_MI_RD    => ETH_MI_RD,
      ETH_MI_WR    => ETH_MI_WR,
      ETH_MI_BE    => ETH_MI_BE,
      ETH_MI_DRD   => ETH_MI_DRD,
      ETH_MI_ARDY  => ETH_MI_ARDY,
      ETH_MI_DRDY  => ETH_MI_DRDY,

      --! Output MI32 interface to user application
      -- (0x0800:0000-0x3FFF:FFFF)
      USER_MI_DWR  => USER_MI_DWR,
      USER_MI_ADDR => USER_MI_ADDR,
      USER_MI_RD   => USER_MI_RD,
      USER_MI_WR   => USER_MI_WR,
      USER_MI_BE   => USER_MI_BE,
      USER_MI_DRD  => USER_MI_DRD,
      USER_MI_ARDY => USER_MI_ARDY,
      USER_MI_DRDY => USER_MI_DRDY
   );

   --! Imitation of TSU MI32 interface (which is actually not used)
   -- -------------------------------------------------------------------------

   tsu_mi_ardy <= tsu_mi_rd OR tsu_mi_wr;
   tsu_mi_drdy <= tsu_mi_rd;


   -- -------------------------------------------------------------------------
   --                             ID component
   -- -------------------------------------------------------------------------

   id_comp_i : entity work.ID_COMP_MI32_NOREC
   generic map (
      --! Constants are defined in host_int_const
      PROJECT_ID       => ID_PROJECT,
      SW_MAJOR         => ID_SW_MAJOR,
      SW_MINOR         => ID_SW_MINOR,
      HW_MAJOR         => ID_HW_MAJOR,
      HW_MINOR         => ID_HW_MINOR,
      PROJECT_TEXT     => PROJECT_TEXT,
      TX_CHANNELS      => ID_TX_CHANNELS,
      RX_CHANNELS      => ID_RX_CHANNELS,

      SYSMON_EN        => true,

      --! Constants are defined in fb100g1_const
      NETCOPE_MAJOR    => NETCOPE_MAJOR,
      NETCOPE_MINOR    => NETCOPE_MINOR,

      --! Generics of fpga entity
      BUILD_TIME       => (others => '0'),
      BUILD_UID        => (others => '0'),

      ICS_FREQUENCY    => X"00FA", -- 250 MHz fixed
      INTERRUPT_IGNORE => X"00FF",

      --! Constants are defined in host_int_const
      MAX_MTU_RX       => MAX_MTU_RX,
      MAX_MTU_TX       => MAX_MTU_TX
   )
   port map (
      --! Basic signals
      CLK              => mi_clk_int,
      RESET            => mi_reset_int,

      --! Misc ports
      COMMAND          => open,
      STATUS           => X"00000000000000000000000000000000",
      WE               => "1111",
      REPEATER         => open,
      SYSMON_ALARM     => open,

      --! Interrupt interface
      INTERRUPT_IN     => id_interrupt_in,
      INTR_RDY_IN      => open,
      INTERRUPT_OUT    => cfg_interrupt_msi_int,
      INTR_SENT        => cfg_interrupt_msi_sent,

      --! MI32 interface
      MI_DWR           => id_mi_dwr,
      MI_ADDR          => id_mi_addr,
      MI_RD            => id_mi_rd,
      MI_WR            => id_mi_wr,
      MI_BE            => id_mi_be,
      MI_DRD           => id_mi_drd,
      MI_ARDY          => id_mi_ardy,
      MI_DRDY          => id_mi_drdy
   );


   -- -------------------------------------------------------------------------
   --                          BootFPGA component
   -- -------------------------------------------------------------------------

   --! BootFPGA component instantiation
   -- -------------------------------------------------------------------------
   bootfpga_i : entity work.bootfpga
   port map (
      --! Clock and reset
      CLK     => mi_clk_int,
      RESET   => mi_reset_int,

      --! Input MI32 interface
      MI_DWR  => boot_mi_dwr,
      MI_ADDR => boot_mi_addr,
      MI_RD   => boot_mi_rd,
      MI_WR   => boot_mi_wr,
      MI_BE   => boot_mi_be,
      MI_DRD  => boot_mi_drd,
      MI_ARDY => boot_mi_ardy,
      MI_DRDY => boot_mi_drdy,

      --! SPI interface to BootFPGA
      BF_CLK  => bf_clk_sig,
      BF_NSS  => bf_nss_sig,
      BF_MOSI => bf_mosi_sig,
      BF_MISO => bf_miso_sig
   );

   --! IBUFs and OBUFs on SPI interface to BootFPGA
   -- -------------------------------------------------------------------------

   --! OBUF to BF_CLK output port
   bf_clk_obuf_i : OBUF
   port map (
      O => BF_CLK,
      I => bf_clk_sig
   );

   --! OBUF to BF_NSS output port
   bf_nss_obuf_i : OBUF
   port map (
      O => BF_NSS,
      I => bf_nss_sig
   );

   --! OBUF to BF_MOSI output port
   bf_mosi_obuf_i : OBUF
   port map (
      O => BF_MOSI,
      I => bf_mosi_sig
   );

   --! IBUF from BF_MISO input port
   bf_miso_ibuf_i : IBUF
   port map (
      O => bf_miso_sig,
      I => BF_MISO
   );


   -- -------------------------------------------------------------------------
   --               Clock domain crossing for DMA MI32 interface
   -- -------------------------------------------------------------------------

   dma_mi_async_i : entity work.mi32_async_handshake
   port map(
      --! Write interface
      CLK_M     => mi_clk_int,
      RESET_M   => mi_reset_int,
      MI_M_DWR  => dma_mi_dwr,
      MI_M_ADDR => dma_mi_addr,
      MI_M_BE   => dma_mi_be,
      MI_M_RD   => dma_mi_rd,
      MI_M_WR   => dma_mi_wr,
      MI_M_DRD  => dma_mi_drd,
      MI_M_ARDY => dma_mi_ardy,
      MI_M_DRDY => dma_mi_drdy,

      --! Read interface
      CLK_S     => clk_user,
      RESET_S   => reset_user,
      MI_S_DWR  => dma_mi_async_dwr_out,
      MI_S_ADDR => dma_mi_async_addr_out,
      MI_S_BE   => dma_mi_async_be_out,
      MI_S_RD   => dma_mi_async_rd_out,
      MI_S_WR   => dma_mi_async_wr_out,
      MI_S_DRD  => dma_mi_async_drd_out,
      MI_S_ARDY => dma_mi_async_ardy_out,
      MI_S_DRDY => dma_mi_async_drdy_out
   );


   -- -------------------------------------------------------------------------
   --                 Separate header interface IS present
   -- -------------------------------------------------------------------------

   separate_header_ifc_gen : if (SEPARATE_HEADER_IFC = true) generate
   begin

      --! Header inserter instantiation
      hins_i : entity work.HINS
      generic map (
         DATA_WIDTH    => 512,
         SOP_POS_WIDTH =>   3,
         HDR_WIDTH     => HDR_WIDTH
      )
      port map (
         --! Clock and reset
         CLK           => clk_user,
         RESET         => reset_user,

         --! Input FLU interface
         RX_DATA       => RX_DATA,
         RX_SOP_POS    => RX_SOP_POS,
         RX_EOP_POS    => RX_EOP_POS,
         RX_SOP        => RX_SOP,
         RX_EOP        => RX_EOP,
         RX_SRC_RDY    => RX_SRC_RDY,
         RX_DST_RDY    => RX_DST_RDY,

         --! Input header interface
         HDR_DATA      => RX_HDATA,
         HDR_READY     => hins_hdr_ready,
         HDR_NEXT      => hins_hdr_next,

         --! Output FLU interface
         TX_DATA       => hins_tx_data,
         TX_SOP_POS    => hins_tx_sop_pos,
         TX_EOP_POS    => hins_tx_eop_pos,
         TX_SOP        => hins_tx_sop,
         TX_EOP        => hins_tx_eop,
         TX_SRC_RDY    => hins_tx_src_rdy,
         TX_DST_RDY    => hins_tx_dst_rdy
      );

      --! Additional logic for header inserter header interface
      hins_hdr_ready <= NOT RX_HSRC_RDY_N;
      RX_HDST_RDY_N  <= NOT hins_hdr_next;

   end generate separate_header_ifc_gen;


   -- -------------------------------------------------------------------------
   --               Separate header interface IS NOT  present
   -- -------------------------------------------------------------------------

   no_separate_header_ifc_gen : if (SEPARATE_HEADER_IFC = false) generate
   begin

      --! Direct connection of input FLU interface to wires towards FLU PIPE 
      hins_tx_data    <= RX_DATA;
      hins_tx_sop_pos <= RX_SOP_POS;
      hins_tx_eop_pos <= RX_EOP_POS;
      hins_tx_sop     <= RX_SOP;
      hins_tx_eop     <= RX_EOP;
      hins_tx_src_rdy <= RX_SRC_RDY;
      RX_DST_RDY      <= hins_tx_dst_rdy;

      --! Imitation of FL interface for headers
      RX_HDST_RDY_N   <= '1'; 

   end generate no_separate_header_ifc_gen;


   -- -------------------------------------------------------------------------
   --                   FLU pipe between HINS and DMA module
   -- -------------------------------------------------------------------------

   hins_dma_flu_pipe_i : entity work.FLU_PIPE
   generic map (
      DATA_WIDTH    => 512,
      SOP_POS_WIDTH =>   3,
      USE_OUTREG    => true,
      FAKE_PIPE     => false
   )
   port map (
      --! Common interface
      CLK           => clk_user,
      RESET         => reset_user,

      --! Input interface
      RX_DATA       => hins_tx_data,
      RX_SOP_POS    => hins_tx_sop_pos,
      RX_EOP_POS    => hins_tx_eop_pos,
      RX_SOP        => hins_tx_sop,
      RX_EOP        => hins_tx_eop,
      RX_SRC_RDY    => hins_tx_src_rdy,
      RX_DST_RDY    => hins_tx_dst_rdy,

      --! Output interface
      TX_DATA       => hins_dma_flu_pipe_tx_data,
      TX_SOP_POS    => hins_dma_flu_pipe_tx_sop_pos,
      TX_EOP_POS    => hins_dma_flu_pipe_tx_eop_pos,
      TX_SOP        => hins_dma_flu_pipe_tx_sop,
      TX_EOP        => hins_dma_flu_pipe_tx_eop,
      TX_SRC_RDY    => hins_dma_flu_pipe_tx_src_rdy,
      TX_DST_RDY    => hins_dma_flu_pipe_tx_dst_rdy
   );


   -- -------------------------------------------------------------------------
   --                                  DMA module
   -- -------------------------------------------------------------------------
   
   dma_module_i : entity work.DMA_MODULE
   generic map (
      --! RX part
      RX_CHANNELS      => conv_integer(ID_RX_CHANNELS),
      RX_BUFFER_SIZE   => 8192,
      RX_FLU_WIDTH     => 512,
      RX_SOP_WIDTH     => 3,
      RX_DATA_ALIGN    => 8,

      --! TX part
      TX_CHANNELS      => conv_integer(ID_TX_CHANNELS),
      TX_BUFFER_SIZE   => 8192,
      TX_FLU_WIDTH     => 512,
      TX_SOP_WIDTH     => 3,
      TX_DATA_ALIGN    => 8)
   port map (
      --! Common interface
      CLK              => clk_user,
      RESET            => reset_user,

      --! Input FrameLink Unaligned interface (from user application)
      FLU_RX_CHANNEL   => "0",
      FLU_RX_DATA      => hins_dma_flu_pipe_tx_data,
      FLU_RX_SOP_POS   => hins_dma_flu_pipe_tx_sop_pos,
      FLU_RX_EOP_POS   => hins_dma_flu_pipe_tx_eop_pos,
      FLU_RX_SOP       => hins_dma_flu_pipe_tx_sop,
      FLU_RX_EOP       => hins_dma_flu_pipe_tx_eop,
      FLU_RX_SRC_RDY   => hins_dma_flu_pipe_tx_src_rdy,
      FLU_RX_DST_RDY   => hins_dma_flu_pipe_tx_dst_rdy,

      --! Output FrameLink Unaligned interface (to user application)
      FLU_TX_CHANNEL   => open,
      FLU_TX_DATA      => TX_DATA,
      FLU_TX_SOP_POS   => TX_SOP_POS,
      FLU_TX_EOP_POS   => TX_EOP_POS,
      FLU_TX_SOP       => TX_SOP,
      FLU_TX_EOP       => TX_EOP,
      FLU_TX_SRC_RDY   => TX_SRC_RDY,
      FLU_TX_DST_RDY   => TX_DST_RDY,

      --! DMA bus - upstream (to PCIe endpoint)
      DMA_UP_DATA      => dma_up_data,
      DMA_UP_HDR       => dma_up_hdr,
      DMA_UP_SOP       => dma_up_sop,
      DMA_UP_EOP       => dma_up_eop,
      DMA_UP_SRC_RDY   => dma_up_src_rdy,
      DMA_UP_DST_RDY   => dma_up_dst_rdy,

      --! DMA bus - downstream (from PCIe endpoint)
      DMA_DOWN_DATA    => dma_down_data,
      DMA_DOWN_HDR     => dma_down_hdr,
      DMA_DOWN_SOP     => dma_down_sop,
      DMA_DOWN_EOP     => dma_down_eop,
      DMA_DOWN_SRC_RDY => dma_down_src_rdy,
      DMA_DOWN_DST_RDY => dma_down_dst_rdy,

      --! SW access for configuration
      MI_DWR           => dma_mi_async_dwr_out,
      MI_ADDR          => dma_mi_async_addr_out,
      MI_BE            => dma_mi_async_be_out,
      MI_RD            => dma_mi_async_rd_out,
      MI_WR            => dma_mi_async_wr_out,
      MI_DRD           => dma_mi_async_drd_out,
      MI_ARDY          => dma_mi_async_ardy_out,
      MI_DRDY          => dma_mi_async_drdy_out,

      --! Interrupt interface
      RX_INTERRUPT     => dma_rx_interrupt,
      TX_INTERRUPT     => dma_tx_interrupt
   );


   -- ----------------------------------------------------------------------
   --            DMA bus between PCIe module and DMA module
   -- ----------------------------------------------------------------------

   dma_bus_i : entity work.dma_bus
   generic map (
      --! Constant is defined in host_int_const
      DOUBLE_PCIE            => DOUBLE_PCIE,

      PIPES                  => true,
      DEBUG_ENABLED          => false
   )
   port map (
      --! DMA module side clock and reset
      USER_CLK               => clk_user,
      USER_RESET             => reset_user,

      --! DMA_UP - user side interface (clocked at USER_CLK)
      USER_DMA_UP_DATA       => dma_up_data,
      USER_DMA_UP_HDR        => dma_up_hdr,
      USER_DMA_UP_SOP        => dma_up_sop,
      USER_DMA_UP_EOP        => dma_up_eop,
      USER_DMA_UP_SRC_RDY    => dma_up_src_rdy,
      USER_DMA_UP_DST_RDY    => dma_up_dst_rdy,

      --! DMA DOWN - user side interface (clocked at USER_CLK)
      USER_DMA_DOWN_DATA     => dma_down_data,
      USER_DMA_DOWN_HDR      => dma_down_hdr,
      USER_DMA_DOWN_SOP      => dma_down_sop,
      USER_DMA_DOWN_EOP      => dma_down_eop,
      USER_DMA_DOWN_SRC_RDY  => dma_down_src_rdy,
      USER_DMA_DOWN_DST_RDY  => dma_down_dst_rdy,

      --! Debug interface (currently not used)
      USER_DMA_DEBUG_PCIE0_BUS_ENABLED => '1',
      USER_DMA_DEBUG_PCIE_BUS_SAMPLE   => '0',
      USER_DMA_DEBUG_PCIE0_BUS_BYTES   => open,
      USER_DMA_DEBUG_PCIE1_BUS_BYTES   => open,

      --! Clock and reset for DMA 0 bus - PCIe side interface
      PCIE_DMA0_CLK          => pcie_dma0_clk,
      PCIE_DMA0_RESET        => pcie_dma0_reset,

      --! DMA UP 0 - PCIe side interface (clocked at PCIE_DMA0_CLK)
      PCIE_DMA0_UP_DATA      => pcie_dma0_up_data,
      PCIE_DMA0_UP_HDR       => pcie_dma0_up_hdr,
      PCIE_DMA0_UP_SOP       => pcie_dma0_up_sop,
      PCIE_DMA0_UP_EOP       => pcie_dma0_up_eop,
      PCIE_DMA0_UP_SRC_RDY   => pcie_dma0_up_src_rdy,
      PCIE_DMA0_UP_DST_RDY   => pcie_dma0_up_dst_rdy,

      --! DMA UP 0 - PCIe side interface (clocked at PCIE_DMA0_CLK)
      PCIE_DMA0_DOWN_DATA    => pcie_dma0_down_data,
      PCIE_DMA0_DOWN_HDR     => pcie_dma0_down_hdr,
      PCIE_DMA0_DOWN_SOP     => pcie_dma0_down_sop,
      PCIE_DMA0_DOWN_EOP     => pcie_dma0_down_eop,
      PCIE_DMA0_DOWN_SRC_RDY => pcie_dma0_down_src_rdy,
      PCIE_DMA0_DOWN_DST_RDY => pcie_dma0_down_dst_rdy,

      --! Clock and reset for DMA 1 bus - PCIe side interface
      PCIE_DMA1_CLK          => pcie_dma1_clk,
      PCIE_DMA1_RESET        => pcie_dma1_reset,

      --! DMA UP 1 - PCIe side interface (clocked at PCIE_DMA1_CLK)
      PCIE_DMA1_UP_DATA      => pcie_dma1_up_data,
      PCIE_DMA1_UP_HDR       => pcie_dma1_up_hdr,
      PCIE_DMA1_UP_SOP       => pcie_dma1_up_sop,
      PCIE_DMA1_UP_EOP       => pcie_dma1_up_eop,
      PCIE_DMA1_UP_SRC_RDY   => pcie_dma1_up_src_rdy,
      PCIE_DMA1_UP_DST_RDY   => pcie_dma1_up_dst_rdy,

      --! DMA UP 1 - PCIe side interface (clocked at PCIE_DMA1_CLK)
      PCIE_DMA1_DOWN_DATA    => pcie_dma1_down_data,
      PCIE_DMA1_DOWN_HDR     => pcie_dma1_down_hdr,
      PCIE_DMA1_DOWN_SOP     => pcie_dma1_down_sop,
      PCIE_DMA1_DOWN_EOP     => pcie_dma1_down_eop,
      PCIE_DMA1_DOWN_SRC_RDY => pcie_dma1_down_src_rdy,
      PCIE_DMA1_DOWN_DST_RDY => pcie_dma1_down_dst_rdy
   );


   -- -------------------------------------------------------------------------
   --                          Interrupt logic
   -- -------------------------------------------------------------------------

   --! DMA interrupts aggregation
   --! WATCH OUT: valid when only one interrupt vector is used

   dma_interrupt <= dma_rx_interrupt OR dma_tx_interrupt;

   --! Interrupt synchronization from clk_user to mi_clk_int clock domain
   -- -------------------------------------------------------------------------

   --! dma_interrupt sample register
   dma_interrupt_reg_p : process(clk_user)
   begin
      if (clk_user'event AND clk_user = '1') then
         if (dma_interrupt = '1') then
            dma_interrupt_reg <= '1';
         elsif (dma_interrupt_synchronized = '1') then
            dma_interrupt_reg <= '0';
         end if;
      end if;
   end process dma_interrupt_reg_p;

   --! dma_interrupt synchronization circuit
   dma_interrupt_sync_i : entity work.ASYNC_GENERAL
   generic map (
      TWO_REG            => true,
      DETECT_RISING_EDGE => false
   )
   port map (
      --! A clock domain (source)
      ACLK               => clk_user,
      ARST               => reset_user,
      ADATAIN            => dma_interrupt_reg,
      AREADY             => dma_interrupt_synchronized,

      --! B clock domain (destination)
      BCLK               => mi_clk_int,
      BRST               => mi_reset_int,
      BDATAOUT           => interrupt_reg
   );

   --! Interrupt output
   -- -------------------------------------------------------------------------

   id_interrupt_in <= (31 downto 1 => '0') & interrupt_reg;


   -- -------------------------------------------------------------------------
   --                           Driving output ports
   -- -------------------------------------------------------------------------

   --! Clock and reser for 100G Ethernet interface 
   DRPCLK     <= clk125;
   RESET      <= pcie_sys_reset;

   --! Clock and reset for user application 
   USER_CLK   <= clk_user;
   USER_RESET <= reset_user;

   --! Clock and reset for MI32 interfaces
   MI_CLK     <= mi_clk_int;
   MI_RESET   <= mi_reset_int;

end full;
