-- dma_module.vhd: Generic RX/TX DMA Module
-- Copyright (C) 2013 CESNET
-- Author(s): Martin Spinler <spinler@cesnet.cz>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--

library ieee;
use ieee.std_logic_1164.all;
use ieee.std_logic_arith.all;
use ieee.std_logic_unsigned.all;
use work.math_pack.all;
use work.dma_pkg.all;

-- -----------------------------------------------------------------------------
--                              Entity declaration
-- -----------------------------------------------------------------------------
entity DMA_MODULE is
   generic (
      --* Number of RX DMA channels
      RX_CHANNELS                : integer := 2;
      --* Size of each RX DMA buffer in bytes
      RX_BUFFER_SIZE             : integer := 8192;
      --* Input FrameLink width in bits
      RX_FLU_WIDTH               : integer := 512;
      --* Input FLU SOP width
      RX_SOP_WIDTH               : integer := 3;
      --* Data align for software
      RX_DATA_ALIGN              : integer := 8;

      --* Number of TX DMA channels
      TX_CHANNELS                : integer := 2;
      --* Size of each TX DMA buffer in bytes
      TX_BUFFER_SIZE             : integer := 8192;
      --* Output FrameLink width in bits
      TX_FLU_WIDTH               : integer := 512;
      --* Ouptut FLU SOP width
      TX_SOP_WIDTH               : integer := 3;
      --* Data align for software
      TX_DATA_ALIGN              : integer := 8;

      --* Generate packet counters
      ENABLE_COUNTERS            : boolean := true;
      --! DMA module is used in 2 PCIe configuration
      DOUBLE_PCIE                : boolean := false;

      --* Pipe generate
      PIPE_DMA_UP_RX_DESC        : boolean := true;
      PIPE_DMA_UP_TX_DESC        : boolean := true
   );
   port (
      -- Common interface
      CLK               : in std_logic;
      RESET             : in std_logic;

      --! Network interfaces
      --* Input FrameLink Unaligned interface
      FLU_RX_CHANNEL    : in  std_logic_vector(log2(RX_CHANNELS)-1 downto 0);
      FLU_RX_DATA       : in  std_logic_vector(RX_FLU_WIDTH-1 downto 0);
      FLU_RX_SOP_POS    : in  std_logic_vector(RX_SOP_WIDTH-1 downto 0);
      FLU_RX_EOP_POS    : in  std_logic_vector(log2(RX_FLU_WIDTH/8)-1 downto 0);
      FLU_RX_SOP        : in  std_logic;
      FLU_RX_EOP        : in  std_logic;
      FLU_RX_SRC_RDY    : in  std_logic;
      FLU_RX_DST_RDY    : out std_logic;

      --! Output FrameLink Unaligned interface
      FLU_TX_CHANNEL    : out std_logic_vector(log2(TX_CHANNELS)-1 downto 0);
      FLU_TX_DATA       : out std_logic_vector(TX_FLU_WIDTH-1 downto 0);
      FLU_TX_SOP_POS    : out std_logic_vector(TX_SOP_WIDTH-1 downto 0);
      FLU_TX_EOP_POS    : out std_logic_vector(log2(TX_FLU_WIDTH/8)-1 downto 0);
      FLU_TX_SOP        : out std_logic;
      FLU_TX_EOP        : out std_logic;
      FLU_TX_SRC_RDY    : out std_logic;
      FLU_TX_DST_RDY    : in  std_logic;

      --! DMA bus - Upstream
      DMA_UP_DATA       : out std_logic_vector(511 downto 0);
      DMA_UP_HDR        : out std_logic_vector(DMA_UPHDR_WIDTH-1 downto 0);
      DMA_UP_SOP        : out std_logic;
      DMA_UP_EOP        : out std_logic;
      DMA_UP_SRC_RDY    : out std_logic;
      DMA_UP_DST_RDY    : in  std_logic;

      --! DMA bus - Downstream
      DMA_DOWN_DATA     : in  std_logic_vector(511 downto 0);
      DMA_DOWN_HDR      : in  std_logic_vector(DMA_DOWNHDR_WIDTH-1 downto 0);
      DMA_DOWN_SOP      : in  std_logic;
      DMA_DOWN_EOP      : in  std_logic;
      DMA_DOWN_SRC_RDY  : in  std_logic;
      DMA_DOWN_DST_RDY  : out std_logic;

      RX_INTERRUPT      : out std_logic;
      TX_INTERRUPT      : out std_logic;

      --! SW Access for configuration
      MI_ADDR           : in  std_logic_vector(31 downto 0);
      MI_ARDY           : out std_logic;
      MI_BE             : in  std_logic_vector(3 downto 0);
      MI_DRD            : out std_logic_vector(31 downto 0);
      MI_DRDY           : out std_logic;
      MI_DWR            : in  std_logic_vector(31 downto 0);
      MI_RD             : in  std_logic;
      MI_WR             : in  std_logic

   );
end entity DMA_MODULE;

-- -----------------------------------------------------------------------------
--                           Architecture declaration
-- -----------------------------------------------------------------------------
architecture behavioral of DMA_MODULE is

   constant RX_BUFFERS              : integer := RX_FLU_WIDTH/8 / RX_DATA_ALIGN;
   constant TX_BUFFERS              : integer := TX_FLU_WIDTH/8 / TX_DATA_ALIGN;

   constant DMA_VER                 : integer := 1;
   constant DMA_FLAGS               : integer := 0;

   -- Configuration register
   signal reg_config_addr           : std_logic_vector(DMA_ADDR_WIDTH-1 downto 0) := (others => '0');
   signal config_addr_rxctrl        : std_logic_vector(DMA_ADDR_WIDTH-1 downto 0);
   signal config_addr_rxdesc        : std_logic_vector(DMA_ADDR_WIDTH-1 downto 0);
   signal config_addr_txctrl        : std_logic_vector(DMA_ADDR_WIDTH-1 downto 0);
   signal config_addr_txdesc        : std_logic_vector(DMA_ADDR_WIDTH-1 downto 0);

   -- RX signals
   signal rx_enable                 : std_logic_vector(RX_CHANNELS-1 downto 0) := (others => '1');
   signal rx_enable_buffer          : std_logic_vector(RX_CHANNELS-1 downto 0) := (others => '1');
   signal rx_discard                : std_logic_vector(RX_CHANNELS-1 downto 0) := (others => '1');
   signal rx_packet_new             : std_logic;
   signal rx_packet_disc            : std_logic;
   signal rx_packet_len             : std_logic_vector(log2(RX_BUFFER_SIZE) downto 0);
   signal rx_packet_channel         : std_logic_vector(log2(RX_CHANNELS)-1 downto 0);

   signal rx_buffer_read            : std_logic_vector(log2(RX_FLU_WIDTH/(RX_DATA_ALIGN*8)) downto 0);
   signal rx_buffer_channel         : std_logic_vector(log2(RX_CHANNELS)-1 downto 0);
   signal rx_buffer_data            : std_logic_vector(RX_FLU_WIDTH-1 downto 0);
   signal rx_buffer_dv              : std_logic;

   signal rx_desc_data              : std_logic_vector(DMA_DESC_WIDTH-1 downto 0);
   signal rx_desc_channel           : std_logic_vector(log2(RX_CHANNELS)-1 downto 0);
   signal rx_desc_rdy               : std_logic;
   signal rx_desc_update            : std_logic;
   signal rx_desc_update_len        : std_logic_vector(PAGE_WIDTH downto 0);

   --TX signals
   signal tx_enable                 : std_logic_vector(TX_CHANNELS-1 downto 0) := (others => '1');
   signal tx_packet_sent            : std_logic;
   signal tx_packet_len             : std_logic_vector(log2(TX_BUFFER_SIZE) downto 0);
   signal tx_packet_channel         : std_logic_vector(log2(TX_CHANNELS)-1 downto 0);

   signal write_data                : std_logic_vector(TX_FLU_WIDTH-1 downto 0);
   signal write_qword               : std_logic_vector(log2(TX_FLU_WIDTH/8 / TX_DATA_ALIGN) downto 0);
   signal write_channel             : std_logic_vector(log2(TX_CHANNELS)-1 downto 0);
   signal write_address             : std_logic_Vector(log2(TX_BUFFER_SIZE/TX_BUFFERS)-1 downto 0);
   signal write_done                : std_logic;
   signal write_done_size           : std_logic_vector(log2(TX_BUFFER_SIZE/TX_DATA_ALIGN) downto 0);
   signal write_done_channel        : std_logic_vector(log2(TX_CHANNELS)-1 downto 0);

   signal tx_desc_data              : std_logic_vector(DMA_DESC_WIDTH-1 downto 0);
   signal tx_desc_channel           : std_logic_vector(log2(TX_CHANNELS)-1 downto 0);
   signal tx_desc_rdy               : std_logic;
   signal tx_desc_update            : std_logic;
   signal tx_desc_update_len        : std_logic_vector(PAGE_WIDTH downto 0);

   -- DMA UP signals
   signal dma_up_switch_data        : std_logic_vector(511 downto 0);
   signal dma_up_switch_hdr         : std_logic_vector(DMA_UPHDR_WIDTH-1 downto 0);
   signal dma_up_switch_sop         : std_logic;
   signal dma_up_switch_eop         : std_logic;
   signal dma_up_switch_src_rdy     : std_logic;
   signal dma_up_switch_dst_rdy     : std_logic;

   signal dma_up_switch_rx_data     : std_logic_vector(511 downto 0);
   signal dma_up_switch_rx_hdr      : std_logic_vector(DMA_UPHDR_WIDTH-1 downto 0);
   signal dma_up_switch_rx_sop      : std_logic;
   signal dma_up_switch_rx_eop      : std_logic;
   signal dma_up_switch_rx_src_rdy  : std_logic;
   signal dma_up_switch_rx_dst_rdy  : std_logic;

   signal dma_up_switch_tx_data     : std_logic_vector(511 downto 0);
   signal dma_up_switch_tx_hdr      : std_logic_vector(DMA_UPHDR_WIDTH-1 downto 0);
   signal dma_up_switch_tx_sop      : std_logic;
   signal dma_up_switch_tx_eop      : std_logic;
   signal dma_up_switch_tx_src_rdy  : std_logic;
   signal dma_up_switch_tx_dst_rdy  : std_logic;

   signal dma_up_ctrl_rx_data       : std_logic_vector(511 downto 0);
   signal dma_up_ctrl_rx_hdr        : std_logic_vector(DMA_UPHDR_WIDTH-1 downto 0);
   signal dma_up_ctrl_rx_sop        : std_logic;
   signal dma_up_ctrl_rx_eop        : std_logic;
   signal dma_up_ctrl_rx_src_rdy    : std_logic;
   signal dma_up_ctrl_rx_dst_rdy    : std_logic;

   signal dma_up_ctrl_tx_data       : std_logic_vector(511 downto 0);
   signal dma_up_ctrl_tx_hdr        : std_logic_vector(DMA_UPHDR_WIDTH-1 downto 0);
   signal dma_up_ctrl_tx_sop        : std_logic;
   signal dma_up_ctrl_tx_eop        : std_logic;
   signal dma_up_ctrl_tx_src_rdy    : std_logic;
   signal dma_up_ctrl_tx_dst_rdy    : std_logic;

   signal dma_up_desc_rx_data       : std_logic_vector(511 downto 0);
   signal dma_up_desc_rx_hdr        : std_logic_vector(DMA_UPHDR_WIDTH-1 downto 0);
   signal dma_up_desc_rx_sop        : std_logic;
   signal dma_up_desc_rx_eop        : std_logic;
   signal dma_up_desc_rx_src_rdy    : std_logic;
   signal dma_up_desc_rx_dst_rdy    : std_logic;

   signal dma_up_desc_rx_pipe_data  : std_logic_vector(511 downto 0);
   signal dma_up_desc_rx_pipe_hdr   : std_logic_vector(DMA_UPHDR_WIDTH-1 downto 0);
   signal dma_up_desc_rx_pipe_sop   : std_logic;
   signal dma_up_desc_rx_pipe_eop   : std_logic;
   signal dma_up_desc_rx_pipe_src_rdy : std_logic;
   signal dma_up_desc_rx_pipe_dst_rdy : std_logic;

   signal dma_up_desc_tx_data       : std_logic_vector(511 downto 0);
   signal dma_up_desc_tx_hdr        : std_logic_vector(DMA_UPHDR_WIDTH-1 downto 0);
   signal dma_up_desc_tx_sop        : std_logic;
   signal dma_up_desc_tx_eop        : std_logic;
   signal dma_up_desc_tx_src_rdy    : std_logic;
   signal dma_up_desc_tx_dst_rdy    : std_logic;

   signal dma_up_desc_tx_pipe_data  : std_logic_vector(511 downto 0);
   signal dma_up_desc_tx_pipe_hdr   : std_logic_vector(DMA_UPHDR_WIDTH-1 downto 0);
   signal dma_up_desc_tx_pipe_sop   : std_logic;
   signal dma_up_desc_tx_pipe_eop   : std_logic;
   signal dma_up_desc_tx_pipe_src_rdy : std_logic;
   signal dma_up_desc_tx_pipe_dst_rdy : std_logic;

   -- DMA Down signals
   signal dma_down_split_data       : std_logic_vector(511 downto 0);
   signal dma_down_split_hdr        : std_logic_vector(DMA_DOWNHDR_WIDTH-1 downto 0);
   signal dma_down_split_sop        : std_logic;
   signal dma_down_split_eop        : std_logic;
   signal dma_down_split_src_rdy    : std_logic;
   signal dma_down_split_dst_rdy    : std_logic;

   signal dma_down_split_rx_data    : std_logic_vector(511 downto 0);
   signal dma_down_split_rx_hdr     : std_logic_vector(DMA_DOWNHDR_WIDTH-1 downto 0);
   signal dma_down_split_rx_sop     : std_logic;
   signal dma_down_split_rx_eop     : std_logic;
   signal dma_down_split_rx_src_rdy : std_logic;
   signal dma_down_split_rx_dst_rdy : std_logic;

   signal dma_down_split_tx_data    : std_logic_vector(511 downto 0);
   signal dma_down_split_tx_hdr     : std_logic_vector(DMA_DOWNHDR_WIDTH-1 downto 0);
   signal dma_down_split_tx_sop     : std_logic;
   signal dma_down_split_tx_eop     : std_logic;
   signal dma_down_split_tx_src_rdy : std_logic;
   signal dma_down_split_tx_dst_rdy : std_logic;

   signal dma_down_ctrl_rx_data     : std_logic_vector(511 downto 0);
   signal dma_down_ctrl_rx_hdr      : std_logic_vector(DMA_DOWNHDR_WIDTH-1 downto 0);
   signal dma_down_ctrl_rx_sop      : std_logic;
   signal dma_down_ctrl_rx_eop      : std_logic;
   signal dma_down_ctrl_rx_src_rdy  : std_logic;
   signal dma_down_ctrl_rx_dst_rdy  : std_logic;

   signal dma_down_ctrl_tx_data     : std_logic_vector(511 downto 0);
   signal dma_down_ctrl_tx_hdr      : std_logic_vector(DMA_DOWNHDR_WIDTH-1 downto 0);
   signal dma_down_ctrl_tx_sop      : std_logic;
   signal dma_down_ctrl_tx_eop      : std_logic;
   signal dma_down_ctrl_tx_src_rdy  : std_logic;
   signal dma_down_ctrl_tx_dst_rdy  : std_logic;

   signal dma_down_desc_rx_data     : std_logic_vector(511 downto 0);
   signal dma_down_desc_rx_hdr      : std_logic_vector(DMA_DOWNHDR_WIDTH-1 downto 0);
   signal dma_down_desc_rx_sop      : std_logic;
   signal dma_down_desc_rx_eop      : std_logic;
   signal dma_down_desc_rx_src_rdy  : std_logic;
   signal dma_down_desc_rx_dst_rdy  : std_logic;

   signal dma_down_desc_tx_data     : std_logic_vector(511 downto 0);
   signal dma_down_desc_tx_hdr      : std_logic_vector(DMA_DOWNHDR_WIDTH-1 downto 0);
   signal dma_down_desc_tx_sop      : std_logic;
   signal dma_down_desc_tx_eop      : std_logic;
   signal dma_down_desc_tx_src_rdy  : std_logic;
   signal dma_down_desc_tx_dst_rdy  : std_logic;

   -- MI for RX controller
   signal mi_rx_addr                : std_logic_vector(31 downto 0);
   signal mi_rx_ardy                : std_logic;
   signal mi_rx_be                  : std_logic_vector(3 downto 0);
   signal mi_rx_drd                 : std_logic_vector(31 downto 0);
   signal mi_rx_drdy                : std_logic;
   signal mi_rx_dwr                 : std_logic_vector(31 downto 0);
   signal mi_rx_rd                  : std_logic;
   signal mi_rx_wr                  : std_logic;

   -- MI for TX controller
   signal mi_tx_addr                : std_logic_vector(31 downto 0);
   signal mi_tx_ardy                : std_logic;
   signal mi_tx_be                  : std_logic_vector(3 downto 0);
   signal mi_tx_drd                 : std_logic_vector(31 downto 0);
   signal mi_tx_drdy                : std_logic;
   signal mi_tx_dwr                 : std_logic_vector(31 downto 0);
   signal mi_tx_rd                  : std_logic;
   signal mi_tx_wr                  : std_logic;

   -- MI for counter
   signal mi_cntr_addr              : std_logic_vector(31 downto 0);
   signal mi_cntr_ardy              : std_logic;
   signal mi_cntr_be                : std_logic_vector(3 downto 0);
   signal mi_cntr_drd               : std_logic_vector(31 downto 0);
   signal mi_cntr_drdy              : std_logic;
   signal mi_cntr_dwr               : std_logic_vector(31 downto 0);
   signal mi_cntr_rd                : std_logic;
   signal mi_cntr_wr                : std_logic;
   signal reg_mi_cntr_drd           : std_logic_vector(31 downto 0);
   signal mux_mi_cntr_drd           : std_logic_vector(63 downto 0);

   -- MI for configuration
   signal mi_conf_addr              : std_logic_vector(31 downto 0);
   signal mi_conf_ardy              : std_logic;
   signal mi_conf_be                : std_logic_vector(3 downto 0);
   signal mi_conf_drd               : std_logic_vector(31 downto 0);
   signal mi_conf_drdy              : std_logic;
   signal mi_conf_dwr               : std_logic_vector(31 downto 0);
   signal mi_conf_rd                : std_logic;
   signal mi_conf_wr                : std_logic;

   signal mux_mi_cntr_rx_in         : std_logic_vector(128*RX_CHANNELS-1 downto 0);
   signal mux_mi_cntr_tx_in         : std_logic_vector(128*TX_CHANNELS-1 downto 0);
   signal mux_mi_cntr_rx            : std_logic_vector(63 downto 0);
   signal mux_mi_cntr_tx            : std_logic_vector(63 downto 0);

   type t_rx_cntr                   is array(RX_CHANNELS-1 downto 0) of std_logic_vector(63 downto 0);
   type t_tx_cntr                   is array(TX_CHANNELS-1 downto 0) of std_logic_vector(63 downto 0);
   signal rx_cntr_new               : t_rx_cntr := (others => (others => '0'));
   signal rx_cntr_disc              : t_rx_cntr := (others => (others => '0'));
   signal tx_cntr_sent              : t_tx_cntr := (others => (others => '0'));

   signal zeros                     : std_logic_vector(511 downto 0) := (others =>'0');

begin

   -- ---------------------------------------------------------------
   --                     Configuration register
   -- ---------------------------------------------------------------

   mi_conf_ardy         <= mi_conf_rd or mi_conf_wr;
   mi_conf_drdy         <= mi_conf_rd;
   mi_conf_drd          <= conv_std_logic_vector(DMA_FLAGS, 8)   & conv_std_logic_vector(DMA_VER, 8) &
                           conv_std_logic_vector(TX_CHANNELS, 8) & conv_std_logic_vector(RX_CHANNELS, 8);

   config_addr_rxdesc   <= reg_config_addr + 4096*0;
   config_addr_rxctrl   <= reg_config_addr + 4096*1;
   config_addr_txdesc   <= reg_config_addr + 4096*2;
   config_addr_txctrl   <= reg_config_addr + 4096*3;

   reg_config_p : process(CLK)
   begin
      if(CLK'event and CLK = '1') then
         if(mi_conf_wr = '1') then
            if(mi_conf_addr(2) = '0') then
               reg_config_addr(31 downto 0)    <= mi_conf_dwr;
            else
               reg_config_addr(63 downto 32)   <= mi_conf_dwr;
            end if;
         end if;
      end if;
   end process;

   -- ---------------------------------------------------------------
   --                         RX Direction
   -- ---------------------------------------------------------------

gen_no_rx : if(RX_CHANNELS = 0) generate

   dma_up_switch_rx_data      <= (others => '0');
   dma_up_switch_rx_hdr       <= (others => '0');
   dma_up_switch_rx_sop       <= '0';
   dma_up_switch_rx_eop       <= '0';
   dma_up_switch_rx_src_rdy   <= '0';

   dma_down_split_rx_dst_rdy  <= '0';

   mi_rx_ardy                 <= mi_rx_rd or mi_rx_wr;
   mi_rx_drdy                 <= mi_rx_rd;
   mi_rx_drd                  <= x"DEAD0C71";

   FLU_RX_DST_RDY             <= '0';

   RX_INTERRUPT               <= '0';

end generate;

gen_rx : if(RX_CHANNELS > 0) generate

   DMA_BUFFER_RX_I : entity work.DMA_BUFFER_RX
   generic map(
      CHANNELS          => RX_CHANNELS,
      BUFFER_SIZE       => RX_BUFFER_SIZE,
      FLU_WIDTH         => RX_FLU_WIDTH,
      SOP_WIDTH         => RX_SOP_WIDTH,
      DATA_ALIGN        => RX_DATA_ALIGN
   )
   port map(
      CLK               => CLK,
      RESET             => RESET,

      ENABLE            => rx_enable,
      DISCARD           => rx_discard,

      RX_CHANNEL        => FLU_RX_CHANNEL,
      RX_DATA           => FLU_RX_DATA,
      RX_SOP_POS        => FLU_RX_SOP_POS,
      RX_EOP_POS        => FLU_RX_EOP_POS,
      RX_SOP            => FLU_RX_SOP,
      RX_EOP            => FLU_RX_EOP,
      RX_SRC_RDY        => FLU_RX_SRC_RDY,
      RX_DST_RDY        => FLU_RX_DST_RDY,

      PACKET_NEW        => rx_packet_new,
      PACKET_DISC       => rx_packet_disc,
      PACKET_CHANNEL    => rx_packet_channel,
      PACKET_LENGTH     => rx_packet_len,

      DMA_READ          => rx_buffer_read,
      DMA_CHANNEL       => rx_buffer_channel,
      DMA_DATA          => rx_buffer_data,
      DMA_DV            => rx_buffer_dv
   );

   DMA_CTRL_RX_I : entity work.DMA_CTRL_RX
   generic map(
      CHANNELS          => RX_CHANNELS,
      BUFFERS           => RX_BUFFERS,
      BUFFER_SIZE       => RX_BUFFER_SIZE,
      DATA_ALIGN        => RX_DATA_ALIGN,
      DOUBLE_PCIE       => DOUBLE_PCIE
   )
   port map(
      CLK               => CLK,
      RESET             => RESET,

      ENABLE            => rx_enable,
      ENABLE_BUFFER     => rx_enable_buffer,
      DISCARD           => rx_discard,
      INTERRUPT         => RX_INTERRUPT,
      CONFIG_ADDR       => config_addr_rxctrl,

      PACKET_NEW        => rx_packet_new,
      PACKET_DISC       => rx_packet_disc,
      PACKET_CHANNEL    => rx_packet_channel,
      PACKET_LENGTH     => rx_packet_len,

      BUFFER_READ       => rx_buffer_read,
      BUFFER_CHANNEL    => rx_buffer_channel,
      BUFFER_DATA       => rx_buffer_data,
      BUFFER_DV         => rx_buffer_dv,

      DESC_DATA         => rx_desc_data,
      DESC_RDY          => rx_desc_rdy,
      DESC_CHANNEL      => rx_desc_channel,
      DESC_UPDATE       => rx_desc_update,
      DESC_UPDATE_LEN   => rx_desc_update_len,

      DMA_UP_DATA       => dma_up_ctrl_rx_data,
      DMA_UP_HDR        => dma_up_ctrl_rx_hdr,
      DMA_UP_SOP        => dma_up_ctrl_rx_sop,
      DMA_UP_EOP        => dma_up_ctrl_rx_eop,
      DMA_UP_SRC_RDY    => dma_up_ctrl_rx_src_rdy,
      DMA_UP_DST_RDY    => dma_up_ctrl_rx_dst_rdy,

      DMA_DOWN_DATA     => dma_down_ctrl_rx_data,
      DMA_DOWN_HDR      => dma_down_ctrl_rx_hdr,
      DMA_DOWN_SOP      => dma_down_ctrl_rx_sop,
      DMA_DOWN_EOP      => dma_down_ctrl_rx_eop,
      DMA_DOWN_SRC_RDY  => dma_down_ctrl_rx_src_rdy,
      DMA_DOWN_DST_RDY  => dma_down_ctrl_rx_dst_rdy,

      MI_ADDR           => mi_rx_addr,
      MI_ARDY           => mi_rx_ardy,
      MI_BE             => mi_rx_be,
      MI_DRD            => mi_rx_drd,
      MI_DRDY           => mi_rx_drdy,
      MI_DWR            => mi_rx_dwr,
      MI_RD             => mi_rx_rd,
      MI_WR             => mi_rx_wr
   );

   dma_desc_manager_rx_i : entity work.dma_desc_manager
   generic map (
      DATA_ALIGN        => RX_DATA_ALIGN,
      CHANNELS          => RX_CHANNELS,
      DMA_ID            => 4
   )
   port map(
      CLK               => CLK,
      RESET             => RESET,

      ENABLE            => rx_enable,
      CONFIG_ADDR       => config_addr_rxdesc,

      DESC_DATA         => rx_desc_data,
      DESC_RDY          => rx_desc_rdy,
      DESC_CHANNEL      => rx_desc_channel,
      DESC_UPDATE       => rx_desc_update,
      DESC_UPDATE_LEN   => rx_desc_update_len,

      DMA_UP_DATA       => dma_up_desc_rx_pipe_data,
      DMA_UP_HDR        => dma_up_desc_rx_pipe_hdr,
      DMA_UP_SOP        => dma_up_desc_rx_pipe_sop,
      DMA_UP_EOP        => dma_up_desc_rx_pipe_eop,
      DMA_UP_SRC_RDY    => dma_up_desc_rx_pipe_src_rdy,
      DMA_UP_DST_RDY    => dma_up_desc_rx_pipe_dst_rdy,

      DMA_DOWN_DATA     => dma_down_desc_rx_data,
      DMA_DOWN_HDR      => dma_down_desc_rx_hdr,
      DMA_DOWN_SOP      => dma_down_desc_rx_sop,
      DMA_DOWN_EOP      => dma_down_desc_rx_eop,
      DMA_DOWN_SRC_RDY  => dma_down_desc_rx_src_rdy,
      DMA_DOWN_DST_RDY  => dma_down_desc_rx_dst_rdy
   );

   gen_pipe_up_rx_desc : if(PIPE_DMA_UP_RX_DESC = true) generate

   dma_pipe_up_rx_descriptor_i : entity work.DMA_PIPE
   generic map(
      DATA_WIDTH        => 512,
      HDR_WIDTH         => 96,
      USE_OUTREG        => true,
      FAKE_PIPE         => false
   )
   port map(
      CLK               => CLK,
      RESET             => RESET,

      RX_DATA           => dma_up_desc_rx_pipe_data,
      RX_HDR            => dma_up_desc_rx_pipe_hdr,
      RX_SOP            => dma_up_desc_rx_pipe_sop,
      RX_EOP            => dma_up_desc_rx_pipe_eop,
      RX_SRC_RDY        => dma_up_desc_rx_pipe_src_rdy,
      RX_DST_RDY        => dma_up_desc_rx_pipe_dst_rdy,

      TX_DATA           => dma_up_desc_rx_data,
      TX_HDR            => dma_up_desc_rx_hdr,
      TX_SOP            => dma_up_desc_rx_sop,
      TX_EOP            => dma_up_desc_rx_eop,
      TX_SRC_RDY        => dma_up_desc_rx_src_rdy,
      TX_DST_RDY        => dma_up_desc_rx_dst_rdy
   );

   end generate;


   not_pipe_up_rx_desc : if(PIPE_DMA_UP_RX_DESC = false) generate

   dma_up_desc_rx_data     <= dma_up_desc_rx_pipe_data;
   dma_up_desc_rx_hdr      <= dma_up_desc_rx_pipe_hdr;
   dma_up_desc_rx_sop      <= dma_up_desc_rx_pipe_sop;
   dma_up_desc_rx_eop      <= dma_up_desc_rx_pipe_eop;
   dma_up_desc_rx_src_rdy  <= dma_up_desc_rx_pipe_src_rdy;
   dma_up_desc_rx_pipe_dst_rdy   <=dma_up_desc_rx_dst_rdy;

   end generate;


   switch_rx_i : entity work.dma_switch
   generic map(
      DATA_WIDTH        => 512,
      HDR_WIDTH         => 96
   )
   PORT map (
      CLK               => CLK,
      RESET             => RESET,

      DMA_TX_DATA       => dma_up_switch_rx_data,
      DMA_TX_HDR        => dma_up_switch_rx_hdr,
      DMA_TX_SOP        => dma_up_switch_rx_sop,
      DMA_TX_EOP        => dma_up_switch_rx_eop,
      DMA_TX_SRC_RDY    => dma_up_switch_rx_src_rdy,
      DMA_TX_DST_RDY    => dma_up_switch_rx_dst_rdy,

      DMA_RX0_DATA      => dma_up_ctrl_rx_data,
      DMA_RX0_HDR       => dma_up_ctrl_rx_hdr,
      DMA_RX0_SOP       => dma_up_ctrl_rx_sop,
      DMA_RX0_EOP       => dma_up_ctrl_rx_eop,
      DMA_RX0_SRC_RDY   => dma_up_ctrl_rx_src_rdy,
      DMA_RX0_DST_RDY   => dma_up_ctrl_rx_dst_rdy,

      DMA_RX1_DATA      => dma_up_desc_rx_data,
      DMA_RX1_HDR       => dma_up_desc_rx_hdr,
      DMA_RX1_SOP       => dma_up_desc_rx_sop,
      DMA_RX1_EOP       => dma_up_desc_rx_eop,
      DMA_RX1_SRC_RDY   => dma_up_desc_rx_src_rdy,
      DMA_RX1_DST_RDY   => dma_up_desc_rx_dst_rdy
   );

   splitter_rx_i : entity work.dma_splitter
   generic map(
      DATA_WIDTH        => 512,
      HDR_WIDTH         => 32,
      HDR_BIT           => 26
   )
   PORT map (
      CLK               => CLK,
      RESET             => RESET,

      DMA_RX_DATA       => dma_down_split_rx_data,
      DMA_RX_HDR        => dma_down_split_rx_hdr,
      DMA_RX_SOP        => dma_down_split_rx_sop,
      DMA_RX_EOP        => dma_down_split_rx_eop,
      DMA_RX_SRC_RDY    => dma_down_split_rx_src_rdy,
      DMA_RX_DST_RDY    => dma_down_split_rx_dst_rdy,

      DMA_TX0_DATA      => dma_down_ctrl_rx_data,
      DMA_TX0_HDR       => dma_down_ctrl_rx_hdr,
      DMA_TX0_SOP       => dma_down_ctrl_rx_sop,
      DMA_TX0_EOP       => dma_down_ctrl_rx_eop,
      DMA_TX0_SRC_RDY   => dma_down_ctrl_rx_src_rdy,
      DMA_TX0_DST_RDY   => dma_down_ctrl_rx_dst_rdy,

      DMA_TX1_DATA      => dma_down_desc_rx_data,
      DMA_TX1_HDR       => dma_down_desc_rx_hdr,
      DMA_TX1_SOP       => dma_down_desc_rx_sop,
      DMA_TX1_EOP       => dma_down_desc_rx_eop,
      DMA_TX1_SRC_RDY   => dma_down_desc_rx_src_rdy,
      DMA_TX1_DST_RDY   => dma_down_desc_rx_dst_rdy
   );

end generate;

   -- ---------------------------------------------------------------
   --                         TX Direction
   -- ---------------------------------------------------------------

gen_no_tx : if(TX_CHANNELS = 0) generate

   dma_up_switch_tx_data      <= (others => '0');
   dma_up_switch_tx_hdr       <= (others => '0');
   dma_up_switch_tx_sop       <= '0';
   dma_up_switch_tx_eop       <= '0';
   dma_up_switch_tx_src_rdy   <= '0';

   dma_down_split_tx_dst_rdy  <= '0';

   mi_tx_ardy                 <= mi_tx_rd or mi_tx_wr;
   mi_tx_drdy                 <= mi_tx_rd;
   mi_tx_drd                  <= X"DEAD1C71";

   FLU_TX_DATA                <= (others => '0');
   FLU_TX_SOP_POS             <= (others => '0');
   FLU_TX_EOP_POS             <= (others => '0');
   FLU_TX_SOP                 <= '0';
   FLU_TX_EOP                 <= '0';
   FLU_TX_SRC_RDY             <= '0';

   TX_INTERRUPT               <= '0';

end generate;

gen_tx : if(TX_CHANNELS > 0) generate

   DMA_BUFFER_TX_I : entity work.DMA_BUFFER_TX
   generic map(
      CHANNELS          => TX_CHANNELS,
      BUFFER_SIZE       => TX_BUFFER_SIZE,
      FLU_WIDTH         => TX_FLU_WIDTH,
      SOP_WIDTH         => TX_SOP_WIDTH,
      DATA_ALIGN        => TX_DATA_ALIGN
   )
   port map(
      CLK               => CLK,
      RESET             => RESET,

      ENABLE            => tx_enable,

      TX_CHANNEL        => FLU_TX_CHANNEL,
      TX_DATA           => FLU_TX_DATA,
      TX_SOP_POS        => FLU_TX_SOP_POS,
      TX_EOP_POS        => FLU_TX_EOP_POS,
      TX_SOP            => FLU_TX_SOP,
      TX_EOP            => FLU_TX_EOP,
      TX_SRC_RDY        => FLU_TX_SRC_RDY,
      TX_DST_RDY        => FLU_TX_DST_RDY,

      PACKET_SENT       => tx_packet_sent,
      PACKET_CHANNEL    => tx_packet_channel,
      PACKET_LENGTH     => tx_packet_len,

      DMA_WRITE         => write_qword,
      DMA_CHANNEL       => write_channel,
      DMA_ADDRESS       => write_address,
      DMA_DATA          => write_data,

      DMA_DONE          => write_done,
      DMA_DONE_SIZE     => write_done_size,
      DMA_DONE_CHANNEL  => write_done_channel
   );

   DMA_CTRL_TX_I : entity work.DMA_CTRL_TX
   generic map(
      CHANNELS          => TX_CHANNELS,
      BUFFERS           => TX_BUFFERS,
      BUFFER_SIZE       => TX_BUFFER_SIZE,
      DATA_ALIGN        => TX_DATA_ALIGN
   )
   port map(
      CLK               => CLK,
      RESET             => RESET,

      ENABLE            => tx_enable,
      INTERRUPT         => TX_INTERRUPT,
      CONFIG_ADDR       => config_addr_txctrl,

      PACKET_SENT       => tx_packet_sent,
      PACKET_CHANNEL    => tx_packet_channel,
      PACKET_LENGTH     => tx_packet_len,

      BUFFER_WRITE      => write_qword,
      BUFFER_ADDRESS    => write_address,
      BUFFER_CHANNEL    => write_channel,
      BUFFER_DATA       => write_data,

      BUFFER_DONE       => write_done,
      BUFFER_DONE_SIZE  => write_done_size,
      BUFFER_DONE_CHANNEL => write_done_channel,

      DESC_DATA         => tx_desc_data,
      DESC_RDY          => tx_desc_rdy,
      DESC_CHANNEL      => tx_desc_channel,
      DESC_UPDATE       => tx_desc_update,
      DESC_UPDATE_LEN   => tx_desc_update_len,

      DMA_UP_DATA       => dma_up_ctrl_tx_data,
      DMA_UP_HDR        => dma_up_ctrl_tx_hdr,
      DMA_UP_SOP        => dma_up_ctrl_tx_sop,
      DMA_UP_EOP        => dma_up_ctrl_tx_eop,
      DMA_UP_SRC_RDY    => dma_up_ctrl_tx_src_rdy,
      DMA_UP_DST_RDY    => dma_up_ctrl_tx_dst_rdy,

      DMA_DOWN_DATA     => dma_down_ctrl_tx_data,
      DMA_DOWN_HDR      => dma_down_ctrl_tx_hdr,
      DMA_DOWN_SOP      => dma_down_ctrl_tx_sop,
      DMA_DOWN_EOP      => dma_down_ctrl_tx_eop,
      DMA_DOWN_SRC_RDY  => dma_down_ctrl_tx_src_rdy,
      DMA_DOWN_DST_RDY  => dma_down_ctrl_tx_dst_rdy,

      MI_ADDR           => mi_tx_addr,
      MI_ARDY           => mi_tx_ardy,
      MI_BE             => mi_tx_be,
      MI_DRD            => mi_tx_drd,
      MI_DRDY           => mi_tx_drdy,
      MI_DWR            => mi_tx_dwr,
      MI_RD             => mi_tx_rd,
      MI_WR             => mi_tx_wr
   );

   dma_desc_manager_tx_i : entity work.dma_desc_manager
   generic map (
      DATA_ALIGN        => TX_DATA_ALIGN,
      CHANNELS          => TX_CHANNELS,
      DMA_ID            => 5
   )
   port map(
      CLK               => CLK,
      RESET             => RESET,

      ENABLE            => tx_enable,
      CONFIG_ADDR       => config_addr_txdesc,

      DESC_DATA         => tx_desc_data,
      DESC_RDY          => tx_desc_rdy,
      DESC_CHANNEL      => tx_desc_channel,
      DESC_UPDATE       => tx_desc_update,
      DESC_UPDATE_LEN   => tx_desc_update_len,

      DMA_UP_DATA       => dma_up_desc_tx_pipe_data,
      DMA_UP_HDR        => dma_up_desc_tx_pipe_hdr,
      DMA_UP_SOP        => dma_up_desc_tx_pipe_sop,
      DMA_UP_EOP        => dma_up_desc_tx_pipe_eop,
      DMA_UP_SRC_RDY    => dma_up_desc_tx_pipe_src_rdy,
      DMA_UP_DST_RDY    => dma_up_desc_tx_pipe_dst_rdy,

      DMA_DOWN_DATA     => dma_down_desc_tx_data,
      DMA_DOWN_HDR      => dma_down_desc_tx_hdr,
      DMA_DOWN_SOP      => dma_down_desc_tx_sop,
      DMA_DOWN_EOP      => dma_down_desc_tx_eop,
      DMA_DOWN_SRC_RDY  => dma_down_desc_tx_src_rdy,
      DMA_DOWN_DST_RDY  => dma_down_desc_tx_dst_rdy
   );

   gen_pipe_up_tx_desc : if(PIPE_DMA_UP_TX_DESC = true) generate

   dma_pipe_up_tx_descriptor_i : entity work.DMA_PIPE
   generic map(
      DATA_WIDTH        => 512,
      HDR_WIDTH         => 96,
      USE_OUTREG        => true,
      FAKE_PIPE         => false
   )
   port map(
      CLK               => CLK,
      RESET             => RESET,

      RX_DATA           => dma_up_desc_tx_pipe_data,
      RX_HDR            => dma_up_desc_tx_pipe_hdr,
      RX_SOP            => dma_up_desc_tx_pipe_sop,
      RX_EOP            => dma_up_desc_tx_pipe_eop,
      RX_SRC_RDY        => dma_up_desc_tx_pipe_src_rdy,
      RX_DST_RDY        => dma_up_desc_tx_pipe_dst_rdy,

      TX_DATA           => dma_up_desc_tx_data,
      TX_HDR            => dma_up_desc_tx_hdr,
      TX_SOP            => dma_up_desc_tx_sop,
      TX_EOP            => dma_up_desc_tx_eop,
      TX_SRC_RDY        => dma_up_desc_tx_src_rdy,
      TX_DST_RDY        => dma_up_desc_tx_dst_rdy
   );

   end generate;

   not_pipe_up_tx_desc : if(PIPE_DMA_UP_TX_DESC = false) generate

   dma_up_desc_tx_data     <= dma_up_desc_tx_pipe_data;
   dma_up_desc_tx_hdr      <= dma_up_desc_tx_pipe_hdr;
   dma_up_desc_tx_sop      <= dma_up_desc_tx_pipe_sop;
   dma_up_desc_tx_eop      <= dma_up_desc_tx_pipe_eop;
   dma_up_desc_tx_src_rdy  <= dma_up_desc_tx_pipe_src_rdy;
   dma_up_desc_tx_pipe_dst_rdy   <=dma_up_desc_tx_dst_rdy;

   end generate;

   switch_tx_i : entity work.dma_switch
   generic map(
      DATA_WIDTH        => 512,
      HDR_WIDTH         => 96
   )
   port map (
      CLK               => CLK,
      RESET             => RESET,

      DMA_TX_DATA       => dma_up_switch_tx_data,
      DMA_TX_HDR        => dma_up_switch_tx_hdr,
      DMA_TX_SOP        => dma_up_switch_tx_sop,
      DMA_TX_EOP        => dma_up_switch_tx_eop,
      DMA_TX_SRC_RDY    => dma_up_switch_tx_src_rdy,
      DMA_TX_DST_RDY    => dma_up_switch_tx_dst_rdy,

      DMA_RX0_DATA      => dma_up_ctrl_tx_data,
      DMA_RX0_HDR       => dma_up_ctrl_tx_hdr,
      DMA_RX0_SOP       => dma_up_ctrl_tx_sop,
      DMA_RX0_EOP       => dma_up_ctrl_tx_eop,
      DMA_RX0_SRC_RDY   => dma_up_ctrl_tx_src_rdy,
      DMA_RX0_DST_RDY   => dma_up_ctrl_tx_dst_rdy,

      DMA_RX1_DATA      => dma_up_desc_tx_data,
      DMA_RX1_HDR       => dma_up_desc_tx_hdr,
      DMA_RX1_SOP       => dma_up_desc_tx_sop,
      DMA_RX1_EOP       => dma_up_desc_tx_eop,
      DMA_RX1_SRC_RDY   => dma_up_desc_tx_src_rdy,
      DMA_RX1_DST_RDY   => dma_up_desc_tx_dst_rdy
   );

   splitter_tx_i : entity work.dma_splitter
   generic map(
      DATA_WIDTH        => 512,
      HDR_WIDTH         => 32,
      HDR_BIT           => 26
   )
   port map (
      CLK               => CLK,
      RESET             => RESET,

      DMA_RX_DATA       => dma_down_split_tx_data,
      DMA_RX_HDR        => dma_down_split_tx_hdr,
      DMA_RX_SOP        => dma_down_split_tx_sop,
      DMA_RX_EOP        => dma_down_split_tx_eop,
      DMA_RX_SRC_RDY    => dma_down_split_tx_src_rdy,
      DMA_RX_DST_RDY    => dma_down_split_tx_dst_rdy,

      DMA_TX0_DATA      => dma_down_ctrl_tx_data,
      DMA_TX0_HDR       => dma_down_ctrl_tx_hdr,
      DMA_TX0_SOP       => dma_down_ctrl_tx_sop,
      DMA_TX0_EOP       => dma_down_ctrl_tx_eop,
      DMA_TX0_SRC_RDY   => dma_down_ctrl_tx_src_rdy,
      DMA_TX0_DST_RDY   => dma_down_ctrl_tx_dst_rdy,

      DMA_TX1_DATA      => dma_down_desc_tx_data,
      DMA_TX1_HDR       => dma_down_desc_tx_hdr,
      DMA_TX1_SOP       => dma_down_desc_tx_sop,
      DMA_TX1_EOP       => dma_down_desc_tx_eop,
      DMA_TX1_SRC_RDY   => dma_down_desc_tx_src_rdy,
      DMA_TX1_DST_RDY   => dma_down_desc_tx_dst_rdy
   );

end generate;

   -- ---------------------------------------------------------------
   --                        Counters
   -- ---------------------------------------------------------------

   mi_cntr_ardy         <= mi_cntr_rd or mi_cntr_wr;
   mi_cntr_drdy         <= mi_cntr_rd;

   no_counters : if(ENABLE_COUNTERS = false) generate
      mi_cntr_drd          <= X"00000000";
   end generate;

   gen_counters : if(ENABLE_COUNTERS = true) generate

      mi_cntr_drd          <= reg_mi_cntr_drd when mi_cntr_addr(2) = '1' else mux_mi_cntr_drd(31 downto 0);

   cntr_rxg : for i in 0 to RX_CHANNELS-1 generate
      mux_mi_cntr_rx_in((i+1)*128-1 downto i*128)  <= rx_cntr_disc(i) & rx_cntr_new(i);

      cntr_rxgp: process (CLK)
      begin
         if(CLK'event and CLK = '1') then
            if(mi_cntr_wr = '1' and mi_cntr_addr(11) = '0' and i = mi_cntr_addr(log2(RX_CHANNELS)-1+4 downto 4)) then
               rx_cntr_new(i)    <= (others => '0');
               rx_cntr_disc(i)   <= (others => '0');
            elsif(rx_packet_new = '1' and i = rx_packet_channel) then
               rx_cntr_new(i)    <= rx_cntr_new(i) + 1;
            elsif(rx_packet_disc = '1' and i = rx_packet_channel) then
               rx_cntr_disc(i)   <= rx_cntr_disc(i) + 1;
            end if;
         end if;
      end process;
   end generate;

   cntr_txg : for i in 0 to TX_CHANNELS-1 generate
      mux_mi_cntr_tx_in((i+1)*128-1 downto i*128)  <= zeros(63 downto 0) & tx_cntr_sent(i);

      cntr_txgp: process (CLK)
      begin
         if(CLK'event and CLK = '1') then
            if(mi_cntr_wr = '1' and mi_cntr_addr(11) = '1' and i = mi_cntr_addr(log2(TX_CHANNELS)-1+4 downto 4)) then
               tx_cntr_sent(i)   <= (others => '0');
            elsif(tx_packet_sent = '1' and i = tx_packet_channel) then
               tx_cntr_sent(i)   <= tx_cntr_sent(i) + 1;
            end if;
         end if;
      end process;
   end generate;

   reg_mi_cntr_drdp : process(CLK)
   begin
      if(CLK'event and CLK = '1') then
         if(mi_cntr_rd = '1') then
            reg_mi_cntr_drd   <= mux_mi_cntr_drd(63 downto 32);
         end if;
      end if;
   end process;

   mux_mi_cntr_drd <= mux_mi_cntr_tx when mi_cntr_addr(11) = '1' else mux_mi_cntr_rx;

   mux_mi_rxi : entity work.GEN_MUX
   generic map(
      DATA_WIDTH  => 64,
      MUX_WIDTH   => RX_CHANNELS*2
   )
   port map(
      SEL      => mi_cntr_addr(log2(RX_CHANNELS*2)-1+3 downto 3),
      DATA_IN  => mux_mi_cntr_rx_in,
      DATA_OUT => mux_mi_cntr_rx
   );

   mux_mi_txi : entity work.GEN_MUX
   generic map(
      DATA_WIDTH  => 64,
      MUX_WIDTH   => TX_CHANNELS*2
   )
   port map(
      SEL      => mi_cntr_addr(log2(TX_CHANNELS*2)-1+3 downto 3),
      DATA_IN  => mux_mi_cntr_tx_in,
      DATA_OUT => mux_mi_cntr_tx
   );

   end generate;

   -- ---------------------------------------------------------------
   --                         Common
   -- ---------------------------------------------------------------

   switch_rxtx_i : entity work.dma_switch
   generic map(
      DATA_WIDTH        => 512,
      HDR_WIDTH         => 96
   )
   PORT map (
      CLK               => CLK,
      RESET             => RESET,

      DMA_TX_DATA       => dma_up_switch_data,
      DMA_TX_HDR        => dma_up_switch_hdr,
      DMA_TX_SOP        => dma_up_switch_sop,
      DMA_TX_EOP        => dma_up_switch_eop,
      DMA_TX_SRC_RDY    => dma_up_switch_src_rdy,
      DMA_TX_DST_RDY    => dma_up_switch_dst_rdy,

      DMA_RX0_DATA      => dma_up_switch_rx_data,
      DMA_RX0_HDR       => dma_up_switch_rx_hdr,
      DMA_RX0_SOP       => dma_up_switch_rx_sop,
      DMA_RX0_EOP       => dma_up_switch_rx_eop,
      DMA_RX0_SRC_RDY   => dma_up_switch_rx_src_rdy,
      DMA_RX0_DST_RDY   => dma_up_switch_rx_dst_rdy,

      DMA_RX1_DATA      => dma_up_switch_tx_data,
      DMA_RX1_HDR       => dma_up_switch_tx_hdr,
      DMA_RX1_SOP       => dma_up_switch_tx_sop,
      DMA_RX1_EOP       => dma_up_switch_tx_eop,
      DMA_RX1_SRC_RDY   => dma_up_switch_tx_src_rdy,
      DMA_RX1_DST_RDY   => dma_up_switch_tx_dst_rdy
   );

   dma_pipe_up_i : entity work.DMA_PIPE
   generic map(
      DATA_WIDTH        => 512,
      HDR_WIDTH         => 96,
      USE_OUTREG        => true
   )
   port map(
      CLK               => CLK,
      RESET             => RESET,

      RX_DATA           => dma_up_switch_data,
      RX_HDR            => dma_up_switch_hdr,
      RX_SOP            => dma_up_switch_sop,
      RX_EOP            => dma_up_switch_eop,
      RX_SRC_RDY        => dma_up_switch_src_rdy,
      RX_DST_RDY        => dma_up_switch_dst_rdy,

      TX_DATA           => DMA_UP_DATA,
      TX_HDR            => DMA_UP_HDR,
      TX_SOP            => DMA_UP_SOP,
      TX_EOP            => DMA_UP_EOP,
      TX_SRC_RDY        => DMA_UP_SRC_RDY,
      TX_DST_RDY        => DMA_UP_DST_RDY
   );

   splitter_rxtx_i : entity work.dma_splitter
   generic map(
      DATA_WIDTH        => 512,
      HDR_WIDTH         => 32,
      HDR_BIT           => 24
   )
   port map (
      CLK               => CLK,
      RESET             => RESET,

      DMA_RX_DATA       => dma_down_split_data,
      DMA_RX_HDR        => dma_down_split_hdr,
      DMA_RX_SOP        => dma_down_split_sop,
      DMA_RX_EOP        => dma_down_split_eop,
      DMA_RX_SRC_RDY    => dma_down_split_src_rdy,
      DMA_RX_DST_RDY    => dma_down_split_dst_rdy,

      DMA_TX0_DATA      => dma_down_split_rx_data,
      DMA_TX0_HDR       => dma_down_split_rx_hdr,
      DMA_TX0_SOP       => dma_down_split_rx_sop,
      DMA_TX0_EOP       => dma_down_split_rx_eop,
      DMA_TX0_SRC_RDY   => dma_down_split_rx_src_rdy,
      DMA_TX0_DST_RDY   => dma_down_split_rx_dst_rdy,

      DMA_TX1_DATA      => dma_down_split_tx_data,
      DMA_TX1_HDR       => dma_down_split_tx_hdr,
      DMA_TX1_SOP       => dma_down_split_tx_sop,
      DMA_TX1_EOP       => dma_down_split_tx_eop,
      DMA_TX1_SRC_RDY   => dma_down_split_tx_src_rdy,
      DMA_TX1_DST_RDY   => dma_down_split_tx_dst_rdy
   );

   dma_pipe_down_i : entity work.DMA_PIPE
   generic map(
      DATA_WIDTH        => 512,
      HDR_WIDTH         => 32,
      USE_OUTREG        => true
   )
   port map(
      CLK               => CLK,
      RESET             => RESET,

      RX_DATA           => DMA_DOWN_DATA,
      RX_HDR            => DMA_DOWN_HDR,
      RX_SOP            => DMA_DOWN_SOP,
      RX_EOP            => DMA_DOWN_EOP,
      RX_SRC_RDY        => DMA_DOWN_SRC_RDY,
      RX_DST_RDY        => DMA_DOWN_DST_RDY,

      TX_DATA           => dma_down_split_data,
      TX_HDR            => dma_down_split_hdr,
      TX_SOP            => dma_down_split_sop,
      TX_EOP            => dma_down_split_eop,
      TX_SRC_RDY        => dma_down_split_src_rdy,
      TX_DST_RDY        => dma_down_split_dst_rdy
   );

   MI_SPLITTER_CTRL_I : entity work.mi_splitter_addr32
   generic map (
      -- Number of output interfaces (RX, TX, counters, config)
      ITEMS                   => 4,
      -- New address width - upper bits will be filled with zeros
      ADDR_WIDTH              => 12,
      -- MI Data width
      DATA_WIDTH              => 32,
      -- Pipe enable
      PIPE                    => true,
      PIPE_OUTREG             => true
   )
   port map (
      -- Common interface
      CLK                     => CLK,
      RESET                   => RESET,
      -- Input MI
      IN_DWR                  => MI_DWR,
      IN_ADDR                 => MI_ADDR(13 downto 0),
      IN_BE                   => MI_BE,
      IN_RD                   => MI_RD,
      IN_WR                   => MI_WR,
      IN_ARDY                 => MI_ARDY,
      IN_DRD                  => MI_DRD,
      IN_DRDY                 => MI_DRDY,
      -- Output MI
      OUT_DWR(31 downto 0)    => mi_rx_dwr,
      OUT_DWR(63 downto 32)   => mi_tx_dwr,
      OUT_DWR(95 downto 64)   => mi_cntr_dwr,
      OUT_DWR(127 downto 96)  => mi_conf_dwr,
      OUT_ADDR(31 downto 0)   => mi_rx_addr,
      OUT_ADDR(63 downto 32)  => mi_tx_addr,
      OUT_ADDR(95 downto 64)  => mi_cntr_addr,
      OUT_ADDR(127 downto 96) => mi_conf_addr,
      OUT_RD(0)               => mi_rx_rd,
      OUT_RD(1)               => mi_tx_rd,
      OUT_RD(2)               => mi_cntr_rd,
      OUT_RD(3)               => mi_conf_rd,
      OUT_WR(0)               => mi_rx_wr,
      OUT_WR(1)               => mi_tx_wr,
      OUT_WR(2)               => mi_cntr_wr,
      OUT_WR(3)               => mi_conf_wr,
      OUT_ARDY(0)             => mi_rx_ardy,
      OUT_ARDY(1)             => mi_tx_ardy,
      OUT_ARDY(2)             => mi_cntr_ardy,
      OUT_ARDY(3)             => mi_conf_ardy,
      OUT_BE(3 downto 0)      => mi_rx_be,
      OUT_BE(7 downto 4)      => mi_tx_be,
      OUT_BE(11 downto 8)     => mi_cntr_be,
      OUT_BE(15 downto 12)    => mi_conf_be,
      OUT_DRD(31 downto 0)    => mi_rx_drd,
      OUT_DRD(63 downto 32)   => mi_tx_drd,
      OUT_DRD(95 downto 64)   => mi_cntr_drd,
      OUT_DRD(127 downto 96)  => mi_conf_drd,
      OUT_DRDY(0)             => mi_rx_drdy,
      OUT_DRDY(1)             => mi_tx_drdy,
      OUT_DRDY(2)             => mi_cntr_drdy,
      OUT_DRDY(3)             => mi_conf_drdy
   );

end architecture;

-- --------------------------------------------------------
