-- dma_ctrl_tx_request.vhd: Request Data Process of TX DMA Controller
-- Copyright (C) 2013 CESNET
-- Author(s): Martin Spinler <spinler@cesnet.cz>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--

library IEEE;
use IEEE.std_logic_1164.all;
use ieee.std_logic_arith.all;
use ieee.std_logic_unsigned.all;
use ieee.numeric_std.all;

library work;
use work.math_pack.all;
use work.dma_pkg.all;

entity dma_ctrl_tx_request is
   generic (
      CHANNELS          : integer;
      BUFFER_SIZE       : integer;
      BUFFERS           : integer;
      DATA_ALIGN        : integer;
      TAG_WIDTH         : integer
   );
  -- ie terface description
   port (
      CLK               : in  std_logic;
      RESET             : in  std_logic;

      BUFFER_WRITE      : out std_logic_vector(log2(BUFFERS) downto 0);
      BUFFER_ADDRESS    : out std_logic_vector(log2(BUFFER_SIZE/DATA_ALIGN)-1 downto 0);
      BUFFER_CHANNEL    : out std_logic_vector(log2(CHANNELS)-1 downto 0);
      BUFFER_DATA       : out std_logic_vector(511 downto 0);

      BUFFER_DONE       : out std_logic;
      BUFFER_DONE_SIZE  : out std_logic_vector(log2(BUFFER_SIZE/DATA_ALIGN) downto 0);
      BUFFER_DONE_CHANNEL : out std_logic_vector(log2(CHANNELS)-1 downto 0);

      SWSTRPTR          : out std_logic_vector(31 downto 0);
      SWSTRPTR_UPDATE   : out std_logic;
      SWSTRPTR_INTERRUPT: out std_logic;

      --! Interface for generating DMA requests
      DMA_TX_DATA       : out std_logic_vector(511 downto 0);
      DMA_TX_HDR        : out std_logic_vector(95 downto 0);
      DMA_TX_SOP        : out std_logic;
      DMA_TX_EOP        : out std_logic;
      DMA_TX_SRC_RDY    : out std_logic;
      DMA_TX_DST_RDY    : in  std_logic;

      --! Interface for receiving DMA requests
      DMA_RX_DATA       : in  std_logic_vector(511 downto 0);
      DMA_RX_HDR        : in  std_logic_vector(31 downto 0);
      DMA_RX_SOP        : in  std_logic;
      DMA_RX_EOP        : in  std_logic;
      DMA_RX_SRC_RDY    : in  std_logic;
      DMA_RX_DST_RDY    : out std_logic;

      DMA_SWSTRPTR      : in  std_logic_vector(31 downto 0);
      DMA_REQ           : in  std_logic;
      DMA_RDY           : out std_logic;
      DMA_INTERRUPT     : in std_logic;
      DMA_LENGTH        : in  std_logic_vector(PAGE_WIDTH-log2(DATA_ALIGN) downto 0);
      DMA_HWADDRESS     : in  std_logic_vector(log2(BUFFER_SIZE/DATA_ALIGN)-1 downto 0);
      DMA_DESC          : in  std_logic_vector(63 downto 0);
      DMA_CHANNEL       : in  std_logic_vector(log2(CHANNELS)-1 downto 0)
    );
end entity;


architecture behavioral of dma_ctrl_tx_request is

   constant zeros                   : std_logic_vector(63 downto 0) := (others => '0');
   constant DMA_LENGTH_WIDTH        : integer := PAGE_WIDTH + 1 - log2(DATA_ALIGN);
   constant BUFFER_WIDTH            : integer := DATA_ALIGN * BUFFERS;
   constant ADDRESS_WIDTH           : integer := log2(BUFFER_SIZE/DATA_ALIGN);

   signal tag                       : std_logic_vector(7 downto 0);
   signal length                    : std_logic_vector(10 downto 0);

   signal done_hwaddress            : std_logic_vector(ADDRESS_WIDTH-1 downto 0);
   signal base_hwaddress            : std_logic_vector(ADDRESS_WIDTH-1 downto 0);
   signal write_hwaddress_w         : std_logic_vector(ADDRESS_WIDTH-1 downto 0);

   signal write_channel             : std_logic_vector(log2(CHANNELS)-1 downto 0);
   signal write_complete            : std_logic;
  
   signal done_channel              : std_logic_vector(log2(CHANNELS)-1 downto 0);

   signal tag_rdy                   : std_logic;
   signal tag_next                  : std_logic_vector(TAG_WIDTH-1 downto 0) := (others => '0');
   signal tag_last                  : std_logic_vector(TAG_WIDTH-1 downto 0) := (others => '0');
   signal reg_tagused               : std_logic_vector(2**TAG_WIDTH-1 downto 0) := (others => '0');
   signal reg_tagdone               : std_logic_vector(2**TAG_WIDTH-1 downto 0) := (others => '0');

   signal pending_dma_req           : std_logic := '0';
   signal pending_rdy               : std_logic;

   signal write_req                 : std_logic;
   signal write_length              : std_logic_vector(DMA_LENGTH_WIDTH-1 downto 0);
   signal write_tag                 : std_logic_vector(TAG_WIDTH-1 downto 0);
   signal dma_exec                  : std_logic;

   signal req_length                : std_logic_vector(DMA_LENGTH_WIDTH-1 downto 0);
   signal done_length               : std_logic_vector(DMA_LENGTH_WIDTH-1 downto 0);
begin

   -- Generating request
   BUFFER_DATA       <= DMA_RX_DATA;
   BUFFER_ADDRESS    <= base_hwaddress + done_hwaddress;
   BUFFER_CHANNEL    <= write_channel;

   BUFFER_DONE       <= pending_rdy;
   BUFFER_DONE_SIZE  <= zeros(log2(BUFFER_SIZE/DATA_ALIGN) downto DMA_LENGTH_WIDTH) & done_length;
   BUFFER_DONE_CHANNEL <= done_channel;

   SWSTRPTR_UPDATE   <= pending_rdy;

   DMA_RX_DST_RDY    <= '1';

   DMA_TX_SOP        <= '1';
   DMA_TX_EOP        <= '1';
   DMA_TX_DATA       <= (others =>'0');
   DMA_TX_SRC_RDY    <= pending_dma_req;
   DMA_RDY           <= tag_rdy and not pending_dma_req;

   dma_exec          <= tag_rdy and not pending_dma_req and DMA_REQ;

   write_req         <= DMA_RX_SRC_RDY;
   write_hwaddress_w <= (done_hwaddress + conv_integer(write_length)) when done_hwaddress + (write_length) /= req_length else (others => '0');
   write_complete    <= '1' when write_req = '1' and (done_hwaddress + write_length = req_length) else '0';

   tag_rdy           <= not reg_tagused(conv_integer(tag_next));
   pending_rdy       <=     reg_tagdone(conv_integer(tag_last));

   write_tag         <= tag(TAG_WIDTH-1 downto 0);
   tag               <= DMA_RX_HDR(DMA_REQUEST_TAG);
   length            <= DMA_RX_HDR(DMA_REQUEST_LENGTH);

   buffer_writep: process (DMA_RX_EOP, DMA_RX_SRC_RDY, length)
   begin
      if(DMA_RX_SRC_RDY = '1') then
         if(DMA_RX_EOP = '1' and length(log2(BUFFERS*DATA_ALIGN/4)-1 downto log2(DATA_ALIGN/4)) /= zeros(log2(BUFFERS)-1 downto 0)) then
            write_length   <= zeros(DMA_LENGTH_WIDTH-1 downto log2(BUFFERS)) & length(log2(BUFFERS*DATA_ALIGN/4)-1 downto log2(DATA_ALIGN/4));
            BUFFER_WRITE   <= "0" & length(log2(BUFFERS*DATA_ALIGN/4)-1 downto log2(DATA_ALIGN/4));
         else
            write_length   <= conv_std_logic_vector(BUFFERS, DMA_LENGTH_WIDTH);
            BUFFER_WRITE   <= conv_std_logic_vector(BUFFERS, log2(BUFFERS)+1);
         end if;
      else
         write_length <=  (others => '0');
         BUFFER_WRITE <=  (others => '0');
      end if;
   end process;

   pdmareqp : process (CLK)
   begin
      if(CLK = '1' and CLK'event) then
         if(dma_exec = '1') then
            pending_dma_req <= '1';

            DMA_TX_HDR(DMA_REQUEST_GLOBAL)  <= DMA_DESC;
            DMA_TX_HDR(DMA_REQUEST_LENGTH)  <= dma_length(PAGE_WIDTH-log2(DATA_ALIGN) downto 0) & "0"; -- TODO
            DMA_TX_HDR(DMA_REQUEST_TYPE)    <= DMA_TYPE_READ;
            DMA_TX_HDR(DMA_REQUEST_TAG)     <= zeros(7-TAG_WIDTH downto 0) & tag_next;
            DMA_TX_HDR(DMA_REQUEST_UNITID)  <= conv_std_logic_vector(1, 8);
         elsif(DMA_TX_DST_RDY = '1') then
            pending_dma_req <= '0';
         end if;
      end if;
   end process;

   gen_tagused: for i in 0 to 2**TAG_WIDTH-1 generate
      reg_tagusedp : process (CLK)
      begin
         if(CLK = '1' and CLK'event) then
            if(write_req = '1' and write_complete ='1' and write_tag = i) then
               reg_tagused(i) <= '0';
               reg_tagdone(i) <= '1';
            elsif(dma_exec = '1' and tag_next = i) then
               reg_tagused(i) <= '1';
            elsif(tag_last = i) then
               reg_tagdone(i) <= '0';
            end if;
         end if;
      end process;
   end generate;

   reg_tagnextp : process (CLK)
   begin
      if(CLK = '1' and CLK'event) then
         if(dma_exec = '1') then
            tag_next       <= tag_next + 1;
         end if;

         if(pending_rdy = '1') then
            tag_last       <= tag_last + 1;
         end if;
      end if;
   end process;

   -- PENDING OPERATIONS

   pending_fifo_addr_i: entity work.DP_DISTMEM
   generic map(
      DATA_WIDTH   	=> ADDRESS_WIDTH,
      ITEMS        	=> 2**TAG_WIDTH,
      DISTMEM_TYPE 	=> 16
   )
   port map(
      WCLK    			=> CLK,
      RESET   			=> RESET,
      DI     			=> DMA_HWADDRESS,
      WE      			=> dma_exec,

      ADDRA   			=> tag_next,
      DOA     			=> open,

      ADDRB   			=> write_tag,
      DOB     			=> base_hwaddress
   );

   pending_fifo_len_i: entity work.DP_DISTMEM
   generic map(
      DATA_WIDTH   	=> DMA_LENGTH_WIDTH,
      ITEMS        	=> 2**TAG_WIDTH,
      DISTMEM_TYPE 	=> 16
   )
   port map(
      WCLK    			=> CLK,
      RESET   			=> RESET,
      DI     			=> dma_length,
      WE      			=> dma_exec,
      ADDRA   			=> tag_next,
      DOA   			=> open,

      ADDRB   			=> write_tag,
      DOB     			=> req_length
   );

   pending_fifo_channel_i: entity work.DP_DISTMEM
   generic map(
      DATA_WIDTH   	=> log2(CHANNELS),
      ITEMS        	=> 2**TAG_WIDTH,
      DISTMEM_TYPE 	=> 16
   )
   port map(
      WCLK    			=> CLK,
      RESET   			=> RESET,
      DI     			=> dma_channel,
      WE      			=> dma_exec,
      ADDRA   			=> tag_next,
      DOA   			=> open,

      ADDRB   			=> write_tag,
      DOB     			=> write_channel
   );

   pending_fifo_tempaddr_i: entity work.SP_DISTMEM
   generic map(
      DATA_WIDTH   	=> ADDRESS_WIDTH,
      ITEMS        	=> 2**TAG_WIDTH,
      DISTMEM_TYPE 	=> 16
   )
   port map(
      WCLK    			=> CLK,
      RESET   			=> RESET,
      DI     			=> write_hwaddress_w,
      ADDR           => write_tag,
      WE      			=> write_req,
      DO             => done_hwaddress
   );

   -- PENDING DONE OPERATIONS (Tag Sequencer)

   pending_fifo_channel_done_i: entity work.DP_DISTMEM
   generic map(
      DATA_WIDTH   	=> log2(CHANNELS),
      ITEMS        	=> 2**TAG_WIDTH,
      DISTMEM_TYPE 	=> 16
   )
   port map(
      WCLK    			=> CLK,
      RESET   			=> RESET,
      DI     			=> dma_channel,
      WE      			=> dma_exec,
      ADDRA   			=> tag_next,
      DOA   			=> open,

      ADDRB   			=> tag_last,
      DOB     			=> done_channel
   );

   pending_fifo_len_done_i: entity work.DP_DISTMEM
   generic map(
      DATA_WIDTH   	=> DMA_LENGTH_WIDTH,
      ITEMS        	=> 2**TAG_WIDTH,
      DISTMEM_TYPE 	=> 16
   )
   port map(
      WCLK    			=> CLK,
      RESET   			=> RESET,
      DI     			=> dma_length,
      WE      			=> dma_exec,
      ADDRA   			=> tag_next,
      DOA   			=> open,

      ADDRB   			=> tag_last,
      DOB     			=> done_length
   );

   pending_fifo_swstrptr_i: entity work.DP_DISTMEM
   generic map(
      DATA_WIDTH   	=> 32,
      ITEMS        	=> 2**TAG_WIDTH,
      DISTMEM_TYPE 	=> 16
   )
   port map(
      WCLK    			=> CLK,
      RESET   			=> RESET,
      DI     			=> DMA_SWSTRPTR,
      WE      			=> dma_exec,

      ADDRA   			=> tag_next,
      DOA     			=> open,

      ADDRB   			=> tag_last,
      DOB     			=> SWSTRPTR
   );

   pending_fifo_interrupt_i: entity work.DP_DISTMEM
   generic map(
      DATA_WIDTH   	=> 1,
      ITEMS        	=> 2**TAG_WIDTH,
      DISTMEM_TYPE 	=> 16
   )
   port map(
      WCLK    			=> CLK,
      RESET   			=> RESET,
      DI(0)  			=> DMA_INTERRUPT,
      WE      			=> dma_exec,

      ADDRA   			=> tag_next,
      DOA     			=> open,

      ADDRB   			=> tag_last,
      DOB(0)  			=> SWSTRPTR_INTERRUPT
   );

end architecture;
