-- dma_ctrl_rx_request.vhd: Request Data Process of RX DMA Controller
-- Copyright (C) 2013 CESNET
-- Author(s): Martin Spinler <spinler@cesnet.cz>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--

library IEEE;
use IEEE.std_logic_1164.all;
use ieee.std_logic_arith.all;
use ieee.std_logic_unsigned.all;
use ieee.numeric_std.all;

library work;
use work.math_pack.all;
use work.dma_pkg.all;

entity dma_ctrl_rx_request is
   generic (
      CHANNELS          : integer;
      BUFFERS           : integer;
      DATA_ALIGN        : integer;
      UNIT_ID           : std_logic_vector(7 downto 0)
   );
   port (
      CLK               : in  std_logic;
      RESET             : in  std_logic;

      --! Interface to HW Buffer
      BUFFER_READ       : out std_logic_vector(log2(BUFFERS) downto 0);
      BUFFER_CHANNEL    : out std_logic_vector(log2(CHANNELS)-1 downto 0);
      BUFFER_DATA       : in  std_logic_vector(511 downto 0);
      BUFFER_DV         : in  std_logic;

      --! Interface for generating DMA requests
      DMA_UP_DATA       : out std_logic_vector(511 downto 0);
      DMA_UP_HDR        : out std_logic_vector(DMA_UPHDR_WIDTH-1 downto 0);
      DMA_UP_SOP        : out std_logic;
      DMA_UP_EOP        : out std_logic;
      DMA_UP_SRC_RDY    : out std_logic;
      DMA_UP_DST_RDY    : in  std_logic;

      --! Interface from NewData process - DMA request
      DMA_LENGTH        : in  std_logic_vector(PAGE_WIDTH - log2(DATA_ALIGN) downto 0);
      DMA_CHANNEL       : in  std_logic_vector(log2(CHANNELS)-1 downto 0);
      DMA_REQ           : in  std_logic;
      DMA_RDY           : out std_logic;
      DMA_DONE          : out std_logic;

      --! Destination address in RAM
      DESC_DATA         : in  std_logic_vector(DMA_DESC_WIDTH-1 downto 0)
   );
end entity;


architecture behavioral of dma_ctrl_rx_request is

   type t_state is (S_IDLE, S_DMA, S_DMA_ONE, S_DMA_START, S_DMA_END);
   signal present_state, next_state : t_state;

   signal hdr_data            : std_logic_vector(95 downto 0);
   signal reg_hdr_data        : std_logic_vector(95 downto 0);

   signal dma_ready           : std_logic;
   signal last_word           : std_logic;
   signal reg_dma_channel     : std_logic_vector(log2(CHANNELS)-1 downto 0);
   signal reg_length          : std_logic_vector(log2(PAGE_SIZE) - log2(DATA_ALIGN) downto 0) := (others => '0');
   signal buffer_read_enable  : std_logic_vector(log2(BUFFERS) downto 0);

   signal pipe_dma_up_data    : std_logic_vector(511 downto 0);
   signal pipe_dma_up_hdr     : std_logic_vector(DMA_UPHDR_WIDTH-1 downto 0);
   signal pipe_dma_up_sop     : std_logic;
   signal pipe_dma_up_eop     : std_logic;
   signal pipe_dma_up_src_rdy : std_logic;
   signal pipe_dma_up_dst_rdy : std_logic;
   signal pipe_dma_done       : std_logic;
   signal pipe_out_dma_done   : std_logic;

   signal data_fifo_rd        : std_logic;
   signal data_fifo_empty     : std_logic;

   signal hdr_fifo_data_in    : std_logic_vector(DMA_UPHDR_WIDTH+2 downto 0);
   signal hdr_fifo_data_out   : std_logic_vector(98 downto 0);

   signal hdr_fifo_wr         : std_logic;
   signal hdr_fifo_full       : std_logic;

begin

   hdr_fifo_data_in              <= pipe_dma_up_hdr & pipe_dma_up_eop & pipe_dma_up_sop & pipe_dma_done;
   pipe_out_dma_done             <= hdr_fifo_data_out(0);
   DMA_UP_SOP                    <= hdr_fifo_data_out(1);
   DMA_UP_EOP                    <= hdr_fifo_data_out(2);
   DMA_UP_HDR                    <= hdr_fifo_data_out(DMA_UPHDR_WIDTH+2 downto 3);

   hdr_data(DMA_REQUEST_GLOBAL)  <= DESC_DATA;
   hdr_data(DMA_REQUEST_LENGTH)  <= DMA_LENGTH(PAGE_WIDTH - log2(DATA_ALIGN) downto 0) & "0";
   hdr_data(DMA_REQUEST_TYPE)    <= DMA_TYPE_WRITE;
   hdr_data(DMA_REQUEST_TAG)     <= conv_std_logic_vector(conv_integer(DMA_CHANNEL), 8);
   hdr_data(DMA_REQUEST_UNITID)  <= UNIT_ID;

   last_word                     <= '1' when reg_length <= 16 else '0';
   buffer_read_enable            <= conv_std_logic_vector(BUFFERS, log2(BUFFERS) + 1) when reg_length > BUFFERS else reg_length(log2(BUFFERS) downto 0);

   DMA_RDY                       <= dma_ready;
   pipe_dma_up_hdr               <= reg_hdr_data;

   DMA_UP_SRC_RDY                <= not data_fifo_empty;
   data_fifo_rd                  <= DMA_UP_DST_RDY and not data_fifo_empty;

   hdr_fifo_wr                   <= pipe_dma_up_src_rdy and not hdr_fifo_full;
   pipe_dma_up_dst_Rdy           <= not hdr_fifo_full;

   hdr_fifo_i : entity work.FIFO
   generic map(
      DATA_WIDTH     => 99,
      ITEMS          => 8
   )
   port map(
      CLK            => CLK,
      RESET          => RESET,
      DATA_IN        => hdr_fifo_data_in,
      WRITE_REQ      => hdr_fifo_wr,
      FULL           => hdr_fifo_full,

      DATA_OUT       => hdr_fifo_data_out,
      READ_REQ       => data_fifo_rd
   );

   data_fifo_i : entity work.FIFO
   generic map(
      DATA_WIDTH     => 512,
      ITEMS          => 8
   )
   port map(
      CLK            => CLK,
      RESET          => RESET,
      DATA_IN        => BUFFER_DATA,
      WRITE_REQ      => BUFFER_DV,

      DATA_OUT       => DMA_UP_DATA,
      READ_REQ       => data_fifo_rd,
      EMPTY          => data_fifo_empty
   );

   reg_requestp : process(CLK, RESET)
   begin
      if (CLK'event AND CLK = '1') then
         DMA_DONE                <= pipe_out_dma_done and data_fifo_rd;

         if(DMA_REQ = '1' and dma_ready = '1') then
            reg_dma_channel      <= DMA_CHANNEL;
            reg_hdr_data         <= hdr_data;
         end if;

         if(DMA_REQ = '1' and dma_ready = '1') then
            reg_length           <= DMA_LENGTH;
         elsif(pipe_dma_up_dst_rdy = '1') then
            if(reg_length <= 8) then
               reg_length        <= (others => '0');
            else
               reg_length        <= reg_length - 8;
            end if;
         end if;

         BUFFER_CHANNEL                <= reg_dma_channel;

         if(pipe_dma_up_dst_rdy = '1') then
            BUFFER_READ <= buffer_read_enable;
         else
            BUFFER_READ <= (others => '0');
         end if;
      end if;
   end process;

   -- --------------- Sync logic -------------------------------------------
   sync_logic : process(CLK, RESET, next_state)
   begin
      if (CLK'event AND CLK = '1') then
         if (RESET = '1') then
            present_state <= S_IDLE;
         else
            present_state <= next_state;
         end if;
      end if;
   end process;

   -- ------------------ Next state logic -------------------------------------
   next_state_logic : process(present_state, DMA_REQ, pipe_dma_up_dst_rdy, DMA_LENGTH, last_word)
   begin
      next_state              <= present_state;

      case (present_state) is
         -- ---------------------------------------------
         when S_IDLE =>
            if(DMA_REQ = '1') then
               if(DMA_LENGTH <= 8) then
                  next_state  <= S_DMA_ONE;
               else
                  next_state  <= S_DMA_START;
               end if;
            end if;
         -- ---------------------------------------------
         when S_DMA_START =>
            if(pipe_dma_up_dst_rdy = '1') then
               next_state     <= S_DMA;

               if(last_word = '1') then
                  next_state  <= S_DMA_END;
               end if;
            end if;
         -- ---------------------------------------------
         when S_DMA_ONE =>
            if(pipe_dma_up_dst_rdy = '1') then
               next_state     <= S_IDLE;

               if(DMA_REQ = '1') then
                  if(DMA_LENGTH <= 8) then
                     next_state <= S_DMA_ONE;
                  else
                     next_state <= S_DMA_START;
                  end if;
               end if;
            end if;
         -- ---------------------------------------------
         when S_DMA_END =>
            if(pipe_dma_up_dst_rdy = '1') then
               next_state       <= S_IDLE;

               if(DMA_REQ = '1') then
                  if(DMA_LENGTH <= 8) then
                     next_state <= S_DMA_ONE;
                  else
                     next_state <= S_DMA_START;
                  end if;
               end if;
            end if;
         -- ---------------------------------------------
         when S_DMA  =>
            if(pipe_dma_up_dst_rdy = '1') then
               if(last_word = '1') then
                  next_state  <= S_DMA_END;
               end if;
            end if;
         -- ---------------------------------------------
      end case;
   end process;

   -- ------------------ Output logic -----------------------------------------
   output_logic: process(present_state, pipe_dma_up_dst_rdy)
   begin
      dma_ready                     <= '0';

      pipe_dma_up_src_rdy           <= '0';
      pipe_dma_up_sop               <= '0';
      pipe_dma_up_eop               <= '0';
      pipe_dma_done                 <= '0';

      case (present_state) is
         -- ---------------------------------------------
         when S_IDLE =>
            dma_ready               <= '1';
         -- ---------------------------------------------
         when S_DMA_START =>
            pipe_dma_up_sop         <= '1';
            pipe_dma_up_src_rdy     <= '1';
         -- ---------------------------------------------
         when S_DMA_ONE =>
            pipe_dma_up_sop         <= '1';
            pipe_dma_up_eop         <= '1';
            pipe_dma_up_src_rdy     <= '1';

            if(pipe_dma_up_dst_rdy = '1') then
               dma_ready            <= '1';
               pipe_dma_done        <= '1';
            end if;
         -- ---------------------------------------------
         when S_DMA_END =>
            pipe_dma_up_eop         <= '1';
            pipe_dma_up_src_rdy     <= '1';

            if(pipe_dma_up_dst_rdy = '1') then
               dma_ready            <= '1';
               pipe_dma_done        <= '1';
            end if;
         -- ---------------------------------------------
         when S_DMA =>
            pipe_dma_up_src_rdy     <= '1';
         -- ---------------------------------------------
      end case;
   end process;

end architecture;
