-- axi2mi.vhd: AXI2MI - Architecture declaration
-- Copyright (C) 2013 CESNET
-- Author(s): Jiri Matousek <xmatou06@stud.fit.vutbr.cz>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--
-- $Id$
--
-- TODO:
--
--


library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_unsigned.all;
use IEEE.std_logic_arith.all;


-- ----------------------------------------------------------------------------
--                      Architecture declaration
-- ----------------------------------------------------------------------------

architecture axi2mi_arch of axi2mi is

   -- Signals declaration -----------------------------------------------------

   -- interconnection of CQ interface and cq_int component 
   signal pipe_cq_int_data_in     : std_logic_vector(340 downto 0);
   signal pipe_cq_int_data_out    : std_logic_vector(340 downto 0);
   signal cq_data_in              : std_logic_vector(255 downto 0);
   signal cq_user_in              : std_logic_vector(84 downto 0);
   signal cq_ready_in             : std_logic;
   signal cq_valid_in             : std_logic;

   -- registered values from CQ interface
   signal reg_cq_data             : std_logic_vector(255 downto 0);
   signal reg_cq_user_first_be    : std_logic_vector(3 downto 0);
   signal reg_cq_user_last_be     : std_logic_vector(3 downto 0);
   signal reg_cq_user_tph_present : std_logic;
   signal reg_cq_user_tph_type    : std_logic_vector(1 downto 0);
   signal reg_cq_user_tph_st_tag  : std_logic_vector(7 downto 0);

   -- interconnection of cq_int and mi_int components
   signal sop                     : std_logic;
   signal valid                   : std_logic;
   signal ready                   : std_logic;
   signal req_type                : std_logic_vector(3 downto 0);
   signal addr                    : std_logic_vector(63 downto 0);
   signal dword_count             : std_logic_vector(10 downto 0);
   signal bar_id                  : std_logic_vector(2 downto 0);
   signal bar_aperture            : std_logic_vector(5 downto 0);
   
   -- interconnection of cq_int and mi_int components
   signal data_lower_addr         : std_logic_vector(4 downto 0);
   signal data_dword_count        : std_logic_vector(10 downto 0);
   signal user_first_be           : std_logic_vector(3 downto 0);
   signal user_last_be            : std_logic_vector(3 downto 0);
 
   -- interconnection of mi_int and cc_int components
   signal read_req                : std_logic;
   signal unsupported_req         : std_logic;
   signal eop                     : std_logic;
   signal completion_sent         : std_logic;
   signal data                    : std_logic_vector(31 downto 0);
   signal wr                      : std_logic;
   signal full                    : std_logic;

   -- interconnection of mi_int component and MI32 interface
   signal mi_dwr_out              : std_logic_vector(31 downto 0);
   signal mi_addr_out             : std_logic_vector(31 downto 0);
   signal mi_be_out               : std_logic_vector(3 downto 0);
   signal mi_rd_out               : std_logic;
   signal mi_wr_out               : std_logic;
   signal mi_drd_out              : std_logic_vector(31 downto 0);
   signal mi_ardy_out             : std_logic;
   signal mi_drdy_out             : std_logic;

   -- interconnection of cc_int component and CC interface
   signal pipe_cc_int_data_in     : std_logic_vector(264 downto 0);
   signal pipe_cc_int_data_out    : std_logic_vector(264 downto 0);
   signal cc_data_out             : std_logic_vector(255 downto 0);
   signal cc_last_out             : std_logic;
   signal cc_keep_out             : std_logic_vector(7 downto 0);
   signal cc_ready_out            : std_logic;
   signal cc_valid_out            : std_logic;

begin


   -- -------------------------------------------------------------------------
   --                       CQ interface connection
   -- -------------------------------------------------------------------------

   -- input data composition
   pipe_cq_int_data_in <=
      CQ_DATA &
      CQ_USER;

   -- pipe instantiation
   pipe_cq_int_i : entity work.pipe
   generic map(
      DATA_WIDTH => 341,
      USE_OUTREG => AXI_PIPE_USE_OUTREG,
      FAKE_PIPE  => not AXI_PIPE
   )
   port map(
      -- Common signals
      CLK         => CLK,
      RESET       => RESET,

      -- Input interface
      IN_DATA     => pipe_cq_int_data_in,
      IN_SRC_RDY  => CQ_VALID,
      IN_DST_RDY  => CQ_READY,

      -- Output interface
      OUT_DATA    => pipe_cq_int_data_out,
      OUT_SRC_RDY => cq_valid_in,
      OUT_DST_RDY => cq_ready_in
   );

   -- output data decomposition
   cq_data_in <= pipe_cq_int_data_out(340 downto 85);
   cq_user_in <= pipe_cq_int_data_out(84 downto 0);


   -- -------------------------------------------------------------------------
   --                           CQ_INT
   -- -------------------------------------------------------------------------

   cq_int_i: entity work.cq_int
   port map(
      -- Common signals
      CLK                     => CLK,
      RESET                   => RESET,

      -- CQ interface - reduced version
      CQ_DATA                 => cq_data_in,
      CQ_USER_FIRST_BE        => cq_user_in(3 downto 0),
      CQ_USER_LAST_BE         => cq_user_in(7 downto 4),
      CQ_USER_SOP             => cq_user_in(40),
      CQ_USER_TPH_PRESENT     => cq_user_in(42),
      CQ_USER_TPH_TYPE        => cq_user_in(44 downto 43),
      CQ_USER_TPH_ST_TAG      => cq_user_in(52 downto 45),
      CQ_VALID                => cq_valid_in,
      CQ_READY                => cq_ready_in,

      -- Registered values from CQ interface
      REG_CQ_DATA             => reg_cq_data,
      REG_CQ_USER_FIRST_BE    => reg_cq_user_first_be,
      REG_CQ_USER_LAST_BE     => reg_cq_user_last_be,
      REG_CQ_USER_TPH_PRESENT => reg_cq_user_tph_present,
      REG_CQ_USER_TPH_TYPE    => reg_cq_user_tph_type,
      REG_CQ_USER_TPH_ST_TAG  => reg_cq_user_tph_st_tag,

      -- Interface with mi_int
      SOP                     => sop,
      VALID                   => valid,
      READY                   => ready,
      REQ_TYPE                => req_type,
      ADDR                    => addr,
      DWORD_COUNT             => dword_count,
      BAR_ID                  => bar_id,
      BAR_APERTURE            => bar_aperture,

      -- Interface with cc_int
      DATA_LOWER_ADDR         => data_lower_addr,
      DATA_DWORD_COUNT        => data_dword_count,
      USER_FIRST_BE           => user_first_be,
      USER_LAST_BE            => user_last_be
   );

   CQ_NP_REQ <= '1';


   -- -------------------------------------------------------------------------
   --                           MI_INT
   -- -------------------------------------------------------------------------

   mi_int_i: entity work.pci_mi_int
   generic map(
      BAR0_BASE_ADDR       => BAR0_BASE_ADDR, 
      BAR1_BASE_ADDR       => BAR1_BASE_ADDR, 
      BAR2_BASE_ADDR       => BAR2_BASE_ADDR, 
      BAR3_BASE_ADDR       => BAR3_BASE_ADDR, 
      BAR4_BASE_ADDR       => BAR4_BASE_ADDR, 
      BAR5_BASE_ADDR       => BAR5_BASE_ADDR, 
      EXP_ROM_BASE_ADDR    => EXP_ROM_BASE_ADDR
   )
   port map(
      -- Common signals 
      CLK                  => CLK,
      RESET                => RESET,

      -- Registered values from CQ interface
      REG_CQ_DATA          => reg_cq_data,
      REG_CQ_USER_FIRST_BE => reg_cq_user_first_be,
      REG_CQ_USER_LAST_BE  => reg_cq_user_last_be,
     
      -- Interface with cq_int component
      SOP                  => sop,
      VALID                => valid,
      READY                => ready,
      REQ_TYPE             => req_type,
      ADDR                 => addr,
      DWORD_COUNT          => dword_count,
      BAR_ID               => bar_id,
      BAR_APERTURE         => bar_aperture,

      -- MI32 Interface
      MI_DWR               => mi_dwr_out,
      MI_ADDR              => mi_addr_out,
      MI_BE                => mi_be_out,
      MI_RD                => mi_rd_out,
      MI_WR                => mi_wr_out,
      MI_DRD               => mi_drd_out,
      MI_ARDY              => mi_ardy_out,
      MI_DRDY              => mi_drdy_out,

      -- Interface with cc_int component
      READ_REQ             => read_req,
      UNSUPPORTED_REQ      => unsupported_req,
      EOP                  => eop,
      COMPLETION_SENT      => completion_sent,      
      DATA                 => data,
      WR                   => wr,
      FULL                 => full
   );


   -- -------------------------------------------------------------------------
   --                       MI interface connection
   -- -------------------------------------------------------------------------

   -- pipe instantiation
   pipe_mi_int_i : entity work.mi_pipe
   generic map(
      USE_OUTREG => MI_PIPE_USE_OUTREG,
      FAKE_PIPE  => not MI_PIPE
   )
   port map(
      -- Common signals
      CLK      => CLK,
      RESET    => RESET,

      -- Input interface
      IN_DWR   => mi_dwr_out,
      IN_ADDR  => mi_addr_out,
      IN_BE    => mi_be_out,
      IN_RD    => mi_rd_out,
      IN_WR    => mi_wr_out,
      IN_DRD   => mi_drd_out,
      IN_ARDY  => mi_ardy_out,
      IN_DRDY  => mi_drdy_out,

      -- Output interface
      OUT_DWR  => MI_DWR,
      OUT_ADDR => MI_ADDR,
      OUT_BE   => MI_BE,
      OUT_RD   => MI_RD,
      OUT_WR   => MI_WR,
      OUT_DRD  => MI_DRD,
      OUT_ARDY => MI_ARDY,
      OUT_DRDY => MI_DRDY
   );


   -- -------------------------------------------------------------------------
   --                           CC_INT
   -- -------------------------------------------------------------------------

   cc_int_i: entity work.cc_int
   generic map(
      FIFO_SIZE    => FIFO_SIZE
   )
   port map(
      -- Common signals
      CLK                     => CLK,
      RESET                   => RESET,

      -- Registered values from CQ interface
      REG_CQ_DESCRIPTOR       => reg_cq_data(127 downto 0),
      REG_CQ_USER_FIRST_BE    => reg_cq_user_first_be,
      REG_CQ_USER_LAST_BE     => reg_cq_user_last_be,
      REG_CQ_USER_TPH_PRESENT => reg_cq_user_tph_present,
      REG_CQ_USER_TPH_TYPE    => reg_cq_user_tph_type,
      REG_CQ_USER_TPH_ST_TAG  => reg_cq_user_tph_st_tag,

      -- Interface with cq_int
      DATA_LOWER_ADDR         => data_lower_addr,
      DATA_DWORD_COUNT        => data_dword_count,
      USER_FIRST_BE           => user_first_be,
      USER_LAST_BE            => user_last_be,

      -- Interface with mi_int component
      READ_REQ                => read_req,
      UNSUPPORTED_REQ         => unsupported_req,
      EOP                     => eop,
      COMPLETION_SENT         => completion_sent,
      DATA                    => data,
      WR                      => wr,
      FULL                    => full,

      -- CC Interface (CC) - reduced version
      CC_DATA                 => cc_data_out,
      CC_LAST                 => cc_last_out,
      CC_KEEP                 => cc_keep_out,
      CC_VALID                => cc_valid_out,
      CC_READY                => cc_ready_out,

      -- Configuration Status Interface - selected parts
      MAX_PAYLOAD_SIZE        => MAX_PAYLOAD_SIZE
   );


   -- -------------------------------------------------------------------------
   --                       CC interface connection
   -- -------------------------------------------------------------------------

   -- input data composition
   pipe_cc_int_data_in <=
      cc_data_out  &
      cc_last_out  &
      cc_keep_out;

   -- pipe instantiation
   pipe_cc_int_i : entity work.pipe
   generic map(
      DATA_WIDTH => 265,
      USE_OUTREG => AXI_PIPE_USE_OUTREG,
      FAKE_PIPE  => not AXI_PIPE
   )
   port map(
      -- Common signals
      CLK         => CLK,
      RESET       => RESET,

      -- Input interface
      IN_DATA     => pipe_cc_int_data_in,
      IN_SRC_RDY  => cc_valid_out,
      IN_DST_RDY  => cc_ready_out,

      -- Output interface
      OUT_DATA    => pipe_cc_int_data_out,
      OUT_SRC_RDY => CC_VALID,
      OUT_DST_RDY => CC_READY
   );

   -- output data decomposition
   CC_DATA <= pipe_cc_int_data_out(264 downto 9);
   CC_USER <= (others => '0');
   CC_LAST <= pipe_cc_int_data_out(8);
   CC_KEEP <= pipe_cc_int_data_out(7 downto 0);


end architecture axi2mi_arch;
