-- key_filter.vhd: Filter out specified fields from input 5-tuple
-- Copyright (C) 2013 Brno University of Technology
-- Author(s): Lukas Kekely <xkekel00@stud.fit.vutbr.cz>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--
-- $Id: key_filter.vhd 4531 2013-12-20 16:29:12Z xkekel00 $
--
-- TODO:
--
--

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_arith.all;
use IEEE.std_logic_unsigned.all;
use IEEE.numeric_std.all;
use WORK.math_pack.all;
-- ----------------------------------------------------------------------------
--                        Entity declaration
-- ----------------------------------------------------------------------------
entity key_filter is
  generic(
    KEY_WIDTH           : integer := 128;
    SRCIP_WIDTH         : integer := 128;
    DSTIP_WIDTH         : integer := 0;
    USE_SRCPORT         : boolean := false;
    USE_DSTPORT         : boolean := false;
    USE_PROTO           : boolean := false;
    IPV4_ONLY           : boolean := false
  );
  port(
    SRCIP               : in std_logic_vector(127 downto 0);
    DSTIP               : in std_logic_vector(127 downto 0);
    SRCPORT             : in std_logic_vector(15 downto 0);
    DSTPORT             : in std_logic_vector(15 downto 0);
    PROTO               : in std_logic_vector(7 downto 0);
    KEY                 : out std_logic_vector(128*2+16*2+8-1 downto 0)
  );
end entity;

-- ----------------------------------------------------------------------------
--                      Architecture declaration
-- ----------------------------------------------------------------------------
architecture arch of key_filter is
  function bool2int(b : boolean) return integer is
  begin
    if b then
      return 1;
    end if;
    return 0;
  end function;
  
  function ipver2length(b : boolean) return integer is
  begin
    if b then
      return 32;
    end if;
    return 128;
  end function;
  
  constant IP_OFFSET      : integer := 128-ipver2length(IPV4_ONLY);
  constant SRCIP_START    : integer := KEY_WIDTH;
  constant DSTIP_START    : integer := SRCIP_START   - SRCIP_WIDTH;
  constant SRCPORT_START  : integer := DSTIP_START   - DSTIP_WIDTH;
  constant DSTPORT_START  : integer := SRCPORT_START - (16*bool2int(USE_SRCPORT));
  constant PROTO_START    : integer := DSTPORT_START - (16*bool2int(USE_DSTPORT));
  
begin
  real_gen : if KEY_WIDTH>0 generate
    -- zero padding
    padding_gen : if (128*2+16*2+8)>KEY_WIDTH generate
      KEY(128*2+16*2+8-1 downto KEY_WIDTH) <= (others => '0');
    end generate;
    
    -- source IP
    srcip_gen : if SRCIP_WIDTH>0 generate
      KEY(SRCIP_START-1 downto SRCIP_START-SRCIP_WIDTH) <= SRCIP(SRCIP_WIDTH+IP_OFFSET-1 downto IP_OFFSET);
    end generate;
    
    -- destination IP
    dstip_gen : if DSTIP_WIDTH>0 generate
      KEY(DSTIP_START-1 downto DSTIP_START-DSTIP_WIDTH) <= DSTIP(DSTIP_WIDTH+IP_OFFSET-1 downto IP_OFFSET);
    end generate;
    
    -- source port
    srcport_gen : if USE_SRCPORT generate
      KEY(SRCPORT_START-1 downto SRCPORT_START-16) <= SRCPORT;
    end generate;
    
    -- destination port
    dstport_gen : if USE_DSTPORT generate
      KEY(DSTPORT_START-1 downto DSTPORT_START-16) <= DSTPORT;
    end generate;
    
    -- protocol
    proto_gen : if USE_PROTO generate
      KEY(PROTO_START-1 downto 0) <= PROTO;
    end generate;
  end generate;
  fake_gen : if KEY_WIDTH=0 generate
    KEY <= (others => '0');
  end generate;
end architecture;
