-- fl_hins_ent.vhd: Entity of FL header inserter into FL data strem
--                  (header size can be at most one data word)
-- Copyright (c) 2014 Brno University of Technology 
-- Author: Jiri Matousek <xmatou06@stud.fit.vutbr.cz>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--
-- $Id: header_insert_ent.vhd 1744 2012-07-10 21:00:50Z xkekel00 $
--
-- TODO:
--


library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_unsigned.all;

use work.math_pack.all;


-- ----------------------------------------------------------------------------
--                           Entity declaration
-- ----------------------------------------------------------------------------

entity fl_hins is
generic (
   --! \brief Data width of input FL payload interface
   --! \details In bits, must be power of 2 and greater than 8.
   DATA_WIDTH           : integer := 512;
   --! \brief Data width of input FL header interface
   --! \details In bits, must be power of 2 and greater than 8.
   HDR_WIDTH            : integer := 128
); 
port (
   --! \name Clock and reset signals
   -- -------------------------------------------------------------------------
   --! \brief Clock signal
   CLK                  : in std_logic;
   --! \brief Reset signal
   RESET                : in std_logic;

   --! \name Input FrameLink payload interface
   -- -------------------------------------------------------------------------
   --! \brief Payload data
   RX_PAYLOAD_DATA      : in  std_logic_vector(DATA_WIDTH-1 downto 0);
   --! \brief Start of frame, active in '0'
   RX_PAYLOAD_SOF_N     : in  std_logic;
   --! \brief Start of part, active in '0'
   RX_PAYLOAD_SOP_N     : in  std_logic;
   --! \brief End of frame, active in '0'
   RX_PAYLOAD_EOF_N     : in  std_logic;
   --! \brief End of part, active in '0'
   RX_PAYLOAD_EOP_N     : in  std_logic;
   --! \brief End of part position
   --! \details Valid only if RX_PAYLOAD_EOP_N is set to '0'.
   RX_PAYLOAD_REM       : in  std_logic_vector(log2(DATA_WIDTH/8)-1 downto 0);
   --! \brief Source is ready, active in '0'
   RX_PAYLOAD_SRC_RDY_N : in  std_logic;
   --! \brief Destination is ready, active in '0'
   RX_PAYLOAD_DST_RDY_N : out std_logic;

   --! \name Input FrameLink header interface
   -- -------------------------------------------------------------------------
   --! \brief Header data
   RX_HDR_DATA          : in  std_logic_vector(HDR_WIDTH-1 downto 0);
   --! \brief Start of frame, active in '0'
   RX_HDR_SOF_N         : in  std_logic;
   --! \brief Start of part, active in '0'
   RX_HDR_SOP_N         : in  std_logic;
   --! \brief End of frame, active in '0'
   RX_HDR_EOF_N         : in  std_logic;
   --! \brief End of part, active in '0'
   RX_HDR_EOP_N         : in  std_logic;
   --! \brief End of part position
   --! \details Valid only if RX_HEADER_EOP_N is set to '0'.
   RX_HDR_REM           : in  std_logic_vector(log2(HDR_WIDTH/8)-1 downto 0);
   --! \brief Source is ready, active in '0'
   RX_HDR_SRC_RDY_N     : in  std_logic;
   --! \brief Destination is ready, active in '0'
   RX_HDR_DST_RDY_N     : out std_logic;

   --! \name Output FrameLink interface
   -- -------------------------------------------------------------------------
   --! \brief Data
   TX_DATA              : out std_logic_vector(DATA_WIDTH-1 downto 0);
   --! \brief Start of frame, active in '0'
   TX_SOF_N             : out std_logic;
   --! \brief Start of part, active in '0'
   TX_SOP_N             : out std_logic;
   --! \brief End of frame, active in '0'
   TX_EOF_N             : out std_logic;
   --! \brief End of part, active in '0'
   TX_EOP_N             : out std_logic;
   --! \brief End of part position
   --! \details Valid only if TX_EOP_N is set to '0'.
   TX_REM               : out std_logic_vector(log2(DATA_WIDTH/8)-1 downto 0);
   --! \brief Source is ready, active in '0'
   TX_SRC_RDY_N         : out std_logic;
   --! \brief Destination is ready, active in '0'
   TX_DST_RDY_N         : in std_logic
);
end entity fl_hins;


