-- network_mod.vhd : network module for one 100 Gbps port of Fiberblaze FB1CG
--                   card
--!
--! \file
--! \brief Network Module for one 100 Gbps port of Fiberblaze FB1CG card
--! \author Jiri Matousek <xmatou06@stud.fit.vutbr.cz>
--! \date 2014
--!
--! \section License
--!
--! Copyright (C) 2014 CESNET
--!
--! Redistribution and use in source and binary forms, with or without
--! modification, are permitted provided that the following conditions
--! are met:
--! 1. Redistributions of source code must retain the above copyright
--!    notice, this list of conditions and the following disclaimer.
--! 2. Redistributions in binary form must reproduce the above copyright
--!    notice, this list of conditions and the following disclaimer in
--!    the documentation and/or other materials provided with the
--!    distribution.
--! 3. Neither the name of the Company nor the names of its contributors
--!    may be used to endorse or promote products derived from this
--!    software without specific prior written permission.
--!
--! This software is provided ``as is'', and any express or implied
--! warranties, including, but not limited to, the implied warranties of
--! merchantability and fitness for a particular purpose are disclaimed.
--! In no event shall the company or contributors be liable for any
--! direct, indirect, incidental, special, exemplary, or consequential
--! damages (including, but not limited to, procurement of substitute
--! goods or services; loss of use, data, or profits; or business
--! interruption) however caused and on any theory of liability, whether
--! in contract, strict liability, or tort (including negligence or
--! otherwise) arising in any way out of the use of this software, even
--! if advised of the possibility of such damage.
--!


library ieee;
use ieee.std_logic_1164.all;
use ieee.std_logic_arith.all;
use ieee.std_logic_unsigned.all;

--! Package with log2 function
use work.math_pack.all;

-- ----------------------------------------------------------------------------
--                            Architecture Declaration
-- ----------------------------------------------------------------------------

architecture full of network_mod is

   --! Constants declaration
   -- -------------------------------------------------------------------------

   --! Number of items of IBUF's FIFOs  
   constant IBUF_DFIFO_SIZE     : integer := (IBUF_DFIFO_BYTES * 8) / 512; 
   constant IBUF_HFIFO_SIZE     : integer := (IBUF_HFIFO_BYTES * 8) / HDR_WIDTH;

   --! Number of items of OBUF's FIFOs  
   constant OBUF_DFIFO_SIZE     : integer := (OBUF_DFIFO_BYTES * 8) / 512; 
   constant OBUF_HFIFO_SIZE     : integer := (OBUF_HFIFO_BYTES * 8) / HDR_WIDTH;

   --! MI splitter constants
   constant MI_SPLIT_DATA_WIDTH : integer := 32;
   constant MI_SPLIT_ITEMS      : integer := 2;

   --! Signals declaration
   -- -------------------------------------------------------------------------

   --! Address decoder signals
   signal ibuf_cs               : std_logic;
   signal obuf_cs               : std_logic;
   signal reg_ibuf_cs           : std_logic;
   signal reg_obuf_cs           : std_logic;

   --! MI Splitter outputs
   signal mi_split_dwr          : std_logic_vector(MI_SPLIT_ITEMS*MI_SPLIT_DATA_WIDTH-1 downto 0);
   signal mi_split_addr         : std_logic_vector(MI_SPLIT_ITEMS*32-1 downto 0);
   signal mi_split_rd           : std_logic_vector(MI_SPLIT_ITEMS-1 downto 0);
   signal mi_split_wr           : std_logic_vector(MI_SPLIT_ITEMS-1 downto 0);
   signal mi_split_be           : std_logic_vector(MI_SPLIT_ITEMS*MI_SPLIT_DATA_WIDTH/8-1 downto 0);
   signal mi_split_drd          : std_logic_vector(MI_SPLIT_ITEMS*MI_SPLIT_DATA_WIDTH-1 downto 0);
   signal mi_split_ardy         : std_logic_vector(MI_SPLIT_ITEMS-1 downto 0);
   signal mi_split_drdy         : std_logic_vector(MI_SPLIT_ITEMS-1 downto 0);

   --! Pipeline registers between the ETH_PHY and IBUF
   signal ibuf_cgmii_rxd        : std_logic_vector(511 downto 0);
   signal ibuf_cgmii_rxc        : std_logic_vector( 63 downto 0);

   --! Synchronized version of REPEATER_CTRL
   signal sync_repeater_ctrl    : std_logic_vector(  1 downto 0);

   --! Repeater input signals (from IBUF's input, OBUF's output and Idle)
   signal ibuf_cgmii_txd        : std_logic_vector(511 downto 0);
   signal ibuf_cgmii_txc        : std_logic_vector( 63 downto 0);
   signal obuf_cgmii_txd        : std_logic_vector(511 downto 0);
   signal obuf_cgmii_txc        : std_logic_vector( 63 downto 0);
   signal idle_cgmii_txd        : std_logic_vector(511 downto 0);
   signal idle_cgmii_txc        : std_logic_vector( 63 downto 0);

   --! Link up signals
   signal ibuf_link_up          : std_logic;
   signal ibuf_link_up_filtered : std_logic;

   --! Packet activity signals
   signal ibuf_incoming_packet  : std_logic;
   signal obuf_outgoing_packet  : std_logic;
   signal packet_activity       : std_logic;


-- ----------------------------------------------------------------------------
--                             Architecture Body
-- ----------------------------------------------------------------------------
  
begin


   -- -------------------------------------------------------------------------
   --                               MI Splitter
   -- -------------------------------------------------------------------------

--   mi_split_ibuf_obuf_i : entity work.MI_SPLITTER_PLUS
--   generic map (
--      DATA_WIDTH    => MI_SPLIT_DATA_WIDTH,
--      ITEMS         => MI_SPLIT_ITEMS,
--      ADDR_CMP_MASK => X"00001000",
--      --! IBUF's base address
----    PORT0_BASE    => X"00000000" (practically X"00008000")
--      --! OBUF's base address
--      PORT1_BASE    => X"00009000",
--      PIPE          => false,
--      PIPE_OUTREG   => false
--   )
--   port map (
--      --! Common interface
--      CLK         => MI_CLK,
--      RESET       => MI_RESET,
--
--      --! Input MI interface
--      IN_DWR      => MI_DWR,
--      IN_ADDR     => MI_ADDR,
--      IN_BE       => MI_BE,
--      IN_RD       => MI_RD,
--      IN_WR       => MI_WR,
--      IN_DRD      => MI_DRD,
--      IN_ARDY     => MI_ARDY,
--      IN_DRDY     => MI_DRDY,
--
--      --! Output MI interfaces
--      OUT_DWR     => mi_split_dwr,
--      OUT_ADDR    => mi_split_addr,
--      OUT_BE      => mi_split_be,
--      OUT_RD      => mi_split_rd,
--      OUT_WR      => mi_split_wr,
--      OUT_DRD     => mi_split_drd,
--      OUT_ARDY    => mi_split_ardy,
--      OUT_DRDY    => mi_split_drdy
--   );

   ibuf_cs <= '1' when MI_ADDR(13 downto 12) = "00" else
              '0';
   obuf_cs <= '1' when MI_ADDR(13 downto 12) = "01" else
              '0';

   mi_cs_regs : process(MI_CLK)
   begin
      if (MI_CLK'event AND MI_CLK = '1') then
         if (mi_split_rd /= "00") then
            reg_ibuf_cs <= ibuf_cs;
            reg_obuf_cs <= obuf_cs;
         end if;
      end if;
   end process mi_cs_regs;

   mi_split_dwr  <= MI_DWR & MI_DWR;
   mi_split_addr <= MI_ADDR & MI_ADDR;
   mi_split_rd   <= '0' & MI_RD when ibuf_cs = '1' else
                    MI_RD & '0' when obuf_cs = '1' else
                    "00";
   mi_split_wr   <= '0' & MI_WR when ibuf_cs = '1' else
                    MI_WR & '0' when obuf_cs = '1' else
                    "00";
   mi_split_be   <= MI_BE & MI_BE;
   MI_DRD        <= mi_split_drd(31 downto  0) when ibuf_cs = '1' else
                    mi_split_drd(63 downto 32) when obuf_cs = '1' else
                    (others => '0');
   MI_ARDY       <= mi_split_ardy(0) OR mi_split_ardy(1);
   MI_DRDY       <= mi_split_drdy(0) when reg_ibuf_cs = '1' else
                    mi_split_drdy(1) when reg_obuf_cs = '1' else
                    '0';

 
   -- -------------------------------------------------------------------------
   --                               CGMII IBUF
   -- -------------------------------------------------------------------------
   
   --! Pipeline to improve RX timing 
   RXD_FFS: process(CGMII_RXCLK)
   begin
      if CGMII_RXCLK'event and CGMII_RXCLK = '1' then
          ibuf_cgmii_rxd <= CGMII_RXD;
          ibuf_cgmii_rxc <= CGMII_RXC;
      end if;
   end process;
   ibuf_cgmii_txd <= ibuf_cgmii_rxd;
   ibuf_cgmii_txc <= ibuf_cgmii_rxc;

   ibuf_cgmii_i: entity work.ibuf_cgmii
   generic map (
      DFIFO_SIZE          => IBUF_DFIFO_SIZE,
      HFIFO_SIZE          => IBUF_HFIFO_SIZE,
      HDR_WIDTH           => HDR_WIDTH,
      MAC_COUNT           => IBUF_MAC_COUNT,
      INBANDFCS           => INBANDFCS,
      CNT_ERROR_LENGTH    => IBUF_CNT_ERROR_LENGTH,
      RESET_BY_INIT       => false,
      CNT_DSP             => true
   )
   port map (
      --! CGMII interface
      CGMII_RXCLK         => CGMII_RXCLK,
      CGMII_RXRESET       => CGMII_RXRESET,
      CGMII_RXD           => ibuf_cgmii_rxd,
      CGMII_RXC           => ibuf_cgmii_rxc,

      --! FrameLink Unaligned interface (data payload output)
      TX_CLK              => TX_CLK,
      TX_RESET            => TX_RESET,
      TX_DATA             => TX_DATA,
      TX_SOP_POS          => TX_SOP_POS,
      TX_EOP_POS          => TX_EOP_POS,
      TX_SOP              => TX_SOP,
      TX_EOP              => TX_EOP,
      TX_SRC_RDY          => TX_SRC_RDY,
      TX_DST_RDY          => TX_DST_RDY,

      --! FrameLink interface (data header output)
      TX_HDATA            => TX_HDATA,
      TX_HSOP_N           => TX_HSOP_N,
      TX_HEOP_N           => TX_HEOP_N,
      TX_HREM             => TX_HREM,
      TX_HSRC_RDY_N       => TX_HSRC_RDY_N,
      TX_HDST_RDY_N       => TX_HDST_RDY_N,

      --! MI32 interface
      MI_CLK              => MI_CLK, 
      MI_RESET            => MI_RESET,
      MI_DWR              => mi_split_dwr((0+1)*MI_SPLIT_DATA_WIDTH-1 downto 0*MI_SPLIT_DATA_WIDTH),
      MI_ADDR             => mi_split_addr((0+1)*32-1 downto 0*32),
      MI_BE               => mi_split_be((0+1)*MI_SPLIT_DATA_WIDTH/8-1 downto 0*MI_SPLIT_DATA_WIDTH/8),
      MI_RD               => mi_split_rd(0),
      MI_WR               => mi_split_wr(0),
      MI_DRD              => mi_split_drd((0+1)*MI_SPLIT_DATA_WIDTH-1 downto 0*MI_SPLIT_DATA_WIDTH),
      MI_ARDY             => mi_split_ardy(0),
      MI_DRDY             => mi_split_drdy(0),

      --! Sampling unit interface
      SAU_CLK             => SAU_CLK,
      SAU_RESET           => SAU_RESET,
      SAU_REQ             => SAU_REQ,
      SAU_ACCEPT          => SAU_ACCEPT,
      SAU_DV              => SAU_DV,

      --! Statistics for PACODAG
      CTRL_STAT_MAC_ERR   => CTRL_STAT_MAC_ERR,
      CTRL_STAT_MINTU_ERR => CTRL_STAT_MINTU_ERR,
      CTRL_STAT_MTU_ERR   => CTRL_STAT_MTU_ERR,
      CTRL_STAT_CRC_ERR   => CTRL_STAT_CRC_ERR,
      CTRL_STAT_FRAME_ERR => CTRL_STAT_FRAME_ERR,
      CTRL_STAT_FRAME_LEN => CTRL_STAT_FRAME_LEN,

      --! PACODAG interface
      CTRL_CLK            => CTRL_CLK,
      CTRL_RESET          => CTRL_RESET,
      CTRL_DATA           => CTRL_DATA,
      CTRL_SOP_N          => CTRL_SOP_N,
      CTRL_EOP_N          => CTRL_EOP_N,
      CTRL_REM            => CTRL_REM,
      CTRL_SRC_RDY_N      => CTRL_SRC_RDY_N,
      CTRL_DST_RDY_N      => CTRL_DST_RDY_N,
      CTRL_SOP            => CTRL_SOP,
      CTRL_STAT_DV        => CTRL_STAT_DV,
      CTRL_RDY            => CTRL_RDY,

      --! State of the link interface
      LINK_UP             => ibuf_link_up,
      INCOMING_PACKET     => ibuf_incoming_packet
   );

 
   -- -------------------------------------------------------------------------
   --                               CGMII OBUF
   -- -------------------------------------------------------------------------

   obuf_cgmii_i : entity work.obuf_cgmii
   generic map (
      DFIFO_ITEMS     => OBUF_DFIFO_SIZE,
      HFIFO_ITEMS     => OBUF_HFIFO_SIZE,
      CNT_DSP         => true
   )
   port map (
      --! CGMII interface
      CGMII_TXCLK     => CGMII_TXCLK,
      CGMII_TXRESET   => CGMII_TXRESET,
      CGMII_TXD       => obuf_cgmii_txd,
      CGMII_TXC       => obuf_cgmii_txc,

      --! FrameLink Unaligned interface
      RX_CLK          => RX_CLK,
      RX_RESET        => RX_RESET,
      RX_DATA         => RX_DATA,
      RX_SOP_POS      => RX_SOP_POS,
      RX_EOP_POS      => RX_EOP_POS,
      RX_SOP          => RX_SOP,
      RX_EOP          => RX_EOP,
      RX_SRC_RDY      => RX_SRC_RDY,
      RX_DST_RDY      => RX_DST_RDY,

      --! MI32 interface
      MI_CLK          => MI_CLK, 
      MI_RESET        => MI_RESET,
      MI_DWR          => mi_split_dwr((1+1)*MI_SPLIT_DATA_WIDTH-1 downto 1*MI_SPLIT_DATA_WIDTH),
      MI_ADDR         => mi_split_addr((1+1)*32-1 downto 1*32),
      MI_BE           => mi_split_be((1+1)*MI_SPLIT_DATA_WIDTH/8-1 downto 1*MI_SPLIT_DATA_WIDTH/8),
      MI_RD           => mi_split_rd(1),
      MI_WR           => mi_split_wr(1),
      MI_DRD          => mi_split_drd((1+1)*MI_SPLIT_DATA_WIDTH-1 downto 1*MI_SPLIT_DATA_WIDTH),
      MI_ARDY         => mi_split_ardy(1),
      MI_DRDY         => mi_split_drdy(1),

      --! Status interface
      OUTGOING_PACKET => obuf_outgoing_packet
   );


   -- -------------------------------------------------------------------------
   --                             Repeater logic
   -- -------------------------------------------------------------------------

   --! Synchronization of REPEATER_CTRL
   sync_repeater_ctrl_i : entity work.ASYNC_BUS_HANDSHAKE
   generic map (
      DATA_WIDTH => 2
   )
   port map (
      --! A clock domain
      ACLK       => MI_CLK,
      ARST       => MI_RESET,
      ADATAIN    => REPEATER_CTRL,
      ASEND      => '1',
      AREADY     => open,

      --! B clock domain
      BCLK       => CGMII_TXCLK,
      BRST       => CGMII_TXRESET,
      BDATAOUT   => sync_repeater_ctrl,
      BLOAD      => '1',
      BVALID     => open
   );

   --! Idle characters vector coposition
   idle_cgmii_txd <= X"0707070707070707" &
                     X"0707070707070707" &
                     X"0707070707070707" &
                     X"0707070707070707" &
                     X"0707070707070707" &
                     X"0707070707070707" &
                     X"0707070707070707" &
                     X"0707070707070707";
   idle_cgmii_txc <= "11111111" &
                     "11111111" &
                     "11111111" &
                     "11111111" &
                     "11111111" &
                     "11111111" &
                     "11111111" &
                     "11111111";

   --! Repeater of TXD
   CGMII_TXD <= obuf_cgmii_txd when (sync_repeater_ctrl(0) = '0') else -- OBUF
                idle_cgmii_txd when (sync_repeater_ctrl(1) = '0') else -- Idle
                ibuf_cgmii_txd;                                        -- IBUF

   --! Repeater of TXC
   CGMII_TXC <= obuf_cgmii_txc when (sync_repeater_ctrl(0) = '0') else -- OBUF
                idle_cgmii_txc when (sync_repeater_ctrl(1) = '0') else -- Idle
                ibuf_cgmii_txc;                                        -- IBUF


   -- -------------------------------------------------------------------------
   --                             LED controllers
   -- -------------------------------------------------------------------------

   --! Glitch filter on ibuf_link_up signal
   -- -------------------------------------------------------------------------
   glitch_filter_i : entity work.glitch_filter
   generic map(
      FILTER_LENGTH   => 4,
      FILTER_SAMPLING => 24,
      INIT            => '0'
   )
   port map (
      CLK             => CGMII_RXCLK,
      RESET           => CGMII_RXRESET,
      DIN             => ibuf_link_up,
      DOUT            => ibuf_link_up_filtered
   );

   --! Assignment of link up valie to the output port
   LINK <= ibuf_link_up_filtered;

   --! Link LED controller
   -- -------------------------------------------------------------------------
   link_led_ctrl_i : entity work.led_ctrl
   generic map (
      CNTR_SIZE => LED_CTRL_CNTR_SIZE
   )
   port map (
      CLK       => CGMII_RXCLK,
      RESET     => CGMII_RXRESET,
      LINE_UP   => ibuf_link_up_filtered,
      PACKET    => '0',
      LED       => LINK_LED
   );

   --! Activity LED controller
   -- -------------------------------------------------------------------------
   activity_led_ctrl_i : entity work.led_ctrl
   generic map (
      CNTR_SIZE => LED_CTRL_CNTR_SIZE
   )
   port map (
      CLK       => CGMII_RXCLK,
      RESET     => CGMII_RXRESET,
      LINE_UP   => '1',
      PACKET    => packet_activity,
      LED       => ACTIVITY_LED
   );

   --! Composition of packet activity signal
   packet_activity <= ibuf_incoming_packet OR obuf_outgoing_packet;

end architecture full;
