-- eth100g.vhd : IP core implementing 100GE interface
--               (encapsulates PCS/PMA layer and network module)
--!
--! \file
--! \brief IP core implementing 100GE interface
--! \author Jiri Matousek <xmatou06@stud.fit.vutbr.cz>
--! \date 2014
--!
--! \section License
--!
--! Copyright (C) 2014 CESNET
--!
--! Redistribution and use in source and binary forms, with or without
--! modification, are permitted provided that the following conditions
--! are met:
--! 1. Redistributions of source code must retain the above copyright
--!    notice, this list of conditions and the following disclaimer.
--! 2. Redistributions in binary form must reproduce the above copyright
--!    notice, this list of conditions and the following disclaimer in
--!    the documentation and/or other materials provided with the
--!    distribution.
--! 3. Neither the name of the Company nor the names of its contributors
--!    may be used to endorse or promote products derived from this
--!    software without specific prior written permission.
--!
--! This software is provided ``as is'', and any express or implied
--! warranties, including, but not limited to, the implied warranties of
--! merchantability and fitness for a particular purpose are disclaimed.
--! In no event shall the company or contributors be liable for any
--! direct, indirect, incidental, special, exemplary, or consequential
--! damages (including, but not limited to, procurement of substitute
--! goods or services; loss of use, data, or profits; or business
--! interruption) however caused and on any theory of liability, whether
--! in contract, strict liability, or tort (including negligence or
--! otherwise) arising in any way out of the use of this software, even
--! if advised of the possibility of such damage.
--!


library ieee;
use ieee.std_logic_1164.all;
use ieee.std_logic_arith.all;
use ieee.std_logic_unsigned.all;

library unisim;
use unisim.vcomponents.all;

--! Package with log2 function
use work.math_pack.all;


-- ----------------------------------------------------------------------------
--                             Entity Declaration
-- ----------------------------------------------------------------------------

entity eth100g is
generic (
   --! \brief Size of packet data FIFO inside IBUF in bytes
   --! \details Must be at least as long as the longest possible packet.
   IBUF_DFIFO_BYTES      : integer := 4096;
   --! \brief Size of control data FIFO inside IBUF in bytes
   IBUF_HFIFO_BYTES      : integer := 1024;
   --! \brief Data width of IBUF's FL interface for SZE headers
   --! \details In bits, must be power of 2 and greater than 8.
   HDR_WIDTH             : integer := 128;
   --! \brief Size of CAM for MAC addresses within IBUF
   --! \details Must be lower or equal to 16.
   IBUF_MAC_COUNT        : integer := 16;
   --! \brief Keep FCS (CRC) for further processing of a packet
   --! \details True = keep, False = remove.
   INBANDFCS             : boolean := false;
   --! \brief Length of link error timeout counter (number of counter bits)
   --! \details If error don't occur for 2^IBUF_CNT_ERROR_LENGTH cycles, the
   --!          link is declared to be up.
   IBUF_CNT_ERROR_LENGTH : integer := 5;

   --! \brief Size of packet data fifo inside OBUFs in bytes
   --! \details Must be at least as long as the longest possible packet.
   --!          Default value is 4096 without 16KB support or 16384 with
   --!          16KB support.
   OBUF_DFIFO_BYTES      : integer := 4096;
   --! \brief Size of helper FIFO inside OBUF in bytes
   OBUF_HFIFO_BYTES      : integer := 1024;

   --! \brief Length of the internal LED controller counter which determines
   --!        period of LED blinks
   LED_CTRL_CNTR_SIZE    : integer := 27
);
port (
   --! \name Clock and reset signals
   -- -------------------------------------------------------------------------
   --! \brief Asynchronous reset
   RESET         : in  std_logic;
   --! \brief DRP clock signal
   --! \details maximum 125 MHz
   DRPCLK        : in  std_logic;

   --! \name CFP2 transceiver reference clock interface
   -- -------------------------------------------------------------------------
   --! \brief Reference clock, positive
   CFP2_GTZ_REFCLK_P : in  std_logic;
   --! \brief Reference clock, negative
   CFP2_GTZ_REFCLK_N : in  std_logic;

   --! \name CFP2 transceiver data interface
   -- -------------------------------------------------------------------------
   --! \brief Transceiver 0 RX data, positive
   CFP2_RX0_RX1_P        : in    std_logic;
   --! \brief Transceiver 0 RX data, negative
   CFP2_RX0_RX1_N        : in    std_logic;
   --! \brief Transceiver 0 TX data, positive
   CFP2_TX0_TX1_P        : out   std_logic;
   --! \brief Transceiver 0 TX data, negative
   CFP2_TX0_TX1_N        : out   std_logic;
   --! \brief Transceiver 1 RX data, positive
   CFP2_RX1_RX2_P        : in    std_logic;
   --! \brief Transceiver 1 RX data, negative
   CFP2_RX1_RX2_N        : in    std_logic;
   --! \brief Transceiver 1 TX data, positive
   CFP2_TX1_TX2_P        : out   std_logic;
   --! \brief Transceiver 1 TX data, negative
   CFP2_TX1_TX2_N        : out   std_logic;
   --! \brief Transceiver 2 RX data, positive
   CFP2_RX2_RX5_P        : in    std_logic;
   --! \brief Transceiver 2 RX data, negative
   CFP2_RX2_RX5_N        : in    std_logic;
   --! \brief Transceiver 2 TX data, positive
   CFP2_TX2_TX5_P        : out   std_logic;
   --! \brief Transceiver 2 TX data, negative
   CFP2_TX2_TX5_N        : out   std_logic;
   --! \brief Transceiver 3 RX data, positive
   CFP2_RX3_RX6_P        : in    std_logic;
   --! \brief Transceiver 3 RX data, negative
   CFP2_RX3_RX6_N        : in    std_logic;
   --! \brief Transceiver 3 TX data, positive
   CFP2_TX3_TX6_P        : out   std_logic;
   --! \brief Transceiver 3 TX data, negative
   CFP2_TX3_TX6_N        : out   std_logic;

   --! \name CFP2 transceiver control interface 
   -- -------------------------------------------------------------------------
   --! \brief Global alarm, active low
   CFP2_GLB_ALRM_N       : in    std_logic;
   --! \brief Module absent
   CFP2_MOD_ABS          : in    std_logic;
   --! \brief Module low power mode
   CFP2_MOD_LOPWR        : out   std_logic;
   --! \brief Module reset, active low
   CFP2_MOD_RST_N        : out   std_logic;
   --! \brief Programmable alarm 1
   CFP2_PRG_ALRM1        : in    std_logic;
   --! \brief Programmable alarm 2
   CFP2_PRG_ALRM2        : in    std_logic;
   --! \brief Programmable alarm 3
   CFP2_PRG_ALRM3        : in    std_logic;
   --! \brief Programmable control 1
   --! \details Set to high when unused
   CFP2_PRG_CNTL1        : out   std_logic;
   --! \brief Programmable control 2
   --! \details Set to high when unused
   CFP2_PRG_CNTL2        : out   std_logic;
   --! \brief Programmable control 3
   --! \details Set to high when unused
   CFP2_PRG_CNTL3        : out   std_logic;
   --! \brief Receiver loss of signal
   CFP2_RX_LOS           : in    std_logic;
   --! \brief Transmitter disable
   CFP2_TX_DIS           : out   std_logic;
   --! \brief MDIO clock
   CFP2_MDC              : out   std_logic;
   --! \brief MDIO data
   CFP2_MDIO             : inout std_logic;

   --! \name Clock and reset for MI32 interfaces
   -- -------------------------------------------------------------------------
   --! \brief Clock signal
   MI_CLK                : in  std_logic;
   --! \brief Reset signal
   MI_RESET              : in  std_logic;

   --! \name PCS/PMA layer MI32 interface
   -- -------------------------------------------------------------------------
   --! \brief Input Data
   ETH_MI_DWR            : in  std_logic_vector(31 downto 0);
   --! \brief Address
   ETH_MI_ADDR           : in  std_logic_vector(31 downto 0);
   --! \brief Read Request
   ETH_MI_RD             : in  std_logic;
   --! \brief Write Request
   ETH_MI_WR             : in  std_logic;
   --! \brief Byte Enable
   ETH_MI_BE             : in  std_logic_vector( 3 downto 0);
   --! \brief Output Data
   ETH_MI_DRD            : out std_logic_vector(31 downto 0);
   --! \brief Address Ready
   ETH_MI_ARDY           : out std_logic;
   --! \brief Data Ready
   ETH_MI_DRDY           : out std_logic;

   --! \name Clock and reset for IBUF's FLU and FL interfaces
   -- -------------------------------------------------------------------------
   --! \brief Output clock
   RX_CLK                : in  std_logic;
   --! \brief Output reset
   RX_RESET              : in  std_logic;

   --! \name IBUF's FrameLink Unaligned interface (data payload output)
   -- -------------------------------------------------------------------------
   --! \brief Payload data
   RX_DATA               : out std_logic_vector(511 downto 0);
   --! \brief Position of the start of the payload
   --! \details Valid only if RX_SOP is set to '1'.
   RX_SOP_POS            : out std_logic_vector(  2 downto 0);
   --! \brief Position of the end of the payload
   --! \details Valid only if RX_EOP is set to '1'.
   RX_EOP_POS            : out std_logic_vector(  5 downto 0);
   --! \brief Start of the payload
   RX_SOP                : out std_logic;
   --! \brief End of the payload
   RX_EOP                : out std_logic;
   --! \brief Source is ready
   RX_SRC_RDY            : out std_logic;
   --! \brief Destination is ready
   RX_DST_RDY            : in  std_logic;

   --! \name IBUF's FrameLink interface (data header output)
   -- -------------------------------------------------------------------------
   --! \brief Header data
   RX_HDATA              : out std_logic_vector(HDR_WIDTH-1 downto 0);
   --! \brief Start of the header, active in '0'
   RX_HSOP_N             : out std_logic;
   --! \brief End of the header, active in '0'
   RX_HEOP_N             : out std_logic;
   --! \brief Position of the end of the header
   --! \details Valid only if RX_HEOP_N is set to '0'.
   RX_HREM               : out std_logic_vector(log2(HDR_WIDTH/8)-1 downto 0);
   --! \brief Source is ready, active in '0'
   RX_HSRC_RDY_N         : out std_logic;
   --! \brief Destination is ready, active in '0'
   RX_HDST_RDY_N         : in  std_logic;

   --! \name OBUF's FrameLink Unaligned interface (data input)
   -- -------------------------------------------------------------------------
   --! \brief Input clock
   TX_CLK                : in  std_logic;
   --! \brief Input reset
   TX_RESET              : in  std_logic;
   --! \brief Packet data
   TX_DATA               : in  std_logic_vector(511 downto 0);
   --! \brief Position of the start of the packet
   --! \details Valid only if TX_SOP is set to '1'.
   TX_SOP_POS            : in  std_logic_vector(  2 downto 0);
   --! \brief Position of the end of the packet
   --! \details Valid only if TX_EOP is set to '1'.
   TX_EOP_POS            : in  std_logic_vector(  5 downto 0);
   --! \brief Start of the packet
   TX_SOP                : in  std_logic;
   --! \brief End of the packet
   TX_EOP                : in  std_logic;
   --! \brief Source is ready
   TX_SRC_RDY            : in  std_logic;
   --! \brief Destination is ready
   TX_DST_RDY            : out std_logic;

   --! \name Network module MI32 interface
   -- -------------------------------------------------------------------------
   --! \brief Input Data
   NET_MI_DWR            : in  std_logic_vector(31 downto 0);
   --! \brief Address
   NET_MI_ADDR           : in  std_logic_vector(31 downto 0);
   --! \brief Read Request
   NET_MI_RD             : in  std_logic;
   --! \brief Write Request
   NET_MI_WR             : in  std_logic;
   --! \brief Byte Enable
   NET_MI_BE             : in  std_logic_vector( 3 downto 0);
   --! \brief Output Data
   NET_MI_DRD            : out std_logic_vector(31 downto 0);
   --! \brief Address Ready
   NET_MI_ARDY           : out std_logic;
   --! \brief Data Ready
   NET_MI_DRDY           : out std_logic;
   
   --! \name Interface LEDs interface
   -- -------------------------------------------------------------------------
   --! \brief 8x1 green LED, active low
   LED_GREEN             : out   std_logic_vector(7 downto 0);
   --! \brief 8x1 red LED, active low
   LED_RED               : out   std_logic_vector(7 downto 0)
);
end entity eth100g;


-- ----------------------------------------------------------------------------
--                            Architecture Declaration
-- ----------------------------------------------------------------------------

architecture full of eth100g is

   --! Signals declaration
   -- -------------------------------------------------------------------------

   --! CGMII RX interface signals
   signal cgmii_rxclk        : std_logic;
   signal cgmii_rxreset      : std_logic;
   signal cgmii_rxd          : std_logic_vector(511 downto 0);
   signal cgmii_rxc          : std_logic_vector(63 downto 0);

   --! CGMII TX interface signals
   signal cgmii_txclk        : std_logic;
   signal cgmii_txreset      : std_logic;
   signal cgmii_txd          : std_logic_vector(511 downto 0);
   signal cgmii_txc          : std_logic_vector(63 downto 0);

   --! CGMII clock stable signal
   signal cgmii_clock_stable : std_logic;

   --! CGMII asynchronous reset
   signal cgmii_async_reset  : std_logic;

   --! Aggregated CFP2 RX and TX data ports
   signal cfp2_rx_p              : std_logic_vector(9 downto 0);
   signal cfp2_rx_n              : std_logic_vector(9 downto 0);
   signal cfp2_tx_p              : std_logic_vector(9 downto 0);
   signal cfp2_tx_n              : std_logic_vector(9 downto 0);

   --! CFP2 control interface signals
   signal cfp2_glb_alrm_n_sig    : std_logic;
   signal cfp2_mod_abs_sig       : std_logic;
   signal cfp2_prg_alrm1_sig     : std_logic; 
   signal cfp2_prg_alrm2_sig     : std_logic; 
   signal cfp2_prg_alrm3_sig     : std_logic; 
   signal cfp2_rx_los_sig        : std_logic;

   --! Signal detect from the transceiver/PMD
   signal signal_det             : std_logic_vector(3 downto 0);

   --! CFP2 MDIO interface signals
   signal cfp2_mdc_sig           : std_logic;
   signal cfp2_mdio_i            : std_logic;
   signal cfp2_mdio_o            : std_logic;
   signal cfp2_mdio_oe           : std_logic;

   --! Control data generator signals (statistics interface)
   signal pcd_stat_frame_len : std_logic_vector(15 downto 0);
   signal pcd_stat_dv        : std_logic;
   signal pcd_rdy            : std_logic;

   --! Control data generator signals (data interface)
   signal pcd_data           : std_logic_vector(HDR_WIDTH-1 downto 0); 
   signal pcd_sop_n          : std_logic;
   signal pcd_eop_n          : std_logic;
   signal pcd_rem            : std_logic_vector(log2(HDR_WIDTH/8)-1 downto 0);
   signal pcd_src_rdy_n      : std_logic;
   signal pcd_dst_rdy_n      : std_logic;

   --! Interface LED signals
   signal link_led               : std_logic;
   signal activity_led           : std_logic;

   --! Input signals to LED OBUFs
   signal led_green_sig      : std_logic_vector(7 downto 0);
   signal led_red_sig        : std_logic_vector(7 downto 0);

   --! Attributes declaration
   -- -------------------------------------------------------------------------
   attribute dont_touch : string;
   attribute dont_touch of cfp2_glb_alrm_n_ibuf_i : label is "true";
   attribute dont_touch of cfp2_mod_abs_ibuf_i    : label is "true";
   attribute dont_touch of cfp2_prg_alrm1_ibuf_i  : label is "true";
   attribute dont_touch of cfp2_prg_alrm2_ibuf_i  : label is "true";
   attribute dont_touch of cfp2_prg_alrm3_ibuf_i  : label is "true";


-- ----------------------------------------------------------------------------
--                             Architecture Body
-- ----------------------------------------------------------------------------
  
begin

 
   -- -------------------------------------------------------------------------
   --                            PCS/PMA layer
   -- -------------------------------------------------------------------------

   --! Aggregation from CFP2 RX data ports
   -- -------------------------------------------------------------------------

   cfp2_rx_p <= CFP2_RX3_RX6_P & CFP2_RX2_RX5_P &
                CFP2_RX1_RX2_P & CFP2_RX0_RX1_P &
                "000000";
   cfp2_rx_n <= CFP2_RX3_RX6_N & CFP2_RX2_RX5_N &
                CFP2_RX1_RX2_N & CFP2_RX0_RX1_N &
                "111111";

   --! eth_phy component instantiation
   -- -------------------------------------------------------------------------

   eth_phy_i : entity work.eth_phy
   port map (
      --! Clock and reset signals
      RESET         => RESET,
      DRPCLK        => DRPCLK,

      --! CGMII interface
      CGMII_RXCLK   => cgmii_rxclk,
      CGMII_RXD     => cgmii_rxd,
      CGMII_RXC     => cgmii_rxc,
      CGMII_TXCLK   => cgmii_txclk,
      CGMII_TXD     => cgmii_txd,
      CGMII_TXC     => cgmii_txc,
      CLK_STABLE    => cgmii_clock_stable,

      --! Transceiver reference clock interface
      CFP2_GTZ_REFCLK_P => CFP2_GTZ_REFCLK_P,
      CFP2_GTZ_REFCLK_N => CFP2_GTZ_REFCLK_N,
      CFP2_GTH_REFCLK_P => '0',
      CFP2_GTH_REFCLK_N => '1',

      --! Transceiver data interface
      CFP2_RX_P     => cfp2_rx_p,
      CFP2_RX_N     => cfp2_rx_n,
      CFP2_TX_P     => cfp2_tx_p,
      CFP2_TX_N     => cfp2_tx_n,

      --! Signal detect
      SIGNAL_DET    => signal_det,

      --! Transceiver management interface
      CFP2_MDC      => cfp2_mdc_sig,
      CFP2_MDIO_I   => cfp2_mdio_i,
      CFP2_MDIO_O   => cfp2_mdio_o,
      CFP2_MDIO_OE  => cfp2_mdio_oe,

      --! Input MI32 interface
      MI_CLK        => MI_CLK,
      MI_RESET      => MI_RESET,
      MI_DWR        => ETH_MI_DWR,
      MI_ADDR       => ETH_MI_ADDR,
      MI_RD         => ETH_MI_RD,
      MI_WR         => ETH_MI_WR,
      MI_BE         => ETH_MI_BE,
      MI_DRD        => ETH_MI_DRD,
      MI_ARDY       => ETH_MI_ARDY,
      MI_DRDY       => ETH_MI_DRDY
   );

   --! CGMII RX and TX resets derivation
   -- -------------------------------------------------------------------------

   --! Deriving asynchronous reset for CGMII-related logic
   cgmii_async_reset <= RESET OR NOT cgmii_clock_stable;

   --! Triple-register the reset to synchronize it to cgmii_rxclk
   cgmii_rxreset_sync_i : entity work.ASYNC_RESET
   generic map(
      TWO_REG    => false
   )
   port map(
      CLK        => cgmii_rxclk,
      ASYNC_RST  => cgmii_async_reset,
      OUT_RST(0) => cgmii_rxreset
   );

   --! Triple-register the reset to synchronize it to cgmii_txclk
   cgmii_txreset_sync_i : entity work.ASYNC_RESET
   generic map(
      TWO_REG    => false
   )
   port map(
      CLK        => cgmii_txclk,
      ASYNC_RST  => cgmii_async_reset,
      OUT_RST(0) => cgmii_txreset
   );

   --! De-aggreagtion to CFP2 TX data ports
   -- -------------------------------------------------------------------------

   --! positive TX channels
   CFP2_TX3_TX6_P <= cfp2_tx_p(9);
   CFP2_TX2_TX5_P <= cfp2_tx_p(8);
   CFP2_TX1_TX2_P <= cfp2_tx_p(7);
   CFP2_TX0_TX1_P <= cfp2_tx_p(6);

   --! negative TX channels
   CFP2_TX3_TX6_N <= cfp2_tx_n(9);
   CFP2_TX2_TX5_N <= cfp2_tx_n(8);
   CFP2_TX1_TX2_N <= cfp2_tx_n(7);
   CFP2_TX0_TX1_N <= cfp2_tx_n(6);

   --! CFP2 transceiver control interface connection through IBUFs/OBUFs
   --! (unused output ports are tied either to '0' or '1')
   -- -------------------------------------------------------------------------

   --! IBUF for GLB_ALRM_N input port
   cfp2_glb_alrm_n_ibuf_i : IBUF
   port map (
      O  => cfp2_glb_alrm_n_sig,
      I  => CFP2_GLB_ALRM_N
   );

   --! IBUF for MOD_ABS input port
   cfp2_mod_abs_ibuf_i : IBUF
   port map (
      O  => cfp2_mod_abs_sig,
      I  => CFP2_MOD_ABS
   );

   --! OBUF for MOD_LOPWR output port
   cfp2_mod_lopwr_obuf_i : OBUF
   port map (
      O  => CFP2_MOD_LOPWR,
      I  => '0'
   );

   --! OBUF for MOD_RST_N output port
   cfp2_mod_rst_n_obuf_i : OBUF
   port map (
      O  => CFP2_MOD_RST_N,
      I  => '1'
   );

   --! IBUF for PRG_ALRM1 input port
   cfp2_prg_alrm1_ibuf_i : IBUF
   port map (
      O  => cfp2_prg_alrm1_sig,
      I  => CFP2_PRG_ALRM1
   );

   --! IBUF for PRG_ALRM2 input port
   cfp2_prg_alrm2_ibuf_i : IBUF
   port map (
      O  => cfp2_prg_alrm2_sig,
      I  => CFP2_PRG_ALRM2
   );

   --! IBUF for PRG_ALRM3 input port
   cfp2_prg_alrm3_ibuf_i : IBUF
   port map (
      O  => cfp2_prg_alrm3_sig,
      I  => CFP2_PRG_ALRM3
   );

   --! OBUF for PRG_CNTL1 output port
   cfp2_prg_cntl1_obuf_i : OBUF
   port map (
      O  => CFP2_PRG_CNTL1,
      I  => '1'
   );

   --! OBUF for PRG_CNTL2 output port
   cfp2_prg_cntl2_obuf_i : OBUF
   port map (
      O  => CFP2_PRG_CNTL2,
      I  => '1'
   );

   --! OBUF for PRG_CNTL3 output port
   cfp2_prg_cntl3_obuf_i : OBUF
   port map (
      O  => CFP2_PRG_CNTL3,
      I  => '1'
   );

   --! IBUF for RX_LOS input port
   cfp2_rx_los_ibuf_i : IBUF
   port map (
      O  => cfp2_rx_los_sig,
      I  => CFP2_RX_LOS
   );

   --! Utilization of some input ports
   signal_det <= not (cfp2_rx_los_sig & cfp2_rx_los_sig &
                      cfp2_rx_los_sig & cfp2_rx_los_sig);

   --! OBUF for TX_DIS output port
   cfp2_tx_dis_obuf_i : OBUF
   port map (
      O  => CFP2_TX_DIS,
      I  => '0'
   );

   --! OBUF for MDC output port
   cfp2_mdc_obuf_i : OBUF
   port map (
      O  => CFP2_MDC,
      I  => cfp2_mdc_sig
   );

   --! IOBUF for MDIO inout port
   cfp2_mdio_iobuf_i : IOBUF
   port map (
      O  => cfp2_mdio_i,
      IO => CFP2_MDIO,
      I  => cfp2_mdio_o,
      T  => cfp2_mdio_oe
   );

 
   -- -------------------------------------------------------------------------
   --                            Network module
   -- -------------------------------------------------------------------------

   network_mod_i : entity work.network_mod
   generic map (
      IBUF_DFIFO_BYTES      => IBUF_DFIFO_BYTES,
      IBUF_HFIFO_BYTES      => IBUF_HFIFO_BYTES,
      HDR_WIDTH             => HDR_WIDTH,
      IBUF_MAC_COUNT        => IBUF_MAC_COUNT,
      INBANDFCS             => INBANDFCS,
      IBUF_CNT_ERROR_LENGTH => IBUF_CNT_ERROR_LENGTH,

      OBUF_DFIFO_BYTES      => OBUF_DFIFO_BYTES,
      OBUF_HFIFO_BYTES      => OBUF_HFIFO_BYTES,

      LED_CTRL_CNTR_SIZE    => LED_CTRL_CNTR_SIZE
   )
   port map (
      --! IBUF's CGMII interface
      CGMII_RXCLK           => cgmii_rxclk,
      CGMII_RXRESET         => cgmii_rxreset,
      CGMII_RXD             => cgmii_rxd,
      CGMII_RXC             => cgmii_rxc,

      --! OBUF's CGMII interface
      CGMII_TXCLK           => cgmii_txclk,
      CGMII_TXRESET         => cgmii_txreset,
      CGMII_TXD             => cgmii_txd,
      CGMII_TXC             => cgmii_txc,

      --! Clock and reset for IBUF's FLU and FL interfaces
      TX_CLK                => RX_CLK,
      TX_RESET              => RX_RESET,

      --! IBUF's FrameLink Unaligned interface (data payload output)
      TX_DATA               => RX_DATA,
      TX_SOP_POS            => RX_SOP_POS,
      TX_EOP_POS            => RX_EOP_POS,
      TX_SOP                => RX_SOP,
      TX_EOP                => RX_EOP,
      TX_SRC_RDY            => RX_SRC_RDY,
      TX_DST_RDY            => RX_DST_RDY,

      --! IBUF's FrameLink interface (data header output)
      TX_HDATA              => RX_HDATA,
      TX_HSOP_N             => RX_HSOP_N,
      TX_HEOP_N             => RX_HEOP_N,
      TX_HREM               => RX_HREM,
      TX_HSRC_RDY_N         => RX_HSRC_RDY_N,
      TX_HDST_RDY_N         => RX_HDST_RDY_N,

      --! OBUF's FrameLink Unaligned interface (data input)
      RX_CLK                => TX_CLK,
      RX_RESET              => TX_RESET,
      RX_DATA               => TX_DATA,
      RX_SOP_POS            => TX_SOP_POS,
      RX_EOP_POS            => TX_EOP_POS,
      RX_SOP                => TX_SOP,
      RX_EOP                => TX_EOP,
      RX_SRC_RDY            => TX_SRC_RDY,
      RX_DST_RDY            => TX_DST_RDY,

      --! Sampling unit interface (no Sampling unit connected)
      SAU_CLK               => open,
      SAU_RESET             => open,
      SAU_REQ               => open,
      SAU_ACCEPT            => '1',
      SAU_DV                => '1',

      --! Control data generator clock and reset
      CTRL_CLK              => open,
      CTRL_RESET            => open,

      --! Statistics interface of control data generator
      CTRL_SOP              => open,
      CTRL_STAT_MAC_ERR     => open,
      CTRL_STAT_MINTU_ERR   => open,
      CTRL_STAT_MTU_ERR     => open,
      CTRL_STAT_CRC_ERR     => open,
      CTRL_STAT_FRAME_ERR   => open,
      CTRL_STAT_FRAME_LEN   => pcd_stat_frame_len,
      CTRL_STAT_DV          => pcd_stat_dv,
      CTRL_RDY              => pcd_rdy,

      --! Data interface of control data generator
      CTRL_DATA             => pcd_data,
      CTRL_SOP_N            => pcd_sop_n,
      CTRL_EOP_N            => pcd_eop_n,
      CTRL_REM              => pcd_rem,
      CTRL_SRC_RDY_N        => pcd_src_rdy_n,
      CTRL_DST_RDY_N        => pcd_dst_rdy_n,

      --! MI32 interface
      MI_CLK                => MI_CLK,
      MI_RESET              => MI_RESET,
      MI_DWR                => NET_MI_DWR,
      MI_ADDR               => NET_MI_ADDR,
      MI_RD                 => NET_MI_RD,
      MI_WR                 => NET_MI_WR,
      MI_BE                 => NET_MI_BE,
      MI_DRD                => NET_MI_DRD,
      MI_ARDY               => NET_MI_ARDY,
      MI_DRDY               => NET_MI_DRDY,

      --! Repeater control interface
      REPEATER_CTRL         => "00",

      --! LED interface
      LINK_LED              => link_led,
      ACTIVITY_LED          => activity_led,

      --! Link presence interface
      LINK            	    => open
   );

   --! Control data generator imitation
   -- -------------------------------------------------------------------------

   pcd_data      <= X"0000000000000000"       & -- timestamp (64 b)
                    X"0000000"                & -- constant '0' (28 b)
                    "0000"                    & -- interface ID (4 b)
                    X"000C"                   & -- SZE header size (16 b)
                    pcd_stat_frame_len +        -- data size (16 b)
                    16 +                        -- + 16 B of SZE header
                    48;                         -- + 48 B of "filling" (added when 128 b header is inserted into 512 b data stream)
   pcd_sop_n     <= '0';
   pcd_eop_n     <= '0';
   pcd_rem       <= (others => '1');

   pcd_rdy       <= NOT pcd_dst_rdy_n;
   pcd_src_rdy_n <= NOT pcd_stat_dv;

   --! Driving green and red interface LEDs (through OBUFs)
   -- -------------------------------------------------------------------------

   --! Compose input for OBUFs to link status LEDs
   led_green_sig(3 downto 0) <= (others => link_led);
   led_red_sig(3 downto 0)   <= (others => '1'); -- red LEDs are switched off

   --! Compose input for OBUFs to link activity LEDs
   led_green_sig(7 downto 4) <= (others => activity_led);
   led_red_sig(7 downto 4)   <= (others => activity_led);

   --! Instantiation of OBUFs to green LEDs
   led_green_obuf_gen : for i in 0 to 7 generate
   begin
      led_green_obuf_i : OBUF
      port map (
         O  => LED_GREEN(i),
         I  => led_green_sig(i)
      );
   end generate led_green_obuf_gen;

   --! Instantiation of OBUFs to red LEDs
   led_red_obuf_gen : for i in 0 to 7 generate
   begin
      led_red_obuf_i : OBUF
      port map (
         O  => LED_RED(i),
         I  => led_red_sig(i)
      );
   end generate led_red_obuf_gen;

end architecture full;
