-- cgmii_enc_fsm.vhd: CGMII_ENC_FSM component.
-- Copyright (C) 2012 CESNET
-- Author(s): Vaclav Hummel <xhumme00@stud.fit.vutbr.cz>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--
--
--


library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_unsigned.all;
use IEEE.std_logic_arith.all;

-- ----------------------------------------------------------------------------
--                        Entity declaration
-- ----------------------------------------------------------------------------
--! CGMII encode control component
entity CGMII_ENC_FSM is
   port (
      -----------------------------------------------------
      --! \name Common interface
      -----------------------------------------------------
      --! Clock signal
      CLK         : in std_logic;
      --! Synchronous reset
      RESET       : in std_logic;

      -----------------------------------------------------
      --! \name Input Intefrace 
      -----------------------------------------------------
      --! SOP_V from component input (not delayed)
      SOP_V       : in std_logic_vector(7 downto 0);
      --! EOP_V from component input (not delayed)
      EOP_V       : in std_logic_vector(7 downto 0);
      --! Delayed SOP_V input
      PIPE_SOP_V     : in std_logic_vector(7 downto 0);
      --! Delayed EOP_V input
      PIPE_EOP_V     : in std_logic_vector(7 downto 0);
      --! Delayed EOP_POS input
      PIPE_EOP_POS   : in std_logic_vector(2 downto 0);

      -----------------------------------------------------
      --! \name Output interface
      -----------------------------------------------------
      --! Lane 0 selection signal
      LANE0_SEL      : out std_logic_vector(3 downto 0);
      --! Lane 1 selection signal 
      LANE1_SEL      : out std_logic_vector(3 downto 0);
      --! Lane 2 selection signal 
      LANE2_SEL      : out std_logic_vector(3 downto 0);
      --! Lane 3 selection signal
      LANE3_SEL      : out std_logic_vector(3 downto 0);
      --! Lane 4 selection signal
      LANE4_SEL      : out std_logic_vector(3 downto 0);
      --! Lane 5 selection signal 
      LANE5_SEL      : out std_logic_vector(3 downto 0);
      --! Lane 6 selection signal 
      LANE6_SEL      : out std_logic_vector(3 downto 0);
      --! Lane 7 selection signal
      LANE7_SEL      : out std_logic_vector(3 downto 0);
      
      --! Get next CRC value
      NEXT_CRC       : out std_logic;

      --! Output XLGMII control bus
      CGMII_CMD     : out std_logic_vector(63 downto 0)
   );
end entity CGMII_ENC_FSM;

-- ----------------------------------------------------------------------------
--                      Architecture declaration
-- ----------------------------------------------------------------------------
architecture full of CGMII_ENC_FSM is

   --Constants -------------------------------------------- 
   --! DATA selection
   constant C_SEL_DATA  : std_logic_vector(3 downto 0) := "0000";
   --! IDLE selection
   constant C_SEL_IDLE  : std_logic_vector(3 downto 0) := "1101";
   --! PREAMBLE selection
   constant C_SEL_PREAM : std_logic_vector(3 downto 0) := "1110";

   --FSM --------------------------------------------------
   --! FSM states type declaration
   type T_FSM_STATE  is (WAIT_FOR_DATA,DATA_TRANS,EOP_SOP,EOP,EOP_CRC);
   --! Actual FSM state
   signal actual_state  : T_FSM_STATE := WAIT_FOR_DATA;
   --! Next FSM state
   signal next_state    : T_FSM_STATE;
  
   --CRC selection signals --------------------------------
   --! CRC selection signal - lane X
   signal crc_lane_sel0          : std_logic_vector(3 downto 0);
   --! CRC selection signal - lane X+1
   signal crc_lane_sel1          : std_logic_vector(3 downto 0);
   --! CRC configuration delay register
   signal crc_lane_sel1_reg      : std_logic_vector(3 downto 0) := C_SEL_IDLE;
   --! XLGMII control signal - lane X
   signal crc_lane_control0      : std_logic_vector(7 downto 0);
   --! XLGMII control signal - lane X+1
   signal crc_lane_control1      : std_logic_vector(7 downto 0);
   --! XLGMII control dela register
   signal crc_lane_control1_reg  :std_logic_vector(7 downto 0) := (others => '1');

begin

   --! \brief Register for actual state value

   act_state_regp:process(CLK)
   begin
      if(CLK = '1' and CLK'event)then
         if(RESET = '1')then
            actual_state <= WAIT_FOR_DATA;
         else
            actual_state <= next_state;
         end if;
      end if;
   end process;

   --! \brief FSM transition process
   fsm_transp:process(actual_state,SOP_V,PIPE_SOP_V,EOP_V,PIPE_EOP_V,PIPE_EOP_POS)
   begin
      --Loop by default
      next_state <= actual_state;

      case actual_state is
         when WAIT_FOR_DATA  =>
            if (PIPE_SOP_V /= "00000000" and SOP_V /= "00000000" and EOP_V /= "00000000") then
               --EOP, SOP detected in the actual flu word
               next_state <= EOP_SOP;
            elsif ((SOP_V = "00000001" and EOP_V = "10000000") or (PIPE_SOP_V /= "00000000" and EOP_V /= "00000000")) then
               --EOP detected in the actual flu word
               next_state <= EOP;
            elsif(SOP_V = "00000001" or PIPE_SOP_V /= "00000000")then
               --SOP_V = "00000001" or pipelined version /= "00000000 detected"
               next_state <= DATA_TRANS;
            end if;

         when DATA_TRANS =>
            --Data transfer is running; now we have to react on actual EOP_V and SOP_V
            --Minimal packet transfer takes 1-2 clocks(Min. frame size without CRC = 60B).
            --46B payload + (6+6) MAC + 2 frame type = 64.
            --Handle with possible situations.

            if((SOP_V /= "00000000" and EOP_V /= "00000000") and EOP_V < SOP_V)then
               --EOP < SOP is detected in actual word.
               next_state <= EOP_SOP;
            elsif(EOP_V /= "00000000")then
               --EOP will be stored in pipeline register in next stage.
               next_state <= EOP;
            end if;

         when EOP =>
            --In this word was detected EOP block. We need to check if the CRC will be splited
            --(CRC+Term. symbol = 5B)
            if((SOP_V = "00000000" and EOP_V = "00000000" and PIPE_EOP_V <= "01000000") or
               (SOP_V = "00000000" and EOP_V = "00000000" and PIPE_EOP_V = "10000000" and PIPE_EOP_POS <= "010"))then
            --CRC with terminate symbol fits into actual word; no other SOP is detected in
            --actual SOP_V.
               next_state <= WAIT_FOR_DATA;
            elsif(PIPE_EOP_V = "10000000" and PIPE_EOP_POS > "010")then
            --CRC with terminate symbol is splitted into two words.
               next_state <= EOP_CRC;
            elsif(SOP_V = "00000001" and EOP_V = "00000000")then
            --In this situation we need to insert preamble on lane 7. Than we countinue with
            --data transfer.CRC and terminate symbol must fit into actual word.
            --CRC is not splitted on two words.
               next_state <= DATA_TRANS;
            elsif(SOP_V = "00000001" and EOP_V = "10000000")then
               --Need to deal with eop in the actual word. Whole packet in the actual word.
               next_state <= EOP;
            elsif(SOP_V /= "00000000")then
            --SOP was detected somewhere on positions 1,2,3,4,5,6,7 and CRC with terminate symbol fits into
            --actual word.CRC is not splitted on two words.
               next_state <= WAIT_FOR_DATA;
            elsif(EOP_V /= "00000000")then
               --Need to dela with eop in the actual word.
               next_state <= EOP;
            end if;

         when EOP_CRC =>
         --In this state we have to finis CRC transfer and start new one if possible.
            if(PIPE_SOP_V = "00000000" and SOP_V = "00000000")then
               --No start here and actual vect. We have to wait on data;
               next_state <= WAIT_FOR_DATA;
            elsif(PIPE_SOP_V = "00000000" and SOP_V = "00000001" and EOP_V = "00000000")then
               --We need to insert preamble. Packet start is on LANE0 of the next word.
               next_state <= DATA_TRANS;
            elsif(PIPE_SOP_V /= "00000000" and EOP_V = "00000000")then
               --We need to insert preamble and data in actual word. Than we could continue with data transfer.
               next_state <= DATA_TRANS;
            elsif(SOP_V /= "00000000" and EOP_V /= "00000000" and EOP_V < SOP_V)then
               --In actual word we need to end one packet and start another
               next_state <= EOP_SOP;
            elsif(EOP_V /= "00000000")then
               --In actual word we need to end packet
               next_state <= EOP;
            elsif(PIPE_SOP_V = "00000000" and SOP_V /= "00000000")then
               --We detected that SOP arrive in next word. We could use WAIT_FOR_DATA state.
               next_state <= WAIT_FOR_DATA;
            end if;
   
         when EOP_SOP =>
         --In this state we have to finish one packet and start another. In the
         --next word 3 situations appear
            --No SOP or EOP detected, whole word is data
            next_state <= DATA_TRANS;

            if(SOP_V /= "00000000" and EOP_V /= "00000000" and EOP_V < SOP_V)then
               --In actual state one packet ends and starts another
               next_state <= EOP_SOP;
            elsif(EOP_V /= "00000000")then
               --In actual state packet ends
               next_state <= EOP;
            end if;

         when others=> null;
      end case;
   end process;

   --! \brief FSM output process
   fsm_outp:process(actual_state,SOP_V,PIPE_SOP_V,EOP_V,PIPE_EOP_V,PIPE_EOP_POS,
                    crc_lane_sel0,crc_lane_sel1,crc_lane_sel1_reg,crc_lane_control0,crc_lane_control1,
                    crc_lane_control1_reg)
   begin
      --Default values ----------------
      --Insert IDLES
      LANE0_SEL <= C_SEL_IDLE;
      LANE1_SEL <= C_SEL_IDLE;
      LANE2_SEL <= C_SEL_IDLE;
      LANE3_SEL <= C_SEL_IDLE;
      LANE4_SEL <= C_SEL_IDLE;
      LANE5_SEL <= C_SEL_IDLE;
      LANE6_SEL <= C_SEL_IDLE;
      LANE7_SEL <= C_SEL_IDLE;
      --CGMII control
      CGMII_CMD <= (others=>'1');
      --CRC32 control
      NEXT_CRC  <= '0';
      ---------------------------------
   
      case actual_state is
         when WAIT_FOR_DATA => 

         --We need to insert PREAMBLE (when SOP_V = "00000001") or 
         --insert preamble and data when PIPE_SOP_V /= "00000000"
            if(SOP_V = "00000001")then
               --Insert preamble on lane 3
               LANE7_SEL <= C_SEL_PREAM;
               --CGMII control - insert zeros (data part)
               CGMII_CMD(63 downto 57) <= (others=>'0');
            elsif(PIPE_SOP_V = "00000010")then
               --Insert preamble before SOP_V position
               LANE0_SEL <= C_SEL_PREAM;
               LANE1_SEL <= C_SEL_DATA; --Insert DATA
               LANE2_SEL <= C_SEL_DATA;
               LANE3_SEL <= C_SEL_DATA;
               LANE4_SEL <= C_SEL_DATA;
               LANE5_SEL <= C_SEL_DATA;
               LANE6_SEL <= C_SEL_DATA;
               LANE7_SEL <= C_SEL_DATA;
               --CGMII control - insert zeros (data part)
               CGMII_CMD(63 downto 1) <= (others=>'0');
            elsif(PIPE_SOP_V = "00000100")then
               --Insert preamble befor SOP_V position
               LANE1_SEL <= C_SEL_PREAM;
               LANE2_SEL <= C_SEL_DATA; --Insert DATA
               LANE3_SEL <= C_SEL_DATA;
               LANE4_SEL <= C_SEL_DATA;
               LANE5_SEL <= C_SEL_DATA;
               LANE6_SEL <= C_SEL_DATA;
               LANE7_SEL <= C_SEL_DATA;
               --CGMII control - insert zeros (data part)
               CGMII_CMD(63 downto 9) <= (others=>'0');
            elsif(PIPE_SOP_V = "00001000")then
               --Insert preamble before SOP_V
               LANE2_SEL <= C_SEL_PREAM;
               LANE3_SEL <= C_SEL_DATA;
               LANE4_SEL <= C_SEL_DATA;
               LANE5_SEL <= C_SEL_DATA;
               LANE6_SEL <= C_SEL_DATA;
               LANE7_SEL <= C_SEL_DATA;
               --CGMII control - insert zeros (data part)
               CGMII_CMD(63 downto 17) <= (others=>'0');
            elsif(PIPE_SOP_V = "00010000")then
               --Insert preamble before SOP_V
               LANE3_SEL <= C_SEL_PREAM;
               LANE4_SEL <= C_SEL_DATA;
               LANE5_SEL <= C_SEL_DATA;
               LANE6_SEL <= C_SEL_DATA;
               LANE7_SEL <= C_SEL_DATA;
               --CGMII control - insert zeros (data part)
               CGMII_CMD(63 downto 25) <= (others=>'0');
            elsif(PIPE_SOP_V = "00100000")then
               --Insert preamble before SOP_V
               LANE4_SEL <= C_SEL_PREAM;
               LANE5_SEL <= C_SEL_DATA;
               LANE6_SEL <= C_SEL_DATA;
               LANE7_SEL <= C_SEL_DATA;
               --CGMII control - insert zeros (data part)
               CGMII_CMD(63 downto 33) <= (others=>'0');
            elsif(PIPE_SOP_V = "01000000")then
               --Insert preamble before SOP_V
               LANE5_SEL <= C_SEL_PREAM;
               LANE6_SEL <= C_SEL_DATA;
               LANE7_SEL <= C_SEL_DATA;
               --CGMII control - insert zeros (data part)
               CGMII_CMD(63 downto 41) <= (others=>'0');
            elsif(PIPE_SOP_V = "10000000")then
               --Insert preamble before SOP_V
               LANE6_SEL <= C_SEL_PREAM;
               LANE7_SEL <= C_SEL_DATA;
               --CGMII control - insert zeros (data part)
               CGMII_CMD(63 downto 49) <= (others=>'0');
            end if;

         when DATA_TRANS => 
            --Data transfer is running; pass data
            LANE0_SEL <= C_SEL_DATA;
            LANE1_SEL <= C_SEL_DATA;
            LANE2_SEL <= C_SEL_DATA;
            LANE3_SEL <= C_SEL_DATA;
            LANE4_SEL <= C_SEL_DATA;
            LANE5_SEL <= C_SEL_DATA;
            LANE6_SEL <= C_SEL_DATA;
            LANE7_SEL <= C_SEL_DATA;
            --CGMII control -> data 
            CGMII_CMD <= (others=>'0');

         when EOP =>
            --In this state is asserted EOP, we need to insert CRC and terminate symbol. We need
            --to insert IFG and PREAMBLE in some cases. SOP can be here. 
            --With respect to EOP_POS insert CRC and terminate symbol.
            --No error during transmit is not expected.
            if(PIPE_EOP_V = "00000001")then
               LANE0_SEL <= crc_lane_sel0;
               LANE1_SEL <= crc_lane_sel1;
               --CGMII control
               CGMII_CMD(7 downto 0)  <= crc_lane_control0;
               CGMII_CMD(15 downto 8) <= crc_lane_control1;
               --Prepare new CRC32 value
               NEXT_CRC <= '1';
            elsif(PIPE_EOP_V = "00000010")then
               LANE0_SEL <= C_SEL_DATA; --Select DATA
               LANE1_SEL <= crc_lane_sel0;
               LANE2_SEL <= crc_lane_sel1;
               --CGMII control
               CGMII_CMD(7 downto 0)     <= "00000000";
               CGMII_CMD(15 downto 8)    <= crc_lane_control0;
               CGMII_CMD(23 downto 16)   <= crc_lane_control1;
               --Prepare new CRC32 value
               NEXT_CRC <= '1';
            elsif(PIPE_EOP_V = "00000100")then
               LANE0_SEL   <= C_SEL_DATA;
               LANE1_SEL   <= C_SEL_DATA; --Select DATA
               LANE2_SEL   <= crc_lane_sel0;
               LANE3_SEL   <= crc_lane_sel1;
               --CGMII control
               CGMII_CMD(15 downto 0)    <= (others=>'0');
               CGMII_CMD(23 downto 16)   <= crc_lane_control0;
               CGMII_CMD(31 downto 24)   <= crc_lane_control1;
               --Prepare new CRC32 value
               NEXT_CRC <= '1';
            elsif(PIPE_EOP_V = "00001000")then
               LANE0_SEL   <= C_SEL_DATA;
               LANE1_SEL   <= C_SEL_DATA; --Select DATA
               LANE2_SEL   <= C_SEL_DATA;
               LANE3_SEL   <= crc_lane_sel0;
               LANE4_SEL   <= crc_lane_sel1;
               --CGMII control
               CGMII_CMD(23 downto 0)    <= (others=>'0');
               CGMII_CMD(31 downto 24)   <= crc_lane_control0;
               CGMII_CMD(39 downto 32)   <= crc_lane_control1;
               --Prepare new CRC32 value
               NEXT_CRC <= '1';
            elsif(PIPE_EOP_V = "00010000")then
               LANE0_SEL   <= C_SEL_DATA;
               LANE1_SEL   <= C_SEL_DATA; --Select DATA
               LANE2_SEL   <= C_SEL_DATA;
               LANE3_SEL   <= C_SEL_DATA;
               LANE4_SEL   <= crc_lane_sel0;
               LANE5_SEL   <= crc_lane_sel1;
               --CGMII control
               CGMII_CMD(31 downto 0)    <= (others=>'0');
               CGMII_CMD(39 downto 32)   <= crc_lane_control0;
               CGMII_CMD(47 downto 40)   <= crc_lane_control1;
               --Prepare new CRC32 value
               NEXT_CRC <= '1';
            elsif(PIPE_EOP_V = "00100000")then
               LANE0_SEL   <= C_SEL_DATA;
               LANE1_SEL   <= C_SEL_DATA; --Select DATA
               LANE2_SEL   <= C_SEL_DATA;
               LANE3_SEL   <= C_SEL_DATA;
               LANE4_SEL   <= C_SEL_DATA;
               LANE5_SEL   <= crc_lane_sel0;
               LANE6_SEL   <= crc_lane_sel1;
               --CGMII control
               CGMII_CMD(39 downto 0)    <= (others=>'0');
               CGMII_CMD(47 downto 40)   <= crc_lane_control0;
               CGMII_CMD(55 downto 48)   <= crc_lane_control1;
               --Prepare new CRC32 value
               NEXT_CRC <= '1';
            elsif(PIPE_EOP_V = "01000000")then
               LANE0_SEL   <= C_SEL_DATA;
               LANE1_SEL   <= C_SEL_DATA;
               LANE2_SEL   <= C_SEL_DATA;
               LANE3_SEL   <= C_SEL_DATA;
               LANE4_SEL   <= C_SEL_DATA;
               LANE5_SEL   <= C_SEL_DATA; --Select DATA
               LANE6_SEL   <= crc_lane_sel0;
               LANE7_SEL   <= crc_lane_sel1;
               --CGMII control
               CGMII_CMD(47 downto 0)    <= (others=>'0');
               CGMII_CMD(55 downto 48)   <= crc_lane_control0;
               CGMII_CMD(63 downto 56)   <= crc_lane_control1;
               --Prepare new CRC32 value
               NEXT_CRC <= '1';
            elsif(PIPE_EOP_V = "10000000")then
               --CRC will be splitted on two words.
               LANE0_SEL   <= C_SEL_DATA;
               LANE1_SEL   <= C_SEL_DATA;
               LANE2_SEL   <= C_SEL_DATA; --Select DATA
               LANE3_SEL   <= C_SEL_DATA;
               LANE4_SEL   <= C_SEL_DATA;
               LANE5_SEL   <= C_SEL_DATA;
               LANE6_SEL   <= C_SEL_DATA;
               LANE7_SEL   <= crc_lane_sel0;
               --CGMII control
               CGMII_CMD(55 downto 0) <= (others=>'0');
               CGMII_CMD(63 downto 56) <= crc_lane_control0;
               --Prepare new CRC value
               if(PIPE_EOP_POS <= "010")then
                  --CRC and terminate symbol fits into actual word.
                  NEXT_CRC <= '1';
               end if;
            end if;

            --Check IF SOP is detected, if so insert preamble
            --In this word packet ends and SOP can be detected as soon as
            --PIPE_SOP_V = "00001000 not earlier, have to check SOP_V too"
            if(PIPE_SOP_V = "00001000")then
               LANE2_SEL <= C_SEL_PREAM;
               CGMII_CMD(23 downto 16) <= "00000001";
               LANE3_SEL <= C_SEL_DATA;
               LANE4_SEL <= C_SEL_DATA;
               LANE5_SEL <= C_SEL_DATA;
               LANE6_SEL <= C_SEL_DATA;
               LANE7_SEL <= C_SEL_DATA;
               CGMII_CMD(63 downto 24) <= (others => '0');
            elsif(PIPE_SOP_V = "00010000")then
               LANE3_SEL <= C_SEL_PREAM;
               CGMII_CMD(31 downto 24) <= "00000001";
               LANE4_SEL <= C_SEL_DATA;
               LANE5_SEL <= C_SEL_DATA;
               LANE6_SEL <= C_SEL_DATA;
               LANE7_SEL <= C_SEL_DATA;
               CGMII_CMD(63 downto 32) <= (others => '0');
            elsif(PIPE_SOP_V = "00100000")then
               LANE4_SEL <= C_SEL_PREAM;
               CGMII_CMD(39 downto 32) <= "00000001";
               LANE5_SEL <= C_SEL_DATA;
               LANE6_SEL <= C_SEL_DATA;
               LANE7_SEL <= C_SEL_DATA;
               CGMII_CMD(63 downto 40) <= (others => '0');
            elsif(PIPE_SOP_V = "01000000")then
               LANE5_SEL <= C_SEL_PREAM;
               CGMII_CMD(47 downto 40) <= "00000001";
               LANE6_SEL <= C_SEL_DATA;
               LANE7_SEL <= C_SEL_DATA;
               CGMII_CMD(63 downto 48) <= (others => '0');
            elsif(PIPE_SOP_V = "10000000")then
               LANE6_SEL <= C_SEL_PREAM;
               CGMII_CMD(55 downto 48) <= "00000001";
               LANE7_SEL <= C_SEL_DATA;
               CGMII_CMD(63 downto 56) <= (others => '0');
            elsif(SOP_V = "00000001")then
               LANE7_SEL      <= C_SEL_PREAM;
               CGMII_CMD(63 downto 56) <= "00000001";
            end if;

         when EOP_CRC =>
            --CRC have to be finished here. SOP could start here. Finish CRC
            LANE0_SEL <= crc_lane_sel1_reg;
            CGMII_CMD(7 downto 0) <= crc_lane_control1_reg;
            --Prepare new CRC value
            NEXT_CRC <= '1';

            --Check where the SOP_V start is.
            if(PIPE_SOP_V = "00000100")then
               --Insert preamble and data.
               --Inser PREAMBLE
               LANE1_SEL <= C_SEL_PREAM;
               CGMII_CMD(15 downto 9) <= (others=>'0');
               --Inser DATA
               LANE2_SEL <= C_SEL_DATA;
               LANE3_SEL <= C_SEL_DATA;
               LANE4_SEL <= C_SEL_DATA;
               LANE5_SEL <= C_SEL_DATA;
               LANE6_SEL <= C_SEL_DATA;
               LANE7_SEL <= C_SEL_DATA;
               CGMII_CMD(63 downto 16) <= (others=>'0');
            elsif(PIPE_SOP_V = "00001000")then
               --Insert PREAMBLE
               LANE2_SEL <= C_SEL_PREAM;
               CGMII_CMD(23 downto 17) <= (others=>'0');
               --Insert DATA
               LANE3_SEL <= C_SEL_DATA;
               LANE4_SEL <= C_SEL_DATA;
               LANE5_SEL <= C_SEL_DATA;
               LANE6_SEL <= C_SEL_DATA;
               LANE7_SEL <= C_SEL_DATA;
               CGMII_CMD(63 downto 24) <= (others=>'0');
            elsif(PIPE_SOP_V = "00010000")then
               --Insert PREAMBLE
               LANE3_SEL <= C_SEL_PREAM;
               CGMII_CMD(31 downto 25) <= (others=>'0');
               --Insert DATA
               LANE4_SEL <= C_SEL_DATA;
               LANE5_SEL <= C_SEL_DATA;
               LANE6_SEL <= C_SEL_DATA;
               LANE7_SEL <= C_SEL_DATA;
               CGMII_CMD(63 downto 32) <= (others=>'0');
            elsif(PIPE_SOP_V = "00100000")then
               --Insert PREAMBLE
               LANE4_SEL <= C_SEL_PREAM;
               CGMII_CMD(39 downto 33) <= (others=>'0');
               --Insert DATA
               LANE5_SEL <= C_SEL_DATA;
               LANE6_SEL <= C_SEL_DATA;
               LANE7_SEL <= C_SEL_DATA;
               CGMII_CMD(63 downto 40) <= (others=>'0');
            elsif(PIPE_SOP_V = "01000000")then
               --Insert PREAMBLE
               LANE5_SEL <= C_SEL_PREAM;
               CGMII_CMD(47 downto 41) <= (others=>'0');
               --Insert DATA
               LANE6_SEL <= C_SEL_DATA;
               LANE7_SEL <= C_SEL_DATA;
               CGMII_CMD(63 downto 48) <= (others=>'0');
            elsif(PIPE_SOP_V = "10000000")then
               --Insert PREAMBLE
               LANE6_SEL <= C_SEL_PREAM;
               CGMII_CMD(55 downto 49) <= (others=>'0');
               --Insert DATA
               LANE7_SEL <= C_SEL_DATA;
               CGMII_CMD(63 downto 56) <= (others=>'0');
            elsif(PIPE_SOP_V = "00000000" and SOP_V = "00000001")then
               --We have to insert preamble on lane 7.
               LANE7_SEL <= C_SEL_PREAM;
               CGMII_CMD(63 downto 57) <= (others=>'0');
            end if;

         when EOP_SOP =>
            --Prepare new CRC value
            NEXT_CRC <= '1';
            
            --Insert IFG after EOP
            if(PIPE_EOP_V = "00000001")then
               LANE0_SEL <= crc_lane_sel0;
               LANE1_SEL <= crc_lane_sel1;
               --CGMII control
               CGMII_CMD(7 downto 0)  <= crc_lane_control0;
               CGMII_CMD(15 downto 8) <= crc_lane_control1;
            elsif(PIPE_EOP_V = "00000010")then
               LANE0_SEL <= C_SEL_DATA; --Select DATA
               LANE1_SEL <= crc_lane_sel0;
               LANE2_SEL <= crc_lane_sel1;
               --CGMII control
               CGMII_CMD(7 downto 0)     <= (others=>'0');
               CGMII_CMD(15 downto 8)    <= crc_lane_control0;
               CGMII_CMD(23 downto 16)   <= crc_lane_control1;
            elsif(PIPE_EOP_V = "00000100")then
               LANE0_SEL <= C_SEL_DATA; --Select DATA
               LANE1_SEL <= C_SEL_DATA; --Select DATA
               LANE2_SEL <= crc_lane_sel0;
               LANE3_SEL <= crc_lane_sel1;
               --CGMII control
               CGMII_CMD(15 downto 0)    <= (others=>'0');
               CGMII_CMD(23 downto 16)   <= crc_lane_control0;
               CGMII_CMD(31 downto 24)   <= crc_lane_control1;
            elsif(PIPE_EOP_V = "00001000")then
               LANE0_SEL <= C_SEL_DATA; --Select DATA
               LANE1_SEL <= C_SEL_DATA; --Select DATA
               LANE2_SEL <= C_SEL_DATA; --Select DATA
               LANE3_SEL <= crc_lane_sel0;
               LANE4_SEL <= crc_lane_sel1;
               --CGMII control
               CGMII_CMD(23 downto 0)    <= (others=>'0');
               CGMII_CMD(31 downto 24)   <= crc_lane_control0;
               CGMII_CMD(39 downto 32)   <= crc_lane_control1;
            elsif(PIPE_EOP_V = "00010000")then
               LANE0_SEL <= C_SEL_DATA; --Select DATA
               LANE1_SEL <= C_SEL_DATA; --Select DATA
               LANE2_SEL <= C_SEL_DATA; --Select DATA
               LANE3_SEL <= C_SEL_DATA; --Select DATA
               LANE4_SEL <= crc_lane_sel0;
               LANE5_SEL <= crc_lane_sel1;
               --CGMII control
               CGMII_CMD(31 downto 0)    <= (others=>'0');
               CGMII_CMD(39 downto 32)   <= crc_lane_control0;
               CGMII_CMD(47 downto 40)   <= crc_lane_control1;
            end if;

            --Insert PREAMBLE before SOP
            if(PIPE_SOP_V = "00001000")then
               --Insert preamble before SOP_V
               LANE2_SEL <= C_SEL_PREAM;
               LANE3_SEL <= C_SEL_DATA;
               LANE4_SEL <= C_SEL_DATA;
               LANE5_SEL <= C_SEL_DATA;
               LANE6_SEL <= C_SEL_DATA;
               LANE7_SEL <= C_SEL_DATA;
               --CGMII control - insert zeros (data part)
               CGMII_CMD(63 downto 17) <= (others=>'0');
            elsif(PIPE_SOP_V = "00010000")then
               --Insert preamble before SOP_V
               LANE3_SEL <= C_SEL_PREAM;
               LANE4_SEL <= C_SEL_DATA;
               LANE5_SEL <= C_SEL_DATA;
               LANE6_SEL <= C_SEL_DATA;
               LANE7_SEL <= C_SEL_DATA;
               --CGMII control - insert zeros (data part)
               CGMII_CMD(63 downto 25) <= (others=>'0');
            elsif(PIPE_SOP_V = "00100000")then
               --Insert preamble before SOP_V
               LANE4_SEL <= C_SEL_PREAM;
               LANE5_SEL <= C_SEL_DATA;
               LANE6_SEL <= C_SEL_DATA;
               LANE7_SEL <= C_SEL_DATA;
               --CGMII control - insert zeros (data part)
               CGMII_CMD(63 downto 33) <= (others=>'0');
            elsif(PIPE_SOP_V = "01000000")then
               --Insert preamble before SOP_V
               LANE5_SEL <= C_SEL_PREAM;
               LANE6_SEL <= C_SEL_DATA;
               LANE7_SEL <= C_SEL_DATA;
               --CGMII control - insert zeros (data part)
               CGMII_CMD(63 downto 41) <= (others=>'0');
            elsif(PIPE_SOP_V = "10000000")then
               --Insert preamble before SOP_V
               LANE6_SEL <= C_SEL_PREAM;
               LANE7_SEL <= C_SEL_DATA;
               --CGMII control - insert zeros (data part)
               CGMII_CMD(63 downto 49) <= (others=>'0');
            end if;

         when others => null;
      end case;
   end process;

   --------------------------------------------------------
   --CRC selection signals
   --------------------------------------------------------
   --! \brief This process generates MUX selection signals for CRC and Terminate symbol insertion.
   crc_mux_selp:process(PIPE_EOP_POS)
   begin
      --Default values (idles)
      crc_lane_sel0 <= C_SEL_IDLE;
      crc_lane_sel1 <= C_SEL_IDLE;
      crc_lane_control0 <= "11111111";
      crc_lane_control1 <= "11111111";

      case PIPE_EOP_POS is
         --CRC with terminate symbol fit into actual 64 bit block.
         when "000" => 
            crc_lane_sel0 <= "0111";
            crc_lane_control0(4 downto 0) <= "00000";

         when "001" =>
            crc_lane_sel0 <= "0110";
            crc_lane_control0(5 downto 0) <= "000000";

         when "010" =>
            crc_lane_sel0 <= "0101";
            crc_lane_control0(6 downto 0) <= "0000000";

         --CRC and terminate symbol are splitted on two parts.
         when "011" => 
            crc_lane_sel0 <= "0100";
            crc_lane_sel1 <= "1100";
            crc_lane_control0 <= "00000000";

         when "100" =>
            crc_lane_sel0 <= "0011";
            crc_lane_sel1 <= "1011";
            crc_lane_control0 <= "00000000";
            crc_lane_control1(0) <= '0';

         when "101" =>
            crc_lane_sel0 <= "0010";
            crc_lane_sel1 <= "1010";
            crc_lane_control0 <= "00000000";
            crc_lane_control1(1 downto 0) <= "00";

         when "110" =>
            crc_lane_sel0 <= "0001";
            crc_lane_sel1 <= "1001";
            crc_lane_control0 <= "00000000";
            crc_lane_control1(2 downto 0) <= "000";

         when "111" =>
            crc_lane_sel0 <= "0000";
            crc_lane_sel1 <= "1000";
            crc_lane_control0 <= "00000000";
            crc_lane_control1(3 downto 0) <= "0000";

         when others => null;
      end case;
   end process;

   --! \brief Delay register for CRC configuration.

   crc_sel_delay_regp:process(CLK)
   begin
      if(CLK = '1' and CLK'event)then
         if(RESET = '1')then
            crc_lane_sel1_reg       <= "0000";
            crc_lane_control1_reg   <= "00000000";
         else
            crc_lane_sel1_reg       <= crc_lane_sel1;
            crc_lane_control1_reg   <= crc_lane_control1;
         end if;
      end if;
   end process;

end architecture full;
