-- mpls.vhd: HFE-M MPLS header analyzer
-- Copyright (C) 2011 CESNET
-- Author(s): Viktor Puš <pus@cesnet.cz>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--
--
--

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_unsigned.all;
use IEEE.std_logic_arith.all;
use IEEE.numeric_std.all;
use work.math_pack.all;

-- ----------------------------------------------------------------------------
--                      Entity declaration
-- ----------------------------------------------------------------------------

entity hfe_m_MPLS is
   generic (
      --* FrameLink width
      DATA_WIDTH  : integer := 64;
      --* Width of WORD_COUNTER
      COUNTER_WIDTH:integer := 4
   );
   port (
      CLK         : in std_logic;
      RESET       : in std_logic;

      --+ FrameLink interface (passive port)
      FL_DATA     : in std_logic_vector(DATA_WIDTH-1 downto 0);
      FL_DREM     : in std_logic_vector(log2(DATA_WIDTH/8)-1 downto 0);
      FL_SRC_RDY_N: in std_logic;
      FL_DST_RDY_N: in std_logic;
      FL_SOF_N    : in std_logic;
      FL_EOF_N    : in std_logic;
      FL_SOP_N    : in std_logic;
      FL_EOP_N    : in std_logic;

      --* Counter of FL words (incremented by 1 with every word)
      WORD_COUNTER   : in std_logic_vector(COUNTER_WIDTH-1 downto 0);

      --* Offset of the MPLS header start from SOF (in Bytes)
      OFFSET_IN      : in std_logic_vector(COUNTER_WIDTH+log2(DATA_WIDTH/8)-1
         downto 0);

      --* Previous layer detected the MPLS header
      ACTIVE_IN      : in std_logic;

      --* Offset of the next header start from SOF (in Bytes)
      OFFSET_OUT     : out std_logic_vector(COUNTER_WIDTH+log2(DATA_WIDTH/8)-1
         downto 0);

      --+ Active for each known next layer protocol
      ACTIVE_IPV4    : out std_logic;
      ACTIVE_IPV6    : out std_logic;
      ACTIVE_MPLS    : out std_logic;

      --* Active for unknown next layer protocol
      ACTIVE_UNKNOWN : out std_logic;

      --* MPLS header has errors
      MALFORMED      : out std_logic;

      --* Still parsing, outputs are not valid yet
      RUNNING        : out std_logic
   );
end entity;

architecture behavioral of hfe_m_mpls is

   signal bottom        : std_logic;
   signal mpls_stuff    : std_logic_vector(6 downto 0);
   signal bottom_vld    : std_logic;

   signal ipver         : std_logic_vector(3 downto 0);
   signal iplen         : std_logic_vector(3 downto 0);
   signal ipver_vld     : std_logic;

   signal sig_active_ipv4:std_logic;
   signal sig_active_ipv6:std_logic;
   signal sig_active_mpls:std_logic;

begin

   --* Get the Bottom of Stack bit
   get_bos_i : entity work.hfe_m_getbyte
   generic map(
      DATA_WIDTH     => DATA_WIDTH,
      COUNTER_WIDTH  => COUNTER_WIDTH,
      BYTE_OFFSET    => 2 -- Bottom of Stack bit is in byte 2
   )
   port map(
      CLK         => CLK,
      RESET       => RESET,

      FL_DATA     => FL_DATA,
      FL_DREM     => FL_DREM,
      FL_SRC_RDY_N=> FL_SRC_RDY_N,
      FL_DST_RDY_N=> FL_DST_RDY_N,
      FL_SOF_N    => FL_SOF_N,
      FL_EOF_N    => FL_EOF_N,
      FL_SOP_N    => FL_SOP_N,
      FL_EOP_N    => FL_EOP_N,

      WORD_COUNTER=> WORD_COUNTER,
      HEADER_OFFSET=>OFFSET_IN,
      ACTIVE      => ACTIVE_IN,
      VALID       => bottom_vld,
      BYTE_OUT(7 downto 1) => mpls_stuff, -- unused
      BYTE_OUT(0) => bottom
   );

   --* Try to get the IP version field from the next protocol
   get_ipver_i : entity work.hfe_m_getbyte
   generic map(
      DATA_WIDTH     => DATA_WIDTH,
      COUNTER_WIDTH  => COUNTER_WIDTH,
      BYTE_OFFSET    => 4 -- Look for IP version in byte 4 (which is AFTER MPLS)
   )
   port map(
      CLK         => CLK,
      RESET       => RESET,

      FL_DATA     => FL_DATA,
      FL_DREM     => FL_DREM,
      FL_SRC_RDY_N=> FL_SRC_RDY_N,
      FL_DST_RDY_N=> FL_DST_RDY_N,
      FL_SOF_N    => FL_SOF_N,
      FL_EOF_N    => FL_EOF_N,
      FL_SOP_N    => FL_SOP_N,
      FL_EOP_N    => FL_EOP_N,

      WORD_COUNTER=> WORD_COUNTER,
      HEADER_OFFSET=>OFFSET_IN,
      ACTIVE      => ACTIVE_IN,
      VALID       => ipver_vld,
      BYTE_OUT(3 downto 0) => iplen, -- ignored
      BYTE_OUT(7 downto 4) => ipver
   );

   -- Compute and output the next header offset
   OFFSET_OUT <= OFFSET_IN + 4; -- MPLS header has 4 bytes

   sig_active_ipv4 <= '1' when ACTIVE_IN = '1' and bottom_vld = '1' and
      ipver_vld = '1' and bottom = '1' and ipver = X"4" else '0';

   sig_active_ipv6 <= '1' when ACTIVE_IN = '1' and bottom_vld = '1' and
      ipver_vld = '1' and bottom = '1' and ipver = X"6" else '0';

   sig_active_mpls <= '1' when ACTIVE_IN = '1' and bottom_vld = '1' and
      bottom = '0' else '0';

   ACTIVE_IPV4 <= sig_active_ipv4;
   ACTIVE_IPV6 <= sig_active_ipv6;
   ACTIVE_MPLS <= sig_active_mpls;

   ACTIVE_UNKNOWN <= '1' when ACTIVE_IN = '1' and bottom_vld = '1' and
      ipver_vld = '1' and not
      (sig_active_ipv4 = '1' or sig_active_ipv6 = '1' or sig_active_mpls = '1')
      else '0';

   MALFORMED <= '0';

   RUNNING <= ACTIVE_IN and not ipver_vld;

end architecture behavioral;
