-- ipv6_gen_ext.vhd: HFE-M IPv6 generic extension header analyzer
-- Copyright (C) 2011 CESNET
-- Author(s): Viktor Puš <pus@cesnet.cz>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--
--
--

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_unsigned.all;
use IEEE.std_logic_arith.all;
use IEEE.numeric_std.all;
use work.math_pack.all;

-- ----------------------------------------------------------------------------
--                      Entity declaration
-- ----------------------------------------------------------------------------

entity hfe_m_ipv6_gen_ext is
   generic (
      --* FrameLink width
      DATA_WIDTH  : integer := 64;
      --* Width of WORD_COUNTER
      COUNTER_WIDTH:integer := 4;
      --* Number of bits of the Hdr Ext Len field
      LEN_BITS    : integer := 6;
      --* Set to true to deactivate the module and set outputs to constant 0s.
      DISABLE     : boolean := false
   );
   port (
      CLK         : in std_logic;
      RESET       : in std_logic;

      --+ FrameLink interface (passive port)
      FL_DATA     : in std_logic_vector(DATA_WIDTH-1 downto 0);
      FL_DREM     : in std_logic_vector(log2(DATA_WIDTH/8)-1 downto 0);
      FL_SRC_RDY_N: in std_logic;
      FL_DST_RDY_N: in std_logic;
      FL_SOF_N    : in std_logic;
      FL_EOF_N    : in std_logic;
      FL_SOP_N    : in std_logic;
      FL_EOP_N    : in std_logic;

      --* Counter of FL words (incremented by 1 with every word)
      WORD_COUNTER   : in std_logic_vector(COUNTER_WIDTH-1 downto 0);

      --* Offset of the header start from SOF (in Bytes)
      OFFSET_IN      : in std_logic_vector(COUNTER_WIDTH+log2(DATA_WIDTH/8)-1
         downto 0);

      --* Previous layer detected the IPv6 extension header
      ACTIVE_IN      : in std_logic;

      --* This is fragmentation header
      FRAGMENTATION  : in std_logic;
      --* Is set to 1 when packet is NOT fragmented or has
      --* fragment offset = 0
      FRAGMENT_FIRST : out std_logic;
      --* Is set to 1 when packet is NOT fragmented or is
      --* the last fragment of the fragmented packet
      FRAGMENT_LAST  : out std_logic;

      --* Offset of the next header start from SOF (in Bytes)
      OFFSET_OUT     : out std_logic_vector(COUNTER_WIDTH+log2(DATA_WIDTH/8)-1
         downto 0);

      --+ Active for each known next layer protocol
      ACTIVE_TCP           : out std_logic;
      ACTIVE_UDP           : out std_logic;
      ACTIVE_ICMP          : out std_logic;
      ACTIVE_IGMP          : out std_logic;
      ACTIVE_MPLS          : out std_logic;
      ACTIVE_SCTP          : out std_logic;
      ACTIVE_IPV6          : out std_logic;
      ACTIVE_IPV4          : out std_logic;
      ACTIVE_ESP           : out std_logic;
      ACTIVE_IPV6_EXT      : out std_logic; -- any extension header (incl. frag)
      ACTIVE_IPV6_EXT_FRAG : out std_logic; -- only for fragmentation header
      ACTIVE_IPV6_NO_NEXT  : out std_logic;
      --* Last observed next-layer protocol
      PROTOCOL             : out std_logic_vector(7 downto 0);

      --* Active for unknown next layer protocol
      ACTIVE_UNKNOWN : out std_logic;

      --* Packet header has errors
      MALFORMED      : out std_logic;

      --* Still running (waiting for correct data word)
      RUNNING        : out std_logic
   );
end entity;

architecture behavioral of hfe_m_ipv6_gen_ext is

   signal length           : std_logic_vector(7 downto 0);
   signal length_trim      : std_logic_vector(8 downto 0);
   signal length_b         : std_logic_vector(10 downto 0);
   signal sig_offset_out   : std_logic_vector(10 downto 0);
   signal length_vld       : std_logic;

   signal fragoff_high     : std_logic_vector(7 downto 0);
   signal fragoff_low      : std_logic_vector(7 downto 0);
   signal fragoff_high_vld : std_logic;
   signal fragoff_low_vld  : std_logic;
   signal fragoff          : std_logic_vector(12 downto 0);
   signal fragoff_zero     : std_logic;

   signal nexth      : std_logic_vector(7 downto 0);
   signal nexth_vld  : std_logic;

   signal sig_active_tcp           : std_logic;
   signal sig_active_udp           : std_logic;
   signal sig_active_icmp          : std_logic;
   signal sig_active_igmp          : std_logic;
   signal sig_active_mpls          : std_logic;
   signal sig_active_sctp          : std_logic;
   signal sig_active_ipv6          : std_logic;
   signal sig_active_ipv4          : std_logic;
   signal sig_active_esp           : std_logic;
   signal sig_active_ipv6_ext      : std_logic;
   signal sig_active_ipv6_ext_frag : std_logic;
   signal sig_active_ipv6_no_next  : std_logic;
   signal sig_active_unknown       : std_logic;

   signal zeros         : std_logic_vector(8 downto 0);
   signal zeros_offset  : std_logic_vector(COUNTER_WIDTH+log2(DATA_WIDTH/8)-1
         downto 0);


begin

   --* Get the Header Length field
   --* (For Fragment Extension Header, this reads reserved byte of all zeros)
   get_length_i : entity work.hfe_m_getbyte
   generic map(
      DATA_WIDTH     => DATA_WIDTH,
      COUNTER_WIDTH  => COUNTER_WIDTH,
      BYTE_OFFSET    => 1 -- Header Length field in in byte 1
   )
   port map(
      CLK         => CLK,
      RESET       => RESET,

      FL_DATA     => FL_DATA,
      FL_DREM     => FL_DREM,
      FL_SRC_RDY_N=> FL_SRC_RDY_N,
      FL_DST_RDY_N=> FL_DST_RDY_N,
      FL_SOF_N    => FL_SOF_N,
      FL_EOF_N    => FL_EOF_N,
      FL_SOP_N    => FL_SOP_N,
      FL_EOP_N    => FL_EOP_N,

      WORD_COUNTER=> WORD_COUNTER,
      HEADER_OFFSET=>OFFSET_IN,
      ACTIVE      => ACTIVE_IN,
      VALID       => length_vld,
      BYTE_OUT    => length
   );

   zeros <= (others => '0');
   -- Trim input to LEN_BITS to save resources
   length_trim <= zeros(8 downto LEN_BITS) & length(LEN_BITS-1 downto 0);

   -- Convert to length in Bytes
   length_b <= (length_trim(7 downto 0) + 1) & "000";

   -- Compute and output the next header offset
   sig_offset_out <= OFFSET_IN + length_b;

   zeros_offset <= (others => '0');
   OFFSET_OUT <= sig_offset_out(COUNTER_WIDTH+log2(DATA_WIDTH/8)-1 downto 0)
                  when not DISABLE else zeros_offset;

   MALFORMED <= '0';

   --* Get the Next Header field
   get_nexth_i : entity work.hfe_m_getbyte
   generic map(
      DATA_WIDTH     => DATA_WIDTH,
      COUNTER_WIDTH  => COUNTER_WIDTH,
      BYTE_OFFSET    => 0 -- Next Header field in in byte 0
   )
   port map(
      CLK         => CLK,
      RESET       => RESET,

      FL_DATA     => FL_DATA,
      FL_DREM     => FL_DREM,
      FL_SRC_RDY_N=> FL_SRC_RDY_N,
      FL_DST_RDY_N=> FL_DST_RDY_N,
      FL_SOF_N    => FL_SOF_N,
      FL_EOF_N    => FL_EOF_N,
      FL_SOP_N    => FL_SOP_N,
      FL_EOP_N    => FL_EOP_N,

      WORD_COUNTER   => WORD_COUNTER,
      HEADER_OFFSET  => OFFSET_IN,
      ACTIVE         => ACTIVE_IN,
      VALID          => nexth_vld,
      BYTE_OUT       => nexth
   );

   --* Get the Fragment Offset field (high)
   get_fragoff_high_i : entity work.hfe_m_getbyte
   generic map(
      DATA_WIDTH     => DATA_WIDTH,
      COUNTER_WIDTH  => COUNTER_WIDTH,
      BYTE_OFFSET    => 2 -- Fragment Offset field starts in byte 2
   )
   port map(
      CLK         => CLK,
      RESET       => RESET,

      FL_DATA     => FL_DATA,
      FL_DREM     => FL_DREM,
      FL_SRC_RDY_N=> FL_SRC_RDY_N,
      FL_DST_RDY_N=> FL_DST_RDY_N,
      FL_SOF_N    => FL_SOF_N,
      FL_EOF_N    => FL_EOF_N,
      FL_SOP_N    => FL_SOP_N,
      FL_EOP_N    => FL_EOP_N,

      WORD_COUNTER   => WORD_COUNTER,
      HEADER_OFFSET  => OFFSET_IN,
      ACTIVE         => ACTIVE_IN,
      VALID          => fragoff_high_vld,
      BYTE_OUT       => fragoff_high
   );

   --* Get the Fragment Offset field (low)
   get_fragoff_low_i : entity work.hfe_m_getbyte
   generic map(
      DATA_WIDTH     => DATA_WIDTH,
      COUNTER_WIDTH  => COUNTER_WIDTH,
      BYTE_OFFSET    => 3 -- Fragment Offset field continues in byte 2
   )
   port map(
      CLK         => CLK,
      RESET       => RESET,

      FL_DATA     => FL_DATA,
      FL_DREM     => FL_DREM,
      FL_SRC_RDY_N=> FL_SRC_RDY_N,
      FL_DST_RDY_N=> FL_DST_RDY_N,
      FL_SOF_N    => FL_SOF_N,
      FL_EOF_N    => FL_EOF_N,
      FL_SOP_N    => FL_SOP_N,
      FL_EOP_N    => FL_EOP_N,

      WORD_COUNTER   => WORD_COUNTER,
      HEADER_OFFSET  => OFFSET_IN,
      ACTIVE         => ACTIVE_IN,
      VALID          => fragoff_low_vld,
      BYTE_OUT       => fragoff_low
   );

   fragoff <= fragoff_high & fragoff_low(7 downto 3);
   -- Detect zero fragment offset (if this is fragmentation header)
   fragoff_zero <= '1' when fragoff = "0000000000000" or
                            FRAGMENTATION = '0'
                   else '0';

   FRAGMENT_FIRST <= fragoff_zero when not DISABLE else '1';
   FRAGMENT_LAST <= '1' when fragoff_low(0) = '0' or FRAGMENTATION = '0'
                             or DISABLE
                    else '0';

   sig_active_tcp <= '1' when ACTIVE_IN = '1' and fragoff_low_vld = '1'
                     and nexth = X"06" and fragoff_zero = '1' else '0';
   sig_active_udp  <= '1' when ACTIVE_IN = '1' and fragoff_low_vld = '1'
                     and nexth = X"11" and fragoff_zero = '1' else '0';
   sig_active_icmp <= '1' when ACTIVE_IN = '1' and fragoff_low_vld = '1'
                     and (nexth = X"01" or nexth = X"3A") and fragoff_zero = '1'
                     else '0';
   sig_active_igmp <= '1' when ACTIVE_IN = '1' and fragoff_low_vld = '1'
                     and nexth = X"02" and fragoff_zero = '1' else '0';
   sig_active_mpls <= '1' when ACTIVE_IN = '1' and fragoff_low_vld = '1'
                     and nexth = X"89" and fragoff_zero = '1' else '0';
   sig_active_sctp <= '1' when ACTIVE_IN = '1' and fragoff_low_vld = '1'
                     and nexth = X"84" and fragoff_zero = '1' else '0';
   sig_active_ipv6 <= '1' when ACTIVE_IN = '1' and fragoff_low_vld = '1'
                     and nexth = X"29" and fragoff_zero = '1' else '0';
   sig_active_ipv4 <= '1' when ACTIVE_IN = '1' and fragoff_low_vld = '1'
                     and nexth = X"04" and fragoff_zero = '1' else '0';
   sig_active_esp  <= '1' when ACTIVE_IN = '1' and fragoff_low_vld = '1'
                     and nexth = X"32" and fragoff_zero = '1' else '0';
   sig_active_ipv6_ext  <= '1' when ACTIVE_IN = '1' and nexth_vld = '1'
                     and (nexth = X"00" or nexth = X"2B" or nexth = X"33"
                          or nexth = X"3C" or nexth = X"87" or nexth = X"2C"
                          or nexth = X"8C")
                     and fragoff_zero = '1' else '0';
   -- Distinguish fragmentation header from any other
   sig_active_ipv6_ext_frag <= '1' when ACTIVE_IN = '1' and nexth_vld = '1'
                     and nexth = X"2C" and fragoff_zero = '1' else '0';
   sig_active_ipv6_no_next  <= '1' when ACTIVE_IN = '1' and nexth_vld = '1'
                     and nexth = X"3B" and fragoff_zero = '1' else '0';

   sig_active_unknown <= '1' when ACTIVE_IN = '1' and nexth_vld = '1'
                     and not
                       (sig_active_tcp  = '1' or sig_active_udp  = '1'
                     or sig_active_icmp = '1' or sig_active_igmp = '1'
                     or sig_active_mpls = '1' or sig_active_sctp = '1'
                     or sig_active_ipv6 = '1' or sig_active_ipv4 = '1'
                     or sig_active_esp  = '1' or sig_active_ipv6_ext  = '1'
                     or sig_active_ipv6_no_next  = '1')
                     else '0';

   ACTIVE_TCP           <= sig_active_tcp when not DISABLE else '0';
   ACTIVE_UDP           <= sig_active_udp when not DISABLE else '0';
   ACTIVE_ICMP          <= sig_active_icmp when not DISABLE else '0';
   ACTIVE_IGMP          <= sig_active_igmp when not DISABLE else '0';
   ACTIVE_MPLS          <= sig_active_mpls when not DISABLE else '0';
   ACTIVE_SCTP          <= sig_active_sctp when not DISABLE else '0';
   ACTIVE_IPV6          <= sig_active_ipv6 when not DISABLE else '0';
   ACTIVE_IPV4          <= sig_active_ipv4 when not DISABLE else '0';
   ACTIVE_ESP           <= sig_active_esp when not DISABLE else '0';
   ACTIVE_IPV6_EXT      <= sig_active_ipv6_ext when not DISABLE else '0';
   ACTIVE_IPV6_EXT_FRAG <= sig_active_ipv6_ext_frag when not DISABLE else '0';
   ACTIVE_IPV6_NO_NEXT  <= sig_active_ipv6_no_next when not DISABLE else '0';

   ACTIVE_UNKNOWN       <= sig_active_unknown when not DISABLE else '0';

   PROTOCOL             <= nexth;

   RUNNING <= (ACTIVE_IN and not length_vld) when not DISABLE else '0';

end architecture behavioral;
