-- ipv6.vhd: HFE-M IPv6 analyzer
-- Copyright (C) 2011 CESNET
-- Author(s): Viktor Puš <pus@cesnet.cz>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--
--
--

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_unsigned.all;
use IEEE.std_logic_arith.all;
use IEEE.numeric_std.all;
use work.math_pack.all;

-- ----------------------------------------------------------------------------
--                      Entity declaration
-- ----------------------------------------------------------------------------

entity hfe_m_ipv6 is
   generic (
      --* FrameLink width
      DATA_WIDTH        : integer := 64;
      --* Width of WORD_COUNTER
      COUNTER_WIDTH     : integer := 4;
      --* Number of bits of the Hdr Ext Len field. Should be 8, lower it
      --* to save resources.
      EXT_LEN_BITS      : integer := 6;
      --* Maximum supported extension headers, max 8
      MAX_EXT_HEADERS   : integer
   );
   port (
      CLK         : in std_logic;
      RESET       : in std_logic;

      --+ FrameLink interface (passive port)
      FL_DATA     : in std_logic_vector(DATA_WIDTH-1 downto 0);
      FL_DREM     : in std_logic_vector(log2(DATA_WIDTH/8)-1 downto 0);
      FL_SRC_RDY_N: in std_logic;
      FL_DST_RDY_N: in std_logic;
      FL_SOF_N    : in std_logic;
      FL_EOF_N    : in std_logic;
      FL_SOP_N    : in std_logic;
      FL_EOP_N    : in std_logic;

      --* Counter of FL words (incremented by 1 with every word)
      WORD_COUNTER   : in std_logic_vector(COUNTER_WIDTH-1 downto 0);

      --* Offset of the header start from SOF (in Bytes)
      OFFSET_IN      : in std_logic_vector(COUNTER_WIDTH+log2(DATA_WIDTH/8)-1
         downto 0);

      --* Previous layer detected the IPv6 header
      ACTIVE_IN      : in std_logic;

      --* Offset of the next header start from SOF (in Bytes)
      OFFSET_OUT     : out std_logic_vector(COUNTER_WIDTH+log2(DATA_WIDTH/8)-1
         downto 0);

      --+ Active for each known next layer protocol
      ACTIVE_TCP           : out std_logic;
      ACTIVE_UDP           : out std_logic;
      ACTIVE_ICMP          : out std_logic;
      ACTIVE_IGMP          : out std_logic;
      ACTIVE_MPLS          : out std_logic;
      ACTIVE_SCTP          : out std_logic;
      ACTIVE_IPV6          : out std_logic;
      ACTIVE_IPV4          : out std_logic;
      ACTIVE_IPV6_NO_NEXT  : out std_logic;
      --* Last observed next-layer protocol
      PROTOCOL             : out std_logic_vector(7 downto 0);
      --* Is set to 1 when packet is NOT fragmented or has
      --* fragment offset = 0
      FRAGMENT_FIRST       : out std_logic;
      --* Is set to 1 when packet is NOT fragmented or is
      --* the last fragment of the fragmented packet
      FRAGMENT_LAST        : out std_logic;

      --* Active for unknown next layer protocol
      ACTIVE_UNKNOWN : out std_logic;

      --* Packet header has errors
      MALFORMED      : out std_logic;

      --* Still parsing, outputs are not valid yet
      RUNNING        : out std_logic
   );
end entity;

architecture behavioral of hfe_m_ipv6 is

   signal version    : std_logic_vector(3 downto 0);
   signal tc         : std_logic_vector(3 downto 0);
   signal version_bad: std_logic;
   signal version_vld: std_logic;

   signal nexth      : std_logic_vector(7 downto 0);
   signal nexth_vld  : std_logic;

   -- Main IPv6 header analysis results
   signal offset_main: std_logic_vector(COUNTER_WIDTH+log2(DATA_WIDTH/8)-1
         downto 0);
   signal sig_active_tcp            : std_logic;
   signal sig_active_udp            : std_logic;
   signal sig_active_icmp           : std_logic;
   signal sig_active_igmp           : std_logic;
   signal sig_active_mpls           : std_logic;
   signal sig_active_sctp           : std_logic;
   signal sig_active_ipv6           : std_logic;
   signal sig_active_ipv4           : std_logic;
   signal sig_active_ipv6_ext       : std_logic;
   signal sig_active_ipv6_ext_frag  : std_logic;
   signal sig_active_ipv6_no_next   : std_logic;
   signal sig_active_unknown        : std_logic;
   signal main_running              : std_logic;

   -- First generic extension header results
   signal geh0_offset_out : std_logic_vector(COUNTER_WIDTH+log2(DATA_WIDTH/8)-1
         downto 0);
   signal geh0_active_tcp          : std_logic;
   signal geh0_active_udp          : std_logic;
   signal geh0_active_icmp         : std_logic;
   signal geh0_active_igmp         : std_logic;
   signal geh0_active_mpls         : std_logic;
   signal geh0_active_sctp         : std_logic;
   signal geh0_active_ipv6         : std_logic;
   signal geh0_active_ipv4         : std_logic;
   signal geh0_active_esp          : std_logic;
   signal geh0_active_ipv6_ext     : std_logic;
   signal geh0_active_ipv6_ext_frag: std_logic;
   signal geh0_active_ipv6_no_next : std_logic;
   signal geh0_active_unknown      : std_logic;
   signal geh0_protocol            : std_logic_vector(7 downto 0);
   signal geh0_frag_first          : std_logic;
   signal geh0_frag_last           : std_logic;
   signal geh0_malformed           : std_logic;
   signal geh0_running             : std_logic;
   -- Second generic extension header results
   signal geh1_offset_out : std_logic_vector(COUNTER_WIDTH+log2(DATA_WIDTH/8)-1
         downto 0);
   signal geh1_active_tcp          : std_logic;
   signal geh1_active_udp          : std_logic;
   signal geh1_active_icmp         : std_logic;
   signal geh1_active_igmp         : std_logic;
   signal geh1_active_mpls         : std_logic;
   signal geh1_active_sctp         : std_logic;
   signal geh1_active_ipv6         : std_logic;
   signal geh1_active_ipv4         : std_logic;
   signal geh1_active_esp          : std_logic;
   signal geh1_active_ipv6_ext     : std_logic;
   signal geh1_active_ipv6_ext_frag: std_logic;
   signal geh1_active_ipv6_no_next : std_logic;
   signal geh1_active_unknown      : std_logic;
   signal geh1_protocol            : std_logic_vector(7 downto 0);
   signal geh1_frag_first          : std_logic;
   signal geh1_frag_last           : std_logic;
   signal geh1_malformed           : std_logic;
   signal geh1_running             : std_logic;
   -- Third generic extension header results
   signal geh2_offset_out : std_logic_vector(COUNTER_WIDTH+log2(DATA_WIDTH/8)-1
         downto 0);
   signal geh2_active_tcp          : std_logic;
   signal geh2_active_udp          : std_logic;
   signal geh2_active_icmp         : std_logic;
   signal geh2_active_igmp         : std_logic;
   signal geh2_active_mpls         : std_logic;
   signal geh2_active_sctp         : std_logic;
   signal geh2_active_ipv6         : std_logic;
   signal geh2_active_ipv4         : std_logic;
   signal geh2_active_esp          : std_logic;
   signal geh2_active_ipv6_ext     : std_logic;
   signal geh2_active_ipv6_ext_frag: std_logic;
   signal geh2_active_ipv6_no_next : std_logic;
   signal geh2_active_unknown      : std_logic;
   signal geh2_protocol            : std_logic_vector(7 downto 0);
   signal geh2_frag_first          : std_logic;
   signal geh2_frag_last           : std_logic;
   signal geh2_malformed           : std_logic;
   signal geh2_running             : std_logic;
   -- Fourth generic extension header results
   signal geh3_offset_out : std_logic_vector(COUNTER_WIDTH+log2(DATA_WIDTH/8)-1
         downto 0);
   signal geh3_active_tcp          : std_logic;
   signal geh3_active_udp          : std_logic;
   signal geh3_active_icmp         : std_logic;
   signal geh3_active_igmp         : std_logic;
   signal geh3_active_mpls         : std_logic;
   signal geh3_active_sctp         : std_logic;
   signal geh3_active_ipv6         : std_logic;
   signal geh3_active_ipv4         : std_logic;
   signal geh3_active_esp          : std_logic;
   signal geh3_active_ipv6_ext     : std_logic;
   signal geh3_active_ipv6_ext_frag: std_logic;
   signal geh3_active_ipv6_no_next : std_logic;
   signal geh3_active_unknown      : std_logic;
   signal geh3_protocol            : std_logic_vector(7 downto 0);
   signal geh3_frag_first          : std_logic;
   signal geh3_frag_last           : std_logic;
   signal geh3_malformed           : std_logic;
   signal geh3_running             : std_logic;
   -- Fifth generic extension header results
   signal geh4_offset_out : std_logic_vector(COUNTER_WIDTH+log2(DATA_WIDTH/8)-1
         downto 0);
   signal geh4_active_tcp          : std_logic;
   signal geh4_active_udp          : std_logic;
   signal geh4_active_icmp         : std_logic;
   signal geh4_active_igmp         : std_logic;
   signal geh4_active_mpls         : std_logic;
   signal geh4_active_sctp         : std_logic;
   signal geh4_active_ipv6         : std_logic;
   signal geh4_active_ipv4         : std_logic;
   signal geh4_active_esp          : std_logic;
   signal geh4_active_ipv6_ext     : std_logic;
   signal geh4_active_ipv6_ext_frag: std_logic;
   signal geh4_active_ipv6_no_next : std_logic;
   signal geh4_active_unknown      : std_logic;
   signal geh4_protocol            : std_logic_vector(7 downto 0);
   signal geh4_frag_first          : std_logic;
   signal geh4_frag_last           : std_logic;
   signal geh4_malformed           : std_logic;
   signal geh4_running             : std_logic;
   -- Sixth generic extension header results
   signal geh5_offset_out : std_logic_vector(COUNTER_WIDTH+log2(DATA_WIDTH/8)-1
         downto 0);
   signal geh5_active_tcp          : std_logic;
   signal geh5_active_udp          : std_logic;
   signal geh5_active_icmp         : std_logic;
   signal geh5_active_igmp         : std_logic;
   signal geh5_active_mpls         : std_logic;
   signal geh5_active_sctp         : std_logic;
   signal geh5_active_ipv6         : std_logic;
   signal geh5_active_ipv4         : std_logic;
   signal geh5_active_esp          : std_logic;
   signal geh5_active_ipv6_ext     : std_logic;
   signal geh5_active_ipv6_no_next : std_logic;
   signal geh5_active_ipv6_ext_frag: std_logic;
   signal geh5_active_unknown      : std_logic;
   signal geh5_protocol            : std_logic_vector(7 downto 0);
   signal geh5_frag_first          : std_logic;
   signal geh5_frag_last           : std_logic;
   signal geh5_malformed           : std_logic;
   signal geh5_running             : std_logic;
   -- Seventh generic extension header results
   signal geh6_offset_out : std_logic_vector(COUNTER_WIDTH+log2(DATA_WIDTH/8)-1
         downto 0);
   signal geh6_active_tcp          : std_logic;
   signal geh6_active_udp          : std_logic;
   signal geh6_active_icmp         : std_logic;
   signal geh6_active_igmp         : std_logic;
   signal geh6_active_mpls         : std_logic;
   signal geh6_active_sctp         : std_logic;
   signal geh6_active_ipv6         : std_logic;
   signal geh6_active_ipv4         : std_logic;
   signal geh6_active_esp          : std_logic;
   signal geh6_active_ipv6_ext     : std_logic;
   signal geh6_active_ipv6_ext_frag: std_logic;
   signal geh6_active_ipv6_no_next : std_logic;
   signal geh6_active_unknown      : std_logic;
   signal geh6_protocol            : std_logic_vector(7 downto 0);
   signal geh6_frag_first          : std_logic;
   signal geh6_frag_last           : std_logic;
   signal geh6_malformed           : std_logic;
   signal geh6_running             : std_logic;
   -- Eighth generic extension header results
   signal geh7_offset_out : std_logic_vector(COUNTER_WIDTH+log2(DATA_WIDTH/8)-1
         downto 0);
   signal geh7_active_tcp          : std_logic;
   signal geh7_active_udp          : std_logic;
   signal geh7_active_icmp         : std_logic;
   signal geh7_active_igmp         : std_logic;
   signal geh7_active_mpls         : std_logic;
   signal geh7_active_sctp         : std_logic;
   signal geh7_active_ipv6         : std_logic;
   signal geh7_active_ipv4         : std_logic;
   signal geh7_active_esp          : std_logic;
   signal geh7_active_ipv6_ext     : std_logic;
   signal geh7_active_ipv6_ext_frag: std_logic;
   signal geh7_active_ipv6_no_next : std_logic;
   signal geh7_active_unknown      : std_logic;
   signal geh7_protocol            : std_logic_vector(7 downto 0);
   signal geh7_frag_first          : std_logic;
   signal geh7_frag_last           : std_logic;
   signal geh7_malformed           : std_logic;
   signal geh7_running             : std_logic;

   -- 33 Bytes after the Next Header field are unimportant
   -- for analysis => for data width up to 256 b, we allow one cycle
   -- of getbyte output latency (simpler getbyte output logic).
   function latency return integer is
   begin
      if DATA_WIDTH <= 256 then
         return 1;
      else
         return 0;
      end if;
   end latency;

begin

   ---------------------------------------------------------------
   -- Main IPv6 header analysis
   ---------------------------------------------------------------

   --* Get the Version field
   get_version_i : entity work.hfe_m_getbyte
   generic map(
      DATA_WIDTH     => DATA_WIDTH,
      COUNTER_WIDTH  => COUNTER_WIDTH,
      BYTE_OFFSET    => 0, -- Version field is in byte 0
      LATENCY        => latency
   )
   port map(
      CLK         => CLK,
      RESET       => RESET,

      FL_DATA     => FL_DATA,
      FL_DREM     => FL_DREM,
      FL_SRC_RDY_N=> FL_SRC_RDY_N,
      FL_DST_RDY_N=> FL_DST_RDY_N,
      FL_SOF_N    => FL_SOF_N,
      FL_EOF_N    => FL_EOF_N,
      FL_SOP_N    => FL_SOP_N,
      FL_EOP_N    => FL_EOP_N,

      WORD_COUNTER=> WORD_COUNTER,
      HEADER_OFFSET=>OFFSET_IN,
      ACTIVE      => ACTIVE_IN,
      VALID       => version_vld,
      BYTE_OUT(3 downto 0) => tc, -- unused
      BYTE_OUT(7 downto 4) => version
   );

   offset_main <= OFFSET_IN + 40;

   --* Get the Next Header field
   get_nexth_i : entity work.hfe_m_getbyte
   generic map(
      DATA_WIDTH     => DATA_WIDTH,
      COUNTER_WIDTH  => COUNTER_WIDTH,
      BYTE_OFFSET    => 6, -- Next Header field is in byte 6
      LATENCY        => latency
   )
   port map(
      CLK         => CLK,
      RESET       => RESET,

      FL_DATA     => FL_DATA,
      FL_DREM     => FL_DREM,
      FL_SRC_RDY_N=> FL_SRC_RDY_N,
      FL_DST_RDY_N=> FL_DST_RDY_N,
      FL_SOF_N    => FL_SOF_N,
      FL_EOF_N    => FL_EOF_N,
      FL_SOP_N    => FL_SOP_N,
      FL_EOP_N    => FL_EOP_N,

      WORD_COUNTER   => WORD_COUNTER,
      HEADER_OFFSET  => OFFSET_IN,
      ACTIVE      => ACTIVE_IN,
      VALID          => nexth_vld,
      BYTE_OUT       => nexth
   );

   sig_active_tcp <= '1' when ACTIVE_IN = '1' and nexth_vld = '1'
                     and nexth = X"06" else '0';
   sig_active_udp  <= '1' when ACTIVE_IN = '1' and nexth_vld = '1'
                     and nexth = X"11" else '0';
   sig_active_icmp <= '1' when ACTIVE_IN = '1' and nexth_vld = '1'
                     and (nexth = X"01" or nexth = X"3A") else '0';
   sig_active_igmp <= '1' when ACTIVE_IN = '1' and nexth_vld = '1'
                     and nexth = X"02" else '0';
   sig_active_mpls <= '1' when ACTIVE_IN = '1' and nexth_vld = '1'
                     and nexth = X"89" else '0';
   sig_active_sctp <= '1' when ACTIVE_IN = '1' and nexth_vld = '1'
                     and nexth = X"84" else '0';
   sig_active_ipv6 <= '1' when ACTIVE_IN = '1' and nexth_vld = '1'
                     and nexth = X"29" else '0';
   sig_active_ipv4 <= '1' when ACTIVE_IN = '1' and nexth_vld = '1'
                     and nexth = X"04" else '0';
   sig_active_ipv6_ext  <= '1' when ACTIVE_IN = '1' and nexth_vld = '1'
                     and (nexth = X"00" or nexth = X"2B" or nexth = X"33"
                          or nexth = X"3C" or nexth = X"87" or nexth = X"2C"
                          or nexth = X"8C")
                     else '0';
   sig_active_ipv6_ext_frag <= '1' when ACTIVE_IN = '1' and nexth_vld = '1'
                     and nexth = X"2C" else '0';
   sig_active_ipv6_no_next  <= '1' when ACTIVE_IN = '1' and nexth_vld = '1'
                     and nexth = X"3B" else '0';

   sig_active_unknown <= '1' when ACTIVE_IN = '1' and nexth_vld = '1' and not
   (sig_active_tcp  = '1' or sig_active_udp  = '1' or
    sig_active_icmp = '1' or sig_active_igmp = '1' or
    sig_active_mpls = '1' or sig_active_sctp = '1' or
    sig_active_ipv6 = '1' or sig_active_ipv4 = '1' or
    sig_active_ipv6_ext = '1' or sig_active_ipv6_no_next = '1')
    else '0';

   main_running <= ACTIVE_IN and not nexth_vld;

   ---------------------------------------------------------------------
   -- Extension header 0
   ---------------------------------------------------------------------
   --* First Extension Header analyzer
   geh0_i : entity work.hfe_m_ipv6_gen_ext
   generic map(
      DATA_WIDTH  => DATA_WIDTH,
      COUNTER_WIDTH=> COUNTER_WIDTH,
      LEN_BITS    => EXT_LEN_BITS,
      DISABLE     => not (MAX_EXT_HEADERS > 0)
   )
   port map(
      CLK         => CLK,
      RESET       => RESET,
      FL_DATA     => FL_DATA,
      FL_DREM     => FL_DREM,
      FL_SRC_RDY_N=> FL_SRC_RDY_N,
      FL_DST_RDY_N=> FL_DST_RDY_N,
      FL_SOF_N    => FL_SOF_N,
      FL_EOF_N    => FL_EOF_N,
      FL_SOP_N    => FL_SOP_N,
      FL_EOP_N    => FL_EOP_N,

      WORD_COUNTER   => WORD_COUNTER,
      OFFSET_IN      => offset_main,
      ACTIVE_IN      => sig_active_ipv6_ext,
      FRAGMENTATION  => sig_active_ipv6_ext_frag,
      OFFSET_OUT     => geh0_offset_out,
      --+ Active for each known next layer protocol
      ACTIVE_TCP           => geh0_active_tcp,
      ACTIVE_UDP           => geh0_active_udp,
      ACTIVE_ICMP          => geh0_active_icmp,
      ACTIVE_IGMP          => geh0_active_igmp,
      ACTIVE_MPLS          => geh0_active_mpls,
      ACTIVE_SCTP          => geh0_active_sctp,
      ACTIVE_IPV6          => geh0_active_ipv6,
      ACTIVE_IPV4          => geh0_active_ipv4,
      ACTIVE_ESP           => geh0_active_esp,
      ACTIVE_IPV6_EXT      => geh0_active_ipv6_ext,
      ACTIVE_IPV6_EXT_FRAG => geh0_active_ipv6_ext_frag,
      ACTIVE_IPV6_NO_NEXT  => geh0_active_ipv6_no_next,
      ACTIVE_UNKNOWN       => geh0_active_unknown,
      PROTOCOL             => geh0_protocol,
      FRAGMENT_FIRST       => geh0_frag_first,
      FRAGMENT_LAST        => geh0_frag_last,
      MALFORMED            => geh0_malformed,
      RUNNING              => geh0_running
   );

   ---------------------------------------------------------------------
   -- Extension header 1
   ---------------------------------------------------------------------
   --* Second Extension Header analyzer
   geh1_i : entity work.hfe_m_ipv6_gen_ext
   generic map(
      DATA_WIDTH  => DATA_WIDTH,
      COUNTER_WIDTH=> COUNTER_WIDTH,
      LEN_BITS    => EXT_LEN_BITS,
      DISABLE     => not (MAX_EXT_HEADERS > 1)
   )
   port map(
      CLK         => CLK,
      RESET       => RESET,
      FL_DATA     => FL_DATA,
      FL_DREM     => FL_DREM,
      FL_SRC_RDY_N=> FL_SRC_RDY_N,
      FL_DST_RDY_N=> FL_DST_RDY_N,
      FL_SOF_N    => FL_SOF_N,
      FL_EOF_N    => FL_EOF_N,
      FL_SOP_N    => FL_SOP_N,
      FL_EOP_N    => FL_EOP_N,

      WORD_COUNTER   => WORD_COUNTER,
      OFFSET_IN      => geh0_offset_out,
      ACTIVE_IN      => geh0_active_ipv6_ext,
      FRAGMENTATION  => geh0_active_ipv6_ext_frag,
      OFFSET_OUT     => geh1_offset_out,
      --+ Active for each known next layer protocol
      ACTIVE_TCP           => geh1_active_tcp,
      ACTIVE_UDP           => geh1_active_udp,
      ACTIVE_ICMP          => geh1_active_icmp,
      ACTIVE_IGMP          => geh1_active_igmp,
      ACTIVE_MPLS          => geh1_active_mpls,
      ACTIVE_SCTP          => geh1_active_sctp,
      ACTIVE_IPV6          => geh1_active_ipv6,
      ACTIVE_IPV4          => geh1_active_ipv4,
      ACTIVE_ESP           => geh1_active_esp,
      ACTIVE_IPV6_EXT      => geh1_active_ipv6_ext,
      ACTIVE_IPV6_EXT_FRAG => geh1_active_ipv6_ext_frag,
      ACTIVE_IPV6_NO_NEXT  => geh1_active_ipv6_no_next,
      ACTIVE_UNKNOWN       => geh1_active_unknown,
      PROTOCOL             => geh1_protocol,
      FRAGMENT_FIRST       => geh1_frag_first,
      FRAGMENT_LAST        => geh1_frag_last,
      MALFORMED            => geh1_malformed,
      RUNNING              => geh1_running
   );

   ---------------------------------------------------------------------
   -- Extension header 2
   ---------------------------------------------------------------------
   --* Third Extension Header analyzer
   geh2_i : entity work.hfe_m_ipv6_gen_ext
   generic map(
      DATA_WIDTH  => DATA_WIDTH,
      COUNTER_WIDTH=> COUNTER_WIDTH,
      LEN_BITS    => EXT_LEN_BITS,
      DISABLE     => not (MAX_EXT_HEADERS > 2)
   )
   port map(
      CLK         => CLK,
      RESET       => RESET,
      FL_DATA     => FL_DATA,
      FL_DREM     => FL_DREM,
      FL_SRC_RDY_N=> FL_SRC_RDY_N,
      FL_DST_RDY_N=> FL_DST_RDY_N,
      FL_SOF_N    => FL_SOF_N,
      FL_EOF_N    => FL_EOF_N,
      FL_SOP_N    => FL_SOP_N,
      FL_EOP_N    => FL_EOP_N,

      WORD_COUNTER   => WORD_COUNTER,
      OFFSET_IN      => geh1_offset_out,
      ACTIVE_IN      => geh1_active_ipv6_ext,
      FRAGMENTATION  => geh1_active_ipv6_ext_frag,
      OFFSET_OUT     => geh2_offset_out,
      --+ Active for each known next layer protocol
      ACTIVE_TCP           => geh2_active_tcp,
      ACTIVE_UDP           => geh2_active_udp,
      ACTIVE_ICMP          => geh2_active_icmp,
      ACTIVE_IGMP          => geh2_active_igmp,
      ACTIVE_MPLS          => geh2_active_mpls,
      ACTIVE_SCTP          => geh2_active_sctp,
      ACTIVE_IPV6          => geh2_active_ipv6,
      ACTIVE_IPV4          => geh2_active_ipv4,
      ACTIVE_ESP           => geh2_active_esp,
      ACTIVE_IPV6_EXT      => geh2_active_ipv6_ext,
      ACTIVE_IPV6_EXT_FRAG => geh2_active_ipv6_ext_frag,
      ACTIVE_IPV6_NO_NEXT  => geh2_active_ipv6_no_next,
      ACTIVE_UNKNOWN       => geh2_active_unknown,
      PROTOCOL             => geh2_protocol,
      FRAGMENT_FIRST       => geh2_frag_first,
      FRAGMENT_LAST        => geh2_frag_last,
      MALFORMED            => geh2_malformed,
      RUNNING              => geh2_running
   );

   ---------------------------------------------------------------------
   -- Extension header 3
   ---------------------------------------------------------------------
   --* Fourth Extension Header analyzer
   geh3_i : entity work.hfe_m_ipv6_gen_ext
   generic map(
      DATA_WIDTH  => DATA_WIDTH,
      COUNTER_WIDTH=> COUNTER_WIDTH,
      LEN_BITS    => EXT_LEN_BITS,
      DISABLE     => not (MAX_EXT_HEADERS > 3)
   )
   port map(
      CLK         => CLK,
      RESET       => RESET,
      FL_DATA     => FL_DATA,
      FL_DREM     => FL_DREM,
      FL_SRC_RDY_N=> FL_SRC_RDY_N,
      FL_DST_RDY_N=> FL_DST_RDY_N,
      FL_SOF_N    => FL_SOF_N,
      FL_EOF_N    => FL_EOF_N,
      FL_SOP_N    => FL_SOP_N,
      FL_EOP_N    => FL_EOP_N,

      WORD_COUNTER   => WORD_COUNTER,
      OFFSET_IN      => geh2_offset_out,
      ACTIVE_IN      => geh2_active_ipv6_ext,
      FRAGMENTATION  => geh2_active_ipv6_ext_frag,
      OFFSET_OUT     => geh3_offset_out,
      --+ Active for each known next layer protocol
      ACTIVE_TCP           => geh3_active_tcp,
      ACTIVE_UDP           => geh3_active_udp,
      ACTIVE_ICMP          => geh3_active_icmp,
      ACTIVE_IGMP          => geh3_active_igmp,
      ACTIVE_MPLS          => geh3_active_mpls,
      ACTIVE_SCTP          => geh3_active_sctp,
      ACTIVE_IPV6          => geh3_active_ipv6,
      ACTIVE_IPV4          => geh3_active_ipv4,
      ACTIVE_ESP           => geh3_active_esp,
      ACTIVE_IPV6_EXT      => geh3_active_ipv6_ext,
      ACTIVE_IPV6_EXT_FRAG => geh3_active_ipv6_ext_frag,
      ACTIVE_IPV6_NO_NEXT  => geh3_active_ipv6_no_next,
      ACTIVE_UNKNOWN       => geh3_active_unknown,
      PROTOCOL             => geh3_protocol,
      FRAGMENT_FIRST       => geh3_frag_first,
      FRAGMENT_LAST        => geh3_frag_last,
      MALFORMED            => geh3_malformed,
      RUNNING              => geh3_running
   );

   ---------------------------------------------------------------------
   -- Extension header 4
   ---------------------------------------------------------------------
   --* Fifth Extension Header analyzer
   geh4_i : entity work.hfe_m_ipv6_gen_ext
   generic map(
      DATA_WIDTH  => DATA_WIDTH,
      COUNTER_WIDTH=> COUNTER_WIDTH,
      LEN_BITS    => EXT_LEN_BITS,
      DISABLE     => not (MAX_EXT_HEADERS > 4)
   )
   port map(
      CLK         => CLK,
      RESET       => RESET,
      FL_DATA     => FL_DATA,
      FL_DREM     => FL_DREM,
      FL_SRC_RDY_N=> FL_SRC_RDY_N,
      FL_DST_RDY_N=> FL_DST_RDY_N,
      FL_SOF_N    => FL_SOF_N,
      FL_EOF_N    => FL_EOF_N,
      FL_SOP_N    => FL_SOP_N,
      FL_EOP_N    => FL_EOP_N,

      WORD_COUNTER   => WORD_COUNTER,
      OFFSET_IN      => geh3_offset_out,
      ACTIVE_IN      => geh3_active_ipv6_ext,
      FRAGMENTATION  => geh3_active_ipv6_ext_frag,
      OFFSET_OUT     => geh4_offset_out,
      --+ Active for each known next layer protocol
      ACTIVE_TCP           => geh4_active_tcp,
      ACTIVE_UDP           => geh4_active_udp,
      ACTIVE_ICMP          => geh4_active_icmp,
      ACTIVE_IGMP          => geh4_active_igmp,
      ACTIVE_MPLS          => geh4_active_mpls,
      ACTIVE_SCTP          => geh4_active_sctp,
      ACTIVE_IPV6          => geh4_active_ipv6,
      ACTIVE_IPV4          => geh4_active_ipv4,
      ACTIVE_ESP           => geh4_active_esp,
      ACTIVE_IPV6_EXT      => geh4_active_ipv6_ext,
      ACTIVE_IPV6_EXT_FRAG => geh4_active_ipv6_ext_frag,
      ACTIVE_IPV6_NO_NEXT  => geh4_active_ipv6_no_next,
      ACTIVE_UNKNOWN       => geh4_active_unknown,
      PROTOCOL             => geh4_protocol,
      FRAGMENT_FIRST       => geh4_frag_first,
      FRAGMENT_LAST        => geh4_frag_last,
      MALFORMED            => geh4_malformed,
      RUNNING              => geh4_running
   );

   ---------------------------------------------------------------------
   -- Extension header 5
   ---------------------------------------------------------------------
   --* Sixth Extension Header analyzer
   geh5_i : entity work.hfe_m_ipv6_gen_ext
   generic map(
      DATA_WIDTH  => DATA_WIDTH,
      COUNTER_WIDTH=> COUNTER_WIDTH,
      LEN_BITS    => EXT_LEN_BITS,
      DISABLE     => not (MAX_EXT_HEADERS > 5)
   )
   port map(
      CLK         => CLK,
      RESET       => RESET,
      FL_DATA     => FL_DATA,
      FL_DREM     => FL_DREM,
      FL_SRC_RDY_N=> FL_SRC_RDY_N,
      FL_DST_RDY_N=> FL_DST_RDY_N,
      FL_SOF_N    => FL_SOF_N,
      FL_EOF_N    => FL_EOF_N,
      FL_SOP_N    => FL_SOP_N,
      FL_EOP_N    => FL_EOP_N,

      WORD_COUNTER   => WORD_COUNTER,
      OFFSET_IN      => geh4_offset_out,
      ACTIVE_IN      => geh4_active_ipv6_ext,
      FRAGMENTATION  => geh4_active_ipv6_ext_frag,
      OFFSET_OUT     => geh5_offset_out,
      --+ Active for each known next layer protocol
      ACTIVE_TCP           => geh5_active_tcp,
      ACTIVE_UDP           => geh5_active_udp,
      ACTIVE_ICMP          => geh5_active_icmp,
      ACTIVE_IGMP          => geh5_active_igmp,
      ACTIVE_MPLS          => geh5_active_mpls,
      ACTIVE_SCTP          => geh5_active_sctp,
      ACTIVE_IPV6          => geh5_active_ipv6,
      ACTIVE_IPV4          => geh5_active_ipv4,
      ACTIVE_ESP           => geh5_active_esp,
      ACTIVE_IPV6_EXT      => geh5_active_ipv6_ext,
      ACTIVE_IPV6_EXT_FRAG => geh5_active_ipv6_ext_frag,
      ACTIVE_IPV6_NO_NEXT  => geh5_active_ipv6_no_next,
      ACTIVE_UNKNOWN       => geh5_active_unknown,
      PROTOCOL             => geh5_protocol,
      FRAGMENT_FIRST       => geh5_frag_first,
      FRAGMENT_LAST        => geh5_frag_last,
      MALFORMED            => geh5_malformed,
      RUNNING              => geh5_running
   );

   ---------------------------------------------------------------------
   -- Extension header 6
   ---------------------------------------------------------------------
   --* Seventh Extension Header analyzer
   geh6_i : entity work.hfe_m_ipv6_gen_ext
   generic map(
      DATA_WIDTH  => DATA_WIDTH,
      COUNTER_WIDTH=> COUNTER_WIDTH,
      LEN_BITS    => EXT_LEN_BITS,
      DISABLE     => not (MAX_EXT_HEADERS > 6)
   )
   port map(
      CLK         => CLK,
      RESET       => RESET,
      FL_DATA     => FL_DATA,
      FL_DREM     => FL_DREM,
      FL_SRC_RDY_N=> FL_SRC_RDY_N,
      FL_DST_RDY_N=> FL_DST_RDY_N,
      FL_SOF_N    => FL_SOF_N,
      FL_EOF_N    => FL_EOF_N,
      FL_SOP_N    => FL_SOP_N,
      FL_EOP_N    => FL_EOP_N,

      WORD_COUNTER   => WORD_COUNTER,
      OFFSET_IN      => geh5_offset_out,
      ACTIVE_IN      => geh5_active_ipv6_ext,
      FRAGMENTATION  => geh5_active_ipv6_ext_frag,
      OFFSET_OUT     => geh6_offset_out,
      --+ Active for each known next layer protocol
      ACTIVE_TCP           => geh6_active_tcp,
      ACTIVE_UDP           => geh6_active_udp,
      ACTIVE_ICMP          => geh6_active_icmp,
      ACTIVE_IGMP          => geh6_active_igmp,
      ACTIVE_MPLS          => geh6_active_mpls,
      ACTIVE_SCTP          => geh6_active_sctp,
      ACTIVE_IPV6          => geh6_active_ipv6,
      ACTIVE_IPV4          => geh6_active_ipv4,
      ACTIVE_ESP           => geh6_active_esp,
      ACTIVE_IPV6_EXT      => geh6_active_ipv6_ext,
      ACTIVE_IPV6_EXT_FRAG => geh6_active_ipv6_ext_frag,
      ACTIVE_IPV6_NO_NEXT  => geh6_active_ipv6_no_next,
      ACTIVE_UNKNOWN       => geh6_active_unknown,
      PROTOCOL             => geh6_protocol,
      FRAGMENT_FIRST       => geh6_frag_first,
      FRAGMENT_LAST        => geh6_frag_last,
      MALFORMED            => geh6_malformed,
      RUNNING              => geh6_running
   );

   ---------------------------------------------------------------------
   -- Extension header 7
   ---------------------------------------------------------------------
   --* Eighth Extension Header analyzer
   geh7_i : entity work.hfe_m_ipv6_gen_ext
   generic map(
      DATA_WIDTH  => DATA_WIDTH,
      COUNTER_WIDTH=> COUNTER_WIDTH,
      LEN_BITS    => EXT_LEN_BITS,
      DISABLE     => not (MAX_EXT_HEADERS > 7)
   )
   port map(
      CLK         => CLK,
      RESET       => RESET,
      FL_DATA     => FL_DATA,
      FL_DREM     => FL_DREM,
      FL_SRC_RDY_N=> FL_SRC_RDY_N,
      FL_DST_RDY_N=> FL_DST_RDY_N,
      FL_SOF_N    => FL_SOF_N,
      FL_EOF_N    => FL_EOF_N,
      FL_SOP_N    => FL_SOP_N,
      FL_EOP_N    => FL_EOP_N,

      WORD_COUNTER   => WORD_COUNTER,
      OFFSET_IN      => geh6_offset_out,
      ACTIVE_IN      => geh6_active_ipv6_ext,
      FRAGMENTATION  => geh6_active_ipv6_ext_frag,
      OFFSET_OUT     => geh7_offset_out,
      --+ Active for each known next layer protocol
      ACTIVE_TCP           => geh7_active_tcp,
      ACTIVE_UDP           => geh7_active_udp,
      ACTIVE_ICMP          => geh7_active_icmp,
      ACTIVE_IGMP          => geh7_active_igmp,
      ACTIVE_MPLS          => geh7_active_mpls,
      ACTIVE_SCTP          => geh7_active_sctp,
      ACTIVE_IPV6          => geh7_active_ipv6,
      ACTIVE_IPV4          => geh7_active_ipv4,
      ACTIVE_ESP           => geh7_active_esp,
      ACTIVE_IPV6_EXT      => geh7_active_ipv6_ext,
      ACTIVE_IPV6_EXT_FRAG => geh7_active_ipv6_ext_frag,
      ACTIVE_IPV6_NO_NEXT  => geh7_active_ipv6_no_next,
      ACTIVE_UNKNOWN       => geh7_active_unknown,
      PROTOCOL             => geh7_protocol,
      FRAGMENT_FIRST       => geh7_frag_first,
      FRAGMENT_LAST        => geh7_frag_last,
      MALFORMED            => geh7_malformed,
      RUNNING              => geh7_running
   );

   ---------------------------------------------------------
   -- Output logic
   ---------------------------------------------------------

   RUNNING <= main_running or geh0_running or geh1_running or geh2_running
              or geh3_running or geh4_running or geh5_running or geh6_running
              or geh7_running;

   -- Detect malformed IPv6 header
   version_bad <= '1' when version /= X"6" and version_vld = '1' else '0';
   MALFORMED <= version_bad or
                (geh0_malformed and sig_active_ipv6_ext) or
                (geh1_malformed and geh0_active_ipv6_ext) or
                (geh2_malformed and geh1_active_ipv6_ext) or
                (geh3_malformed and geh2_active_ipv6_ext) or
                (geh4_malformed and geh3_active_ipv6_ext) or
                (geh5_malformed and geh4_active_ipv6_ext) or
                (geh6_malformed and geh5_active_ipv6_ext) or
                (geh7_malformed and geh6_active_ipv6_ext);

   -- Offset of the last active Extension Header analyzer
   OFFSET_OUT <= geh7_offset_out when geh6_active_ipv6_ext = '1' else
                 geh6_offset_out when geh5_active_ipv6_ext = '1' else
                 geh5_offset_out when geh4_active_ipv6_ext = '1' else
                 geh4_offset_out when geh3_active_ipv6_ext = '1' else
                 geh3_offset_out when geh2_active_ipv6_ext = '1' else
                 geh2_offset_out when geh1_active_ipv6_ext = '1' else
                 geh1_offset_out when geh0_active_ipv6_ext = '1' else
                 geh0_offset_out when sig_active_ipv6_ext = '1' else
                 offset_main;

   ACTIVE_TCP <= geh7_active_tcp when geh6_active_ipv6_ext = '1' else
                 geh6_active_tcp when geh5_active_ipv6_ext = '1' else
                 geh5_active_tcp when geh4_active_ipv6_ext = '1' else
                 geh4_active_tcp when geh3_active_ipv6_ext = '1' else
                 geh3_active_tcp when geh2_active_ipv6_ext = '1' else
                 geh2_active_tcp when geh1_active_ipv6_ext = '1' else
                 geh1_active_tcp when geh0_active_ipv6_ext = '1' else
                 geh0_active_tcp when sig_active_ipv6_ext = '1' else
                 sig_active_tcp;

   ACTIVE_UDP <= geh7_active_udp when geh6_active_ipv6_ext = '1' else
                 geh6_active_udp when geh5_active_ipv6_ext = '1' else
                 geh5_active_udp when geh4_active_ipv6_ext = '1' else
                 geh4_active_udp when geh3_active_ipv6_ext = '1' else
                 geh3_active_udp when geh2_active_ipv6_ext = '1' else
                 geh2_active_udp when geh1_active_ipv6_ext = '1' else
                 geh1_active_udp when geh0_active_ipv6_ext = '1' else
                 geh0_active_udp when sig_active_ipv6_ext = '1' else
                 sig_active_udp;

   ACTIVE_ICMP<= geh7_active_icmp when geh6_active_ipv6_ext = '1' else
                 geh6_active_icmp when geh5_active_ipv6_ext = '1' else
                 geh5_active_icmp when geh4_active_ipv6_ext = '1' else
                 geh4_active_icmp when geh3_active_ipv6_ext = '1' else
                 geh3_active_icmp when geh2_active_ipv6_ext = '1' else
                 geh2_active_icmp when geh1_active_ipv6_ext = '1' else
                 geh1_active_icmp when geh0_active_ipv6_ext = '1' else
                 geh0_active_icmp when sig_active_ipv6_ext = '1' else
                 sig_active_icmp;

   ACTIVE_IGMP<= geh7_active_igmp when geh6_active_ipv6_ext = '1' else
                 geh6_active_igmp when geh5_active_ipv6_ext = '1' else
                 geh5_active_igmp when geh4_active_ipv6_ext = '1' else
                 geh4_active_igmp when geh3_active_ipv6_ext = '1' else
                 geh3_active_igmp when geh2_active_ipv6_ext = '1' else
                 geh2_active_igmp when geh1_active_ipv6_ext = '1' else
                 geh1_active_igmp when geh0_active_ipv6_ext = '1' else
                 geh0_active_igmp when sig_active_ipv6_ext = '1' else
                 sig_active_igmp;

   ACTIVE_MPLS<= geh7_active_mpls when geh6_active_ipv6_ext = '1' else
                 geh6_active_mpls when geh5_active_ipv6_ext = '1' else
                 geh5_active_mpls when geh4_active_ipv6_ext = '1' else
                 geh4_active_mpls when geh3_active_ipv6_ext = '1' else
                 geh3_active_mpls when geh2_active_ipv6_ext = '1' else
                 geh2_active_mpls when geh1_active_ipv6_ext = '1' else
                 geh1_active_mpls when geh0_active_ipv6_ext = '1' else
                 geh0_active_mpls when sig_active_ipv6_ext = '1' else
                 sig_active_mpls;

   ACTIVE_SCTP<= geh7_active_sctp when geh6_active_ipv6_ext = '1' else
                 geh6_active_sctp when geh5_active_ipv6_ext = '1' else
                 geh5_active_sctp when geh4_active_ipv6_ext = '1' else
                 geh4_active_sctp when geh3_active_ipv6_ext = '1' else
                 geh3_active_sctp when geh2_active_ipv6_ext = '1' else
                 geh2_active_sctp when geh1_active_ipv6_ext = '1' else
                 geh1_active_sctp when geh0_active_ipv6_ext = '1' else
                 geh0_active_sctp when sig_active_ipv6_ext = '1' else
                 sig_active_sctp;

   ACTIVE_IPV6<= geh7_active_ipv6 when geh6_active_ipv6_ext = '1' else
                 geh6_active_ipv6 when geh5_active_ipv6_ext = '1' else
                 geh5_active_ipv6 when geh4_active_ipv6_ext = '1' else
                 geh4_active_ipv6 when geh3_active_ipv6_ext = '1' else
                 geh3_active_ipv6 when geh2_active_ipv6_ext = '1' else
                 geh2_active_ipv6 when geh1_active_ipv6_ext = '1' else
                 geh1_active_ipv6 when geh0_active_ipv6_ext = '1' else
                 geh0_active_ipv6 when sig_active_ipv6_ext = '1' else
                 sig_active_ipv6;

   ACTIVE_IPV4<= geh7_active_ipv4 when geh6_active_ipv6_ext = '1' else
                 geh6_active_ipv4 when geh5_active_ipv6_ext = '1' else
                 geh5_active_ipv4 when geh4_active_ipv6_ext = '1' else
                 geh4_active_ipv4 when geh3_active_ipv6_ext = '1' else
                 geh3_active_ipv4 when geh2_active_ipv6_ext = '1' else
                 geh2_active_ipv4 when geh1_active_ipv6_ext = '1' else
                 geh1_active_ipv4 when geh0_active_ipv6_ext = '1' else
                 geh0_active_ipv4 when sig_active_ipv6_ext = '1' else
                 sig_active_ipv4;

   ACTIVE_IPV6_NO_NEXT<=geh7_active_ipv6_no_next when geh6_active_ipv6_ext = '1' else
                 geh6_active_ipv6_no_next when geh5_active_ipv6_ext = '1' else
                 geh5_active_ipv6_no_next when geh4_active_ipv6_ext = '1' else
                 geh4_active_ipv6_no_next when geh3_active_ipv6_ext = '1' else
                 geh3_active_ipv6_no_next when geh2_active_ipv6_ext = '1' else
                 geh2_active_ipv6_no_next when geh1_active_ipv6_ext = '1' else
                 geh1_active_ipv6_no_next when geh0_active_ipv6_ext = '1' else
                 geh0_active_ipv6_no_next when sig_active_ipv6_ext = '1' else
                 sig_active_ipv6_no_next;

   ACTIVE_UNKNOWN <= geh7_active_unknown when geh6_active_ipv6_ext = '1' else
                 geh6_active_unknown when geh5_active_ipv6_ext = '1' else
                 geh5_active_unknown when geh4_active_ipv6_ext = '1' else
                 geh4_active_unknown when geh3_active_ipv6_ext = '1' else
                 geh3_active_unknown when geh2_active_ipv6_ext = '1' else
                 geh2_active_unknown when geh1_active_ipv6_ext = '1' else
                 geh1_active_unknown when geh0_active_ipv6_ext = '1' else
                 geh0_active_unknown when sig_active_ipv6_ext = '1' else
                 sig_active_unknown;

   PROTOCOL <= geh7_protocol when geh6_active_ipv6_ext = '1' else
               geh6_protocol when geh5_active_ipv6_ext = '1' else
               geh5_protocol when geh4_active_ipv6_ext = '1' else
               geh4_protocol when geh3_active_ipv6_ext = '1' else
               geh3_protocol when geh2_active_ipv6_ext = '1' else
               geh2_protocol when geh1_active_ipv6_ext = '1' else
               geh1_protocol when geh0_active_ipv6_ext = '1' else
               geh0_protocol when sig_active_ipv6_ext = '1' else
               nexth;

   -- If none of extension headers reported Fragment Offset > 0
   FRAGMENT_FIRST <= geh7_frag_first and geh6_frag_first and
                     geh5_frag_first and geh4_frag_first and
                     geh3_frag_first and geh2_frag_first and
                     geh1_frag_first and geh0_frag_first;

   -- If none of extension headers reported More Fragments = 1
   FRAGMENT_LAST  <= geh7_frag_last and geh6_frag_last and
                     geh5_frag_last and geh4_frag_last and
                     geh3_frag_last and geh2_frag_last and
                     geh1_frag_last and geh0_frag_last;

end architecture behavioral;
