/*!
 * \file	ibufctl.c
 * \brief	Tool for controlling the input buffer for MII interface (IBUF)
 * \author	Andrej Hank <xhanka00@liberouter.org>
 * \author	Miroslav Vadkerti <xvadke010@liberouter.org>
 * \author	Peter Stranak <stranak@liberouter.org>
 * \date	2006-2011
 *
 * Copyright (C) 2006-2011 CESNET
 * LICENSE TERMS
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 * 3. Neither the name of the Company nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * ALTERNATIVELY, provided that this notice is retained in full, this
 * product may be distributed under the terms of the GNU General Public
 * License (GPL), in which case the provisions of the GPL apply INSTEAD
 * OF those given above.
 *
 * This software is provided ``as is'', and any express or implied
 * warranties, including, but not limited to, the implied warranties of
 * merchantability and fitness for a particular purpose are disclaimed.
 * In no event shall the company or contributors be liable for any
 * direct, indirect, incidental, special, exemplary, or consequential
 * damages (including, but not limited to, procurement of substitute
 * goods or services; loss of use, data, or profits; or business
 * interruption) however caused and on any theory of liability, whether
 * in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even
 * if advised of the possibility of such damage.
 *
 * $Id$
 *
 */

#include <err.h>
#include <unistd.h>

#include <combo.h>
#include <commlbr.h>

__RCSID ("$Id$");

/* Number of MAC addresses in old IBUF (version < 2.4) */
#define DEFAULT_MAC_COUNT 16

/* 2XFP2 flag */
bool xfp = false;

/* new XGMII_IBUF flag */
bool ibuf_old = false;
/* new link status register */
bool link_status = false;

/* 1G4 card flag */
bool c_1g4 = false;

/* Print etherStatsTable counters */
bool output_addr_space = true;
bool ether_stats = false;

/* Number of MAC addresses that can be placed into CAM */
bool has_mac_count = false;
static uint16_t mac_count = DEFAULT_MAC_COUNT;

/*!
 * \def	VERSION
 * \brief	File version
 */
#define VERSION "2.6"

/*
 * \var	verbose
 * \brief	Control verbosity
 */
extern int verbose;

/* \brief acceptable command line arguments */
#define ARGUMENTS	":a:b:d:e:i:l:m:s:u:x:v:AchMorVS"

/* EMAC component name */
#define EMAC_COMP_NAME	"EMAC"
/* EMAC Mode Configuration offset */
#define EMAC_CTRL_OFFS	0x100
/* EMAC Mode Configuration data for 10/100/1000 Mbps */
#define EMAC_SPEED_10	0x14000000
#define EMAC_SPEED_100	0x54000000
#define EMAC_SPEED_1000	0x94000000

/*! COMBO6_4SFPRO add on card name */
#define COMBO6_SFPRO	"sfpro"
/*! COMBO6_2XFP2 add on card name */
#define COMBO6_2XFP2	"xfp2"
/*! COMBOI_10G1 add on card name */
#define COMBOI_10G2	"10G1"
/*! COMBOI_10G2 add on card name */
#define COMBOI_10G4	"10G2"
/*! COMBOI_1G4 add on card name */
#define COMBOI_1G4	"1G4"
/*! UNKNOWN add on card name */
#define COMBO_UNKNOWN	"unknown"

/*!
 * \def	IBUF_COMP_NAME
 * \brief	IBUF component name
 */
#define IBUF_COMP_NAME	"IBUF"

/*!
 * \def	MAC_BUFSIZE
 * \brief	MAC buffer size
 */
#define MAC_BUFSIZE	20

/**/
#define IBUF_CTRL 0x002C

/* Total received packets with bad CRC (etherStatsCRCAlignErrors) */
#define ES_CRC_ERR_LO	0x000
#define ES_CRC_ERR_HI	0x038

/* Total received packets over set MTU (etherStatsPkts) */
#define ES_PKTS_LO		0x004
#define ES_PKTS_HI		0x03C

/* Total received packets below set minimal length (etherStatsUndersizePkts) */
#define ES_UNDERSIZE_LO	0x008
#define ES_UNDERSIZE_HI	0x040

/* Total received packets with broadcast address (etherStatsBroadcastPkts) */
#define ES_BCAST_LO		0x00C
#define ES_BCAST_HI		0x044

/* Total received packets with multicast address (etherStatsMulticastPkts) */
#define ES_MCAST_LO		0x010
#define ES_MCAST_HI		0x048

/* Total received fragment frames (etherStatsFragments) */
#define ES_FRAGMENTS_LO	0x014
#define ES_FRAGMENTS_HI	0x04C

/* Total received jabber frames (etherStatsJabbers) */
#define ES_JABBERS_LO	0x018
#define ES_JABBERS_HI	0x050

/* Total transfered octets (etherStatsOctets) */
#define ES_OCTETS_LO	0x01C
#define ES_OCTETS_HI	0x054

/* Total received frames with length 64B (etherStatsPkts64Octets) */
#define ES_FRAMES_64_LO	0x020
#define ES_FRAMES_64_HI 0x058

/* Total received frames with length from 65B to 127B (etherStatsPkts65to127Octets) */
#define ES_FRAMES_65_127_LO	0x024
#define ES_FRAMES_65_127_HI	0x05C

/* Total received frames with length from 128B to 255B (etherStatsPkts128to255Octets) */
#define ES_FRAMES_128_255_LO	0x028
#define ES_FRAMES_128_255_HI	0x060

/* Total received frames with length from 256B to 511B (etherStatsPkts256to511Octets) */
#define ES_FRAMES_256_511_LO	0x02C
#define ES_FRAMES_256_511_HI	0x064

/* Total received frames with length from 512B to 1023B (etherStatsPkts512to1023Octets) */
#define ES_FRAMES_512_1023_LO	0x030
#define ES_FRAMES_512_1023_HI	0x068

/* Total received frames with length from 1024B to 1518B (etherStatsPkts1024to1518Octets) */
#define ES_FRAMES_1024_1518_LO	0x034
#define ES_FRAMES_1024_1518_HI	0x06C


/*!
 * \enum	commands
 * \brief	List of available commands
 */
enum commands {
	CMD_ENABLE,
	CMD_RESET,
	CMD_SET_MASK,
	CMD_SET_SPEED,
	CMD_SET_MAX_LENGTH,
	CMD_SET_MIN_LENGTH,
	CMD_MAC_CHECK_MODE,
	CMD_SHOW_MACS,
	CMD_CLEAR_MACS,
	CMD_SET_MACS,
	CMD_PRINT_STATUS
};

/*!
 * \brief	Convert hex number to integer
 *
 * \param number	Converted hex number
 *
 * \return	converted value on success, -1 on error
 */
int
hex2num(char number)
{
	if (number >= '0' && number <= '9') {
		return number - '0';
	}
	if (number >= 'a' && number <= 'f') {
		return number - 'a' + 10;
	}
	if (number >= 'A' && number <= 'F') {
		return number - 'A' + 10;
	}
	return -1;
}

/*!
 * \brief	Convert HW address string (MAC) to 2 32bit integer.
 *		Lo contains lower 32 bits, hi higher 16 bits
 *		and validity bit (set to false if not used)
 *
 * \param txt	Converted MAC string (aa:bb:cc:dd:ee:ff)
 * \param addr	MAC address
 *		(hi -> higher 16 bits and optional validity bit,
 *		lo -> lower 32 bits of MAC)
 * \param valid	True -> valid bit 1, False -> valid bit 0
 *
 * \retval	0 on success
 * \retval	-1 on error
 */
int
str2hwaddr(char *txt, cs_ibuf_mac_t *addr, int valid)
{
	int byte, a, b;

	for (byte = 0; byte < 6; byte++) {
		a = hex2num(*txt++);
		if (a < 0) {
			return -1;
		}
		b = hex2num(*txt++);
		if (b < 0) {
			return -1;
		}
		if (byte > 1) {
			addr->lo = addr->lo << 8;
			addr->lo |= (a << 4) | b;
		} else {
			addr->hi = addr->hi << 8;
			addr->hi |= (a << 4) | b;
		}
		if (byte < 5 && *txt++ != ':') {
			return -1;
		}
	}

	/* Set validity bit */
	if (valid) {
		addr->hi += 0x10000;
	}

	return 0;
}

/*!
 * \brief 	Convert 48bit MAC address stored in lower and higher
 *		integer into MAC address string. Txt must be an > 18 byte array.
 *
 * \param txt	MAC string (aa:bb:cc:dd:ee:ff) from conversion
 * \param addr	MAC address
 *		(hi -> higher 16 bits and optional validity bit,
 *		lo -> lower 32 bits of MAC)
 * \param valid	Returns validity bit
 */
void
hwaddr2str(char *txt, cs_ibuf_mac_t *addr, int *valid)
{
	char str[MAC_BUFSIZE] = {0};	/* temp string */
	int i;	/* cycle variable */

	if (addr->hi > 0xFFFF) {
		addr->hi -= 0x10000;
		*valid = true;
	} else {
		*valid = false;
	}

	/* print */
	sprintf(str, "%04x%08x", addr->hi, addr->lo);
	VERBOSE(CL_VERBOSE_LIBRARY, "Extracted HW address: %s\n", str);

	/* insert colons */
	for (i = 0; i < 13; i++) {
		if (i % 2 == 0 && i != 0 && i != 12) {
			*txt++ = ':';
		}
		*txt++ = str[i];
	}
}

/*!
 * \brief       read MTU from desing.xml
 *
 * \param dev   COMBO device
 *
 * \retval	mtu from desing.xml
 * \retval	0 if no mtu found
 */
int
design_mtu(cs_device_t *dev, int j)
{
	cs_component_t *comp;
	float *mtu_ptr;
	int ret;

        ret = cs_component_find(dev, &comp, NULL, "IBUF", j);
        if ((ret < 0) || (comp == NULL)) {
                return 0;
        }

        /* preset IBUF MTU */
        mtu_ptr = cs_component_upriv_get(comp);
        if (mtu_ptr)
        	return (int)(*mtu_ptr);

	return 0;
}

#define BOARD_UNKNOWN 0
#define BOARD_FB8XG   1
/*!
 * \brief	Identify FPGA board.
 * \param path	Path to device file to use
 * \return		One of BOARD_? values.
 */
int
identify_board(char *path)
{
	int board_id = BOARD_UNKNOWN;
	char *board, *card, *chip;
	cs_device_t *device;
	if (cs_attach_noex(&device, path) == 0) {
		if (cs_identify(device, &board, &card, &chip) == 0) {
			if (strcmp(board,"FB8XG") == 0)
				board_id = BOARD_FB8XG;
		}
		cs_detach(&device);
	}
	return board_id;
}

/*!
 * \brief	Display usage of program
 */
void
usage(int board_id)
{
	printf("Usage: %s [-AchMorVS] [-a mode] [-b base_addr] [-d path] [-e 0|1]\n"
			"               [-i interface] [-l length] [-m mask] [-s speed]\n"
			"               [-u length] [-v level] [-x file]\n\n", getprogname());
	printf("Only one command may be used at a time.\n");
	printf("-a mode        Set MAC check mode\n");
	printf("-b base_addr   IBUF base address (hex)\n");
	printf("-d path        Path to device file to use\n");
	printf("-e 0|1         Enable(1)/disable(0) interface\n");
	printf("-i interface   Select interface (default 0)\n");
	printf("-l length      The minimal frame length allowed\n");
	printf("-m mask        Set error mask 0-31\n");
	if(board_id == BOARD_UNKNOWN)
		printf("-s speed       Set speed to 10|100|1000 Mb/s (usable only for 1 Gb/s copper\n"
		       "               interfaces)\n");
	printf("-u length      The maximal frame length allowed in bytes\n");
	printf("-x file        File design.xml used for component space mapping\n");
	printf("-v level       Verbose level (0-2)\n");
	printf("-A             Print status of all interfaces or affect all interfaces if used\n"
			"               with other option\n");
	printf("-c             Clear MAC list\n");
	printf("-h             Show this text\n");
	printf("-M             Read list of MAC addresses (max %d) with validity bit from stdin\n", CS_IBUF_MACLIST_SIZE);
	printf("-o             Show MAC list\n");
	printf("-r             Reset IBUF frame counters\n");
	printf("-V             Show version\n");
	printf("-S             Show etherStats counters\n");
}

/*!
 * \brief Print IBUF speed
 *
 * \param speed Speed flag in IBUF status register
 */
void
print_speed(int speed)
{
	switch (speed) {
	case 0x1:
		printf("100 Mb/s\n");
		break;
	case 0x2:
		printf("1 Gb/s\n");
		break;
	case 0x3:
		printf("10 Gb/s\n");
		break;
	case 0x4:
		printf("40 Gb/s\n");
		break;
	case 0x5:
		printf("100 Gb/s\n");
		break;
	default:
		printf("10 Mb/s\n");
		break;
	}
}

/*!
 * \brief Print IBUF etherStatsTable counters
 *
 * \param dev Device
 * \param space Space with mapped ibuf component
 */
void
print_ether_stats(cs_device_t *dev, cs_space_t *space)
{
	int offset = 0x100;

	if (output_addr_space) {
		offset = 0x0;

		/* switch output address space */
		cs_space_write_4(dev, space, IBUF_CTRL, 0x3);
	}

	printf("----------------------------- IBUF etherStatsTable ----\n");
//	printf("etherStatsDropEvents          : %llu\n")
	printf("etherStatsOctets              : %llu\n", ((long long unsigned)
		(cs_space_read_4(dev, space, ES_OCTETS_HI + offset)) << 32) + cs_space_read_4(dev, space, ES_OCTETS_LO + offset));

	printf("etherStatsPkts                : %llu\n", ((long long unsigned)
		(cs_space_read_4(dev, space, ES_PKTS_HI + offset)) << 32) + cs_space_read_4(dev, space, ES_PKTS_LO + offset));

	printf("etherStatsBroadcastPkts       : %llu\n", ((long long unsigned)
		(cs_space_read_4(dev, space, ES_BCAST_HI + offset)) << 32) + cs_space_read_4(dev, space, ES_BCAST_LO + offset));

	printf("etherStatsMulticastPkts       : %llu\n", ((long long unsigned)
		(cs_space_read_4(dev, space, ES_MCAST_HI + offset)) << 32) + cs_space_read_4(dev, space, ES_MCAST_LO + offset));

	printf("etherStatsCRCAlignErrors      : %llu\n", ((long long unsigned)
		(cs_space_read_4(dev, space, ES_CRC_ERR_HI + offset)) << 32) + cs_space_read_4(dev, space, ES_CRC_ERR_LO + offset));

	printf("etherStatsUndersizePkts       : %llu\n", ((long long unsigned)
		(cs_space_read_4(dev, space, ES_UNDERSIZE_HI + offset)) << 32) + cs_space_read_4(dev, space, ES_UNDERSIZE_LO + offset));

	//	printf("etherStatsOversizePkts        : %llu\n");

	printf("etherStatsFragments           : %llu\n", ((long long unsigned)
		(cs_space_read_4(dev, space, ES_FRAGMENTS_HI + offset)) << 32) + cs_space_read_4(dev, space, ES_FRAGMENTS_LO + offset));

	printf("etherStatsJabbers             : %llu\n", ((long long unsigned)
		(cs_space_read_4(dev, space, ES_JABBERS_HI + offset)) << 32) + cs_space_read_4(dev, space, ES_JABBERS_LO + offset));

//	printf("etherStatsCollisions          : %llu\n");

	printf("etherStatsPkts64Octets        : %llu\n", ((long long unsigned)
		(cs_space_read_4(dev, space, ES_FRAMES_64_HI + offset)) << 32) + cs_space_read_4(dev, space, ES_FRAMES_64_LO + offset));

	printf("etherStatsPkts65to127Octets   : %llu\n", ((long long unsigned)
		(cs_space_read_4(dev, space, ES_FRAMES_65_127_HI + offset)) << 32) + cs_space_read_4(dev, space, ES_FRAMES_65_127_LO + offset));

	printf("etherStatsPkts128to255Octets  : %llu\n", ((long long unsigned)
		(cs_space_read_4(dev, space, ES_FRAMES_128_255_HI + offset)) << 32) + cs_space_read_4(dev, space, ES_FRAMES_128_255_LO + offset));

	printf("etherStatsPkts256to511Octets  : %llu\n", ((long long unsigned)
		(cs_space_read_4(dev, space, ES_FRAMES_256_511_HI + offset)) << 32) + cs_space_read_4(dev, space, ES_FRAMES_256_511_LO + offset));

	printf("etherStatsPkts512to1023Octets : %llu\n", ((long long unsigned)
		(cs_space_read_4(dev, space, ES_FRAMES_512_1023_HI + offset)) << 32) + cs_space_read_4(dev, space, ES_FRAMES_512_1023_LO + offset));

	printf("etherStatsPkts1024to1518Octets: %llu\n", ((long long unsigned)
		(cs_space_read_4(dev, space, ES_FRAMES_1024_1518_HI + offset)) << 32) + cs_space_read_4(dev, space, ES_FRAMES_1024_1518_LO + offset));

	if (output_addr_space) {
		/* switch back from output address space */
		cs_space_write_4(dev, space, IBUF_CTRL, 0x3);
	}
}

/*!
 * \brief	Print IBUF counters
 *
 * \param data	Data to print in cs_ibuf_t structure
 * \param ifc	Printed interface
 */
void
print_ibuf(cs_ibuf_t data, int ifc)
{
	char*	text;

	printf("-------------------------------------- IBUF Status ----\n");
	/* IBUF was probably selected by its base adress */
	if (ifc < 0) {
		printf("Interface number           : unknown\n");
	} else {
		printf("Interface number           : %d\n", ifc);
	}
	printf("Link status                : %s \n",
				(data.ibuf_stat & 0x00000080) ? "ON" : "OFF");
	printf("IBUF                       : %s\n", (data.reg_en != 1) ? "DISABLED" : "ENABLED");

	VERBOSE(CL_VERBOSE_LIBRARY, "IBUF status: 0x%x", data.ibuf_stat);
	printf("HFIFO overflow occurred    : %s \n", (data.ibuf_stat & 0x00000001) ? "True" : "False");

	/* Speed is on bits 4-6 */
	printf("IBUF speed                 : ");
	print_speed((data.ibuf_stat & 0x70) >> 4);

	printf("------------------------ IBUF Packets/Frames Stats ----\n");
	printf("Packets                    : %llu\n", (long long unsigned) data.cnt_packets);
	printf("Received                   : %llu\n", (long long unsigned) data.cnt_recv);
	printf("Discarded packets          : %llu\n", (long long unsigned) data.cnt_error);
	printf("Buf overflow               : %llu\n", (long long unsigned) data.cnt_overflow);
	printf("Error packets              : %llu\n", (long long unsigned) (data.cnt_error - data.cnt_overflow));

	printf("------------------------------------ IBUF Settings ----\n");
	VERBOSE(CL_VERBOSE_LIBRARY, "Error mask: 0x%x", data.err_mask);
	printf("Frame error from MII  [1]  : %s",
			(data.err_mask & 0x00000001) ? "enabled\n" : "disabled\n");
	printf("CRC check             [2]  : %s",
			(data.err_mask & 0x00000002) ? "enabled\n" : "disabled\n");
	printf("Minimum frame length  [4]  : %s\n"
			"* length                   : %d B\n",
			(data.err_mask & 0x00000004) ? "enabled" : "disabled",
			data.ibuf_frame_min_len);
	printf("MTU frame length      [8]  : %s\n"
			"* length                   : %d B",
			(data.err_mask & 0x00000008) ? "enabled" : "disabled",
			data.ibuf_frame_mtu);
	if (~data.ibuf_frame_buffer_size == 0) {
		printf(" (max unlimited)\n");
	} else {
		printf(" (max %d B)\n", data.ibuf_frame_buffer_size);
	}

	/* get MAC mode string */
	if (data.ibuf_mac_check == 0x3) {
		text = "[0x3] MAC check, broadcast, multicast";
	} else if (data.ibuf_mac_check == 0x2) {
		text = "[0x2] MAC check, broadcast";
	} else if (data.ibuf_mac_check == 0x1) {
		text = "[0x1] MAC check";
	} else { /* 0x0 */
		text = "[0x0] promiscuous";
	}
	printf("MAC address check     [16] : %s\n"
			"* mode                     : %s\n",
			(data.err_mask & 0x00000010) ? "enabled" : "disabled", text);
	printf("Maximal MAC addresses      : %d\n", mac_count);
}

/*!
 * \brief	Print IBUF counters
 *
 * \param data		Data to print in cs_ibuf_t structure
 * \param emac_data	EMAC Mode Configuration data for speed info
 * \param ifc		Printed interface
 */
void
print_ibuf_xgmii(cs_ibuf_t data, u_int32_t emac_data, int ifc)
{
	char	*text;

	printf("-------------------------------------- IBUF Status ----\n");
	/* IBUF was probably selected by its base adress */
	if (ifc < 0) {
		printf("Interface number           : unknown\n");
	} else {
		printf("Interface number           : %d\n", ifc);
	}
	/* TODO: is link status supported by all IBUFs? */
	printf("Link status                : %s \n",
				(data.ibuf_stat & 0x00000080) ? "ON" : "OFF");
	printf("IBUF                       : %s\n", (data.reg_en != 1) ? "DISABLED" : "ENABLED");
	VERBOSE(CL_VERBOSE_LIBRARY, "IBUF status: 0x%x", data.ibuf_stat);
	printf("PACODAG overflow occurred  : %s \n",
			(data.ibuf_stat & 0x00000001) ? "True" : "False");
	printf("DFIFO overflow occurred    : %s \n",
			(data.ibuf_stat & 0x00000002) ? "True" : "False");

	printf("IBUF speed                 : ");

	if (c_1g4) {
		/* IBUF was probably selected by its base adress - in emac_data is a garbage*/
		if (ifc < 0) {
			printf("unknown\n");
		} else {
			/* Speed is set in two highests bits */
			print_speed(emac_data >> 30);
		}
	} else {
		print_speed((data.ibuf_stat & 0x70) >> 4);
	}

	printf("------------------------ IBUF Packets/Frames Stats ----\n");
	printf("Packets                    : %llu\n", (long long unsigned) data.cnt_packets);
	printf("Received                   : %llu\n", (long long unsigned) data.cnt_recv);
	printf("Discarded                  : %llu\n", (long long unsigned) data.cnt_error);
	printf("Buf overflow               : %llu\n", (long long unsigned) data.cnt_overflow);
	printf("Error packets              : %llu\n", (long long unsigned) (data.cnt_error - data.cnt_overflow));

	printf("------------------------------------ IBUF Settings ----\n");
	VERBOSE(CL_VERBOSE_LIBRARY, "Error mask: 0x%x", data.err_mask);
	printf("Frame error from MII  [1]  : %s",
			(data.err_mask & 0x00000001) ? "enabled\n" : "disabled\n");
	printf("CRC check             [2]  : %s",
			(data.err_mask & 0x00000002) ? "enabled\n" : "disabled\n");
	printf("Minimum frame length  [4]  : %s\n"
			"* length                   : %d B\n",
			(data.err_mask & 0x00000004) ? "enabled" : "disabled",
			data.ibuf_frame_min_len);
	printf("MTU frame length      [8]  : %s\n"
			"* length                   : %d B",
			(data.err_mask & 0x00000008) ? "enabled" : "disabled",
			data.ibuf_frame_mtu);
	if (~data.ibuf_frame_buffer_size == 0) {
		printf(" (max unlimited)\n");
	} else {
		printf(" (max %d B)\n", data.ibuf_frame_buffer_size);
	}

	/* get MAC mode string */
	if (data.ibuf_mac_check == 0x3) {
		text = "[0x3] MAC check, broadcast, multicast";
	} else if (data.ibuf_mac_check == 0x2) {
		text = "[0x2] MAC check, broadcast";
	} else if (data.ibuf_mac_check == 0x1) {
		text = "[0x1] MAC check";
	} else { /* 0x0 */
		text = "[0x0] promiscuous";
	}
	printf("MAC address check     [16] : %s\n"
			"* mode                     : %s\n",
			(data.err_mask & 0x00000010) ? "enabled" : "disabled", text);
	printf("Maximal MAC addresses      : %d\n", mac_count);
}

/*!
 * \brief	Print old IBUF counters
 *
 * \param dev		COMBO device
 * \param ibuf_space	COMBO space with mapped ibuf component
 * \param ifc		Printed interface
 */
void
print_ibuf_old_xgmii(cs_device_t *dev, cs_space_t *ibuf_space, int ifc)
{
	printf("-------------------------------------- IBUF Status ----\n");
	/* IBUF was probably selected by its base adress */
	if (ifc < 0) {
		printf("Interface number           : unknown\n");
	} else {
		printf("Interface number           : %d\n", ifc);
	}
	printf("IBUF                       : %s\n",
			(cs_space_read_4(dev, ibuf_space, 0x20) != 1) ? "DISABLED" : "ENABLED");
	printf("IBUF speed                 : 10 Gb/s\n");
	printf("------------------------ IBUF Packets/Frames Stats ----\n");
	printf("Packets                    : %u\n", cs_space_read_4(dev, ibuf_space, 0x0));
	printf("Received                   : %u\n", cs_space_read_4(dev, ibuf_space, 0x4));
	printf("Discarded packets          : %u\n", cs_space_read_4(dev, ibuf_space, 0x8));

	printf("------------------------------------ IBUF Settings ----\n");
	printf("Maximal packet length      : %d B\n", cs_space_read_4(dev, ibuf_space, 0x24));
}

/*!
 * \brief	Map IBUF (also EMAC, if necessary) component
 *
 * \param dev		COMBO device
 * \param ibuf_space	COMBO space with mapped IBUF component
 * \param emac_space	COMBO space with mapped EMAC component
 * \param emac_data	EMAC Mode Configuration data for speed info
 * \param ifc		Used interface
 *
 * \retval	0 on success
 * \retval	-1 on error
 * \retval	1 when there is no IBUF on selected interface
 */
int
map_ifc(cs_device_t *dev, cs_space_t **ibuf_space, cs_space_t **emac_space,
		u_int32_t *emac_data, int ifc)
{
	u_int32_t	cver;
	cs_component_t	*comp_id;

	/* find information about IBUF component */
	if (cs_component_find(dev, &comp_id, NULL, IBUF_COMP_NAME, ifc) != 0) {
		/* component not found */
		return 1;
	}

	/* map IBUF space */
	if (cs_component_space(comp_id, ibuf_space) != 0) {
		warnx("Failed to map IBUF component.");
		return -1;
	}

	/* on 2XFP2 cards detect IBUF component version */
	if (xfp == true) {
		cver = cs_component_version(comp_id);
		/* check for newer XGMII_IBUF's - version 2.1 and higher*/
		VERBOSE(CL_VERBOSE_BASIC, "IBUF_XGMII %u.%u detected",
				CS_CVER_MAJOR(cver), CS_CVER_MINOR(cver));
		if ((CS_CVER_MAJOR(cver) == 2 && CS_CVER_MINOR(cver) == 0) ||
				CS_CVER_MAJOR(cver) < 2) {
			ibuf_old = true;
		}
		/* link status reqister from version 2.3 */
		if ((CS_CVER_MAJOR(cver) == 2 && CS_CVER_MINOR(cver) > 2) ||
				CS_CVER_MAJOR(cver) > 2) {
			link_status = true;
		}
		
		/* MAC_COUNT register is from version 2.4 */
		if ((CS_CVER_MAJOR(cver) == 2 && CS_CVER_MINOR(cver) > 3) ||
				CS_CVER_MAJOR(cver) > 2) {
			has_mac_count = true;
		}
	}

	/* on 1G4 cards map EMAC for speed control and get EMAC Mode Configuration data */
	if (c_1g4 == true) {
		if (cs_component_find(dev, &comp_id, NULL, EMAC_COMP_NAME, ifc) != 0) {
			warnx("Component EMAC with index %d not found in your design.", ifc);
			return -1;
		}
		if (cs_component_space(comp_id, emac_space) != 0) {
			warnx("Failed to map EMAC component.");
			return -1;
		}
		*emac_data = (u_int32_t) cs_space_read_4(dev, *emac_space, EMAC_CTRL_OFFS);
	}
	return 0;
}

/*!
 * \brief	Load design
 *
 * \param dev		COMBO device
 * \param xmlfile	Path to file
 *
 * \retval	0 on success
 * \retval	-1 on errors
 */
int
load_design(cs_device_t *dev, char *xmlfile)
{
	int	xerr = 0;

	/* load design xml from file */
	if (xmlfile != NULL) {
		/* load data about design from specified design.xml */
		xerr = cs_design_reload(dev, CS_DESIGN_XML, xmlfile);
		if (xerr < 0) {
			warnx("Loading file %s failed : %s.", xmlfile, strerror(-xerr));
		}
	} else {
		/* load data about design from corresponding design.xml */
		xerr = cs_design_reload(dev, CS_DESIGN_NONE, NULL);
		if (xerr < 0) {
			warnx("Loading design.xml failed. Please check your installation.");
		}
	}
	return xerr;
}

/*!
 * \brief	Do requested operation on selected device
 *
 * \param command	Requested operation
 * \param param		Parameters for the operation
 * \param dev		COMBO device
 * \param ibuf_space	COMBO space with mapped IBUF component
 * \param emac_space	COMBO space with mapped EMAC component
 * \param data		IBUF data structure
 * \param emac_data	EMAC Mode Configuration data for speed info
 * \param ifc		Used interface
 *
 * \retval	0 on success
 * \retval	-1 on error
 */
int
execute_operation(enum commands command, int param, cs_device_t *dev,
		cs_space_t *ibuf_space, cs_space_t *emac_space, cs_ibuf_t data,
		u_int32_t emac_data, int ifc)
{
	int		i, valid_bit, ret;
	cs_phy_t	*phy;		/* phyter */
	char		mac_buffer[MAC_BUFSIZE];/* maximum buffer size */
	cs_ibuf_mac_t	mac_list[CS_IBUF_MACLIST_SIZE];	/* list for MAC addresses to get/set */

	/* Clean all addresses in list */
	memset(mac_list, 0, sizeof(mac_list));
	
	/* Get number of mac addresses stored in IBUF */
	if (has_mac_count) {
		mac_count = (uint16_t) ((data.ibuf_stat & 0xF800000) >> 23);
	}

	switch (command) {
		/* If enable/disable IBUF required */
		case CMD_ENABLE:
			if (param == 0) {
				VERBOSE(CL_VERBOSE_BASIC, "Disabling IBUF.");
				cs_ibuf_disable(dev, ibuf_space);
			} else {
				VERBOSE(CL_VERBOSE_BASIC, "Enabling IBUF.");
				cs_ibuf_enable(dev, ibuf_space);
			}
			break;

		/* If reset frame counters required */
		case CMD_RESET:
			VERBOSE(CL_VERBOSE_BASIC, "Resetting frame counters.");
			cs_ibuf_reset(dev, ibuf_space);
			break;

		/* If set error mask required */
		case CMD_SET_MASK:
			VERBOSE(CL_VERBOSE_BASIC, "Setting Error Mask to: %d\n", param);
			cs_ibuf_err_mask(dev, ibuf_space, param, data.reg_en);
			break;

		/* If set IBUF speed required */
		case CMD_SET_SPEED:
			VERBOSE(CL_VERBOSE_BASIC, "Setting Speed to: %d\n", param);
			/* if other than 1G4 card detected */
			if (!c_1g4) {
				warnx("You can change speed only on ComboI-1G4 cards.");
				return -1;
			}
			if (ifc < 0) {
				warnx("You can't address IBUF with '-b address'"
						" parameter while setting its speed on ComboI-1G4"
						" cards. Use parameter '-i interface' instead.");
				return -1;
			}
			/* attach phyter */
			phy = cs_phy_attach(dev, NULL, NULL, NULL, ifc);
			/* phy->dbytes == 1	--> SFP		--> Optical
			   else			--> GBIC	--> Copper
			   according to phyterctl
			*/
			if (phy != NULL) {
				if (phy->dbytes != 1) {
					/* advertise speed in phyter */
					ret = phy->phy_service(phy, CS_PHY_ADVSPEED, (u_int32_t*)&param, NULL, NULL);
					if (ret != 0) {
						cs_phy_detach(phy);
						warnx("Unsupported operation for interface %d.", ifc);
						return -1;
					}

					/* advertise speed in EMAC */
					if (param == 1000) {
						cs_space_write_4(dev, emac_space, EMAC_CTRL_OFFS, EMAC_SPEED_1000);
					} else if (param == 100) {
						cs_space_write_4(dev, emac_space, EMAC_CTRL_OFFS, EMAC_SPEED_100);
					} else if (param == 10) {
						cs_space_write_4(dev, emac_space, EMAC_CTRL_OFFS, EMAC_SPEED_10);
					}
				} else {
					printf("Optical phyter found on interface %d. Skipping ...\n", ifc);
				}
				cs_phy_detach(phy);
			} else {
				printf("No compatible phyter found on interface %d\n", ifc);
			}
			break;

		/* If maximal frame length specified */
		case CMD_SET_MAX_LENGTH:
			VERBOSE(CL_VERBOSE_BASIC, "Setting frame MTU to: %d\n", param);
			if (data.ibuf_frame_min_len > (unsigned)param) {
				warnx("Maximal frame length can't be smaller then minimal frame length.");
				return -1;
			} else if ((data.ibuf_frame_buffer_size != (unsigned)~0) &&
					(data.ibuf_frame_buffer_size < (unsigned)param)) {
				warnx("Maximal frame length can't be larger then frame buffer size (%u B).",
					data.ibuf_frame_buffer_size);
				return -1;
			}
			cs_ibuf_frame_length(dev, ibuf_space, param, 1, data.reg_en);
			break;

		/* If minimal frame length specified*/
		case CMD_SET_MIN_LENGTH:
			VERBOSE(CL_VERBOSE_BASIC, "Setting minimal frame length to: %d\n", param);
			if (data.ibuf_frame_mtu < (unsigned)param) {
				warnx("Minimal frame length can't be larger then maximal frame length.");
				return -1;
			}
			cs_ibuf_frame_length(dev, ibuf_space, param, 0, data.reg_en);
			break;

		/* If MAC check mode change required */
		case CMD_MAC_CHECK_MODE:
			VERBOSE(CL_VERBOSE_BASIC, "Setting MAC check mode to: %x\n",
					(unsigned)param);
			cs_ibuf_mac_check(dev, ibuf_space, (u_int32_t)param, data.reg_en);
			break;

		/* If MAC list print required */
		case CMD_SHOW_MACS:
			VERBOSE(CL_VERBOSE_BASIC, "Querying for MAC list");
			/* read MAC addresses */
			cs_ibuf_mac_get_list(dev, ibuf_space, mac_list, data.reg_en, mac_count);
			/* print all MAC addresses */
			printf("IBUF MAC address list: \n");
			for (i = 0; i < mac_count; i++) {
				/* convert to string */
				hwaddr2str(mac_buffer, &mac_list[i], &valid_bit);
				printf("  %2d. %s  valid: %s\n", i + 1, mac_buffer, (valid_bit) ? "true" : "false");
			}
			break;

		/* If clear MAC list required */
		case CMD_CLEAR_MACS:
			VERBOSE(CL_VERBOSE_BASIC, "Clearing MAC list");
			cs_ibuf_mac_set_list (dev, ibuf_space, mac_list, data.reg_en, mac_count);
			break;

		/* If MAC list specified form stdin */
		case CMD_SET_MACS:
			printf("Enter list of max %d MAC addresses. After the last address "
					"send EOF.\nPlease use the format \"aa:bb:cc:dd:ee:ff x\" "
					"where x is the valid flag (0-1):\n", mac_count);
			for (i = 0; i < mac_count; i++) {
				ret = scanf("%17s %d", mac_buffer, &valid_bit);
				VERBOSE(CL_VERBOSE_LIBRARY, "Exit scanf ret=%d", ret);
				if (ret == EOF) {
					break;
				}
				if (valid_bit != 0 && valid_bit != 1) {
					warnx("Bad valid flag.");
					return -1;
				}
				if (str2hwaddr(mac_buffer, &mac_list[i], valid_bit) == -1) {
					warnx("Bad MAC address %s.", mac_buffer);
					return -1;
				}
				VERBOSE(CL_VERBOSE_BASIC, "Read MAC string %s > HI: 0x%x  LO: 0x%x.",
						mac_buffer, mac_list[i].hi, mac_list[i].lo);
			}
			cs_ibuf_mac_set_list (dev, ibuf_space, mac_list, data.reg_en, mac_count);
			break;

		/* If print IBUF status required */
		case CMD_PRINT_STATUS:
			if (xfp && ibuf_old) {
				print_ibuf_old_xgmii(dev, ibuf_space, ifc);
			} else if (xfp) {
				print_ibuf_xgmii(data, emac_data, ifc);
			} else {
				print_ibuf(data, ifc);
			}

			if (ether_stats) {
				print_ether_stats(dev, ibuf_space);
			}
			break;

		default:
			break;
	}
	return 0;
}

/*!
 * \brief	Detach device (also unload design) and exit program
 *
 * \param ret	Exit value
 * \param dev	COMBO device
 */
void
detach_and_exit(int ret, cs_device_t *dev)
{
	cs_design_free(dev);
	cs_detach(&dev);
	exit(ret);
}

/*!
 * \brief	Program main function
 *
 * \param argc		number of arguments
 * \param *argv[]	array with the arguments
 *
 * \retval	EXIT_SUCCESS on success
 * \retval	EXIT_FAILURE on error
 */
int
main(int argc, char *argv[])
{
	cs_device_t	*dev;			/* COMBO device */
	cs_space_t	*ibuf_space;		/* IBUF component space */
	cs_space_t	*emac_space;		/* EMAC component space */
	u_int32_t	base_address = 0;	/* component address in design */
	cs_ibuf_t	data;			/* IBUF data structure */
	u_int32_t	emac_data = 0;		/* EMAC Mode Configuration data */
	long		ifc = 0;		/* interface number */
	char		*file = CS_PATH_DEV(0);	/* default COMBO device */
	char		*xmlfile = NULL;	/* design.xml file with path */
	int		c;			/* temp variable */
	long		tmp;			/* temp variable */
	char		*card;			/* cs_identify param */
	bool		all = false;		/* indicates to affect all ifaces */
	bool		help = false;
	char		cmds = 0;		/* how many operations requested */
	char		interfaces = 0;
	enum commands	command = CMD_PRINT_STATUS;
	long		param = 0;
	int			mtu = 0;
	u_int32_t	ifcs = 0;

	/* process command line arguments */
	opterr = 0;
	while ((c = getopt(argc, argv, ARGUMENTS)) != -1) {
		switch(c) {
			case 'a':
				if (cl_strtol(optarg, &param) ||
						param < 0 || param > 3) {
					errx(EXIT_FAILURE, "Wrong MAC check mode. Must be <0, 3>.");
				}
				command = CMD_MAC_CHECK_MODE;
				cmds++;
				break;
			case 'b':
				if (cl_strtol(optarg, &tmp) ||
						tmp <= 0) {
					errx(EXIT_FAILURE, "Wrong base address.");
				}
				base_address = tmp;
				interfaces++;
				break;
			case 'd':
				file = optarg;
				break;
			case 'e':
				if (cl_strtol(optarg, &param) ||
						(param != 0 && param != 1)) {
					errx(EXIT_FAILURE, "Wrong enable value (0|1).");
				}
				command = CMD_ENABLE;
				cmds++;
				break;
			case 'i':
				if (cl_strtol(optarg, &ifc) || ifc < 0) {
					errx(EXIT_FAILURE, "Wrong interface number.");
				}
				interfaces++;
				break;
			case 'l':
				if (cl_strtol(optarg, &param) ||
						param <= 0) {
					errx(EXIT_FAILURE, "Wrong minimal frame length.");
				}
				command = CMD_SET_MIN_LENGTH;
				cmds++;
				break;
			case 'm':
				if (cl_strtol(optarg, &param) ||
						param < 0 || param > 31) {
					errx(EXIT_FAILURE, "Wrong error mask.");
				}
				command = CMD_SET_MASK;
				cmds++;
				break;
			case 's':
				if (cl_strtol(optarg, &param) ||
						(param != 10 && param != 100 && param != 1000)) {
					errx(EXIT_FAILURE, "Wrong speed. Must be one of 10|100|1000.");
				}
				command = CMD_SET_SPEED;
				cmds++;
				break;
			case 'u':
				if (cl_strtol(optarg, &param) ||
						param <= 0) {
					errx(EXIT_FAILURE, "Wrong maximal frame length.");
				}
				command = CMD_SET_MAX_LENGTH;
				cmds++;
				break;
			case 'v':
				if (cl_strtol(optarg, &tmp) ||
						tmp < 0) {
					errx(EXIT_FAILURE, "Invalid verbosity level attribute. Please specify"
							" number between 0 and 2.");
				}
				verbose = (tmp <= 2)? tmp : 2;
				break;
			case 'x':
				xmlfile = optarg;
				break;
			case 'A':
				all = true;
				interfaces++;
				break;
			case 'c':
				command = CMD_CLEAR_MACS;
				cmds++;
				break;
			case 'h':
				help = true;
				break;
			case 'M':
				command = CMD_SET_MACS;
				cmds++;
				break;
			case 'o':
				command = CMD_SHOW_MACS;
				cmds++;
				break;
			case 'r':
				command = CMD_RESET;
				cmds++;
				break;
			case 'V':
				printf("IBUF control tool - version %s\n", VERSION);
				return EXIT_SUCCESS;
			case 'S':
				ether_stats = true;
				break;
			case '?':
				errx(EXIT_FAILURE, "Unknown argument '%c'", optopt);
			case ':':
				errx(EXIT_FAILURE, "Missing parameter for argument '%c'", optopt);
			default:
				errx(EXIT_FAILURE, "Unknown error");
		}
	}
	argc -= optind;
	argv += optind;

	if (argc != 0) {
		errx(EXIT_FAILURE, "stray arguments");
	}

	if (help) {
		usage(identify_board(file));
		return EXIT_SUCCESS;
	}

	if (cmds > 1) {
		errx(EXIT_FAILURE, "More than one operation requested. Please select just one.");
	}

	if (interfaces > 1) {
		errx(EXIT_FAILURE, "Combination of parameters '-b', '-A', '-i' detected. "
				"Please don't combine them.");
	}

	/* attach device and map address spaces */
	if (cs_attach_noex(&dev, file) != 0) {
		err(EXIT_FAILURE, "cs_attach_noex failed");
	}

	/* find information about component (depending on Combo card) */
	// back compatibility to very old cards removed!!!
	xfp = true;
	/* if (cs_identify(dev, NULL, &card, NULL) != 0) {
		warnx("cs_identify() failed");
		detach_and_exit(EXIT_FAILURE, dev);
	}
	VERBOSE(CL_VERBOSE_BASIC, "Addon: %s", card);
	if (!strncmp(card, COMBOI_1G4, sizeof(COMBOI_1G4) - 1)) {
		xfp = true;
		c_1g4 = true;
	} else if (!strncmp(card, COMBOI_10G2, sizeof(COMBOI_10G2) - 1)) {
		xfp = true;
	} else if (!strncmp(card, COMBOI_10G4, sizeof(COMBOI_10G4) - 1)) {
		xfp = true;
	} else if (!strncmp(card, COMBO6_2XFP2, sizeof(COMBO6_2XFP2) - 1) ||
			!strncmp(card, COMBO_UNKNOWN, sizeof(COMBO_UNKNOWN) - 1)) {
		xfp = true;
	} */

	/* if all interfaces requested */
	if (all) {
		if (load_design(dev, xmlfile) < 0) {
			detach_and_exit(EXIT_FAILURE, dev);
		}

		if (cs_component_number(dev, NULL, IBUF_COMP_NAME, &ifcs) != 0) {
			warnx("cs_component_number() failed");
			detach_and_exit(EXIT_FAILURE, dev);
		}

		/* go through all IBUFs */
		for (ifc = 0; ifc < ifcs; ifc++) {
			tmp = map_ifc(dev, &ibuf_space, &emac_space, &emac_data, ifc);
			if (tmp == 1) {
				continue;
			} else if (tmp < 0) {
				detach_and_exit(EXIT_FAILURE, dev);
			}

			/* read all data in IBUF registers */
			data = cs_ibuf_read(dev, ibuf_space, xfp);

			mtu = design_mtu(dev, ifc);
                        if(mtu!=0){
                                data.ibuf_frame_buffer_size = mtu;
			}

			if (execute_operation(command, param, dev, ibuf_space, emac_space, data, emac_data, ifc) != 0) {
				detach_and_exit(EXIT_FAILURE, dev);
			}

			if (command == CMD_PRINT_STATUS) {
				printf("\n");
			}
		}
	} else {
		if (base_address != 0) {
			/* Map space of the component specified by base address */
			VERBOSE(CL_VERBOSE_BASIC, "Mapping space at base 0x%x", base_address);
			if (cs_space_map(dev, &ibuf_space, CS_SPACE_FPGA, CS_MAP_ALL, base_address, 0) != 0) {
				warnx("cs_space_map failed to map ibuf_space");
				detach_and_exit(EXIT_FAILURE, dev);
			}
			/* Interface number is now invalid */
			ifc = -1;
		} else {
			/* Map space of the component specified by design */
			if (load_design(dev, xmlfile) < 0) {
				detach_and_exit(EXIT_FAILURE, dev);
			}

			if (map_ifc(dev, &ibuf_space, &emac_space, &emac_data, ifc) != 0) {
				warnx("Component IBUF with index %ld not found in your design.", ifc);
				detach_and_exit(EXIT_FAILURE, dev);
			}
		}

		/* read all data in IBUF registers */
		data = cs_ibuf_read(dev, ibuf_space, xfp);

                mtu = design_mtu(dev, ifc);
                if(mtu!=0){
                        data.ibuf_frame_buffer_size = mtu;
		}

		if (execute_operation(command, param, dev, ibuf_space, emac_space, data, emac_data, ifc) != 0) {
			if (base_address != 0) {
				cs_space_unmap(dev, &ibuf_space);
			}
			detach_and_exit(EXIT_FAILURE, dev);
		}
	}

	if (base_address != 0) {
		cs_space_unmap(dev, &ibuf_space);
	}
	detach_and_exit(EXIT_SUCCESS, dev);
	return EXIT_SUCCESS;
}
