/*
 * test_bus.c: Test basic bus access methods in libcombo
 * Copyright (C) 2003-2008 CESNET
 * Author(s): Jachym Holecek <freza@liberouter.org>
 *
 * LICENSE TERMS
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 * 3. Neither the name of the Company nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * ALTERNATIVELY, provided that this notice is retained in full, this
 * product may be distributed under the terms of the GNU General Public
 * License (GPL) version 2 or later, in which case the provisions
 * of the GPL apply INSTEAD OF those given above.
 *
 * This software is provided ``as is'', and any express or implied
 * warranties, including, but not limited to, the implied warranties of
 * merchantability and fitness for a particular purpose are disclaimed.
 * In no event shall the company or contributors be liable for any
 * direct, indirect, incidental, special, exemplary, or consequential
 * damages (including, but not limited to, procurement of substitute
 * goods or services; loss of use, data, or profits; or business
 * interruption) however caused and on any theory of liability, whether
 * in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even
 * if advised of the possibility of such damage.
 *
 * $Id$
 *
 * This program offers simple peek/poke functionality to every
 * supported space. Always generates 32bit bus access.
 *
 */

#include <err.h>
#include <stdio.h>
#include <stdint.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <unistd.h>
#include <errno.h>

#include <combo.h>
#include <commlbr.h>

/* CVS ID tag for ident(1) program */
__RCSID("$Id$");

/*
 * Match space by name.
 */

struct {
	cs_target_t 	code;
	const char 	*name;
} known_spaces[] = {
	{ CS_SPACE_CPLD, 	"cpld" 		},
	{ CS_SPACE_FPGA, 	"fpga" 		},
	{ CS_SPACE_PLX, 	"plx" 		},	/* only alias for br */
	{ CS_SPACE_BRIDGE, 	"br" 		},
	{ CS_SPACE_PLX_EEPROM, 	"plxee"		},
	{ CS_SPACE_PCI_CONF, 	"pciconf" 	},
	{ CS_SPACE_NONE, 	NULL 		},
};

int
space_by_name(cs_target_t *t, const char *name)
{
	int 			i;

	for (i = 0; known_spaces[i].code != CS_SPACE_NONE; i++)
		if (strcasecmp(name, known_spaces[i].name) == 0) {
			*t = known_spaces[i].code;
			return 0;
		}

	return 1;
}

void
space_list()
{
	int 			i;

	for (i = 0; known_spaces[i].code != CS_SPACE_NONE; i++)
		printf("%s\n", known_spaces[i].name);
}

/*
 * Make sure user does not provide garbage.
 */
unsigned long int
xstrtoul(const char * str, int base)
{
	char 			*end;
	unsigned long int 	val;

	val = strtoul(str, &end, base);
	if (*end != '\0' || str[0] == '\0')
		errx(1, "%s: not a number of base %d", str, base);

	return val;
}

/*
 * Main.
 */

#define ARGUMENTS 		"ac:d:fhils:w:W"

void
usage(const char *me)
{
	printf("Usage: %s [-ahilW] [-c n] [-d s] [-s s] [-w x] addr [val]\n", me);
        printf("-a     Print address\n");
	printf("-c n   Read 'n' (dec) 32bit values (default = 1)\n");
	printf("-d s   Set device file path to 's'\n");
	printf("-f     Force write even if device is used by another program\n");
	printf("-h     Show this text\n");
	printf("-i     Print index\n");
	printf("-l     List known spaces\n");
	printf("-s s   Select space, 'fpga' is default\n");
	printf("-w x   Write value 'x' (hex), read otherwise\n");
	printf("-W     Write 4-byte values read from stdin\n");
	printf("addr   Hexadecimal offset to selected space\n");
	printf("val    Write value 'val' (hex), same as -w val\n");
}

int
main(int argc, char *argv[])
{
	cs_device_t 		*d;
	cs_space_t 		*s;
	cs_target_t 		space;
	cs_size_t 		mapsize;
	u_int32_t 		data, offs;
	int 			do_write, count;
	int 			do_write_ff;
	int			force_write;
	int 			c, i, address, index;
	char 			*file;

	do_write 	= 0;
	count 		= 1;
	mapsize 	= 4;
	do_write_ff 	= 0;
	force_write	= 0;
	address		= 0;
	index		= 0;
	space 		= CS_SPACE_FPGA;
	file 		= CS_PATH_DEV(0);


	while((c = getopt(argc, argv, ARGUMENTS)) != -1) {
		switch (c) {
		case 'a':
			address = 1;
			break;

		case 'c': 		/* Set read count 	*/
			count = xstrtoul(optarg, 10);
			if (count < 0)
				errx(1, "invalid count");
			break;

		case 'd': 		/* Select device 	*/
			file = optarg;
			break;

                case 'f':		/* Force write		*/
                        force_write = 1;
                        break;

		case 'h': 		/* Help text 		*/
			usage(argv[0]);
			return 0;

		case 'i':
			index = 1;
			break;

		case 'l': 		/* List spaces 		*/
			space_list();
			return 0;

		case 's': 		/* Select space 	*/
			if (space_by_name(&space, optarg) != 0)
				errx(1, "%s: no such space", optarg);
			break;

		case 'w': 		/* Write value 		*/
			data = xstrtoul(optarg, 16);
			do_write = 1;
			break;

		case 'W': 		/* Write values from file */
			do_write = 1;
			do_write_ff = 1;
			break;

		default:
			errx(1, "unknown option -%c", (char) optopt);
			return 1;
		}
	}

	argc -= optind;
	argv += optind;

	if (argc == 0)
		errx(1, "address missing");
	if (argc > 2)
		errx(1, "stray arguments");

	offs = xstrtoul(argv[0], 16);

	if (argc == 2) {
		if (do_write)
			errx(1, "inconsistent usage");

		data = xstrtoul(argv[1], 16);
		do_write = 1;
	}

	if (cs_attach(&d, file) != 0) {
	        if (errno == EBUSY && (!do_write || force_write)) {
	                if (cs_attach_noex(&d, file) != 0)
                		err(1, "cs_attach_noex failed");
                } else {
                	err(1, "cs_attach failed");
                }
        }

	/*
	 * XXX: extend to read from file. Add MAP_ALL as a special-case
	 * XXX: size for cs_space_map.
	 */
	if (do_write_ff)
		mapsize = CS_MAP_ALL;
	else
		mapsize = count*4;

	if (cs_space_map(d, &s, space, mapsize, offs, 0) != 0)
		errx(1, "cs_space_map failed");

	/*
	 * Do the actual work.
	 */

	if (do_write) {
		if (do_write_ff) {
			u_int32_t 	xoffs = 0;
			int		c;

			while (! feof(stdin)) {
				do {
					c = getc(stdin);
				} while (isspace(c));
				if (feof(stdin))
					break;
				ungetc(c, stdin);
				if (fscanf(stdin, "%08x", &data) != 1 && \
				    feof(stdin) == 0) {
					warnx("invalid input token");
					goto fail;
				}
				printf("%08x%c", data, \
				    ((xoffs + 4) % 32 == 0) ? '\n' : ' ');
				fflush(stdout);
				cs_space_write_4(d, s, xoffs, data);
				xoffs += 4;
			}
			printf("\n");
		} else {
			cs_space_write_4(d, s, 0, data);
		}
	} else {
		for (i = 0; i < count; i++) {
			if (address && (i % 4) == 0)
				printf("%08x: ", offs + i*4); 
			else if (index && (i % 8) == 0)
				printf("%02x: ", i);
			data = (u_int32_t) cs_space_read_4(d, s, i*4);
			printf("%08x%c", data, \
			    (((i + 1) % (address ? 4 : 8)) == 0 || i == (count - 1) ? \
			     '\n' : ' '));
		}
	}

fail:
	cs_space_unmap(d, &s);
	cs_detach(&d);
	return 0;
}
