/*!
 */
#include <stdlib.h>
#include <stdio.h>
#include <libsze2.h>
#include <time.h>

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <err.h>
#include <string.h>
#include <unistd.h>
#include <signal.h>

#include "compat.c"
#include "afilterctl_input.h"
#include "afilterctl_main.h"


#define DTREE_PATH "/proc/device-tree"
#define INTERFACES    2

#define UNKNOWN "unknown"
#define PROBEIDFILE "probeid.txt"
#define INIT_VECTORS [124, 222, 233]
#define BUFFER 10000
#define DEFAULTPROBEID 54321
#define FILTERSIZE 65536
#define MASKSID2RID 0xFFFF


char server[1024]="127.0.0.1";
char port[10]="22220";
char probeidfile[1024]=PROBEIDFILE;
char designxmlfile[1024]=DTREE_PATH;
uint32_t probeid=DEFAULTPROBEID;
uint32_t ridfull[FILTERSIZE]={0};


/// initialize communication with HW filter
int filter_init(struct hwio_dev *dev, struct hwio_comp **comp, const char *fn) {
     if(hwio_init(dev,NULL,fn)!=HWIO_EOK) {
        fprintf(stderr,"Unable to initialize device!\n");
        return EXIT_FAILURE;
    }
    struct hwio_comp_spec comp_compat_tab[]={
        {"ug4-150", "axi_advanced_filter", HWIO_VER3(1, 0, HWIO_VERSION_NA)},
        {"combov2", "Advanced_filter",     HWIO_VER2(1, 0)},
        {"debug",   "Advanced_filter",     HWIO_VER2(1, 0)},
        HWIO_COMPAT_END 
    };
    if(hwio_byspec(dev,comp_compat_tab,comp)!=HWIO_EOK || *comp==NULL) {
        fprintf(stderr,"Unable to find advanced filter inside device!\n");
        hwio_deinit(dev);
        return EXIT_FAILURE;
    }
    return EXIT_SUCCESS;
}

/// deinitialize communication with HW filter
void filter_deinit(struct hwio_dev *dev, struct hwio_comp *comp) {
    hwio_comp_free(comp);
    hwio_deinit(dev);
}

void print_help(void)
{
        printf("afilternetctl\n");
        printf("=========\n");
        printf("Implements CCCI\n");
        printf("Connects to MF, waits for Probe ID and answers with designated Probe ID\n");
        printf("and it awaits requests from MF and resposds with status of performed operations\n");
        printf("\n");
        printf("USAGE ./afilternetctl ARGUMENTS\n");
        printf("    -h             Print this help\n");
        printf("    -s IPADDRESS   IP address of MF server\n");
        printf("    -p NUMBER      Destination port number\n");
        printf("    -x FILE        design.xml file\n");
        printf("    -i ID          Probe ID\n");
        printf("    -o FILE        Output file with final Probe ID\n");
    exit(EXIT_SUCCESS);
}

int arguments(int argc, char *argv[])
{
    // initial settings
    char opt;

    // argument parsing loop
    while ((opt = getopt(argc, argv, "hs:p:i:o:x:")) != -1) {
        switch (opt) {
        case 'h':  // help
            if (argc != 2)
                errx(1, "write(): Buffer written just partially");
            print_help();
        case 's':  // Server
            strcpy(server, optarg);
            break;
        case 'p':  // port
            strcpy(port, optarg);
            break;
        case 'o':  // probe id file
            strcpy(probeidfile, optarg);
            break;
        case 'x':  // probe id file
            strcpy(designxmlfile, optarg);
            break;
         case 'i':  // probe id file
            probeid = strtoul(optarg, NULL, 0);
            break;

        default:  // invalid arguments
            errx(1,"invalid arguments");
        }
    }
    return 1;
}

void signal_callback_handler(int signum) {

   if (signum == SIGINT) {
	  exit(signum);
   }
}


void sockinfo(char *fmt, int fd)
{
	union {
	    struct sockaddr_in sin;
	    struct sockaddr_in6 sin6;
	    struct sockaddr sa;
	} socket;
	union {
	    char in[INET_ADDRSTRLEN];
	    char in6[INET6_ADDRSTRLEN];
	    char a[sizeof(UNKNOWN)];
	} addr[2];
        unsigned int port[2], i, length;

	for (i = 0; i < 2; i++) {
		length = sizeof(socket);
		if (((i == 0) ? getsockname : getpeername)(fd,
		    (struct sockaddr *)&socket, &length) == -1)
			err(1, (i == 0) ? "getsockname()" : "getpeername()");
		switch (socket.sa.sa_family) {
		case AF_INET:
			if (inet_ntop(AF_INET, &socket.sin.sin_addr,
			    addr[i].in, sizeof(addr[i].in)) == NULL)
				err(1, "inet_ntop()");
			port[i] = ntohs(socket.sin.sin_port);
			break;
		case AF_INET6:
			if (inet_ntop(AF_INET6, &socket.sin6.sin6_addr,
			    addr[i].in6, sizeof(addr[i].in6)) == NULL)
				err(1, "inet_ntop()");
			port[i] = ntohs(socket.sin6.sin6_port);
			break;
		default:
			strlcpy(addr[i].a, UNKNOWN, sizeof(addr[i].a));
			port[i] = 0;
			break;
		}
	}
	printf(fmt, addr[0].a, port[0], addr[1].a, port[1]);
}

uint32_t findemptyrid(uint32_t * ridfull) {
   static uint32_t ridpointer = 0;
   while (ridfull[ridpointer]) {
      ridpointer = (ridpointer + 1) % FILTERSIZE;
   }
   return ridpointer;
}

void service(struct hwio_comp *comp, int servicefd)
{
   char netbuf[BUFFER];
   net_rule_t netrule;
   unsigned int n;
   unsigned int act_bytes = 0;
   unsigned int r = 0;

   while ((n = read(servicefd, netbuf+act_bytes, CCCIREQUESTSIZE - act_bytes)) > 0) {
      act_bytes += n;
      if (act_bytes >= CCCIREQUESTSIZE)
         break;
   }
   if (act_bytes != CCCIREQUESTSIZE)
      errx(1, "service(): Actual CCCI REQUEST size of %d does not match expected size of %d", act_bytes, CCCIREQUESTSIZE);
   parse_netbuf2netrule(netbuf, &netrule);

   netrule.status = ACTIONSTATUSFAIL;

   switch  (netrule.action) {
      case NETACTIONDELETE:
         netrule.frule.id = MASKSID2RID & netrule.sid;
         if (DEL_RULE_SUCCESS == flt_del_rule(comp,&(netrule.frule))) {
            netrule.status = ACTIONSTATUSSUCCESS;
            ridfull[netrule.frule.id] = 0;
         }
         break;
      case NETACTIONADD:
         netrule.frule.id = findemptyrid(ridfull);
         if (ADD_RULE_SUCCESS == flt_add_rule(comp,&(netrule.frule))) {
            netrule.status = ACTIONSTATUSSUCCESS;
            ridfull[netrule.frule.id] = 1;
         }
         break;
      default:  // invalid arguments
         errx(1,"invalid arguments");
         break;
   }

   parse_netrule2netbuf(&netrule, netbuf);

   r = write(servicefd, (unsigned char *) netbuf, CCCIRESPONSESIZE);
   if (r == -1) err(1, "write() failed");
   if (r != CCCIRESPONSESIZE) errx(1, "write(): Buffer written just partially");

}


int main(int argc, char **argv) {

    ccci_header_t ccci_header;
    struct hwio_dev dev;
    struct hwio_comp *comp=NULL;
    int connectfd;
    int gai_error;
    struct addrinfo hints, *res, *res0;

    if (!arguments(argc, argv))
        return EXIT_FAILURE;

    signal_callback_handler(SIGINT);
    signal(SIGINT, signal_callback_handler);

    memset(&hints, 0, sizeof(hints));
    hints.ai_family = AF_UNSPEC;
    hints.ai_socktype = SOCK_STREAM;
    if ((gai_error = getaddrinfo(server, port, &hints, &res0)) != 0)
		errx(1, "getaddrinfo(): %s", gai_strerror(gai_error));
    connectfd = -1;
    for (res = res0; res != NULL; res = res->ai_next) {
		printf("socket(...)\n");
		if ((connectfd = socket(res->ai_family, res->ai_socktype,
		    res->ai_protocol)) == -1) {
			warn("socket()");
			connectfd = -1;
			continue;
		}
		printf("connect(...)\n");
		if (connect(connectfd, res->ai_addr, res->ai_addrlen) == -1) {
			close(connectfd);
			connectfd = -1;
			warn("connect()");
		} else
			break;
    }
    if (connectfd < 0)
       errx(1, "getaddrinfo(): Interface not found");

    sockinfo("service(...): %s/%d -> %s/%d\n", connectfd);

    /* Init hw filter */
    filter_init(&dev, &comp, designxmlfile);

    // Wait for probe ID
    receive_ccci_header(connectfd, &ccci_header);

    // Respond with probe ID
    if (ccci_header.version != SUPPORTED_CCCI_VERSION) {
       ccci_header.version = SUPPORTED_CCCI_VERSION;
       send_ccci_header(connectfd,&ccci_header);
	   errx(1, "CCCI_VERSION not supported");
    }

    if (ccci_header.probe_id == 0)
       ccci_header.probe_id = probeid;
    send_ccci_header(connectfd,&ccci_header);

    // Write received probe ID into file
    FILE * pfile = fopen(probeidfile, "w");
//    fwrite((char *) &ccci_header.probe_id, 1, sizeof(ccci_header.probe_id), pfile);
    fprintf(pfile,"%d",ccci_header.probe_id);
    fclose(pfile);


    int i = 0;
    while (i<1) {
       service(comp, connectfd);
       i++;
    }

    // Start processing MF requests



    filter_deinit(&dev, comp);
    close(connectfd);
    return 0;
}


