/*!
 * \file afilterctl_main.h
 * \brief Tool for controlling advanced filter - configure filter
 * \author Lukas Kekely <xkekel00@stud.fit.vutbr.cz>
 * \date 2012
 *
 * Copyright (C) 2012 Brno University of Technology
 *
 * LICENSE TERMS
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 * 3. Neither the name of the Company nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * ALTERNATIVELY, provided that this notice is retained in full, this
 * product may be distributed under the terms of the GNU General Public
 * License (GPL), in which case the provisions of the GPL apply INSTEAD
 * OF those given above.
 *
 * This software is provided ``as is'', and any express or implied
 * warranties, including, but not limited to, the implied warranties of
 * merchantability and fitness for a particular purpose are disclaimed.
 * In no event shall the company or contributors be liable for any
 * direct, indirect, incidental, special, exemplary, or consequential
 * damages (including, but not limited to, procurement of substitute
 * goods or services; loss of use, data, or profits; or business
 * interruption) however caused and on any theory of liability, whether
 * in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even
 * if advised of the possibility of such damage.
 *
 * $Id: afilterctl_main.h 3122 2013-02-06 01:41:51Z xkekel00 $
 *
 */

#ifndef AFILTERCTL_MAIN_H_INCLUDED
#define AFILTERCTL_MAIN_H_INCLUDED

#include <stdio.h>
#include <stdint.h>
#include "afilterctl_cfg.h"
#include "hwio.h"

/// status flags masks
#define MASK_GLOBAL_BUSY      0x00000001
#define MASK_RULE_TYPE(i)     (1<<(i))

/*!
 * Packet counters of HW interface.
 */
typedef struct {
    uint64_t deny;   /*!< Number of denied packets */
    uint64_t allow;  /*!< Number of allowed packets */
} flt_cnt_status;

/*!
 * Status flags of HW filter.
 */
typedef struct {
    uint32_t disable;        /*!< Filter status: rule type disabled flags */
    uint32_t ignore;         /*!< Filter status: rule type ignored flags */
    uint32_t full;           /*!< Filter status: rule type full tables flags*/
    uint32_t busy;           /*!< Filter status: rule type busy flags*/
} flt_status_flags;

/*!
 * Dirrect access iterator into HW filter Cockoo hash tables.
 */
typedef struct {
    uint32_t type;    /*!< type of rule */
    uint32_t table;   /*!< hash table number with given rule type */
    uint32_t item;    /*!< item address inside given hash table */
} flt_dir_iterator;

/*!
 * Maximal rule of HW filter.
 */
typedef struct {
    uint32_t srcip[4];  /*!< source IP address */
    uint32_t dstip[4];  /*!< destination IP address */
    uint16_t srcport;   /*!< source port number */
    uint16_t dstport;   /*!< destination port number */
    uint8_t proto;      /*!< protocol number */
    uint32_t id;        /*!< identificator of rule */
    int type;           /*!< type of rule */
    uint8_t srcipmask;      /*!< Numeric mask of SRC IP*/
    uint8_t dstipmask;      /*!< Numeric mask of DST IP*/
} flt_rule;


/**
* Reads packet counters of interfaces from HW filtering unit.
* @param comp Pointer to initialized representation of filter component inside device.
* @param reset Specifies if statuses will be reseted after read (0=don't reset, 1=reset).
* @param ints Number of HW filter interfaces.
* @param res Pointer to array of readed interface statuses (must be allocated to ints fields).
* @return Return 0 on success, or 1 on write/read error.
*/
int flt_read_counters(struct hwio_comp *comp, int reset, int ints, flt_cnt_status *res);
/**
* Reads packet counters of one interface from HW filter.
* @param comp Pointer to initialized representation of filter component inside device.
* @param iface Number of interface.
* @param s Pointer to readed interface status.
* @return Return 0 on success, or 1 on write/read error.
*/
int flt_read_ifc_counters(struct hwio_comp *comp, int iface, flt_cnt_status *s);
/**
* Resets packet counters of one interface from HW filter.
* @param comp Pointer to initialized representation of filter component inside device.
* @param iface Number of interface.
* @return Return 0 on success, or 1 on write/read error.
*/
int flt_reset_ifc_counters(struct hwio_comp *comp, int iface);
/**
* Reads status register from HW filter with busy and full flags for each rule table.
* @param comp Pointer to initialized representation of filter component inside device.
* @param res Pointer to array of readed status flags (must be allocated).
* @return Return 0 on success, or 1 on write/read error.
*/
int flt_read_status(struct hwio_comp *comp, flt_status_flags *res);
/**
* Parse rule from string into rule structure.
* @param comp Pointer to initialized representation of filter component inside device.
* @param str String with rule representation.
* @param res Pointer to parsed rule structure.
* @return Return 0 on success, or error code otherwise.
*/
int flt_parse_rule(struct hwio_comp *comp, char *str, flt_rule *res);
#define PARSE_RULE_ERROR   1
#define PARSE_RULE_EMPTY   2
#define PARSE_RULE_HWERROR 3
/**
* Adds rule into HW filter.
* @param comp Pointer to initialized representation of filter component inside device.
* @param r Pointer to structure with rule.
* @return Return 0 on success, or error code otherwise.
*/
int flt_add_rule(struct hwio_comp *comp, flt_rule *r);
#define ADD_RULE_SUCCESS 0
#define ADD_RULE_FULL    1
#define ADD_RULE_BUSY    2
/**
* Removes rule from HW filter.
* @param comp Pointer to initialized representation of filter component inside device.
* @param r Pointer to structure with rule.
* @return Return 0 on success, or error code otherwise.
*/
int flt_del_rule(struct hwio_comp *comp, flt_rule *r);
#define DEL_RULE_SUCCESS 0
#define DEL_RULE_BUSY    1
/**
* Waits for filter to be ready.
* @param comp Pointer to initialized representation of filter component inside device.
* @param mask Mask of rule types, which must be ready.
* @return Return 1 on success (filter ready), or 0 otherwise (waiting to long).
*/
int flt_wait_ready(struct hwio_comp *comp, unsigned mask);
/**
* Fills data registers of HW filter.
* @param comp Pointer to initialized representation of filter component inside device.
* @param r Pointer to structure with rule.
* @param write_data If nonzero, data (id) field is also written.
* @return Return 0 on success, or error code otherwise (currently returns 0 every time).
*/
int flt_write_registers(struct hwio_comp *comp, flt_rule *r, int write_data);
/**
* Reads data registers of HW filter.
* @param comp Pointer to initialized representation of filter component inside device.
* @param r Pointer to structure with rule.
* @return Return 0 on success, or 1 on write/read error.
*/
int flt_read_registers(struct hwio_comp *comp, flt_rule *r);
/**
* Calls add command of HW filter.
* @param comp Pointer to initialized representation of filter component inside device.
* @param mask Mask defining which types of rules are added.
* @return Return 0 on success, or 1 on write/read error.
*/
int flt_add_rule_cmd(struct hwio_comp *comp, unsigned mask);
/**
* Calls remove command of HW filter.
* @param comp Pointer to initialized representation of filter component inside device.
* @param mask Mask defining which types of rules are removed.
* @return Return 0 on success, or 1 on write/read error.
*/
int flt_del_rule_cmd(struct hwio_comp *comp, unsigned mask);
/**
* Fills register with source IP address.
* @param comp Pointer to initialized representation of filter component inside device.
* @param ip IP address in binary form.
* @return Return 0 on success, or 1 on write/read error.
*/
int flt_write_srcip(struct hwio_comp *comp, uint32_t *ip);
/**
* Fills register with destination IP address.
* @param comp Pointer to initialized representation of filter component inside device.
* @param ip IP address in binary form.
* @return Return 0 on success, or 1 on write/read error.
*/
int flt_write_dstip(struct hwio_comp *comp, uint32_t *ip);
/**
* Fills register with port numbers.
* @param comp Pointer to initialized representation of filter component inside device.
* @param sp Source port number.
* @param dp Destination port number.
* @return Return 0 on success, or 1 on write/read error.
*/
int flt_write_ports(struct hwio_comp *comp, uint16_t sp, uint16_t dp);
/**
* Fills register with protocol number.
* @param comp Pointer to initialized representation of filter component inside device.
* @param p Protocol number.
* @return Return 0 on success, or 1 on write/read error.
*/
int flt_write_proto(struct hwio_comp *comp, uint8_t p);
/**
* Fills register with rule identificator.
* @param comp Pointer to initialized representation of filter component inside device.
* @param id Identificator of rule.
* @return Return 0 on success, or 1 on write/read error.
*/
int flt_write_data(struct hwio_comp *comp, uint32_t id);
/**
* Reads register with source IP address.
* @param comp Pointer to initialized representation of filter component inside device.
* @param ip IP address in binary form.
* @return Return 0 on success, or 1 on write/read error.
*/
int flt_read_srcip(struct hwio_comp *comp, uint32_t *ip);
/**
* Reads register with destination IP address.
* @param comp Pointer to initialized representation of filter component inside device.
* @param ip IP address in binary form.
* @return Return 0 on success, or 1 on write/read error.
*/
int flt_read_dstip(struct hwio_comp *comp, uint32_t *ip);
/**
* Reads register with port numbers.
* @param comp Pointer to initialized representation of filter component inside device.
* @param sp Source port number.
* @param dp Destination port number.
* @return Return 0 on success, or 1 on write/read error.
*/
int flt_read_ports(struct hwio_comp *comp, uint16_t *sp, uint16_t *dp);
/**
* Reads register with protocol number.
* @param comp Pointer to initialized representation of filter component inside device.
* @param p Protocol number.
* @return Return 0 on success, or 1 on write/read error.
*/
int flt_read_proto(struct hwio_comp *comp, uint8_t *p);
/**
* Reads register with rule identificator.
* @param comp Pointer to initialized representation of filter component inside device.
* @param id Identificator of rule.
* @return Return 0 on success, or 1 on write/read error.
*/
int flt_read_data(struct hwio_comp *comp, uint32_t *id);
/**
* Reads init vector register from HW filter.
* @param comp Pointer to initialized representation of filter component inside device.
* @param iv Init vector, array of seeds for hash functions.
* @return Return 0 on success, or 1 on write/read error.
*/
int flt_read_initv(struct hwio_comp *comp, uint32_t *iv);
/**
* Fills init vector register inside HW filter.
* @param comp Pointer to initialized representation of filter component inside device.
* @param iv Init vector, array of seeds for hash functions.
* @return Return 0 on success, or 1 on write/read error.
*/
int flt_write_initv(struct hwio_comp *comp, uint32_t *iv);
/**
* Parse init vector register from string representation.
* @param comp Pointer to initialized representation of filter component inside device.
* @param str String with init vector representation.
* @param iv Init vector, array of seeds for hash functions.
* @return Return 0 on success, or error code otherwise.
*/
int flt_parse_initv(struct hwio_comp *comp, char *str, uint32_t *iv);
#define PARSE_INITV_ERROR    1
#define PARSE_INITV_NOEFF    2
/**
* Calls stop command of HW filter.
* @param comp Pointer to initialized representation of filter component inside device.
* @param mask Mask defining which types of rules are stopped from adding reconfiguration.
* @return Return 0 on success, or 1 on write/read error.
*/
int flt_stop_cmd(struct hwio_comp *comp, unsigned mask);
/**
* Calls ignore command of HW filter.
* @param comp Pointer to initialized representation of filter component inside device.
* @param mask Mask defining which types of rules are ignored in matching process.
* @return Return 0 on success, or 1 on write/read error.
*/
int flt_ignore_cmd(struct hwio_comp *comp, unsigned mask);


/**
* Initialize direct iterator into HW filter hash tables.
* @param rt Type of rules accessible by this iterator.
* @param it Pointer to initialized iterator.
*/
void flt_dir_iterator_init(int rt, flt_dir_iterator *it);
/**
* Next iterator, should be empty field.
* @param comp Pointer to initialized representation of filter component inside device.
* @param it Pointer to changed iterator.
*/
void flt_dir_iterator_next(struct hwio_comp *comp, flt_dir_iterator *it);
/**
* Transform dirrect iterator into dirrect address.
* @param fc Pointer to static filter rule types configuration.
* @param it Pointer to iterator to transform.
* @return Dirrect access 32-bit address.
*/
uint32_t flt_dir_iterator_2addr(flt_config *fc, flt_dir_iterator *it);
/**
* Finds and reads next valid rule from HW filter hash tables into data registers of HW filter and returns it.
* @param comp Pointer to initialized representation of filter component inside device.
* @param it Pointer to iterator used for read, iterator is changed.
* @param r Pointer to rule extracted from HW filter.
* @return Return 1 on success, or 0 when there is no next rule left.
*/
int flt_dir_iterator_nextrule(struct hwio_comp *comp, flt_dir_iterator *it, flt_rule *r);
/**
* Clears field inside HW filter hash tables at address given by iterator.
* @param comp Pointer to initialized representation of filter component inside device.
* @param it Pointer to used iterator for clear, iterator is NOT changed.
* @return Return 0 on success, or 1 on write/read error.
*/
int flt_dir_iterator_clear(struct hwio_comp *comp, flt_dir_iterator *it);
/**
* Clears all rules inside HW filter hash tables of type given by iterator.
* @param comp Pointer to initialized representation of filter component inside device.
* @param it Pointer to used iterator for clear, iterator is changed.
* @return Return 0 on success, or 1 on write/read error.
*/
int flt_dir_iterator_clearall(struct hwio_comp *comp, flt_dir_iterator *it);

/**
* Reads rule types configuration from HW filter block.
* @param comp Pointer to initialized representation of filter component inside device.
* @param cfg Pointer to allocated field where rule types configuration will be stored.
* @param iv Pointer to red number of init vectors.
* @return Return 0 on failure, or number of rule types in filter.
*/
int flt_read_config(struct hwio_comp *comp, uint32_t *cfg, int *iv);

#endif // FILTERCTL_MAIN_H_INCLUDED
