/*!
 * \file afilterctl_input.c
 * \brief Tool for controlling advanced filter - read configuration module
 * \author Lukas Kekely <xkekel00@stud.fit.vutbr.cz>
 * \date 2012
 *
 * Copyright (C) 2012 Brno University of Technology
 *
 * LICENSE TERMS
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 * 3. Neither the name of the Company nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * ALTERNATIVELY, provided that this notice is retained in full, this
 * product may be distributed under the terms of the GNU General Public
 * License (GPL), in which case the provisions of the GPL apply INSTEAD
 * OF those given above.
 *
 * This software is provided ``as is'', and any express or implied
 * warranties, including, but not limited to, the implied warranties of
 * merchantability and fitness for a particular purpose are disclaimed.
 * In no event shall the company or contributors be liable for any
 * direct, indirect, incidental, special, exemplary, or consequential
 * damages (including, but not limited to, procurement of substitute
 * goods or services; loss of use, data, or profits; or business
 * interruption) however caused and on any theory of liability, whether
 * in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even
 * if advised of the possibility of such damage.
 *
 * $Id: afilterctl_input.c 3333 2013-03-07 15:02:26Z izadnik $
 *
 */

#include "afilterctl_input.h"
#include "afilterctl_main.h"
#include <arpa/inet.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <fcntl.h>
#include <err.h>
#include <arpa/inet.h>

/// function to parse ip address
#define ADDR_IPv4  1
#define ADDR_IPv6  2
#define ADDR_EMPTY 3
#define ADDR_NM    0


void receive_ccci_header(int fd, ccci_header_t * ccci_header)
{
	char * buf = (char *) ccci_header;
	unsigned int n;
	unsigned int act_bytes = 0;
	while ((n = read(fd, buf+act_bytes, CCCIHEADERLEN - act_bytes)) > 0) {
		act_bytes += n;
        if (act_bytes >= CCCIHEADERLEN)
           break;
    }
    if (act_bytes != CCCIHEADERLEN)
       errx(1, "receive_ccci_header: Actual CCCI header size does not match expected size");
}

void send_ccci_header(int servicefd, ccci_header_t * ccci_header)
{
    int r;
    r = write(servicefd, (unsigned char *) ccci_header, CCCIHEADERLEN);
	if (r == -1) err(1, "write()");
	if (r != (CCCIHEADERLEN) ) errx(1, "write(): Buffer written just partially");
}

void parse_netrule2netbuf(net_rule_t * netrule, char * netbuf) {

   *netbuf = netrule->action;
   *(netbuf+NETOFFSETSTATUS) = netrule->status;

   *((uint32_t*) (netbuf+NETOFFSETSRCIP0)) = netrule->frule.srcip[0];
   *((uint32_t*) (netbuf+NETOFFSETSRCIP1)) = netrule->frule.srcip[1];
   *((uint32_t*) (netbuf+NETOFFSETSRCIP2)) = netrule->frule.srcip[2];
   *((uint32_t*) (netbuf+NETOFFSETSRCIP3)) = netrule->frule.srcip[3];
   *((uint32_t*) (netbuf+NETOFFSETDSTIP0)) = netrule->frule.dstip[0];
   *((uint32_t*) (netbuf+NETOFFSETDSTIP1)) = netrule->frule.dstip[1];
   *((uint32_t*) (netbuf+NETOFFSETDSTIP2)) = netrule->frule.dstip[2];
   *((uint32_t*) (netbuf+NETOFFSETDSTIP3)) = netrule->frule.dstip[3];
   *((uint16_t*) (netbuf+NETOFFSETSRCPORT))= netrule->frule.srcport ;
   *((uint16_t*) (netbuf+NETOFFSETDSTPORT))= netrule->frule.dstport ;
   *(            (netbuf+NETOFFSETPROTO))  = netrule->frule.proto   ;
   *(        (netbuf+NETOFFSETSRCIPMASK))  = netrule->frule.srcipmask;
   *(            (netbuf+NETOFFSETIPVER))  = netrule->ipver;
   *((uint32_t*) (netbuf+NETOFFSETID))     = netrule->frule.id      ;
                   *(netbuf+NETOFFSETTYPE) = (uint8_t) netrule->frule.type ;

   *((uint32_t*) (netbuf+NETOFFSETSID))    = netrule->sid          ;
}

void parse_netbuf2netrule(char * netbuf, net_rule_t * netrule) {

   netrule->action = (uint8_t) *netbuf;
   netrule->status = 0x0;

   netrule->frule.srcip[0]  = *((uint32_t*) (netbuf+NETOFFSETSRCIP0));
   netrule->frule.srcip[1]  = *((uint32_t*) (netbuf+NETOFFSETSRCIP1));
   netrule->frule.srcip[2]  = *((uint32_t*) (netbuf+NETOFFSETSRCIP2));
   netrule->frule.srcip[3]  = *((uint32_t*) (netbuf+NETOFFSETSRCIP3));
   netrule->frule.dstip[0]  = *((uint32_t*) (netbuf+NETOFFSETDSTIP0));
   netrule->frule.dstip[1]  = *((uint32_t*) (netbuf+NETOFFSETDSTIP1));
   netrule->frule.dstip[2]  = *((uint32_t*) (netbuf+NETOFFSETDSTIP2));
   netrule->frule.dstip[3]  = *((uint32_t*) (netbuf+NETOFFSETDSTIP3));
   netrule->frule.srcport   = *((uint16_t*) (netbuf+NETOFFSETSRCPORT));
   netrule->frule.dstport   = *((uint16_t*) (netbuf+NETOFFSETDSTPORT));
   netrule->frule.proto     = *((uint8_t*)  (netbuf+NETOFFSETPROTO));
   netrule->frule.srcipmask = *((uint8_t*)  (netbuf+NETOFFSETSRCIPMASK));
   netrule->frule.id        = *((uint32_t*) (netbuf+NETOFFSETID));  // rid
   netrule->frule.type      = *((uint8_t*)  (netbuf+NETOFFSETTYPE));

   netrule->sid             = *((uint32_t*) (netbuf+NETOFFSETSID));
   netrule->ipver           = *((uint8_t*) (netbuf+NETOFFSETIPVER));

   if (netrule->frule.type == 0) {
      netrule->frule.type = 3;
      if (netrule->ipver == 4) {
         if (netrule->frule.srcipmask == 24) {
            netrule->frule.type = 0;
	 }
      }
      else {
         if (netrule->frule.srcipmask == 48) {
            netrule->frule.type = 1;
	 }
	 else if (netrule->frule.srcipmask == 64) {
            netrule->frule.type = 2;
	 }
      }
   }
}


int get_ip(const char *str, char *ip) {
    memset(ip,0x00,16);
    if(inet_pton(AF_INET,str,ip+12)) // try IPv4
        return ADDR_IPv4;
    else if(inet_pton(AF_INET6,str,ip)) // if not IPv4, try IPv6
        return ADDR_IPv6;
    else
        while((*str)!='\0')
            if(!isspace(*str))
                return ADDR_NM;
            else
                str++;
    return ADDR_EMPTY;
}


/// function to correctly convert IP address from binary to string
void ip2string(unsigned char *ip, char *str) {
    unsigned char *bin=ip+12;
    int domain=AF_INET;
    for(int i=0; i<12; i++)
        if(ip[i]) {
            bin=ip;
            domain=AF_INET6;
            break;
        }
    inet_ntop(domain,bin,str,IPSTR_LEN);
}


/// function to parse unsigned integer
#define UINT_OK    1
#define UINT_EMPTY 2
#define UINT_NM    0
int get_unsigned(const char *str, uint32_t *res) {
    char *end;
    while((*str)!='\0')
        if(!isspace(*str))
            break;
        else
            str++;
    if((*str)=='\0')
        return UINT_EMPTY;
    *res=strtoul(str,&end,0);
    while((*end)!='\0')
        if(!isspace(*end))
            return UINT_NM;
        else
            end++;
    return UINT_OK;
}

int get_port(const char *str, uint16_t *res) {
    uint32_t r;
    int x=get_unsigned(str,&r);
    *res=htons(r);
    if(x==UINT_OK&&r>0xFFFF)
        return UINT_NM;
    return x;
}
int get_proto(const char *str, uint8_t *res) {
    uint32_t r;
    int x=get_unsigned(str,&r);
    *res=r;
    if(x==UINT_OK&&r>0xFF)
        return UINT_NM;
    return x;
}


/// function to parse rule type
int get_type(const char *str, int *res) {
    uint32_t r;
    int x=get_unsigned(str,&r);
    *res=r;
    if(x==UINT_EMPTY)
        return 0;
    return 1;
}


/// function to replace all semicolons with '\0'
int split_semi(char *str) {
    int c=0;
    while((*str)!='\0') {
        if((*str)==';') {
            *str='\0';
            c++;
        }
        str++;
    }
    return c;
}


/// parse rule values from split string
int flt_get_rule(char *str, flt_rule *res, unsigned *empty) {
    int ipv4_only=1;
    if(split_semi(str)!=6) {
        return 0;
    }
    *empty=0;
    int x;
    // rule type
    if(!get_type(str,&(res->type))) {
        return 0;
    }
    str=str+strlen(str)+1;
    // srcip address
    x=get_ip(str,(char *)(res->srcip));
    if(x==ADDR_NM) {
        return 0;
    }
    if(x==ADDR_EMPTY)
        *empty|=EMPTY_SRCIP;
    else if(x==ADDR_IPv6)
        ipv4_only=0;
    str=str+strlen(str)+1;
    // dstip address
    x=get_ip(str,(char *)(res->dstip));
    if(x==ADDR_NM) {
        return 0;
    }
    if(x==ADDR_EMPTY)
        *empty|=EMPTY_DSTIP;
    else if(x==ADDR_IPv6)
        ipv4_only=0;
    str=str+strlen(str)+1;
    // srcport number
    x=get_port(str,&(res->srcport));
    if(x==UINT_NM) {
        return 0;
    }
    if(x==UINT_EMPTY)
        *empty|=EMPTY_SRCPORT;
    str=str+strlen(str)+1;
    // dstport number
    x=get_port(str,&(res->dstport));
    if(x==UINT_NM) {
        return 0;
    }
    if(x==UINT_EMPTY)
        *empty|=EMPTY_DSTPORT;
    str=str+strlen(str)+1;
    // protocol number
    x=get_proto(str,&(res->proto));
    if(x==UINT_NM) {
        return 0;
    }
    if(x==UINT_EMPTY)
        *empty|=EMPTY_PROTO;
    str=str+strlen(str)+1;
    // data
    x=get_unsigned(str,&(res->id));
    if(x==UINT_NM) {
        return 0;
    }
    if(x==UINT_EMPTY)
        *empty|=EMPTY_DATA;
    if(!ipv4_only)
        *empty|=EMPTY_IPV4_ONLY;
    return 1;
}

/// parse init vector
int flt_get_initv(char *str, uint32_t *res, unsigned *empty) {
    int parts=split_semi(str)+1;
    int x;
    *empty=0;
    for(int i=0; i<parts; i++) {
        x=get_unsigned(str,res+i);
        if(x==UINT_NM)
            return 0;
        if(x==UINT_EMPTY)
            *empty|=(1<<i);
        str+=strlen(str)+1;
    }
    return parts;
}


