/*!
 * \file phys.c
 * \brief Physical address mapping of userland buffers.
 * \author Jachym Holecek <freza@liberouter.org>
 * \date 2003,2004
 *
 * Copyright (C) 2003,2004 CESNET
 */
/* LICENSE TERMS
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 * 3. Neither the name of the Company nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * ALTERNATIVELY, provided that this notice is retained in full, this
 * product may be distributed under the terms of the GNU General Public
 * License (GPL) version 2 or later, in which case the provisions
 * of the GPL apply INSTEAD OF those given above.
 *
 * This software is provided ``as is'', and any express or implied
 * warranties, including, but not limited to, the implied warranties of
 * merchantability and fitness for a particular purpose are disclaimed.
 * In no event shall the company or contributors be liable for any
 * direct, indirect, incidental, special, exemplary, or consequential
 * damages (including, but not limited to, procurement of substitute
 * goods or services; loss of use, data, or profits; or business
 * interruption) however caused and on any theory of liability, whether
 * in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even
 * if advised of the possibility of such damage.
 *
 * $Id$
 *
 */

#include <sys/ioctl.h>
#include <sys/mman.h>

#include <err.h>
#include <errno.h>
#include <fcntl.h>
#include <stdlib.h>
#include <unistd.h>

#include <commlbr.h>
#include <combo6.h>
#include "combosix.h"
#include "cs_local.h"

__RCSID("$Id$");

/* XXX: until Linux version is done? */
#if defined(__NetBSD__) || defined(DOC_HIDDEN)

/**
 * \brief Allocate physical memory area
 * \param d combo6 device structure
 * \param len physical memory area length
 * \param phys mallocated physical memory area
 * \retval zero on success otherwise a negative error code
 */
int
cs_phys_alloc(cs_device_t *d, size_t len, cs_phys_t **phys)
{
	struct combo6_phys 	r;
	cs_phys_t 		*p;
	int 			i, ret;

	p = (cs_phys_t *)malloc(sizeof(cs_phys_t));
	if (p == NULL)
		return -ENOMEM;

	p->p_data = (u_int8_t *)malloc(len);
	if (p->p_data == NULL) {
		ret = -ENOMEM;
		goto fail_0;
	}

	p->p_size 	= len;
	p->p_nseg 	= 0;

	r.cp_req 	= PHYS_ALLOC;
	r.cp_buf 	= p->p_data;
	r.cp_len 	= p->p_size;

	if (-1 == ioctl(d->dv_file, COMBO6_IOC_PHYS, &r)) {
		ret = -errno;
		goto fail_1;
	}

	p->p_nseg = r.cp_cnt;
	p->p_segs = (cs_seg_t *)malloc(r.cp_cnt * sizeof(cs_seg_t));
	if (p->p_segs == NULL) {
		ret = -ENOMEM;
		goto fail_1;
	}

	for (i = 0; i < (int) r.cp_cnt; i++) {
		p->p_segs[i].sg_addr = r.cp_seg[i].sg_addr;
		p->p_segs[i].sg_size = r.cp_seg[i].sg_size;
	}

	*phys = p;
	return 0;

fail_1:
	free(p->p_data);
fail_0:
	free(p);
	return ret;
}

/**
 * \brief Sync physical memory area before op?
 * \param d combo6 device structure
 * \param p physical memory area to free
 * \retval zero on success otherwise a negative error code
 */
int
cs_phys_sync_before(cs_device_t *d, cs_phys_t *p)
{
	struct combo6_phys 	r;

	r.cp_req 		= PHYS_PRESYNC;
	r.cp_buf 		= p->p_data;
	r.cp_len 		= p->p_size;
	/* no need to fill remaining fields */

	if (-1 == ioctl(d->dv_file, COMBO6_IOC_PHYS, &r))
		return -errno;

	return 0;
}

/**
 * \brief Sync physical memory area after op?
 * \param d combo6 device structure
 * \param p physical memory area to free
 * \retval zero on success otherwise a negative error code
 */
int
cs_phys_sync_after(cs_device_t *d, cs_phys_t *p)
{
	struct combo6_phys 	r;

	r.cp_req 		= PHYS_POSTSYNC;
	r.cp_buf 		= p->p_data;
	r.cp_len 		= p->p_size;
	/* no need to fill remaining fields */

	if (-1 == ioctl(d->dv_file, COMBO6_IOC_PHYS, &r))
		return -errno;

	return 0;
}

/**
 * \brief Free a physical memory area
 * \param d combo6 device structure
 * \param p physical memory area to free
 * \retval zero on success otherwise a negative error code
 */
int
cs_phys_free(cs_device_t *d, cs_phys_t **p)
{
	struct combo6_phys 	r;

	r.cp_req 		= PHYS_FREE;
	r.cp_buf 		= (*p)->p_data;
	r.cp_len 		= (*p)->p_size;
	/* no need to fill remaining fields */

	if (-1 == ioctl(d->dv_file, COMBO6_IOC_PHYS, &r))
		return -errno;

	free((*p)->p_data);
	free((*p)->p_segs);
	free(*p);

	*p = NULL;
	return 0;
}

#endif /* __NetBSD__ */
