/*! 
 * \file tcamctrl.c
 * \brief Functions for controlling CS_TCAM_CTRL component
 * \author Andrej Hank <xhanka00@liberouter.org>
 * \date 2007-07-12
 */
/*
 * Copyright (C) 2007 CESNET
 *
 * LICENSE TERMS
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 * 3. Neither the name of the Company nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * ALTERNATIVELY, provided that this notice is retained in full, this
 * product may be distributed under the terms of the GNU General Public
 * License (GPL), in which case the provisions of the GPL apply INSTEAD
 * OF those given above.
 *
 * This software is provided ``as is'', and any express or implied
 * warranties, including, but not limited to, the implied warranties of
 * merchantability and fitness for a particular purpose are disclaimed.
 * In no event shall the company or contributors be liable for any
 * direct, indirect, incidental, special, exemplary, or consequential
 * damages (including, but not limited to, procurement of substitute
 * goods or services; loss of use, data, or profits; or business
 * interruption) however caused and on any theory of liability, whether
 * in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even
 * if advised of the possibility of such damage.
 *
 * $Id$
 *
 */

#include <err.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <commlbr.h>

#include "tcamctrl.h"

__RCSID ("$Id$");

/*!
 *\brief		Init instruction cache from file
 *\param dev		Device
 *\param space		Space
 *\param src_file_name	Source file
 *\return		0 - OK
 */
int cs_tcam_ctrl_init_inst_cache (cs_device_t * dev, cs_space_t * space, char *src_file_name)
{
        u_int32_t           value = 0;
        char                buffer[18];
        FILE               *src_file;   /* source file */
        int                 count;

        /* Load file. */
        if ((src_file = fopen (src_file_name, "r")) == NULL)
                errx (1, "Error open source file %s.", src_file_name);

        count = 0;
	VERBOSE (CL_VERBOSE_LIBRARY, "Instruction cache initiation starting...");
        while (fgets (buffer, 17, src_file) != NULL) {
                if (sscanf (buffer, "%016x", &value) != EOF) {
                        cs_space_write_4 (dev, space, count * 4, value);
                        VERBOSE (CL_VERBOSE_LIBRARY, "0x%08x -> 0x%08x", value, count * 4);
                }
                count++;
        }
        VERBOSE (CL_VERBOSE_LIBRARY, "Initiation done with %d items", count);
        fclose (src_file);

        return 0;
}

/*!
 *\brief		Clear search reg
 *\param dev		Device
 *\param space		Space
 */
void cs_tcam_ctrl_clear_search_reg (cs_device_t * dev, cs_space_t * space)
{
	u_int32_t	value;
	
	value = cs_space_read_4 (dev, space, CS_TCAM_CTRL);
	value |= 1 << 12;
	cs_space_write_4 (dev, space, CS_TCAM_CTRL, value);
	VERBOSE (CL_VERBOSE_LIBRARY, "0x%08x -> 0x%08x", value, CS_TCAM_CTRL);
	value &= 0xFFFFEFFF;
	cs_space_write_4 (dev, space, CS_TCAM_CTRL, value);
	VERBOSE (CL_VERBOSE_LIBRARY, "0x%08x -> 0x%08x", value, CS_TCAM_CTRL);
}

/*!
 *\brief		Allocate unit	
 *\param dev		Device
 *\param space		Space
 */
void cs_tcam_ctrl_get_unit_control (cs_device_t * dev, cs_space_t * space)
{
	u_int32_t	value;

	value = cs_space_read_4 (dev, space, CS_TCAM_CTRL);
	value |= 1;
	cs_space_write_4 (dev, space, CS_TCAM_CTRL, value);
	VERBOSE (CL_VERBOSE_LIBRARY, "0x%08x -> 0x%08x", value, CS_TCAM_CTRL);

	while ((value = (cs_space_read_4 (dev, space, CS_TCAM_STAT) & 1) != 1))
		usleep(100);
			
	VERBOSE (CL_VERBOSE_LIBRARY, "Unit is allocated");
}

/*!
 *\brief		Free unit	
 *\param dev		Device
 *\param space		Space
 */
void cs_tcam_ctrl_give_unit_control (cs_device_t * dev, cs_space_t * space)
{
	u_int32_t	value;

	value = cs_space_read_4 (dev, space, CS_TCAM_CTRL);
	value = value & 0xFFFFFFFE;
	cs_space_write_4 (dev, space, CS_TCAM_CTRL, value);
	VERBOSE (CL_VERBOSE_LIBRARY, "0x%08x -> 0x%08x", value, CS_TCAM_CTRL);

	VERBOSE (CL_VERBOSE_LIBRARY, "Unit is free");
}

/*! 
 * \brief Make row structure from string
 * 
 * \param buffer String with row
 * \param row Output row
 * 
 * \return 
 * 	- ture if susseccfuly converted
 * 	- false if failed
 */
bool cs_tcam_ctrl_get_row_from_string (char *buffer, cs_tcam_ctrl_row_t *row) {
	u_int32_t first, second, third;
	if (sscanf(buffer, "%01x%08x%08x", &third, &second, &first) != 3) {
		MSG(2, "String to row conversion failed");
		return false;
	}
	MSG(2, "String to row conversion done");
	
	row->first = first;
	row->second = second;
	row->third = third;

	return true;
}

/*! 
 * \brief Reads one line from file and returns read row
 * 
 * \param row Output row
 * \param file Input file
 * 
 * \return 
 * 	- ture if susseccfuly converted
 * 	- false if failed
 */
bool cs_tcam_ctrl_get_row_from_file (cs_tcam_ctrl_row_t *row, FILE *file) {
	char buffer[20];
	
	if (fgets(buffer, 20, file) == NULL)
		return false;
	MSG(2, "\"%s\" read from file", buffer);
	return cs_tcam_ctrl_get_row_from_string(buffer, row);
}

/*! 
 * \brief Reads multiple lines from file and stores into array
 * 
 * Does not control allocated memory
 *
 * \param rows Output rows
 * \param count Count of rows to get
 * \param file Input file
 * 
 * \return 
 * 	- ture if susseccfuly read and stored
 * 	- false if failed
 */
bool cs_tcam_ctrl_get_multiple_rows_from_file (cs_tcam_ctrl_row_t *rows, int count, FILE *file) {
	int i;
	cs_tcam_ctrl_row_t row;

	for(i = 0; i < count; i++) {
		if(!cs_tcam_ctrl_get_row_from_file(&row, file))	
			return false;
		else
			rows[i] = row;	
			/* TODO check */
	}
	return true;
}

/*! 
 * \brief Write parameters and command type to perform specific command
 * 
 * \param dev Device
 * \param space Space
 * \param cmd_params Command parameters
 * \param command_type
 */
void cs_tcam_ctrl_write_command_with_params (cs_device_t * dev, cs_space_t * space, int cmd_params, cs_tcam_ctrl_command_t command_type) {

	u_int32_t value;

	value = cs_space_read_4 (dev, space, CS_TCAM_CMD);
	value &= 0x0000FFF0;
	value |= (cmd_params << 16) + command_type;
	cs_space_write_4 (dev, space, CS_TCAM_CMD, value);
	VERBOSE (CL_VERBOSE_LIBRARY, "0x%08x -> 0x%08x", value, CS_TCAM_CMD);
}

/*!
 *\brief		Write one row to address
 *\param dev		Device
 *\param space		Space
 *\param addr		Address
 *\param row		Row to write	
 *\param addr_offset    Offset to add to address beeing written into address reg
 */
void cs_tcam_ctrl_write_row (cs_device_t * dev, cs_space_t * space, u_int32_t addr, cs_tcam_ctrl_row_t row, u_int32_t addr_offset)
{
	u_int64_t		sum;
	u_int32_t		check, carry;
	int 			overflow = 0;
	u_int32_t		*reg = NULL;
	int 			i;
	
	/* get sum, and add offset, we must work with 128b value */
	/* control overflow */
	if(addr_offset) {
		carry = addr_offset;
		for(i = 0; i < 3; i++) {
			if(i == 0) {
				reg = &(row.first);
			} else if (i == 1) {
				reg = &(row.second);
			} else if (i == 2) {
				reg = &(row.third);
			}
			
			sum = *reg + (u_int64_t)carry;
			check = *reg + (u_int64_t)carry;
			*reg = check;
			
			if(sum >> 32) { /* oveflow detected */
				overflow = 1;
			}
			else { /* not overflow, can stop now */
				break;
			}

			if(overflow)
				carry = 1;
		}
	}
	
	cs_space_write_4 (dev, space, addr, row.first);
	cs_space_write_4 (dev, space, addr + 4, row.second);
	cs_space_write_4 (dev, space, addr + 8, row.third);
	cs_space_write_4 (dev, space, addr + 12, 0);
	VERBOSE (CL_VERBOSE_LIBRARY, "0x%01x%08x%08x -> 0x%08x", row.third, row.second, row.first, addr);
	
}

/*!
 *\brief		Read one row from addr
 *\param dev		Device
 *\param space		Space
 *\param addr		Address
 *
 * \return Read row
 */
cs_tcam_ctrl_row_t cs_tcam_ctrl_read_row (cs_device_t * dev, cs_space_t * space, u_int32_t addr)
{
	cs_tcam_ctrl_row_t result;

	result.first = cs_space_read_4 (dev, space, addr);
	result.second = cs_space_read_4 (dev, space, addr + 4);
	result.third = cs_space_read_4 (dev, space, addr + 8);
	VERBOSE (CL_VERBOSE_LIBRARY, "0x%08x <- 0x%01x%08x%08x", addr, result.third, result.second, result.first);

	return result;
}

/*!
 * \brief 		Performs single data/mask write operation
 * \param dev		Device
 * \param space		Space
 * \param cmd_params	Operation parameters
 * \param data		Data row
 * \param addr		Address row
 * \param operation	Operation type - data/mask
 */
void cs_tcam_ctrl_single_write (cs_device_t * dev, cs_space_t * space, int cmd_params, cs_tcam_ctrl_row_t data, cs_tcam_ctrl_row_t addr, cs_tcam_ctrl_operation_t operation)
{
	/* write address into register */
	if(operation == CS_TCAM_CTRL_OPERATION_DATA)
		cs_tcam_ctrl_write_row (dev, space, CS_TCAM_ADR, addr, 0);
	else if(operation == CS_TCAM_CTRL_OPERATION_MASK)
		cs_tcam_ctrl_write_row (dev, space, CS_TCAM_ADR, addr, CS_TCAM_MASK_OFFSET);

	/* write data into register */
	cs_tcam_ctrl_write_row (dev, space, CS_TCAM_DATA, data, 0);

	/* write parameters and command type */
	cs_tcam_ctrl_write_command_with_params(dev, space, cmd_params, CS_TCAM_CTRL_WRITE_COMMAND);
}


/*!
 * \brief 		Performs single write operation into global register
 * \param dev		Device
 * \param space		Space
 * \param cmd_params	Operation parameters
 * \param addr		Address (probably dummy)
 * \param data		Data
 * \param reg_num	Register number
 */
void cs_tcam_ctrl_single_write_gmask (cs_device_t * dev, cs_space_t * space, int cmd_params, cs_tcam_ctrl_row_t data, cs_tcam_ctrl_row_t addr, int reg_num)
{
	/* don't read address, write immediately */
	cs_space_write_4 (dev, space, CS_TCAM_ADR, CS_TCAM_GMASK_REG + reg_num);
	cs_space_write_4 (dev, space, CS_TCAM_ADR + 4, 0);
	cs_space_write_4 (dev, space, CS_TCAM_ADR + 8, 0);
	cs_space_write_4 (dev, space, CS_TCAM_ADR + 12, 0);
	VERBOSE (CL_VERBOSE_LIBRARY, "0x%08x -> 0x%08x", CS_TCAM_GMASK_REG + reg_num, CS_TCAM_ADR);

        cs_tcam_ctrl_write_row (dev, space, CS_TCAM_DATA, addr, 0); /* dummy address write - TODO really needed ? */
	/* write data into register */
	cs_tcam_ctrl_write_row (dev, space, CS_TCAM_DATA, data, 0); 

	/* write parameters and command type */
	cs_tcam_ctrl_write_command_with_params(dev, space, cmd_params, CS_TCAM_CTRL_WRITE_COMMAND);
}

/*!
 * \brief Search operation
 * \param dev		Device
 * \param space		Space
 * \param cmd_params	Operation parameters
 * \param addr		Address row
 * \param data		Pointer to array of 4 values of cs_tcam_ctrl_row_t type
 * \param match_addr	If found, returned address where match found
 *
 * \return 
 * 	- true if found
 * 	- false if not found
 */
bool cs_tcam_ctrl_search (cs_device_t * dev, cs_space_t * space, int cmd_params, cs_tcam_ctrl_row_t addr, cs_tcam_ctrl_row_t *data, u_int32_t *match_addr) {


	u_int32_t 	value;

	/* write address into register */
	cs_tcam_ctrl_write_row (dev, space, CS_TCAM_ADR, addr, 0);

	/* write data into register */
	cs_tcam_ctrl_write_row (dev, space, CS_TCAM_DATA, data[0], 0);
	cs_tcam_ctrl_write_row (dev, space, CS_TCAM_DATA + 16, data[1], 0);
	cs_tcam_ctrl_write_row (dev, space, CS_TCAM_DATA + 32, data[2], 0);
	cs_tcam_ctrl_write_row (dev, space, CS_TCAM_DATA + 48, data[3], 0);

	/* write parameters and command type */
	cs_tcam_ctrl_write_command_with_params(dev, space, cmd_params, CS_TCAM_CTRL_SEARCH_COMMAND);

	/* wait for operation to be completed */
	cs_tcam_ctrl_wait_for_command_completion(dev, space);

	value = cs_space_read_4 (dev, space, CS_TCAM_SEARCH);

	/* clear search reg */
	cs_tcam_ctrl_clear_search_reg(dev, space);

	if ((value & 1) == 1) {
		*match_addr = (value & 0x3FFFFF00)/256;
		return true;	
	} else
		return false;
}

/*!
 * \brief Read data/mask operation
 * \param dev		Device
 * \param space		Space
 * \param cmd_params	Operation parameters
 * \param addr		Address
 * \param operation	Operation type - data/mask
 *
 * \return Read row
 */
cs_tcam_ctrl_row_t cs_tcam_ctrl_single_read (cs_device_t * dev, cs_space_t * space, int cmd_params, cs_tcam_ctrl_row_t addr, cs_tcam_ctrl_operation_t operation)
{
	cs_tcam_ctrl_row_t result;

	/* write address into register */
	if(operation == CS_TCAM_CTRL_OPERATION_DATA) {
		cs_tcam_ctrl_write_row (dev, space, CS_TCAM_ADR, addr, 0);

	} else if(operation == CS_TCAM_CTRL_OPERATION_MASK) {
		cs_tcam_ctrl_write_row (dev, space, CS_TCAM_ADR, addr, CS_TCAM_MASK_OFFSET);
	}


	/* write parameters and command type */
	cs_tcam_ctrl_write_command_with_params(dev, space, cmd_params, CS_TCAM_CTRL_READ_COMMAND);

	/* wait for operation to be completed */
	cs_tcam_ctrl_wait_for_command_completion(dev, space);
	
	/* read data from data registers */
	result.first = cs_space_read_4 (dev, space, CS_TCAM_DATA);
	result.second = cs_space_read_4 (dev, space, CS_TCAM_DATA + 4);
	result.third = cs_space_read_4 (dev, space, CS_TCAM_DATA + 8);

	return result;
}

/*! 
 * \brief Print content of row to output
 * 
 * \param row Row to print
 * \param output Output stream
 */
void cs_tcam_ctrl_print_row (cs_tcam_ctrl_row_t row, FILE *output) {
	fprintf(output, "0x%01x%08x%08x\n", row.third, row.second, row.first);
}


/*!
 * \brief 		Read mask operation from global registers
 * \param dev		Device
 * \param space		Space
 * \param cmd_params	Operation parameters
 * \param reg_num	Register number
 *
 * \return Read register mask row
 */
cs_tcam_ctrl_row_t cs_tcam_ctrl_single_read_gmask (cs_device_t * dev, cs_space_t * space, int cmd_params, int reg_num)
{
	cs_tcam_ctrl_row_t result;

	/* don't read address, write immediately */
	cs_space_write_4 (dev, space, CS_TCAM_ADR, CS_TCAM_GMASK_REG + reg_num);
	cs_space_write_4 (dev, space, CS_TCAM_ADR + 4, 0);
	cs_space_write_4 (dev, space, CS_TCAM_ADR + 8, 0);
	cs_space_write_4 (dev, space, CS_TCAM_ADR + 12, 0);
	VERBOSE (CL_VERBOSE_LIBRARY, "0x%08x -> 0x%08x", CS_TCAM_GMASK_REG + reg_num, CS_TCAM_ADR);
	
	/* write parameters and command type */
	cs_tcam_ctrl_write_command_with_params(dev, space, cmd_params, CS_TCAM_CTRL_READ_COMMAND);

	/* wait for operation to be completed */
	cs_tcam_ctrl_wait_for_command_completion(dev, space);
	
	/* read data from data registers */
	result.first = cs_space_read_4 (dev, space, CS_TCAM_DATA);
	result.second = cs_space_read_4 (dev, space, CS_TCAM_DATA + 4);
	result.third = cs_space_read_4 (dev, space, CS_TCAM_DATA + 8);

	return result;
}

/*! 
 * \brief Common operation for burst commands - writes address of burst register
 * and compulsory parameters -a -n
 * 
 * \param dev Device
 * \param space Space
 * \param burst_count Burst count, range 4 - 511
 * \param burst_addr Burst address
 * \param reg Register (read/write)
 *
 * \return 
 * 	- true if successfully passed checks and initialized
 * 	- false if failed - count, address out of range
 */
bool cs_tcam_ctrl_burst_init (cs_device_t * dev, cs_space_t * space, int burst_count, int burst_addr, u_int32_t reg) 
{
	u_int32_t value;

	if(burst_count < CS_TCAM_MIN_BURST_COUNT || burst_count > CS_TCAM_MAX_BURST_COUNT) {
		warnx("burst count out of range %d - %d", CS_TCAM_MIN_BURST_COUNT, CS_TCAM_MAX_BURST_COUNT);
		return false;
	}

	if(burst_addr > CS_TCAM_MAX_BURST_ADDR) {
		warnx("burst address out of range 0 - 0x%08x", CS_TCAM_MAX_BURST_ADDR);
		return false;
	}

	/* addr of burst read/write reg */
	cs_space_write_4 (dev, space, CS_TCAM_ADR, reg);
	cs_space_write_4 (dev, space, CS_TCAM_ADR + 4, 0);
	cs_space_write_4 (dev, space, CS_TCAM_ADR + 8, 0);
	cs_space_write_4 (dev, space, CS_TCAM_ADR + 12, 0);
	VERBOSE (CL_VERBOSE_LIBRARY, "0x%08x -> 0x%08x", reg, CS_TCAM_ADR);

	/* write burst address and count */
	value = 0;
	value += (burst_count << 19) + burst_addr; 
	cs_space_write_4 (dev, space, CS_TCAM_DATA, value);
	cs_space_write_4 (dev, space, CS_TCAM_DATA + 4, 0);
	cs_space_write_4 (dev, space, CS_TCAM_DATA + 8, 0);
	cs_space_write_4 (dev, space, CS_TCAM_DATA + 12, 0);
	VERBOSE (CL_VERBOSE_LIBRARY, "0x%08x -> 0x%08x", value, CS_TCAM_DATA);

	return true;
}

/*! 
 * \brief Wati for command completion
 * 
 * \param dev Device
 * \param space Space
 */
void cs_tcam_ctrl_wait_for_command_completion(cs_device_t * dev, cs_space_t * space){	
	/* wait for operation to be completed */
	while(!(cs_space_read_4 (dev, space, CS_TCAM_STAT) & 16))
		usleep(100);
}

/*!
 * \brief 		Burst data/mask write operation
 * \param dev		Device
 * \param space		Space
 * \param cmd_params	Operation parameters
 * \param rows 		Pointer to array of burst_count values of type
 * cs_tcam_ctrl_row_t 
 * \param burst_count	Burst count, range 4 - 511
 * \param burst_addr 	Burst address = first row number
 * \param operation	Operation type - data/mask
 *
 * \return 
 * 	- true if successfully written
 * 	- flse if failed
 */
bool cs_tcam_ctrl_burst_write (cs_device_t * dev, cs_space_t * space, int cmd_params, cs_tcam_ctrl_row_t *rows, int burst_count, int burst_addr, cs_tcam_ctrl_operation_t operation)
{
	int 		i;
	
	if(!cs_tcam_ctrl_burst_init(dev, space, burst_count, burst_addr, CS_TCAM_BWRITE_REG))
		return false;

	/* write command without params */
	VERBOSE (CL_VERBOSE_LIBRARY, "0x%08x -> 0x%08x", 1, CS_TCAM_CMD);
	cs_space_write_4 (dev, space, CS_TCAM_CMD, 1);		

	/* wait for operation to be completed */
	cs_tcam_ctrl_wait_for_command_completion(dev, space);
	
	/* don't read address, write immediately */
	if(operation == CS_TCAM_CTRL_OPERATION_DATA) {
		VERBOSE (CL_VERBOSE_LIBRARY, "0x%08x -> 0x%08x", 0, CS_TCAM_ADR);
		cs_space_write_4 (dev, space, CS_TCAM_ADR, 0);
	} else if (operation == CS_TCAM_CTRL_OPERATION_MASK) {
		VERBOSE (CL_VERBOSE_LIBRARY, "0x%08x -> 0x%08x", CS_TCAM_MASK_OFFSET, CS_TCAM_ADR);
		cs_space_write_4 (dev, space, CS_TCAM_ADR, CS_TCAM_MASK_OFFSET);
	}

	cs_space_write_4 (dev, space, CS_TCAM_ADR + 4, 0);
	cs_space_write_4 (dev, space, CS_TCAM_ADR + 8, 0);
	cs_space_write_4 (dev, space, CS_TCAM_ADR + 12, 0);

	/* fill with data from file */
	for (i = 0; i < burst_count; i++){
		cs_tcam_ctrl_write_row (dev, space, CS_TCAM_DATA + i*16, rows[i], 0);
		/*usleep(1000);*/
	}
	
	/* write command */
	cs_tcam_ctrl_write_command_with_params(dev, space, cmd_params, CS_TCAM_CTRL_BURST_WRITE_COMMAND);
	
	/* wait for operation to be completed */
	cs_tcam_ctrl_wait_for_command_completion(dev, space);

	return true;
}

/*!
 * \brief 		Burst read data operation, pointer to burst_count of
 * read rows are returned through parameter rows
 * \param dev		Device
 * \param space		Space
 * \param cmd_params	Operation parameters
 * \param burst_count	Burst count, range 4 - 511
 * \param burst_addr 	Burst address
 * \param rows		Pointer to allocated memory to store output -
 * burst_count read rows
 * \param operation	Operation type - burst data read/burst mask read
 *
 * \return 
 * 	- true if successfully read
 * 	- flse if failed
 */
bool cs_tcam_ctrl_burst_read (cs_device_t * dev, cs_space_t * space, int cmd_params, int burst_count, int burst_addr, cs_tcam_ctrl_row_t *rows, cs_tcam_ctrl_operation_t operation)
{
	int 		i;
	
	if(!cs_tcam_ctrl_burst_init(dev, space, burst_count, burst_addr, CS_TCAM_BREAD_REG))
		return false;

	/* write command without params */
	VERBOSE (CL_VERBOSE_LIBRARY, "0x%08x -> 0x%08x", 1, CS_TCAM_CMD);
	cs_space_write_4 (dev, space, CS_TCAM_CMD, 1);		

	/* wait for operation to be completed */
	cs_tcam_ctrl_wait_for_command_completion(dev, space);

	/* don't read address, write immediately */
	if(operation == CS_TCAM_CTRL_OPERATION_DATA) {
		VERBOSE (CL_VERBOSE_LIBRARY, "0x%08x -> 0x%08x", 0, CS_TCAM_ADR);
		cs_space_write_4 (dev, space, CS_TCAM_ADR, 0);
	} else if(operation == CS_TCAM_CTRL_OPERATION_MASK) {
		VERBOSE (CL_VERBOSE_LIBRARY, "0x%08x -> 0x%08x", CS_TCAM_MASK_OFFSET, CS_TCAM_ADR);
		cs_space_write_4 (dev, space, CS_TCAM_ADR, CS_TCAM_MASK_OFFSET);
	}
	cs_space_write_4 (dev, space, CS_TCAM_ADR + 4, 0);
	cs_space_write_4 (dev, space, CS_TCAM_ADR + 8, 0);
	cs_space_write_4 (dev, space, CS_TCAM_ADR + 12, 0);

	/* write command */
	cs_tcam_ctrl_write_command_with_params(dev, space, cmd_params, CS_TCAM_CTRL_BURST_READ_COMMAND);

	/* wait for operation to be completed */
	cs_tcam_ctrl_wait_for_command_completion(dev, space);

	/* read result */ 
	for (i = 0; i < burst_count; i++){
		rows[i] = cs_tcam_ctrl_read_row (dev, space, CS_TCAM_DATA + i * 16);
	}

	return true;
}

/*! 
 * \brief Initialize TCAM_CTRL
 * 
 * \param dev Device
 * \param space Space
 */
void cs_tcam_ctrl_init (cs_device_t * dev, cs_space_t * space) {

	u_int32_t value;
	int i;

	/* get control */
	cs_tcam_ctrl_get_unit_control(dev, space);

	cs_space_write_4 (dev, space, CS_TCAM_CTRL, 0x00000F11);
	cs_space_write_4 (dev, space, CS_TCAM_CTRL, 0x00000F01);

	/* write address */
	cs_space_write_4 (dev, space, CS_TCAM_ADR, 0x00180038);
	cs_space_write_4 (dev, space, CS_TCAM_ADR + 4, 0);
	cs_space_write_4 (dev, space, CS_TCAM_ADR + 8, 0);
	cs_space_write_4 (dev, space, CS_TCAM_ADR + 12, 0);

	/* write data */
	cs_space_write_4 (dev, space, CS_TCAM_DATA, 0x00015592);
	cs_space_write_4 (dev, space, CS_TCAM_DATA + 4, 0);
	cs_space_write_4 (dev, space, CS_TCAM_DATA + 8, 0);
	cs_space_write_4 (dev, space, CS_TCAM_DATA + 12, 0);

	/* write parameters and command type */
	value = cs_space_read_4 (dev, space, CS_TCAM_CMD);
	value &= 0x0000FFF0;
	value |= 1;
	cs_space_write_4 (dev, space, CS_TCAM_CMD, value);
	
	/*cs_tcam_ctrl_wait_for_command_completion(dev, space);*/
	
	/* SINGLE WRITE operation - Write TCAM internal mask registers -
	 * addresses 180020-18002F, data FFFFFFFFFFFFFFFFF */
	
	for (i = 0; i < 16; i++) { 
		/* write address */
		cs_space_write_4 (dev, space, CS_TCAM_ADR, 0x00180020 + i);
		cs_space_write_4 (dev, space, CS_TCAM_ADR + 4, 0);
		cs_space_write_4 (dev, space, CS_TCAM_ADR + 8, 0);
		cs_space_write_4 (dev, space, CS_TCAM_ADR + 12, 0);

		/* write data */
		cs_space_write_4 (dev, space, CS_TCAM_DATA, 0xFFFFFFFF);
		cs_space_write_4 (dev, space, CS_TCAM_DATA + 4, 0xFFFFFFFF);
		cs_space_write_4 (dev, space, CS_TCAM_DATA + 8, 0xFFFFFFFF);
		cs_space_write_4 (dev, space, CS_TCAM_DATA + 12, 0xFFFFFFFF);
		
		/* write parameters and command type */
		value = cs_space_read_4 (dev, space, CS_TCAM_CMD);
		value &= 0x0000FFF0;
		value |= 1;
		cs_space_write_4 (dev, space, CS_TCAM_CMD, value);
		
		/*cs_tcam_ctrl_wait_for_command_completion(dev, space);*/
	}

	cs_tcam_ctrl_clear_search_reg (dev, space);
}

/*! 
 * \brief Print multiple lines from array
 * 
 * \param rows Rows to print
 * \param count Number of rows
 * \param output File to print
 */
void cs_tcam_ctrl_print_multiple_rows(cs_tcam_ctrl_row_t *rows, int count, FILE * output) {
	int i;

	for(i = 0; i < count; i++)
		cs_tcam_ctrl_print_row(rows[i], output);
}
