/*!
 * \file 	ssramctrl.c
 * \brief 	Function definitions for controlling ssram_ctrl
 * \author 	Andrej Hank <xhanka00@liberouter.org>
 * \date 	2006, 2007
 */
/*
 * Copyright (C) 2006, 2007 CESNET
 *
 * LICENSE TERMS
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 * 3. Neither the name of the Company nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * ALTERNATIVELY, provided that this notice is retained in full, this
 * product may be distributed under the terms of the GNU General Public
 * License (GPL), in which case the provisions of the GPL apply INSTEAD
 * OF those given above.
 *
 * This software is provided ``as is'', and any express or implied
 * warranties, including, but not limited to, the implied warranties of
 * merchantability and fitness for a particular purpose are disclaimed.
 * In no event shall the company or contributors be liable for any
 * direct, indirect, incidental, special, exemplary, or consequential
 * damages (including, but not limited to, procurement of substitute
 * goods or services; loss of use, data, or profits; or business
 * interruption) however caused and on any theory of liability, whether
 * in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even
 * if advised of the possibility of such damage.
 *
 * $Id$
 *
 */

#include <err.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <commlbr.h>

#include "ssramctrl.h"

__RCSID ("$Id$");

/*!
 *\brief		Allocate unit	
 *
 *\param dev		Device
 *\param space		Space
 */
void cs_ssramctrl_get_unit_control (cs_device_t * dev, cs_space_t * space)
{
	u_int32_t	value;

	value = cs_space_read_4 (dev, space, CS_SSRAM_CTRL_REG);
	value |= 1;
	cs_space_write_4 (dev, space, CS_SSRAM_CTRL_REG, value);

	while ((value = (cs_space_read_4 (dev, space, CS_SSRAM_STAT_REG) & 1) != 1))
		usleep(100);
		
	VERBOSE (CL_VERBOSE_LIBRARY, "Unit is allocated");
}

/*!
 *\brief		Free unit	
 *
 *\param dev		Device
 *\param space		Space
 */
void cs_ssramctrl_give_unit_control (cs_device_t * dev, cs_space_t * space)
{
	u_int32_t	value;

	value = cs_space_read_4 (dev, space, CS_SSRAM_CTRL_REG);
	value = value & 0xFFFFFFFE;
	cs_space_write_4 (dev, space, CS_SSRAM_CTRL_REG, value);

	VERBOSE (CL_VERBOSE_LIBRARY, "Unit is free");
}

/*!
 *\brief		Get row from string (9 hex digits)
 *
 *\param buffer 	Buffer with value
 *\param row		Return read row
 *
 *\return
 *	1 - if successfully read
 *	0 - if failed
 */
int cs_ssramctrl_get_row_from_string (char *buffer, cs_ssramctrl_row_t *row)
{
	u_int32_t	low, high;
	
	/* fill row from buffer */
	if (sscanf(buffer, "%01x%08x", &high, &low) != 2) {
		warnx ("cs_ssramctrl_get_row_from_string - wrong string conversion %s.", buffer);
		return 0;
	}
	row->low = low;
	row->high = high;

	return 1;
}

/*!
 *\brief		Read one row from addr
 *
 *\param dev		Device
 *\param space		Space
 *\param addr		Row number. Address is masked by CS_SSRAM_ROW_OFFSET_MASK.
 *
 * \return 		One SSRAM row.
 */
cs_ssramctrl_row_t cs_ssramctrl_read_row (cs_device_t * dev, cs_space_t * space, u_int32_t addr)
{
	cs_ssramctrl_row_t row;
	
	addr&=CS_SSRAM_ROW_OFFSET_MASK;
	row.high = cs_space_read_4 (dev, space, CS_SSRAM_MEM + addr * 8 + 4);
	row.low = cs_space_read_4 (dev, space, CS_SSRAM_MEM + addr * 8 );
	VERBOSE (CL_VERBOSE_LIBRARY, "0x%01x%08x <- 0x%08x", row.high, row.low, CS_SSRAM_MEM + addr * 8);

	return row;
}

/*!
 *\brief Write into ssram values from file
 *
 *\param space		Space
 *\param addr   	Starting row number
 *\param src_file_name  Filename
 */
void cs_ssramctrl_write_file (cs_device_t * dev, cs_space_t * space, u_int32_t addr, char *src_file_name) {
	
	FILE *src_file;
	cs_ssramctrl_row_t row;
	int row_count = 0;
	char buffer[11];
	
	/* Load file. */
	if ((src_file = fopen (src_file_name, "r")) == NULL)
		errx (1, "Error open source file %s.", src_file_name);
	
	/* write whole file */
	VERBOSE(CL_VERBOSE_LIBRARY, "Writing from file %s", src_file_name);
        while (fgets (buffer, 11, src_file) != NULL) {
		if(!cs_ssramctrl_get_row_from_string(buffer, &row)) {
			fclose(src_file);
			errx (1, "Error parsing data file.");
		}

		cs_ssramctrl_write_row(dev, space, row, addr + row_count);
                row_count++;
        }
	
        fclose (src_file);
}

/*! 
 * \brief Write byte write select
 * 
 * \param dev Device
 * \param space Space
 * \param byte_write_sel Value of byte write select
 */
void cs_ssramctrl_write_byte_write_select (cs_device_t * dev, cs_space_t * space, u_int32_t byte_write_sel) {
	/* 7-4 bit @ CS_SSRAM_CTRL_REG */
	byte_write_sel = byte_write_sel << 4;
	byte_write_sel |= (cs_space_read_4(dev, space, CS_SSRAM_CTRL_REG) & 0xFFFFFF0F);
	cs_space_write_4(dev, space, CS_SSRAM_CTRL_REG, byte_write_sel);
	VERBOSE (CL_VERBOSE_LIBRARY, "0x%01x -> 0x%08x", byte_write_sel, CS_SSRAM_CTRL_REG);
}

/*!
 * \brief Set base address for rw operation
 * 
 * \param dev Dev
 * \param space Space
 * \param rw_base Base row number. It is masked by CS_SSRAM_ROW_SEGMENT_MASK.
 */
void cs_ssramctrl_set_base_addr (cs_device_t * dev, cs_space_t * space, u_int32_t rw_base) {
	rw_base&=CS_SSRAM_ROW_SEGMENT_MASK;
	
	cs_space_write_4 (dev, space, CS_SSRAM_BASEADDR_REG, rw_base);
	VERBOSE (CL_VERBOSE_LIBRARY, "0x%08x -> 0x%08x", rw_base, CS_SSRAM_BASEADDR_REG);
}

/*!
 * \brief Write one row into SSRAM memory. Base address must be
 * set by cs_ssramctrl_set_base_addr function
 * 
 * \param dev Dev
 * \param space Space
 * \param row Data to write
 * \param addr Row number.  Address is masked by CS_SSRAM_ROW_OFFSET_MASK.
 */
void cs_ssramctrl_write_row (cs_device_t * dev, cs_space_t * space, cs_ssramctrl_row_t row, u_int32_t addr) {
	addr&=CS_SSRAM_ROW_OFFSET_MASK;

	cs_space_write_4(dev, space, CS_SSRAM_MEM + addr * 8, row.low);
	cs_space_write_4(dev, space, CS_SSRAM_MEM + addr * 8 + 4, row.high);
	VERBOSE (CL_VERBOSE_LIBRARY, "0x%01x%08x -> 0x%08x", row.high, row.low, CS_SSRAM_MEM + addr * 8);
}

/*!
 * \brief Print one row to output
 * 
 * \param row Row to print
 * \param output Output stream
 */
void cs_ssramctrl_print_row (cs_ssramctrl_row_t row, FILE *output) {
	fprintf(output, "0x%01x%08x\n", row.high, row.low);
}
