/**
 * \file phy.h
 * \brief Interface for phyter device.
 * \author Miroslav Vadkerti <thrix@liberouter.org>
 * \date 2007-2008
 */

/*
 * Copyright (C) 2007-2008  CESNET
 *
 * LICENSE TERMS
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 * 3. Neither the name of the Company nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * This software is provided ``as is'', and any express or implied
 * warranties, including, but not limited to, the implied warranties of
 * merchantability and fitness for a particular purpose are disclaimed.
 * In no event shall the company or contributors be liable for any
 * direct, indirect, incidental, special, exemplary, or consequential
 * damages (including, but not limited to, procurement of substitute
 * goods or services; loss of use, data, or profits; or business
 * interruption) however caused and on any theory of liability, whether
 * in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even
 * if advised of the possibility of such damage.
 *
 * $Id$
 *
 */

#ifndef _LIBEROUTER_PHY_H_
#define _LIBEROUTER_PHY_H_

/* combo include */
#include "../combosix.h"
#include "phydevs.h"


/*!
 * \defgroup phy_component interface for XFP and SFP phyters abstraction.
 *
 * \ingroup components
 * \{
 *
 * \brief This module implements functions for controlling PHYTERI2C component (4 phyters).
 */

/*! \brief Phyter commands */
enum cs_phy_commands {
	/* GBIC & partly SFP commands */
	CS_PHY_WRITE, /* param1 address, param2 written value, return 0 ok, -1 error (i2c only) */
	CS_PHY_MDIOWR, /* write to to specified MMD's register (10Gb phys only)
                          param1 - pointer to u_int32_t array[3] specifying device addr,
                          register addr and written data */
	CS_PHY_MDIORD, /* read mdio register of specified MMD (10Gb phys only)
                          param1 - MMD device adress
                          param2 - register address */
	CS_PHY_RESET, /* software reset phyter, no parameter */
	CS_PHY_ADVSPEED, /* advertise only param1 speed and restart auto-negotiation */
	CS_PHY_GETSPEED, /* returns speed, if half duplex incremented by 1, no param */
	CS_PHY_ISOLATE, /* isolate phyter, no param */
	CS_PHY_SETSPEED, /* set speed without auto-negotiation */
	CS_PHY_ALLREGS, /* get all registers for interface (not for 10Gb phys) */
	CS_PHY_TRANSREGS, /* get 10GbE transceiver registers for interface (connected through PHYTERI2C component) */
	CS_PHY_GETLINK,	/* get link status 0 - down; 1 - up */
	CS_PHY_TXENABLE, /* enable/disable TX transmitting  */
	CS_PHY_STXDISABLE, /* soft disable TX transmitting */
	CS_PHY_SPOWERDOWN, /* soft disable TX transmitting */
	CS_PHY_ANEGSUPP, /* autonegotiation supported 0 - no; 1 - yes */
	CS_PHY_ANEGADV, /* autonegotiation advertised 0 - no; 1 - yes */
	CS_PHY_ANEG, /* autonegotiation state 0 - off, 1 - on */
	CS_PHY_SUPP10F, /* supported 10BASE-T Full Duplex 0 - no, 1 - yes */
	CS_PHY_SUPP10H, /* supported 10BASE-T Half Duplex 0 - no, 1 - yes */
	CS_PHY_SUPP1000F,  /* supported 1000BASE-T Full Duplex 0 - no, 1 - yes */
	CS_PHY_ADV10F, /* advertised 10BASE-T Full Duplex 0 - no, 1 - yes */
	CS_PHY_ADV10H, /* advertised 10BASE-T Half Duplex 0 - no, 1 - yes */
	CS_PHY_ADV100F, /* advertised 100BASE-T Full Duplex 0 - no, 1 - yes */
	CS_PHY_ADV100H, /* advertised 100BASE-T Half Duplex 0 - no, 1 - yes */
	CS_PHY_ADV1000F  /* advertised 1000BASE-T Full Duplex 0 - no, 1 - yes */
};

/*!
 * \brief Used as a PHY driver instance, contains everything you can access and do with phy.
 */
typedef struct cs_phy {
	cs_device_t 	*dev_i2c;   /* Combo device for PHYTERI2C component */
	cs_device_t 	*dev_mdio;   /* Combo device for PHYTERMDIO component */
	cs_space_t 	*space_i2c; /* Combo space for PHYTERI2C component */
	cs_space_t 	*space_mdio; /* Combo space for PHYTERMDIO component */
	struct phy_desc	*desc;	/* PHY description - must be dereferenced as (phy_desc_t*) */
	u_int32_t	dbytes; /* EEPROM data length */
	u_int32_t	link;   /* link attached to the driver */
	/* phyter services - see phyter commands*/
	int (*phy_service)(struct cs_phy *, int, u_int32_t*, u_int32_t*, u_int32_t*);

	u_int32_t	reallink; /* temporary workaround for 10g4 transceivers */
} cs_phy_t;

/*!
 * \brief Used to match a PHY.
 */
typedef struct phy_desc {
	u_int32_t 	oui;		/* the PHY's OUI */
	u_int32_t	model;		/* the PHY's model */
	const char	*vname;		/* the PHY's vendor name */
	const char 	*mname;		/* the PHY's model name */
	char		*tvname;	/* the transceiver's vendor name */
	char		*tmname;	/* the transceiver's model name */
	char		*tserial;	/* the transceiver's serial number */
	u_int32_t	rx_stat;	/* latched RX status */
	u_int32_t	tx_stat;	/* latched TX status */
	u_int32_t	recv_sig;	/* recieve signal detect (1) / loss (0) */
	u_int32_t	tx_enabled;	/* TX enabled (1) */
	u_int32_t	dev_id;   	/* device id */
	u_int32_t	dev_rev;   	/* device revision */
	u_int16_t	rx_pwr;		/* RX input power as number of 0.1uW */
	cs_phy_t	*driver;	/* phyter driver */
} phy_desc_t;

/* Public interface */
cs_phy_t*
cs_phy_attach(cs_device_t *dev_i2c, cs_device_t *dev_mdio, cs_space_t *space_i2c,
		cs_space_t *space_mdio, u_int32_t link);

void
cs_phy_detach(cs_phy_t * phy);

/*!
 * \brief Function for LED initialization when it is necessary
 *
 * \param dev	combo device
 *
 * \retval	zero on success, otherwise error
 */
int
cs_phy_init_leds(cs_device_t *dev);

/*!
 * \brief AEL2005 Initialization for ComboI-10G4 card
 */
/*  */
void phy_init_ael2005(cs_phy_t * phy);

/*!
 * \brief AEL2005 Initialization for NetFPGA 10G4 card
 */
/* */
void phy_init_netfpga_ael2005(cs_phy_t * phy);

/*!
 * \brief VSC8486 Initialization
 */
/*  */
void phy_init_vsc8486(cs_phy_t * phy);

/*! \} */

#endif /* _LIBEROUTER_PHY_H_ */
