/*!
 * \file phy.c
 * \brief Implementation of phyter component.
 * \author Miroslav Vadkerti <thrix@liberouter.org>
 * \date 2007-2009
 */

/*
 * Copyright (C) 2007-2009  CESNET
 *
 * LICENSE TERMS
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 * 3. Neither the name of the Company nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * This software is provided ``as is'', and any express or implied
 * warranties, including, but not limited to, the implied warranties of
 * merchantability and fitness for a particular purpose are disclaimed.
 * In no event shall the company or contributors be liable for any
 * direct, indirect, incidental, special, exemplary, or consequential
 * damages (including, but not limited to, procurement of substitute
 * goods or services; loss of use, data, or profits; or business
 * interruption) however caused and on any theory of liability, whether
 * in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even
 * if advised of the possibility of such damage.
 *
 * $Id$
 *
 */

#include "../combo.h"
#include <unistd.h>
#include <err.h>
#include <math.h>
#include <arpa/inet.h>
#include <commlbr.h>
#include "phy.h"
#include "phydevs.h"


__RCSID("$Id$");

/* MDIO manageable devices (MMDs) */
#define MDIO_PMA		0x01
#define MDIO_PCS		0x03
#define MDIO_PHY		0x04
#define MDIO_I2C		0x1E

/* MMD registers */
/*! PCS status register */
#define PCS_STAT		0x1
/*! PCS status register 2 */
#define PCS_STAT2	   0x8
/*! PMA status register 2 */
#define PMA_STAT2		0x8
/*! PMA status register 3 */
#define PMA_STAT3		0xA
/*! PMA control register 3 */
#define PMA_CTRL1		0x0
/*! PMA control register 3 */
#define PMA_CTRL3		0x9

/* Bits */
#define PCS_STAT_LINK		0x0004
#define PMA_STAT2_TX		0x0800 /* 1 << 11 */
#define PMA_STAT2_RX		0x0400 /* 1 << 10 */
#define PMA_CTRL1_LW		0x0800 /* bit 11 - PMA low power mode control */

/* Generic GBIC registers */
/*! Basic mode control register (rw) */
#define	GBIC_CTRL	0x00
/* Bits */
#define	GBIC_CTRL_RESET		0x8000	/* reset */
#define	GBIC_CTRL_LOOP		0x4000	/* loopback */
#define	GBIC_CTRL_S100		0x2000	/* speed (10/100) select (low bit) */
#define	GBIC_CTRL_AUTOEN	0x1000	/* autonegotiation enable */
#define	GBIC_CTRL_PDOWN		0x0800	/* power down */
#define	GBIC_CTRL_ISO		0x0400	/* isolate */
#define	GBIC_CTRL_STARTNEG	0x0200	/* restart autonegotiation */
#define	GBIC_CTRL_FDX		0x0100	/* Set duplex mode */
#define	GBIC_CTRL_CTEST		0x0080	/* collision test */
#define GBIC_CTRL_S1000		0x0040	/* speed select */

/*! Status register (rw) */
#define	GBIC_STAT	0x01
/* Bits */
/*! Link lost since last read (read twice for current) */
#define GBIC_STAT_LINK		0x0004
#define GBIC_STAT_ANEGSUPP	0x0008
#define GBIC_STAT_ANEG		0x0020
#define GBIC_STAT_SUPP10F	0x0800
#define GBIC_STAT_SUPP10H	0x0400

/*! Identification register 1 (ro) */
#define	GBIC_IDR1		0x02

/*! Identification register 2 (ro) */
#define	GBIC_IDR2		0x03

/*! Extended PHY Specific Status Register */
#define	GBIC_EXTS		0x1B

/*! Auto-Negotiation Advertisement Register (ANAR) */
#define	GBIC_ANAR		0x04
/* Bits */
#define GBIC_ANAR_100F		0x0100
#define GBIC_ANAR_100H		0x0080
#define GBIC_ANAR_10F		0x0040
#define GBIC_ANAR_10H		0x0020
#define GBIC_ANAR_ALL		0x01D0
#define GBIC_ANAR_NONE		0x0001

/*! 1000BASE-T Control Register (KTCR) */
#define	GBIC_KTCR		0x09
/* Bits */
#define GBIC_KTCR_1000F		0x0200
#define GBIC_KTCR_1000H		0x0100
#define GBIC_KTCR_ALL		0x0300
#define GBIC_KTCR_NONE		0x0000

/*! Extended Status Register (ESR) */
#define	GBIC_ESR		0xF
/* Bits */
#define GBIC_ESR_SUPP1000F	0x1000

/* Used Broadcom registers and masks */
#define BRGPHY_MII_AUXSTS	0x19	/* AUX status */
#define BRGPHY_RES_1000FD	0x0700	/* 1000baseT full duplex */
#define BRGPHY_RES_1000HD	0x0600	/* 1000baseT half duplex */
#define BRGPHY_RES_100FD	0x0500	/* 100baseT full duplex */
#define BRGPHY_RES_100T4	0x0400	/* 100baseT4 */
#define BRGPHY_RES_100HD	0x0300	/* 100baseT half duplex */
#define BRGPHY_RES_10FD		0x0200	/* 10baseT full duplex */
#define BRGPHY_RES_10HD		0x0100	/* 10baseT half duplex */

/* Generic SFP registers */
/*! Connector type */
#define SFP_TYPE		0x2
/*! Identification register 1 - address 37 */
#define SFP_IDR1		0x25
/*! Identification register 2 */
#define SFP_IDR2		0x26
/*! Identification register 3 */
#define SFP_IDR3		0x27
/*! Diagnostic Monitoring Type */
#define SFP_DMT			0x5C

/* SFP Diagnostics - Calibration Constants */
/*! RX optical power calibration data */
#define SFP_DIAG_RX_PWRC4	0x38
#define SFP_DIAG_RX_PWRC3	0x3C
#define SFP_DIAG_RX_PWRC2	0x40
#define SFP_DIAG_RX_PWRC1	0x44
#define SFP_DIAG_RX_PWRC0	0x48

/* SFP Diagnostics - Real Time Diagnostic Registers */
/*! Measured RX input power */
#define SFP_DIAG_RX_PWR	0x68

/*! Maximum SFP EEPROM registers (16b) */
#define MAX_GBIC_REGS		32
/*! Maximum GBIC EEPROM registes (8b) */
#define MAX_SFP_REGS		128
/*! Maximum 10GbE transceiver EEPROM registes (8b) */
#define MAX_10GB_TRANS_REGS	256

#define XFP_CONTROL_REG		110
#define XFP_CRB_STXDISABLE	0x40
#define XFP_CRB_SPOWERDOWN	0x08

#define XFP_ENHANCED_OPTS_REG 221
#define XFP_EOB_STXDISABLE	0x40
#define XFP_EOB_SPOWERDOWN	0x20

#define SFP_CONTROL_REG		110
#define SFP_CRB_STXDISABLE	0x40

#define SFP_ENHANCED_OPTS_REG 93
#define SFP_EOB_STXDISABLE	0x40


/* macros used for driver list specification - see components/phy.c */
#define PHY_DESC(a, b, c)	{ PHY_OUI_ ## a, PHY_MODEL_ ## a ## _ ## b, \
	PHY_OUI_NAME_ ## a, PHY_STR_ ## a ## _ ## b, NULL, NULL, NULL, 0, 0, 0, 0, 0, 0, 0, (c) }
#define PHY_END			{ 0, 0, NULL, NULL, NULL, NULL, NULL, 0, 0, 0, 0, 0, 0, 0, NULL }
/* used to specify driver */
#define PHY_DRIVER(a)		{ NULL, NULL, NULL, NULL, NULL, 0, 0, a }

/*! COMBO6_4SFPRO add on card name */
#define COMBO6_SFPRO "sfpro"
/*! COMBO6_2XFP2 add on card name */
#define COMBO6_2XFP2 "xfp2"
/*! COMBOI-10G2 add on card name */
#define COMBOI_10G2 "10G1"
/*! COMBOI-1G4 add on card name */
#define COMBOI_1G4 "1G4"
/*! COMBOI-10G4 add on card name */
#define COMBOI_10G4 "10G2"
/*! NetFPGA 10G4 card name */
#define NETFPGA_10G4 "NetFPGA 10G4"

/*! Name of component with SW I2C bus */
#define COMP_STRING_I2C "PHYTERI2C"
/*! Name of component with HW I2C bus */
#define COMP_STRING_I2C_HW "I2C_HW"

/*! Name of component with MDIO bus */
#define COMP_STRING_MDIO "PHYTERMDIO"

/*! Address of I2C HW controller for transceivers on COMBOI-10G2 for IFC0 */
#define I2C_HW_10G2_IFC0 0x38
/*! Address of I2C HW controller for transceivers on COMBOI-10G2 for IFC1 */
#define I2C_HW_10G2_IFC1 0x30

/*! I2C master adresses */
#define I2CMS_ADDRESS_IF1              0x34
#define I2CMS_ADDRESS_IF2              0x3C
#define I2CMS_DEVICE_ADDRESS           0x40

/*! I2C master adresses - the one on upper card */
#define I2CMS_2_DEVICE_ADDRESS1        0x50
#define I2CMS_2_DEVICE_ADDRESS2        0x51

#define AEL_PHY_CONTROL_REG            0x0200
#define AEL_PHY_TRANS_REG              0x0204
#define AEL_PHY_PER_LINK_OFFSET        0x0008

/*! uProgram initialization addresses */
#define PHY_INIT_AEL2005_ADDR	0xCC00

#define VSC8486_PMA_CONFIG1_ADDR	0x8000
#define VSC8486_PMA_CONFIG1_INIT	0xB5DD

/* Drivers - prototypes */
int phy_broadcom_service(cs_phy_t *phy, int command, u_int32_t *param1,
	u_int32_t *param2, u_int32_t *param3);
int phy_marvell_service(cs_phy_t *phy, int command, u_int32_t *param1,
	u_int32_t *param2, u_int32_t *param3);
int phy_sfp_service(cs_phy_t *phy, int command, u_int32_t *param1,
	u_int32_t *param2, u_int32_t *param3);
int phy_vsc_service(cs_phy_t *phy, int command, u_int32_t *param1,
	u_int32_t *param2, u_int32_t *param3);
int phy_ael_service(cs_phy_t *phy, int command, u_int32_t *param1,
	u_int32_t *param2, u_int32_t *param3);

/* DRIVERS */
/* Broadcom GBIC driver */
cs_phy_t gbic_broadcomdrv = PHY_DRIVER(&phy_broadcom_service);
/* Marvell GBIC driver */
cs_phy_t gbic_marvelldrv = PHY_DRIVER(&phy_marvell_service);
/* generic SFP driver */
cs_phy_t sfp_gendrv = PHY_DRIVER(&phy_sfp_service);
/* generic 10Gb driver */
cs_phy_t vsc_gendrv = PHY_DRIVER(&phy_vsc_service);
/* AEL2005 "driver" */
cs_phy_t ael_gendrv = PHY_DRIVER(&phy_ael_service);

/*===========================================================================*/
/* RECOGNIZED PHYTERS on 4SFPRO add on cards */
phy_desc_t i2c_phys[] = {
	/*	VENDOR		TYPE		DRIVER	*/
	PHY_DESC(xxBROADCOM,	BCM5461,	&gbic_broadcomdrv),
	PHY_DESC(MARVELL,	E1111,		&gbic_marvelldrv),
	PHY_DESC(FINISAR,	GENERIC,	&sfp_gendrv),
	PHY_DESC(AGILENT,	GENERIC,	&sfp_gendrv),
	PHY_DESC(NONE,		GENERIC,	&sfp_gendrv),
	PHY_END /* as default use driver according to EEPROM data length */
};

/*===========================================================================*/

/* phyter on 2XFP add on cards - VITESSE phys */
phy_desc_t mdio_phys[] = {
	PHY_DESC(VITESSE, VSC8476, &vsc_gendrv),
	PHY_DESC(VITESSE, VSC8486, &vsc_gendrv),
	PHY_DESC(NETLOGIC, AEL2005, &ael_gendrv),
	PHY_END
};

/******************************* Initilization *******************************/
/* Initilization sequence flags */
#define INIT_SEQ_FLAG_NONE		0
#define INIT_SEQ_FLAG_U_PROGRAM		1
#define INIT_SEQ_FLAG_END		2

/* Initilization command structure */
typedef struct INIT_SEQUENCE {
	u_int32_t address : 16;
	u_int32_t data : 16;
	u_int32_t wait_time : 30; /* in milliseconds */
	u_int32_t flag : 2;
} INIT_SEQUENCE;

/* initialization sequences */
/* AEL2005 */
/* ComboI 10G4 */
static const INIT_SEQUENCE init_sequence_AEL2005[] = {
	{0x0000, 0xA040, 500, 0},		/* Chip reset */
	{0xC017, 0xFEB0, 0, 0},			/* Flip RX_LOS polariy (for SFP+) */
	{0xC013, 0xF341, 0, 0},			/* Invert lxmit clock (undocumented) */
	{0xC01F, 0x040C, 0, 0},			/* Invert HSTXDATA polarity on all channels (schematics page #) */
	{0xC007, 0x5500, 0, 0},			/* Invert XAUI RXDATA0, RXDATA2, TDATA0, TXDATA2 (schematics page 10) */
	{0xC210, 0x8000, 0, 0},			/* Reset datapath (undocumented) */
	{0xC210, 0x8100, 0, 0},
	{0xC210, 0x8000, 0, 0},
	{0xC210, 0x0000, 500, 0},
	{0xC003, 0x0181, 0, 0},			/* Enable CDR (undocumented) */
	{0xC010, 0x448A, 0, 0},			/* Mask out high BER input from LOS signal (undocumented) */
	{0xC04A, 0x5200, 500, 0},		/* Enable the embedded uC and pause */
	{0, 0, 0, INIT_SEQ_FLAG_U_PROGRAM},	/* initilize phyter's u-program */
	{0xCA00, 0x0080, 0, 0},			/* Unpause & start program */
	{0xCA12, 0x0000, 0, 0},			/* Enable uController */
	{0, 0, 0, INIT_SEQ_FLAG_END}		/* End initilization sequence */
};

/* AEL2005 */
/* NetFPGA 10G4 */
static const INIT_SEQUENCE init_sequence_netfpga_AEL2005[] = {
	{0x0000, 0xA040, 500, 0},		/* Chip reset */
	{0xC017, 0xFEB0, 0, 0},			/* Flip RX_LOS polariy (for SFP+) */
	{0xC013, 0xF341, 0, 0},			/* Invert lxmit clock (undocumented) */
	{0xC01F, 0x0428, 0, 0},			/* Invert HSRXDATA polarity on all channels (schematics page  3,4,5,6 #) */
	{0xC007, 0x0000, 0, 0},			/* Do not invert XAUI RXDATA0, RXDATA2, TDATA0, TXDATA2 */
	{0xC210, 0x8000, 0, 0},			/* Reset datapath (undocumented) */
	{0xC210, 0x8100, 0, 0},
	{0xC210, 0x8000, 0, 0},
	{0xC210, 0x0000, 500, 0},
	{0xC003, 0x0181, 0, 0},			/* Enable CDR (undocumented) */
	{0xC010, 0x448A, 0, 0},			/* Mask out high BER input from LOS signal (undocumented) */
	{0xC04A, 0x5200, 500, 0},		/* Enable the embedded uC and pause */
	{0, 0, 0, INIT_SEQ_FLAG_U_PROGRAM},	  /* initilize phyter's u-program */
	{0xCA00, 0x0080, 0, 0},			/* Unpause & start program */
	{0xCA12, 0x0000, 0, 0},			/* Enable uController */
	{0, 0, 0, INIT_SEQ_FLAG_END}	/* End initilization sequence */
};


/* initialization micro programs */
/* AEL2005 */
static const u_int16_t init_uprog_AEL2005[] = {
	0x20c5, 0x3c05, 0x6536, 0x2fe4, 0x3cd4, 0x6624, 0x2015, 0x3145,
	0x6524, 0x27ff, 0x300f, 0x2c8b, 0x300b, 0x4009, 0x400e, 0x2f52,
	0x3002, 0x1002, 0x2202, 0x3012, 0x1002, 0x2662, 0x3012, 0x1002,
	0xd01e, 0x2862, 0x3012, 0x1002, 0x2004, 0x3c84, 0x6436, 0x2007,
	0x3f87, 0x8676, 0x40b7, 0xa746, 0x4047, 0x5673, 0x29c2, 0x3002,
	0x13d2, 0x8bbd, 0x28f2, 0x3012, 0x1002, 0x2122, 0x3012, 0x1002,
	0x5cc3, 0x0314, 0x2982, 0x3002, 0x1002, 0xd019, 0x20c2, 0x3012,
	0x1002, 0x2a04, 0x3c74, 0x6435, 0x2fa4, 0x3cd4, 0x6624, 0x5563,
	0x2d82, 0x3002, 0x13d2, 0x464d, 0x28f2, 0x3012, 0x1002, 0x20c2,
	0x3012, 0x1002, 0x2fb4, 0x3cd4, 0x6624, 0x5563, 0x2d82, 0x3002,
	0x13d2, 0x2eb2, 0x3002, 0x1002, 0x2002, 0x3012, 0x1002, 0x0004,
	0x2982, 0x3002, 0x1002, 0x2122, 0x3012, 0x1002, 0x5cc3, 0x0317,
	0x2f52, 0x3002, 0x1002, 0x2982, 0x3002, 0x1002, 0x22cd, 0x301d,
	0x28f2, 0x3012, 0x1002, 0x21a2, 0x3012, 0x1002, 0x5aa3, 0x2e02,
	0x3002, 0x1312, 0x2d42, 0x3002, 0x1002, 0x2ff7, 0x30f7, 0x20c4,
	0x3c04, 0x6724, 0x2807, 0x31a7, 0x20c4, 0x3c24, 0x6724, 0x1002,
	0x2807, 0x3187, 0x20c4, 0x3c24, 0x6724, 0x2fe4, 0x3cd4, 0x6437,
	0x20c4, 0x3c04, 0x6724, 0x1002, 0x2514, 0x3c64, 0x6436, 0xdff4,
	0x6436, 0x1002, 0x40a4, 0x643c, 0x4016, 0x8c6c, 0x2b24, 0x3c24,
	0x6435, 0x1002, 0x2b24, 0x3c24, 0x643a, 0x4025, 0x8a5a, 0x1002,
	0x27c1, 0x3011, 0x1001, 0xc7a0, 0x0100, 0xc502, 0x53ac, 0xc503,
	0xd5d5, 0xc600, 0x2a6d, 0xc601, 0x2a4c, 0xc602, 0x0111, 0xc60c,
	0x5900, 0xc710, 0x0700, 0xc718, 0x0700, 0xc720, 0x4700, 0xc801,
	0x7f50, 0xc802, 0x7760, 0xc803, 0x7fce, 0xc804, 0x5700, 0xc805,
	0x5f11, 0xc806, 0x4751, 0xc807, 0x57e1, 0xc808, 0x2700, 0xc809,
	0x0000, 0xc821, 0x0002, 0xc822, 0x0014, 0xc832, 0x1186, 0xc847,
	0x1e02, 0xc013, 0xf341, 0xc01a, 0x0446, 0xc024, 0x1000, 0xc025,
	0x0a00, 0xc026, 0x0c0c, 0xc027, 0x0c0c, 0xc029, 0x00a0, 0xc030,
	0x0a00, 0xc03c, 0x001c, 0xc005, 0x7a06, 0x0000, 0x27c1, 0x3011,
	0x1001, 0xc620, 0x0000, 0xc621, 0x003f, 0xc622, 0x0000, 0xc623,
	0x0000, 0xc624, 0x0000, 0xc625, 0x0000, 0xc627, 0x0000, 0xc628,
	0x0000, 0xc62c, 0x0000, 0x0000, 0x2806, 0x3cb6, 0xc161, 0x6134,
	0x6135, 0x5443, 0x0303, 0x6524, 0x000b, 0x1002, 0x2104, 0x3c24,
	0x2105, 0x3805, 0x6524, 0xdff4, 0x4005, 0x6524, 0x1002, 0x5dd3,
	0x0306, 0x2ff7, 0x38f7, 0x60b7, 0xdffd, 0x000a, 0x1002, 0x0000,
};

/*!
 * \brief Autodetect and initialize phyter driver for phyters on COMBOI-1G4 cards
 *
 * \param dev Combo device for I2C communication
 * \param dev_mdio Combo device for MDIO communication
 * \param space Combo space for I2C communication
 * \param space_mdio Combo space for MDIO communication
 * \param link Selected interface
 *
 * \return Pointer to phyter driver
 * \return NULL if no driver found
 */
cs_phy_t *
phy_1g4_attach(cs_device_t *dev, cs_device_t *dev_mdio, cs_space_t *space, cs_space_t *space_mdio, u_int32_t link)
{
	int k, l;		/* cycle variables */
	int dbytes;		/* EEPROM data length */
	char *string;		/* temporary pointer to string */
	cs_component_t *comp_id;/* combo component */
	cs_phy_t *phy = NULL;	/* returned driver */
	u_int32_t oui;		/* detected oui */
	u_int32_t model = 0x0;	/* detected model */
	u_int32_t udata;	/* temporary unsigned value */
	u_int32_t addr;		/* adress on I2C bus */
	u_int32_t type;		/* adress on I2C bus */
	u_int32_t dmt;		/* Digital Monitoring Type byte */
	u_int32_t rx_pwrm;  /* Measured RX input power */
	u_int32_t rx_pwr_cd[5];	/* RX Optical Power Calibration Data */
	u_int16_t rx_pwr = 0;	/* Computed RX input power */
	float rx_pwrc;

	/* TODO - implement mdio EMAC */
	dev_mdio = NULL;
	space_mdio = NULL;

	if (space == NULL) {
		/* find PHYTERI2C compoent and map it-s space */
		if (cs_component_find(dev, &comp_id, NULL, COMP_STRING_I2C_HW, link)) {
			VERBOSE(CL_VERBOSE_LIBRARY, "Component %s on index %d not found in your design."
				" Please check corresponding design.xml file.\n", COMP_STRING_I2C_HW, link);
			return NULL;
		}

		/* map phy space with PHYTERI2C */
		if (cs_component_space(comp_id, &space)) {
			VERBOSE(CL_VERBOSE_LIBRARY, "Failed to map space for %s component.\n", COMP_STRING_I2C_HW);
		}
	}

	VERBOSE(CL_VERBOSE_LIBRARY, "Available drivers: %zd",
		sizeof(i2c_phys) / sizeof(cs_phy_t) - 1);

	/* set I2C device address on I2C bus to SFP module */
	addr = 0xA0;
	VERBOSE(CL_VERBOSE_LIBRARY, "Trying I2C bus address 0x%X", addr);
	cs_i2c_set_addr(addr);
	cs_i2c_init(dev, space);

	/* set device data length */
	dbytes = 1;
	cs_i2c_set_data_bytes(dbytes);

	/* if copper -> address is 0xAC, dbytes == 2 */
	cs_i2c_read(dev, space, link, SFP_TYPE, &type);
	/* if no phyter detected */
	if (type == 0xFF) {
		return NULL;
	/* if not LC Optical - Copper */
	} else if (type != 7) {
		dbytes = 2;
		cs_i2c_set_data_bytes(dbytes);
		cs_i2c_set_addr(0xAC);
	}

	/* detect device OUI for 8bit devices use the SFP scheme */
	if (dbytes == 1) {
		cs_i2c_read(dev, space, link, SFP_IDR1, &oui);
		cs_i2c_read(dev, space, link, SFP_IDR2, &udata);
		oui = (oui << 8) | udata;
		cs_i2c_read(dev, space, link, SFP_IDR3, &udata);
		oui = (oui << 8) | udata;
	/* detect device OUI for 16bit EEPROMs */
	} else {
		cs_i2c_read(dev, space, link, GBIC_IDR1, &udata);
		oui = (0xFFFF & udata) << 6;
		cs_i2c_read(dev, space, link, GBIC_IDR2, &udata);
		oui = oui | ((0xFC00 & udata) >> 10);
		model = 0x1F & (udata >> 4);
	}

	/* detect RX Input Power in digital diagnostic monitoring if implemented */
	cs_i2c_set_data_bytes(1);
	cs_i2c_read(dev, space, link, SFP_DMT, &dmt);
	VERBOSE(CL_VERBOSE_LIBRARY, "DMT byte: %d", dmt);
	if (dmt & 0x40) {
		/* digital diagnostic monitoring is implemented */
		cs_i2c_set_addr(0xA2);

		/* read 16b measured RX input power */
		cs_i2c_set_data_bytes(2);
		cs_i2c_read(dev, space, link, SFP_DIAG_RX_PWR, &rx_pwrm);
		VERBOSE(CL_VERBOSE_LIBRARY, "RX_PWR: %d", rx_pwrm);

		/* read 5 x 32b calibration constants for RX input power */
		cs_i2c_set_data_bytes(4);
		cs_i2c_read(dev, space, link, SFP_DIAG_RX_PWRC0, &rx_pwr_cd[0]);
		VERBOSE(CL_VERBOSE_LIBRARY, "RX_PWRC0: %8x", rx_pwr_cd[0]);

		cs_i2c_read(dev, space, link, SFP_DIAG_RX_PWRC1, &rx_pwr_cd[1]);
		VERBOSE(CL_VERBOSE_LIBRARY, "RX_PWRC1: %8x", rx_pwr_cd[1]);

		cs_i2c_read(dev, space, link, SFP_DIAG_RX_PWRC2, &rx_pwr_cd[2]);
		VERBOSE(CL_VERBOSE_LIBRARY, "RX_PWRC2: %8x", rx_pwr_cd[2]);

		cs_i2c_read(dev, space, link, SFP_DIAG_RX_PWRC3, &rx_pwr_cd[3]);
		VERBOSE(CL_VERBOSE_LIBRARY, "RX_PWRC3: %8x", rx_pwr_cd[3]);

		cs_i2c_read(dev, space, link, SFP_DIAG_RX_PWRC4, &rx_pwr_cd[4]);
		VERBOSE(CL_VERBOSE_LIBRARY, "RX_PWRC4: %8x", rx_pwr_cd[4]);

		/* compute the Measured RX input power */
		for(k = 4, rx_pwr = 0; k >= 0; k--) {
			memcpy (&rx_pwrc, &rx_pwr_cd[k], 4);
			rx_pwr += (rx_pwrc * pow(rx_pwrm, k));
		}
		VERBOSE(CL_VERBOSE_LIBRARY, "RX_PWR: %u", rx_pwr);

		/* set i2c values back */
		cs_i2c_set_addr(0xA0);
	}
	cs_i2c_set_data_bytes(dbytes);


	VERBOSE(CL_VERBOSE_LIBRARY, "Detected OUI: 0x%X", oui);
	VERBOSE(CL_VERBOSE_LIBRARY, "Detected Model: 0x%X", model);
	for (k = 0; ; k++) {
		/* model matched */
		if (i2c_phys[k].oui == oui) {
			/* model matched */
			if (i2c_phys[k].model == model) {
				/* Phyter and model according to OUI and MUI */
				phy = i2c_phys[k].driver;
				phy->dev_i2c = dev;
				phy->space_i2c = space;
				phy->dev_mdio = NULL;
				phy->space_mdio = NULL;
				phy->desc = (struct phy_desc*)&i2c_phys[k];
				phy->dbytes = dbytes;
				phy->link = link;
				/* verbose results  */
				VERBOSE(CL_VERBOSE_LIBRARY, "Vendor: %s",
					phy->desc->vname);
				VERBOSE(CL_VERBOSE_LIBRARY, "Model: %s",
					phy->desc->mname);
				/* get gbic strings */
				cs_i2c_set_addr(0xA0);
				string = (char *)malloc(17 * sizeof(char));
				for (l = 0; l < 16; l++) {
					cs_i2c_read(dev, space, link, 19 + l, &udata);
					string[l] = udata & 0xFF;
				}
				string[16] = 0;
				phy->desc->tvname = string;
				VERBOSE(CL_VERBOSE_LIBRARY, "GBIC Vendor: %s", string);
				string = (char *)malloc(17 * sizeof(char));
				for (l = 0; l < 17; l++) {
					cs_i2c_read(dev, space, link, 39 + l, &udata);
					string[l] = udata & 0xFF;
				}
				string[16] = 0;
				phy->desc->tmname = string;
				VERBOSE(CL_VERBOSE_LIBRARY, "GBIC Model: %s", string);
				cs_i2c_set_addr(0xAC);
			
				phy->desc->rx_pwr = rx_pwr;
				VERBOSE(CL_VERBOSE_LIBRARY, "RX Input Power: %d", rx_pwr)
				break;
			/* if generic SFP matched */
			} else if (i2c_phys[k].model == 0x0) {
				/* get PHY's strings */
				string = (char *)malloc(17 * sizeof(char));
				for (l = 0; l < 16; l++) {
					cs_i2c_read(dev, space, link, 20 + l, &udata);
					string[l] = udata & 0xFF;
				}
				string[16] = 0;
				phy->desc->vname = string;
				VERBOSE(CL_VERBOSE_LIBRARY, "Vendor: %s", string);
				string = (char *)malloc(17 * sizeof(char));
				for (l = 0; l < 17; l++) {
					cs_i2c_read(dev, space, link, 40 + l, &udata);
					string[l] = udata & 0xFF;
				}
				string[16] = 0;
				phy->desc->mname = string;
				VERBOSE(CL_VERBOSE_LIBRARY, "Model: %s", string);

				/* get gbic strings */
				cs_i2c_set_addr(0xAC);
				string = (char *)malloc(17 * sizeof(char));
				for (l = 0; l < 16; l++) {
					cs_i2c_read(dev, space, link, 20 + l, &udata);
					string[l] = udata & 0xFF;
				}
				string[16] = 0;
				phy->desc->tvname = string;
				VERBOSE(CL_VERBOSE_LIBRARY, "GBIC Vendor: %s", string);
				string = (char *)malloc(17 * sizeof(char));
				for (l = 0; l < 17; l++) {
					cs_i2c_read(dev, space, link, 40 + l, &udata);
					string[l] = udata & 0xFF;
				}
				string[16] = 0;
				phy->desc->tmname = string;
				VERBOSE(CL_VERBOSE_LIBRARY, "GBIC Model: %s", string);
				cs_i2c_set_addr(0xA0);
			}
		}
	}

	return phy;
}

/*!
 * \brief Autodetect and initialize phyter driver for phyters on SFP cards
 *
 * \param dev Combo device
 * \param space Combo space
 * \param link Selected interface
 * \addr
 *
 * \return Pointer to phyter driver
 * \return NULL if no driver found
 */
cs_phy_t *
phy_sfp_attach(cs_device_t *dev, cs_space_t *sp, u_int32_t link)
{
	int k, l;		/* cycle variables */
	int dbytes;		/* EEPROM data length */
	char *string;		/* temporary pointer to string */
	cs_component_t *comp_id;/* combo component */
	cs_phy_t *phy = NULL;	/* returned driver */
	cs_space_t *space = sp; /* component space */
	u_int32_t oui;		/* detected oui */
	u_int32_t model = 0x0;	/* detected model */
	u_int32_t udata;	/* temporary unsigned value */
	u_int32_t addr;		/* adress on I2C bus */

	if (space == NULL) {
		/* find PHYTERI2C compoent and map it-s space */
		if (cs_component_find(dev, &comp_id, NULL, COMP_STRING_I2C, 0) != 0) {
			VERBOSE(CL_VERBOSE_LIBRARY, "Component %s not found in your design."
				" Please check corresponding design.xml file.\n", COMP_STRING_I2C);
			return NULL;
		}

		/* map phy space with PHYTERI2C */
		if (cs_component_space(comp_id, &space) != 0) {
			VERBOSE(CL_VERBOSE_LIBRARY, "Failed to map space for %s component.\n", COMP_STRING_I2C);
		}
	}

	VERBOSE(CL_VERBOSE_LIBRARY, "Available drivers: %zd",
		sizeof(i2c_phys) / sizeof(cs_phy_t) - 1);

	/* set I2C device address on I2C bus */
	addr = 0xAC;
	VERBOSE(CL_VERBOSE_LIBRARY, "Trying I2C bus address 0x%X", addr);
	cs_i2c_set_addr(addr);

	/* get device data length */
	/* try 0xAC eeproms */
	dbytes = cs_i2c_detect_dlength(dev, space, link, addr);
	if (dbytes < 0) {
		/* try 0xA0 eeproms */
		addr = 0xA0;
		VERBOSE(CL_VERBOSE_LIBRARY, "Trying I2C bus address 0x%X", addr);
		cs_i2c_set_addr(addr);
		dbytes = cs_i2c_detect_dlength(dev, space, link, addr);
		dbytes++;
		VERBOSE(CL_VERBOSE_LIBRARY, "dbytes detected %d", dbytes);
		if (dbytes != 1) {
			VERBOSE(CL_VERBOSE_LIBRARY, "No acceptable phyter EEPROMS detected.");
			return NULL;
		}
	} else {
		dbytes++;
	}
	VERBOSE(CL_VERBOSE_LIBRARY, "Device data length detected: %d bits", dbytes * 8);

	/* set device data length */
	cs_i2c_set_data_bytes(dbytes);

	/* detect device OUI for 8bit devices use the SFP scheme */
	if (dbytes == 1) {
		cs_i2c_read(dev, space, link, SFP_IDR1, &oui);
		cs_i2c_read(dev, space, link, SFP_IDR2, &udata);
		oui = (oui << 8) | udata;
		cs_i2c_read(dev, space, link, SFP_IDR3, &udata);
		oui = (oui << 8) | udata;
	/* detect device OUI for 16bit EEPROMs */
	} else {
		cs_i2c_read(dev, space, link, GBIC_IDR1, &udata);
		oui = (0xFFFF & udata) << 6;
		cs_i2c_read(dev, space, link, GBIC_IDR2, &udata);
		oui = oui | ((0xFC00 & udata) >> 10);
		model = 0x1F & (udata >> 4);
	}

	VERBOSE(CL_VERBOSE_LIBRARY, "Detected OUI: 0x%X", oui);
	VERBOSE(CL_VERBOSE_LIBRARY, "Detected Model: 0x%X", model);
	for (k = 0; ; k++) {
		/* model matched */
		if (i2c_phys[k].oui == oui) {
			/* model matched */
			if (i2c_phys[k].model == model) {
				/* Phyter and model according to OUI and MUI */
				phy = i2c_phys[k].driver;
				phy->dev_i2c = dev;
				phy->space_i2c = space;
				phy->dev_mdio = NULL;
				phy->space_mdio = NULL;
				phy->desc = (struct phy_desc *)&i2c_phys[k];
				phy->dbytes = dbytes;
				phy->link = link;
				/* verbose results  */
				VERBOSE(CL_VERBOSE_LIBRARY, "Vendor: %s",
					phy->desc->vname);
				VERBOSE(CL_VERBOSE_LIBRARY, "Model: %s",
					phy->desc->mname);
				/* get gbic strings */
				cs_i2c_set_addr(0xA0);
				string = (char *)malloc(17 * sizeof(char));
				for (l = 0; l < 16; l++) {
					cs_i2c_read(dev, space, link, 20 + l, &udata);
					string[l] = udata & 0xFF;
				}
				string[16] = 0;
				phy->desc->tvname = string;
				VERBOSE(CL_VERBOSE_LIBRARY, "GBIC Vendor: %s", string);
				string = (char *)malloc(17 * sizeof(char));
				for (l = 0; l < 17; l++) {
					cs_i2c_read(dev, space, link, 40 + l, &udata);
					string[l] = udata & 0xFF;
				}
				string[16] = 0;
				phy->desc->tmname = string;
				VERBOSE(CL_VERBOSE_LIBRARY, "GBIC Model: %s", string);
				cs_i2c_set_addr(0xAC);
				break;
			/* if generic SFP matched */
			} else if (i2c_phys[k].model == 0x0) {
				/* get PHY's strings */
				string = (char *)malloc(17 * sizeof(char));
				for (l = 0; l < 16; l++) {
					cs_i2c_read(dev, space, link, 20 + l, &udata);
					string[l] = udata & 0xFF;
				}
				string[16] = 0;
				phy->desc->vname = string;
				VERBOSE(CL_VERBOSE_LIBRARY, "Vendor: %s", string);
				string = (char *)malloc(17 * sizeof(char));
				for (l = 0; l < 17; l++) {
					cs_i2c_read(dev, space, link, 40 + l, &udata);
					string[l] = udata & 0xFF;
				}
				string[16] = 0;
				phy->desc->mname = string;
				VERBOSE(CL_VERBOSE_LIBRARY, "Model: %s", string);

				/* get gbic strings */
				cs_i2c_set_addr(0xAC);
				string = (char *)malloc(17 * sizeof(char));
				for (l = 0; l < 16; l++) {
					cs_i2c_read(dev, space, link, 20 + l, &udata);
					string[l] = udata & 0xFF;
				}
				string[16] = 0;
				phy->desc->tvname = string;
				VERBOSE(CL_VERBOSE_LIBRARY, "GBIC Vendor: %s", string);
				string = (char *)malloc(17 * sizeof(char));
				for (l = 0; l < 17; l++) {
					cs_i2c_read(dev, space, link, 40 + l, &udata);
					string[l] = udata & 0xFF;
				}
				string[16] = 0;
				phy->desc->tmname = string;
				VERBOSE(CL_VERBOSE_LIBRARY, "GBIC Model: %s", string);
				cs_i2c_set_addr(0xA0);
			}
		}
	}

	return phy;
}

/*!
 * \brief Autodetect and initialize phyter driver for phyters on XFP cards
 *
 * \param dev_i2c Combo I2C device
 * \param dev_i2c Combo MDIO device
 * \param space Combo I2C space
 * \param space Combo MDIO space
 * \param link Selected interface
 *
 * \return Pointer to phyter driver
 * \return NULL if no driver found
 */
cs_phy_t *
phy_xfp_attach(cs_device_t *dev_i2c, cs_device_t *dev_mdio, cs_space_t *sp_i2c,
	cs_space_t *sp_mdio, u_int32_t link)
{
	int dev_id;
	int stat;
	cs_phy_t *phy = NULL;		/* returned driver */
	cs_component_t *comp_id_mdio;	/* component ID */
	cs_component_t *comp_id_i2c;	/* component ID */
	cs_space_t *space_i2c = sp_i2c;
	cs_space_t *space_mdio = sp_mdio;
	u_int32_t slink = link;

	if (space_mdio == NULL) {
		/* find PHYTERMDIO compoent and map it-s space */
		if (cs_component_find(dev_mdio, &comp_id_mdio, NULL, COMP_STRING_MDIO, 0) != 0) {
			VERBOSE(CL_VERBOSE_LIBRARY, "Component %s not found in your design."
				" Please check corresponding design.xml file.\n", COMP_STRING_MDIO);
			return NULL;
		}

		/* map phy space with PHYTERMDIO */
		if (cs_component_space(comp_id_mdio, &space_mdio) != 0) {
			VERBOSE(CL_VERBOSE_LIBRARY, "Failed to map space for %s component.\n",
				COMP_STRING_MDIO);
			return NULL;
		}
	}

	/* PHYTERI2C component needed for transceiver access */
	if (space_i2c == NULL) {
		if (cs_component_find(dev_i2c, &comp_id_i2c, NULL, COMP_STRING_I2C, (slink == 0) ? 1 : 2) != 0) {
			VERBOSE(CL_VERBOSE_LIBRARY, "Component %s not found in your design. Transceiver EEPROM"
				" will not be available.\n", COMP_STRING_I2C);
		}

		/* map phy space with PHYTERI2C */
		if (cs_component_space(comp_id_i2c, &space_i2c) != 0) {
			VERBOSE(CL_VERBOSE_LIBRARY, "Failed to map space for %s component. Transceiver EEPROM"
						" will not be available.\n", COMP_STRING_I2C);
		}

		cs_i2c_set_reg(0x4);
		cs_i2c_set_addr(0xA0);
		cs_i2c_init(dev_i2c, space_i2c);
	}

	/* read device id */
	dev_id = cs_mdio_read(dev_mdio, space_mdio, slink, MDIO_PMA, 0xE800);
	VERBOSE(CL_VERBOSE_LIBRARY, "Detected device id: 0x%X", dev_id);

	if (dev_id == PHY_MODEL_VITESSE_VSC8476) {
		VERBOSE(CL_VERBOSE_LIBRARY, "Attaching VSC8476 driver");
		/* attach VSC8476 PHY driver */
		phy = mdio_phys[0].driver;
		phy->dev_mdio = dev_mdio;
		phy->space_mdio = space_mdio;
		phy->dev_i2c = dev_i2c;
		phy->space_i2c = space_i2c;
		phy->desc = &mdio_phys[0];
		phy->dbytes = 0;
		phy->link = slink;
	} else if (dev_id == PHY_MODEL_VITESSE_VSC8486) {
		VERBOSE(CL_VERBOSE_LIBRARY, "Attaching VSC8486 driver");
		/* attach VSC8786 PHY driver */
		phy = mdio_phys[1].driver;
		phy->dev_mdio = dev_mdio;
		phy->space_mdio = space_mdio;
		phy->dev_i2c = dev_i2c;
		phy->space_i2c = space_i2c;
		phy->desc = &mdio_phys[1];
		phy->dbytes = 0;
		phy->link = slink;
	} else {
		VERBOSE(CL_VERBOSE_LIBRARY, "Unknown device ID 0x%X, no driver attached!", dev_id);
		return NULL;
	}

	phy->desc->dev_id = dev_id;

	/* read device rev */
	phy->desc->dev_rev = cs_mdio_read(dev_mdio, space_mdio, slink, MDIO_PMA, 0xE801);
	VERBOSE(CL_VERBOSE_LIBRARY, "Detected device revision: 0x%X", phy->desc->dev_rev);

	/* Read RX/TX latched status */
	cs_mdio_read(dev_mdio, space_mdio, slink, MDIO_PMA, PMA_STAT2);
	stat = cs_mdio_read(dev_mdio, space_mdio, slink, MDIO_PMA, PMA_STAT2);
	phy->desc->tx_stat = stat & PMA_STAT2_TX;
	phy->desc->rx_stat = stat & PMA_STAT2_RX;

	/* Read recieve signal detect */
	phy->desc->recv_sig = cs_mdio_read(dev_mdio, space_mdio, slink, MDIO_PMA, PMA_STAT3) & 1;

	/* Read TX state */
	phy->desc->tx_enabled = !(cs_mdio_read(dev_mdio, space_mdio, slink, MDIO_PMA, PMA_CTRL1) & PMA_CTRL1_LW);

	return phy;
}

/*!
 * \brief Autodetect and initialize phyter driver for phyters on ComboI-10G2 cards
 *
 * \param dev_i2c Combo I2C device
 * \param dev_i2c Combo MDIO device
 * \param space Combo I2C space
 * \param space Combo MDIO space
 * \param link Selected interface
 *
 * \return Pointer to phyter driver
 * \return NULL if no driver found
 */
cs_phy_t *
phy_10g2_attach(cs_device_t *dev_i2c, cs_device_t *dev_mdio,
	cs_space_t *sp_i2c, cs_space_t *sp_mdio, u_int32_t link)
{
	int dev_id;
	int stat;
	cs_phy_t *phy = NULL;		/* returned driver */
	cs_component_t *comp_id_mdio;	/* component ID */
	cs_space_t *space_i2c = sp_i2c;
	cs_space_t *space_mdio = sp_mdio;
	u_int32_t slink = 0;

	if (space_mdio == NULL) {
		/* find PHYTERMDIO compoent and map it-s space */
		if (cs_component_find(dev_mdio, &comp_id_mdio, NULL, COMP_STRING_MDIO, link) != 0) {
			VERBOSE(CL_VERBOSE_LIBRARY, "Component %s not found in your design."
				" Please check corresponding design.xml file.\n", COMP_STRING_MDIO);
			return NULL;
		}

		/* map phy space with PHYTERMDIO */
		if (cs_component_space(comp_id_mdio, &space_mdio) != 0) {
			VERBOSE(CL_VERBOSE_LIBRARY, "Failed to map space for %s component.\n",
				COMP_STRING_MDIO);
			return NULL;
		}
	}

	/* PHYTERI2C component needed for transceiver access */
	if (space_i2c == NULL) {
		if (link == 0) {
			/* map space for HW I2C controller */
			VERBOSE(CL_VERBOSE_LIBRARY, "Map interface 0 controller");
			if (cs_space_map(dev_i2c, &space_i2c, CS_SPACE_BRIDGE,
				0x1000, I2C_HW_10G2_IFC1, 0) != 0) {
					errx(1, "Failed to map I2C HW "
						"component at base "
						"address 0x0!\n");
			}
			cs_i2c_set_addr(0xA0);
			cs_i2c_init(dev_i2c, space_i2c);
		} else if (link == 1) {
			VERBOSE(CL_VERBOSE_LIBRARY, "Map interface 1 controller");
			/* map fpga space for PHYTERI2C component at specified base address */
			if (cs_space_map(dev_i2c, &space_i2c, CS_SPACE_BRIDGE,
				0x1000, I2C_HW_10G2_IFC0, 0) != 0) {
					errx(1, "Failed to map I2C HW "
						"component at base "
						"address 0x0!\n");
			}
			cs_i2c_set_addr(0xA0);
			cs_i2c_init(dev_i2c, space_i2c);
		} else {
			VERBOSE(CL_VERBOSE_LIBRARY, "Wrong link number: %d", link);
			return NULL;
		}
	}

	/* read device id */
	dev_id = cs_mdio_read(dev_mdio, space_mdio, slink, MDIO_PMA, 0xE800);
	VERBOSE(CL_VERBOSE_LIBRARY, "Detected device id: 0x%X", dev_id);

	if (dev_id == PHY_MODEL_VITESSE_VSC8476) {
		VERBOSE(CL_VERBOSE_LIBRARY, "Attaching VSC8476 driver");
		/* attach VSC8476 PHY driver */
		phy = mdio_phys[0].driver;
		phy->dev_mdio = dev_mdio;
		phy->space_mdio = space_mdio;
		phy->dev_i2c = dev_i2c;
		phy->space_i2c = space_i2c;
		phy->desc = &mdio_phys[0];
		phy->dbytes = 0;
		phy->link = slink;
	} else if (dev_id == PHY_MODEL_VITESSE_VSC8486) {
		VERBOSE(CL_VERBOSE_LIBRARY, "Attaching VSC8486 driver");
		/* attach VSC8786 PHY driver */
		phy = mdio_phys[1].driver;
		phy->dev_mdio = dev_mdio;
		phy->space_mdio = space_mdio;
		phy->dev_i2c = dev_i2c;
		phy->space_i2c = space_i2c;
		phy->desc = &mdio_phys[1];
		phy->dbytes = 0;
		phy->link = slink;
	} else {
		VERBOSE(CL_VERBOSE_LIBRARY, "Unknown device ID 0x%X, no driver attached!", dev_id);
		return NULL;
	}

	phy->desc->dev_id = dev_id;

	/* read device rev */
	phy->desc->dev_rev = cs_mdio_read(dev_mdio, space_mdio, slink, MDIO_PMA, 0xE801);
	VERBOSE(CL_VERBOSE_LIBRARY, "Detected device revision: 0x%X", phy->desc->dev_rev);

	/* Read RX/TX latched status */
	cs_mdio_read(dev_mdio, space_mdio, slink, MDIO_PMA, PMA_STAT2);
	stat = cs_mdio_read(dev_mdio, space_mdio, slink, MDIO_PMA, PMA_STAT2);
	phy->desc->tx_stat = stat & PMA_STAT2_TX;
	phy->desc->rx_stat = stat & PMA_STAT2_RX;

	/* Read recieve signal detect */
	phy->desc->recv_sig = cs_mdio_read(dev_mdio, space_mdio, slink, MDIO_PMA, PMA_STAT3) & 1;

	/* Read TX state */
	phy->desc->tx_enabled = !(cs_mdio_read(dev_mdio, space_mdio, slink, MDIO_PMA, PMA_CTRL1) & PMA_CTRL1_LW);

	return phy;
}

/*!
 * \brief Autodetect and initialize phyter driver for phyters on COMBOI-10G4TXT cards
 *
 * \param dev_i2c Combo I2C device
 * \param dev_i2c Combo MDIO device
 * \param space Combo I2C space
 * \param space Combo MDIO space
 * \param link Selected interface
 *
 * \return Pointer to phyter driver
 * \return NULL if no driver found
 */
cs_phy_t *
phy_10g4_attach(cs_device_t *dev_i2c, cs_device_t *dev_mdio,
	cs_space_t *sp_i2c, cs_space_t *sp_mdio, u_int32_t link)
{
	int dev_id;
	int stat;
	cs_phy_t *phy = NULL;		/* returned driver */
	cs_component_t *comp_id_mdio;	/* component ID */
	cs_space_t *space_i2c = sp_i2c;
	cs_space_t *space_mdio = sp_mdio;
	u_int32_t slink = (link & 1) << 1;

	if (space_mdio == NULL) {
		/* find PHYTERMDIO component and map it-s space */
		if (cs_component_find(dev_mdio, &comp_id_mdio, NULL, COMP_STRING_MDIO, link / 2)) {
			VERBOSE(CL_VERBOSE_LIBRARY, "Component %s not found in your design."
				" Please check corresponding design.xml file.\n", COMP_STRING_MDIO);
			return NULL;
		}

		/* map phy space with PHYTERMDIO */
		if (cs_component_space(comp_id_mdio, &space_mdio)) {
			VERBOSE(CL_VERBOSE_LIBRARY, "Failed to map space for %s component.\n",
				COMP_STRING_MDIO);
			return NULL;
		}
	}

	/* I2C component needed for transceiver access */
	if (space_i2c == NULL) {
		if (link > 3) {
			VERBOSE(CL_VERBOSE_LIBRARY, "Wrong link number: %d", link);
			return NULL;
		} else {
			VERBOSE(CL_VERBOSE_LIBRARY, "Map interface controller");
			if (cs_space_map(dev_i2c, &space_i2c, CS_SPACE_BRIDGE,
				4, I2CMS_ADDRESS_IF1, 0)) {
					errx(1, "Failed to map I2C "
						"component at address %X!\n",
						I2CMS_ADDRESS_IF1);
			}
		}
	}
	cs_i2cms_init(dev_i2c, space_i2c);

	/* Try AEL2005 device id */
	dev_id = cs_mdio_read(dev_mdio, space_mdio, slink, MDIO_PHY, 0x0002) << 16;
	dev_id |= cs_mdio_read(dev_mdio, space_mdio, slink, MDIO_PHY, 0x0003);
	VERBOSE(CL_VERBOSE_LIBRARY, "Detected device id: 0x%X", dev_id);

	if (dev_id == PHY_MODEL_NETLOGIC_AEL2005) {
		VERBOSE(CL_VERBOSE_LIBRARY, "Attaching AEL2005 driver");
	   phy = mdio_phys[2].driver;
	   phy->dev_mdio = dev_mdio;
	   phy->space_mdio = space_mdio;
	   phy->dev_i2c = dev_i2c;
	   phy->space_i2c = space_i2c;
	   phy->desc = &mdio_phys[2];
	   phy->dbytes = 0;
	   phy->link = slink;
	   phy->reallink = link;
	   phy->desc->dev_id = dev_id;

	   /* read device rev - n/a */
	   phy->desc->dev_rev = 0;
	   VERBOSE(CL_VERBOSE_LIBRARY, "Detected device revision: n/a");

	   /* Read RX/TX latched status */
	   cs_mdio_read(dev_mdio, space_mdio, slink, MDIO_PMA, PMA_STAT2);
	   stat = cs_mdio_read(dev_mdio, space_mdio, slink, MDIO_PMA, PMA_STAT2);
	   phy->desc->tx_stat = stat & PMA_STAT2_TX;
	   phy->desc->rx_stat = stat & PMA_STAT2_RX;

	   /* Read recieve signal detect */
	   phy->desc->recv_sig = cs_mdio_read(dev_mdio, space_mdio, slink, MDIO_PMA, PMA_STAT3) & 1;

	   /* Read TX state */
	   phy->desc->tx_enabled = !(cs_mdio_read(dev_mdio, space_mdio, slink, MDIO_PMA, PMA_CTRL1) & PMA_CTRL1_LW);

	   return phy;
	} else {
	   /* Try Vitesse device id */
	   dev_id = cs_mdio_read(dev_mdio, space_mdio, slink, MDIO_PMA, 0xE800);
	   VERBOSE(CL_VERBOSE_LIBRARY, "Detected device id: 0x%X", dev_id);

	   if (dev_id == PHY_MODEL_VITESSE_VSC8486) {
		   /* attach VSC8486 PHY driver */
		   VERBOSE(CL_VERBOSE_LIBRARY, "Attaching VSC8486 driver");
		   /* TODO - FIXME */
		   /* AEL driver is used, because the VSC driver doesn't implement correct transceiver */
		   /* communication on the 10G4 card (see CS_PHY_TRANSREGS case in phy_ael_service() */
		   phy = mdio_phys[2].driver;
		   phy->dev_mdio = dev_mdio;
		   phy->space_mdio = space_mdio;
		   phy->dev_i2c = dev_i2c;
		   phy->space_i2c = space_i2c;
		   phy->desc = &mdio_phys[1];
		   phy->dbytes = 0;
		   phy->link = slink;
           phy->reallink = link;
	   } else {
		   VERBOSE(CL_VERBOSE_LIBRARY, "Unknown device ID 0x%X, no driver attached!", dev_id);
		   return NULL;
	   }

	   phy->desc->dev_id = dev_id;

	   /* read device rev */
	   phy->desc->dev_rev = cs_mdio_read(dev_mdio, space_mdio, slink, MDIO_PMA, 0xE801);
	   VERBOSE(CL_VERBOSE_LIBRARY, "Detected device revision: 0x%X", phy->desc->dev_rev);

	   /* Read RX/TX latched status */
	   cs_mdio_read(dev_mdio, space_mdio, slink, MDIO_PMA, PMA_STAT2);
	   stat = cs_mdio_read(dev_mdio, space_mdio, slink, MDIO_PMA, PMA_STAT2);
	   phy->desc->tx_stat = stat & PMA_STAT2_TX;
	   phy->desc->rx_stat = stat & PMA_STAT2_RX;

	   /* Read recieve signal detect */
	   phy->desc->recv_sig = cs_mdio_read(dev_mdio, space_mdio, slink, MDIO_PMA, PMA_STAT3) & 1;

	   /* Read TX state */
	   phy->desc->tx_enabled = !(cs_mdio_read(dev_mdio, space_mdio, slink, MDIO_PMA, PMA_CTRL1) & PMA_CTRL1_LW);

	   return phy;
	}

}

/*!
 * \brief Detach phyter driver
 *
 * \param phy Phyter driver
 */
void
cs_phy_detach(cs_phy_t * phy)
{
	free(phy->desc->tvname);
	free(phy->desc->tmname);
}

/*!
 * \brief Autodetect and initialize phyter driver for phyters on NetFPGA 10G4 cards
 *
 * \param dev_i2c Combo I2C device
 * \param dev_i2c Combo MDIO device
 * \param space Combo I2C space
 * \param space Combo MDIO space
 * \param link Selected interface
 *
 * \return Pointer to phyter driver
 * \return NULL if no driver found
 */
cs_phy_t *
phy_netfpga10g4_attach(cs_device_t *dev_i2c, cs_device_t *dev_mdio,
	cs_space_t *sp_i2c, cs_space_t *sp_mdio, u_int32_t link)
{
	int dev_id;
	int stat;
	cs_phy_t *phy = NULL;		   /* returned driver */
	cs_component_t *comp_id_mdio;   /* component ID */
	cs_space_t *space_i2c = sp_i2c;
	cs_space_t *space_mdio = sp_mdio;
//	 u_int32_t slink = (link & 1) << 1;
	u_int32_t slink = link;

	if (space_mdio == NULL) {
		/* find PHYTERMDIO component and map it-s space */
		if (cs_component_find(dev_mdio, &comp_id_mdio, NULL, COMP_STRING_MDIO, 0)) {
			VERBOSE(CL_VERBOSE_LIBRARY, "Component %s not found in your design."
				" Please check corresponding design.xml file.\n", COMP_STRING_MDIO);
			return NULL;
		}

		/* map phy space with PHYTERMDIO */
		if (cs_component_space(comp_id_mdio, &space_mdio)) {
			VERBOSE(CL_VERBOSE_LIBRARY, "Failed to map space for %s component.\n",
					COMP_STRING_MDIO);
			return NULL;
		}
	}

	/* I2C component needed for transceiver access */
	if (space_i2c == NULL) {
		if (link > 3) {
			VERBOSE(CL_VERBOSE_LIBRARY, "Wrong link number: %d", link);
			return NULL;
		} else {
			VERBOSE(CL_VERBOSE_LIBRARY, "Map interface controller");
			if (cs_space_map(dev_i2c, &space_i2c, CS_SPACE_BRIDGE,
					4, I2CMS_ADDRESS_IF1, 0)) {
				errx(1, "Failed to map I2C "
					"component at address %X!\n",
					I2CMS_ADDRESS_IF1);
			}
		}
	}
	cs_i2cms_init(dev_i2c, space_i2c);

	/* Try AEL2005 device id */
	dev_id = cs_mdio_read(dev_mdio, space_mdio, slink, MDIO_PHY, 0x0002) << 16;
	dev_id |= cs_mdio_read(dev_mdio, space_mdio, slink, MDIO_PHY, 0x0003);
	VERBOSE(CL_VERBOSE_LIBRARY, "Detected device id: 0x%X", dev_id);

	if (dev_id == PHY_MODEL_NETLOGIC_AEL2005) {
		VERBOSE(CL_VERBOSE_LIBRARY, "Attaching AEL2005 driver");
		phy = mdio_phys[2].driver;
		phy->dev_mdio = dev_mdio;
		phy->space_mdio = space_mdio;
		phy->dev_i2c = dev_i2c;
		phy->space_i2c = space_i2c;
		phy->desc = &mdio_phys[2];
		phy->dbytes = 0;
		phy->link = slink;
		phy->desc->dev_id = dev_id;

		/* read device rev - n/a */
		phy->desc->dev_rev = 0;
		VERBOSE(CL_VERBOSE_LIBRARY, "Detected device revision: n/a");

		/* Read RX/TX latched status */
		cs_mdio_read(dev_mdio, space_mdio, slink, MDIO_PMA, PMA_STAT2);
		stat = cs_mdio_read(dev_mdio, space_mdio, slink, MDIO_PMA, PMA_STAT2);
		phy->desc->tx_stat = stat & PMA_STAT2_TX;
		phy->desc->rx_stat = stat & PMA_STAT2_RX;

		/* Read recieve signal detect */
		phy->desc->recv_sig = cs_mdio_read(dev_mdio, space_mdio, slink, MDIO_PMA, PMA_STAT3) & 1;

		/* Read TX state */
		phy->desc->tx_enabled = !(cs_mdio_read(dev_mdio, space_mdio, slink, MDIO_PMA, PMA_CTRL1) & PMA_CTRL1_LW);

		return phy;
	} else {
		VERBOSE(CL_VERBOSE_LIBRARY, "Unknown device ID 0x%X, no driver attached!", dev_id);
		return NULL;
	}
}

/*!
 * \brief Autodetect card type and attach i2c (COMBO-4SFPRO) or mdio (COMBO-2XFP2) phyter
 *
 * \param dev Combo device
 * \param space Combo space
 * \param link Selected interface
 *
 * \return Pointer to phyter driver
 * \return NULL if attach failed - unknown board
 */
cs_phy_t *
cs_phy_attach(cs_device_t *dev_i2c, cs_device_t *dev_mdio,
	cs_space_t *space_i2c, cs_space_t *space_mdio, u_int32_t link)
{
	char *board, *card, *chip;	/* cs_identify params */

	/* if no mdio device -> use same as for i2c */
	if (dev_mdio == NULL)
		dev_mdio = dev_i2c;

	/* check for combo6x board */
	if (cs_identify(dev_mdio, &board, &card, &chip) != 0)
		return NULL;

	VERBOSE(CL_VERBOSE_LIBRARY, "Addon: %s", card);
	/* 4SFPRO */
	if (!strcmp(card, COMBO6_SFPRO)) {
		return phy_sfp_attach(dev_i2c, space_i2c, link);
	/* 2XFP2 - cards with identification prefix xfp2 */
	} else if (!strncmp(card, COMBO6_2XFP2, sizeof(COMBO6_2XFP2) - 1)) {
		/* on link 0 is a i2c phyter */
		if (link == 0)
			return phy_sfp_attach(dev_i2c, space_i2c, link);
		/* link 2 -> link 0 on mdio bus
		   link 1 -> link 1 on mdio bus */
		return phy_xfp_attach(dev_i2c, dev_mdio, space_i2c, space_mdio, 2 - link);
	/* COMBOI-10G2 */
	} else if (!strncmp(card, COMBOI_10G2, sizeof(COMBOI_10G2) - 1)) {
		/* link -> used only for separation of transceivers, separated mdio controllers for links */
		return phy_10g2_attach(dev_i2c, dev_mdio, space_i2c, space_mdio, link);
	} else if (!strcmp(card, COMBOI_1G4)) {
		return phy_1g4_attach(dev_i2c, dev_mdio,  space_i2c, space_mdio, link);
	} else if (!strcmp(card, COMBOI_10G4)) {
		return phy_10g4_attach(dev_i2c, dev_mdio, space_i2c, space_mdio, link);
	} else if (!strcmp(board, NETFPGA_10G4)) {
		return phy_netfpga10g4_attach(dev_i2c, dev_mdio, space_i2c, space_mdio, link);
	} else {
		VERBOSE(2, "No compatible addon detected!");
		return NULL;
	}
}

/*!
 *
 * \brief Read 64 registers of a Broadcom EEPROM.
 *
 * \param dev Pointer to combo device.
 * \param phy Pointer to mapped phyter combo space.
 * \param sel Selected phyter.
 * \param regs Array for storing 16 registers.
 *
 * \retval 0 on success, 1 value if succeded.
 */
int
phy_sfp_allregs(cs_phy_t *phy, u_int8_t *regs)
{
	int i;		/* cycle variable */
	u_int16_t *regs_16b = (u_int16_t *)regs;
	u_int32_t data;	/* read data */

	/* read all registers */
	cs_i2c_set_addr(0xA0);
	/* 96 x 8b value */
	if (phy->dbytes == 1) {
		for (i = 0; i < 96; i++) {
			if (cs_i2c_read(phy->dev_i2c, phy->space_i2c,
				phy->link, i, &data)) {
					return -1;
			}
			VERBOSE(CL_VERBOSE_LIBRARY, "Register %d read data "
				"0x%02X", i, data);
			regs[i] = data;
		}
		return 96;
	/* 64 x 16b value */
	} else {
		for (i = 0; i < MAX_SFP_REGS >> 1; i++) {
			if (cs_i2c_read(phy->dev_i2c, phy->space_i2c,
				phy->link, i, &data)) {
					return -1;
			}
			VERBOSE(CL_VERBOSE_LIBRARY, "Register %d read data "
				"0x%04X", i, data);
			regs_16b[i] = data;
		}
		return MAX_SFP_REGS >> 1;
	}

	/* return success, this return will never occur */
	return MAX_SFP_REGS;
}

/*!
 * \brief Read 16 registers of a Marvell EEPROM.
 *
 * \param dev Pointer to combo device.
 * \param phy Pointer to mapped phyter combo space.
 * \param sel Selected phyter.
 * \param regs Array for storing 16 registers.
 *
 * \retval 0 on success, 1 value if succeded.
 */
int
phy_gbic_allregs(cs_phy_t *phy, u_int16_t *regs)
{
	int i;			/* cycle variable */
	u_int32_t data;		/* read data */

	/* read all registers */
	for (i = 0; i < MAX_GBIC_REGS >> 1; i++) {
		if (cs_i2c_read(phy->dev_i2c, phy->space_i2c, phy->link, i, &data)) {
			return -1;
		}
		VERBOSE(CL_VERBOSE_LIBRARY, "Register %d read data 0x%X", i, data);
		regs[i] = data;
	}

	/* return success */
	return MAX_GBIC_REGS;
}

/*!
 * \brief Read 256 registers of 10GbE transceiver EEPROM (connected through I2C bus in PHYTERI2C - link 2 & 3)
 *
 * \param dev Pointer to combo device.
 * \param phy Pointer to mapped phyter combo space.
 * \param sel Selected phyter.
 * \param regs Array for storing 16 registers.
 * \param link .
 *
 * \retval 0 on success, 1 value if succeded.
 */
int
phy_10gb_trans_allregs(cs_phy_t *phy, u_int8_t *regs, u_int32_t *link)
{
	int i;					/* cycle variable */
	int ret; 				/* return value */
	int tlink = (*link == 0) ? 3 : 2;	/* for link 0 -> bus link 3  */
	int disc = 0;				/* this flag indicates that transceiver is disconnected */
	u_int32_t data;				/* read data */

	VERBOSE(CL_VERBOSE_BASIC, "Reading all transceiver regs on bus link: %d", tlink);
	/* read all registers */
	for (i = 0; i < MAX_10GB_TRANS_REGS; i++) {

		ret = cs_i2c_read(phy->dev_i2c, phy->space_i2c, tlink, i, &data);

		if (disc == 0 && i != 0) {
			if (regs[i - 1] != data)
				disc = 1;
		}

		if (ret < 0)
			return -1;
		else
			regs[i] = data;
	}

	/* return success */
	if (disc == 0)
		return -1;
	else
		return MAX_GBIC_REGS;
}

/*!
 * \brief Phyter services for Marvell driver.
 *
 * \param dev Pointer to combo device.
 * \param phy Pointer to mapped phyter combo space.
 * \param sel Selected phyter.
 * \param regs Array for storing 16 registers.
 *
 * \retval 0 on success, 1 value if succeded.
 */
int
phy_marvell_service(cs_phy_t *phy, int command, u_int32_t *param1,
	u_int32_t *param2, u_int32_t *param3)
{
	int ret = 0;
	u_int32_t utemp, anar, ktcr;
	param3 = NULL;

	switch (command) {
	case CS_PHY_WRITE:	/* write a value to PHY EEPROM */
		VERBOSE(CL_VERBOSE_LIBRARY, "Write 0x%X to address 0x%X", *param2, *param1);
		cs_i2c_write(phy->dev_i2c, phy->space_i2c, phy->link, *param1, *param2);
		break;

	case CS_PHY_RESET:	/* reset interface */
		/* enable sgmii */
		cs_i2c_write(phy->dev_i2c, phy->space_i2c, phy->link, GBIC_EXTS, 0x9084);
		cs_i2c_write(phy->dev_i2c, phy->space_i2c, phy->link, GBIC_CTRL, GBIC_CTRL_RESET | GBIC_CTRL_AUTOEN);
		/* advertise speed and restart autonegotiation */
		cs_i2c_write(phy->dev_i2c, phy->space_i2c, phy->link, GBIC_KTCR, GBIC_KTCR_ALL);
		cs_i2c_write(phy->dev_i2c, phy->space_i2c, phy->link, GBIC_ANAR, GBIC_ANAR_ALL);
		cs_i2c_write(phy->dev_i2c, phy->space_i2c, phy->link, 0x16, 0x1);
		cs_i2c_write(phy->dev_i2c, phy->space_i2c, phy->link,
			GBIC_CTRL, GBIC_CTRL_RESET);
		cs_i2c_write(phy->dev_i2c, phy->space_i2c, phy->link, 0x16, 0x0);
		break;

	case CS_PHY_ADVSPEED:	/* advertise speed (+1 for half) */
		anar = GBIC_ANAR_NONE;
		ktcr = GBIC_KTCR_NONE;
		switch (*param1) {
		case 1000:
			ktcr = GBIC_KTCR_1000F;
			break;
		case 100:
			anar = GBIC_ANAR_100F;
			break;
		case 10:
			anar = GBIC_ANAR_10F;
			break;
		case 1001:
			ktcr = GBIC_KTCR_1000H;
			break;
		case 101:
			anar = GBIC_ANAR_100H;
			break;
		case 11:
			anar = GBIC_ANAR_10H;
			break;
		}
		/* enable sgmii */
		cs_i2c_write(phy->dev_i2c, phy->space_i2c, phy->link, GBIC_EXTS, 0x9084);
		cs_i2c_write(phy->dev_i2c, phy->space_i2c, phy->link, GBIC_CTRL, GBIC_CTRL_RESET | GBIC_CTRL_AUTOEN);
		/* advertise speed and restart autonegotiation */
		cs_i2c_write(phy->dev_i2c, phy->space_i2c, phy->link, GBIC_KTCR, ktcr);
		/* ANAR register must end with 1 */
		if (anar & 1)
			cs_i2c_write(phy->dev_i2c, phy->space_i2c, phy->link, GBIC_ANAR, anar);
		else
			cs_i2c_write(phy->dev_i2c, phy->space_i2c, phy->link, GBIC_ANAR, anar + 1);
		cs_i2c_write(phy->dev_i2c, phy->space_i2c, phy->link, 0x16, 0x1);
		cs_i2c_write(phy->dev_i2c, phy->space_i2c, phy->link,
			GBIC_CTRL, GBIC_CTRL_RESET);
		cs_i2c_write(phy->dev_i2c, phy->space_i2c, phy->link, 0x16, 0x0);
		break;


	case CS_PHY_SETSPEED: 	/* set speed manually, auto-negotiation off */
		utemp = GBIC_CTRL_RESET | GBIC_CTRL_FDX; /* reset + 10 full duplex */
		if (*param1 & 0x1)
			utemp &= 0xFEFF; /* if half duplex, unset bit 8 */
		switch (*param1 & 0xFFFE) {
		case 1000:
			utemp &= GBIC_CTRL_S1000;
			break;
		case 100:
			utemp &= GBIC_CTRL_S100;
			break;
		}
		VERBOSE(CL_VERBOSE_LIBRARY, "CTRL: 0x%X", utemp);
		cs_i2c_write(phy->dev_i2c, phy->space_i2c, phy->link, GBIC_CTRL, utemp);
		break;

	case CS_PHY_GETSPEED:
		cs_i2c_read(phy->dev_i2c, phy->space_i2c, phy->link, 17, &utemp);
		switch ((utemp >> 14) & 3) {
		case 2:
			ret = 1000;
			break;
		case 1:
			ret = 100;
			break;
		case 0:
			ret = 10;
			break;
		}
		if ((utemp >> 13) & 1)
			return ret;
		else
			return ret + 1;

	case CS_PHY_GETLINK:
		cs_i2c_read(phy->dev_i2c, phy->space_i2c, phy->link, GBIC_STAT, &utemp);
		cs_i2c_read(phy->dev_i2c, phy->space_i2c, phy->link, GBIC_STAT, &utemp);
		if (utemp & GBIC_STAT_LINK)
			return 1;
		else
			return 0;

	case CS_PHY_ISOLATE:
		VERBOSE(CL_VERBOSE_LIBRARY, "Write 0x%X to address 0x%X", GBIC_CTRL_ISO, GBIC_CTRL);
		cs_i2c_write(phy->dev_i2c, phy->space_i2c, phy->link, GBIC_CTRL, GBIC_CTRL_ISO);
		break;

	case CS_PHY_ALLREGS:
		phy_gbic_allregs(phy, (u_int16_t *)param1);
		break;

	case CS_PHY_ANEGSUPP:
		cs_i2c_read(phy->dev_i2c, phy->space_i2c, phy->link, GBIC_STAT, &utemp);
		VERBOSE(CL_VERBOSE_LIBRARY, "Read value 0x%X from address 0x%X", utemp, GBIC_STAT);
		if (utemp & GBIC_STAT_ANEGSUPP)
			return 1;
		else
			return 0;

	case CS_PHY_ANEGADV:
		cs_i2c_read(phy->dev_i2c, phy->space_i2c, phy->link, GBIC_CTRL, &utemp);
		VERBOSE(CL_VERBOSE_LIBRARY, "Read value 0x%X from address 0x%X", utemp, GBIC_CTRL);
		if (utemp & GBIC_CTRL_AUTOEN)
			return 1;
		else
			return 0;

	case CS_PHY_ANEG:
		cs_i2c_read(phy->dev_i2c, phy->space_i2c, phy->link, GBIC_STAT, &utemp);
		VERBOSE(CL_VERBOSE_LIBRARY, "Read value 0x%X from address 0x%X", utemp, GBIC_STAT);
		if (utemp & GBIC_STAT_ANEG)
			return 1;
		else
			return 0;

	case CS_PHY_SUPP10H:
		cs_i2c_read(phy->dev_i2c, phy->space_i2c, phy->link, GBIC_STAT, &utemp);
		VERBOSE(CL_VERBOSE_LIBRARY, "Read value 0x%X from address 0x%X", utemp, GBIC_STAT);
		if (utemp & GBIC_STAT_SUPP10H) return 1;
		else return 0;

	case CS_PHY_SUPP10F:
		cs_i2c_read(phy->dev_i2c, phy->space_i2c, phy->link, GBIC_STAT, &utemp);
		VERBOSE(CL_VERBOSE_LIBRARY, "Read value 0x%X from address 0x%X", utemp, GBIC_STAT);
		if (utemp & GBIC_STAT_SUPP10F)
			return 1;
		else
			return 0;

	case CS_PHY_SUPP1000F:
		cs_i2c_read(phy->dev_i2c, phy->space_i2c, phy->link, GBIC_ESR, &utemp);
		VERBOSE(CL_VERBOSE_LIBRARY, "Read value 0x%X from address 0x%X", utemp, GBIC_ESR);
		if (utemp & GBIC_ESR_SUPP1000F)
			return 1;
		else
			return 0;

	case CS_PHY_ADV10H:
		cs_i2c_read(phy->dev_i2c, phy->space_i2c, phy->link, GBIC_ANAR, &utemp);
		VERBOSE(CL_VERBOSE_LIBRARY, "Read value 0x%X from address 0x%X", utemp, GBIC_ANAR);
		if (utemp & GBIC_ANAR_10H)
			return 1;
		else
			return 0;

	case CS_PHY_ADV10F:
		cs_i2c_read(phy->dev_i2c, phy->space_i2c, phy->link, GBIC_ANAR, &utemp);
		VERBOSE(CL_VERBOSE_LIBRARY, "Read value 0x%X from address 0x%X", utemp, GBIC_ANAR);
		if (utemp & GBIC_ANAR_10F)
			return 1;
		else
			return 0;

	case CS_PHY_ADV100H:
		cs_i2c_read(phy->dev_i2c, phy->space_i2c, phy->link, GBIC_ANAR, &utemp);
		VERBOSE(CL_VERBOSE_LIBRARY, "Read value 0x%X from address 0x%X", utemp, GBIC_ANAR);
		if (utemp & GBIC_ANAR_100H)
			return 1;
		else
			return 0;

	case CS_PHY_ADV100F:
		cs_i2c_read(phy->dev_i2c, phy->space_i2c, phy->link, GBIC_ANAR, &utemp);
		VERBOSE(CL_VERBOSE_LIBRARY, "Read value 0x%X from address 0x%X", utemp, GBIC_ANAR);
		if (utemp & GBIC_ANAR_100F)
			return 1;
		else
			return 0;

	case CS_PHY_ADV1000F:
		cs_i2c_read(phy->dev_i2c, phy->space_i2c, phy->link, GBIC_KTCR, &utemp);
		VERBOSE(CL_VERBOSE_LIBRARY, "Read value 0x%X from address 0x%X", utemp, GBIC_KTCR);
		if (utemp & GBIC_KTCR_1000F)
			return 1;
		else
			return 0;

	default:
		return -1;
	}

	return 0;
}

/*!
 * \brief Phyter services for Broadcom driver.
 *
 * \param dev Pointer to combo device.
 * \param phy Pointer to mapped phyter combo space.
 * \param sel Selected phyter.
 * \param regs Array for storing 16 registers.
 *
 * \retval 0 on success, -1 value if not supported operation.
 */
int
phy_broadcom_service(cs_phy_t *phy, int command, u_int32_t *param1,
	u_int32_t *param2, u_int32_t *param3)
{
	u_int32_t utemp, anar, ktcr;
	param3 = NULL;

	switch(command) {
	case CS_PHY_WRITE:
		VERBOSE(CL_VERBOSE_LIBRARY, "Write 0x%X to address 0x%X", *param2, *param1);
		cs_i2c_write(phy->dev_i2c, phy->space_i2c, phy->link, *param1, *param2);
		break;

	case CS_PHY_RESET:
		cs_i2c_write(phy->dev_i2c, phy->space_i2c, phy->link, GBIC_KTCR, GBIC_KTCR_ALL);
		cs_i2c_write(phy->dev_i2c, phy->space_i2c, phy->link, GBIC_ANAR, GBIC_ANAR_ALL);
		cs_i2c_write(phy->dev_i2c, phy->space_i2c, phy->link, GBIC_CTRL, GBIC_CTRL_RESET | GBIC_CTRL_AUTOEN);
		break;

	case CS_PHY_ISOLATE:
		VERBOSE(CL_VERBOSE_LIBRARY, "Write 0x%X to address 0x%X", GBIC_CTRL_ISO, GBIC_CTRL);
		cs_i2c_write(phy->dev_i2c, phy->space_i2c, phy->link, GBIC_CTRL, GBIC_CTRL_ISO);
		break;

	case CS_PHY_ADVSPEED:	/* advertise speed (+1 for half) and restart aneg */
		anar = GBIC_ANAR_NONE;
		ktcr = GBIC_KTCR_NONE;
		switch(*param1) {
		case 1000:
			ktcr = GBIC_KTCR_1000F;
			break;
		case 100:
			anar = GBIC_ANAR_100F;
			break;
		case 10:
			anar = GBIC_ANAR_10F;
			break;
		case 1001:
			ktcr = GBIC_KTCR_1000H;
			break;
		case 101:
			anar = GBIC_ANAR_100H;
			break;
		case 11:
			anar = GBIC_ANAR_10H;
			break;
		}
		cs_i2c_write(phy->dev_i2c, phy->space_i2c, phy->link, GBIC_KTCR, ktcr);
		cs_i2c_write(phy->dev_i2c, phy->space_i2c, phy->link, GBIC_ANAR, anar);
		cs_i2c_write(phy->dev_i2c, phy->space_i2c, phy->link,
			GBIC_CTRL, GBIC_CTRL_AUTOEN | GBIC_CTRL_STARTNEG);
		break;

	case CS_PHY_SETSPEED:	/* set speed manually, auto-negotiation off */
		utemp = GBIC_CTRL_RESET | GBIC_CTRL_FDX; /* reset + 10 full duplex */
		if (*param1 & 0x1)
			utemp &= 0xFEFF; /* if half duplex, unset bit 8 */
		switch(*param1 & 0xFFFE) {
		case 1000:
			utemp &= GBIC_CTRL_S1000;
			break;
		case 100:
			utemp &= GBIC_CTRL_S100;
			break;
		}
		VERBOSE(CL_VERBOSE_LIBRARY, "CTRL: 0x%X", utemp);
		cs_i2c_write(phy->dev_i2c, phy->space_i2c, phy->link, GBIC_CTRL, utemp);
		break;

	case CS_PHY_GETSPEED:
		cs_i2c_read(phy->dev_i2c, phy->space_i2c, phy->link, BRGPHY_MII_AUXSTS, &utemp);
		/* print interface speed */
		if (utemp & BRGPHY_RES_1000FD)
			return 1000;
		else if (utemp & BRGPHY_RES_1000HD)
			return 1001;
		else if (utemp & BRGPHY_RES_100FD)
			return 100;
		else if (utemp & BRGPHY_RES_100HD)
			return 101;
		else if (utemp & BRGPHY_RES_10FD)
			return 10;
		else if (utemp & BRGPHY_RES_10HD)
			return 11;
		break;

	case CS_PHY_GETLINK:
		cs_i2c_read(phy->dev_i2c, phy->space_i2c, phy->link, GBIC_STAT, &utemp);
		cs_i2c_read(phy->dev_i2c, phy->space_i2c, phy->link, GBIC_STAT, &utemp);
		if (utemp & GBIC_STAT_LINK)
			return 1;
		else
			return 0;

	case CS_PHY_ALLREGS:
		phy_gbic_allregs(phy, (u_int16_t *)param1);
		break;

	default:
		return -1;
	}

	return 0;
}

/*!
 * \brief Generic phyter services for SFP driver.
 *
 * \param phy Pointer to mapped phyter combo space.
 * \param command Command for service.
 * \param param1 Pointer to first parameter.
 * \param param2 Pointer to second parameter.
 *
 * \retval 0 on success, -1 value if not supported operation.
 */
int
phy_sfp_service(cs_phy_t *phy, int command, u_int32_t *param1,
	u_int32_t *param2, u_int32_t *param3)
{
	int dbytes, ret;
	u_int32_t utemp;
	param3 = NULL;

	switch(command) {
	case CS_PHY_WRITE:
		VERBOSE(CL_VERBOSE_LIBRARY, "Write 0x%X to address 0x%X", *param2, *param1);
		cs_i2c_write(phy->dev_i2c, phy->space_i2c, phy->link, *param1, *param2);
		break;

	/* return 1 if link up, 0 if link down, 2 if diagnostic EEPROM not found */
	case CS_PHY_GETLINK:
		/* if LOS (lost of signal) - link down (read from diagnostic EEPROM - I2C adress 0xA2) */
		cs_i2c_set_addr(0xA2);
		/* check if diagnostic eeprom found */
		dbytes = cs_i2c_detect_dlength(phy->dev_i2c, phy->space_i2c, phy->link, 0xA2);
		if (dbytes < 0)
			return 2;
		ret = cs_i2c_read(phy->dev_i2c, phy->space_i2c, phy->link, 110, &utemp);
		if (ret != 0)
			return 2;
		cs_i2c_set_addr(0xA0);
		if (utemp & 0x2)
			return 0;
		else
			return 1;

	case CS_PHY_ALLREGS:
		phy_sfp_allregs(phy, (u_int8_t *)param1);
		break;

	case CS_PHY_GETSPEED:
		return 1002;

	default:
		return -1;
	}

	return 0;
}

/*!
 * \brief Generic phyter services for 10gb VITESSE phyter (VSC8467)
 *
 * \param dev Pointer to combo device.
 * \param phy Pointer to mapped phyter combo space.
 * \param sel Selected phyter.
 * \param regs Array for storing 16 registers.
 *
 * \retval 0 on success, -1 value if not supported operation.
 */
int
phy_vsc_service(cs_phy_t *phy, int command, u_int32_t *param1,
	u_int32_t *param2, u_int32_t *param3)
{
	u_int32_t utemp;
	u_int32_t bit;

	switch (command) {
	case CS_PHY_RESET:
		cs_mdio_write(phy->dev_mdio, phy->space_mdio, phy->link, 0x1, 0x0, 0x8000);
		break;

	case CS_PHY_TRANSREGS:
		cs_i2c_set_data_bytes(1);
		cs_i2c_enable_clk_stretch(1);
		return phy_10gb_trans_allregs(phy, (u_int8_t *)param1, param2);

	case CS_PHY_MDIOWR:
		VERBOSE(CL_VERBOSE_LIBRARY, "Write: device 0x%X reg 0x%X data 0x%X",
			*param1 , *param2, *param3);
		cs_mdio_write(phy->dev_mdio, phy->space_mdio, phy->link,
			*param1, *param2, *param3);
		break;

	case CS_PHY_MDIORD:
		utemp = cs_mdio_read(phy->dev_mdio, phy->space_mdio, phy->link, *param1, *param2);
		VERBOSE(CL_VERBOSE_LIBRARY, "Read: device 0x%X reg 0x%X rdata 0x%X",
			*param1, *param2, utemp);
		return (int)utemp; /* utemp is max 16bit */

	case CS_PHY_GETLINK:
		/* Read last link state, which is not important */
		cs_mdio_read(phy->dev_mdio, phy->space_mdio, phy->link, MDIO_PCS, PCS_STAT2);
		cs_mdio_read(phy->dev_mdio, phy->space_mdio, phy->link, MDIO_PCS, PCS_STAT);
		/* Read current link state */
		utemp = cs_mdio_read(phy->dev_mdio, phy->space_mdio, phy->link, MDIO_PCS, PCS_STAT);
		if (utemp & PCS_STAT_LINK)
			return 1;
		else
			return 0;

	case CS_PHY_TXENABLE:
		utemp = cs_mdio_read(phy->dev_mdio, phy->space_mdio, phy->link, MDIO_PMA, PMA_CTRL1);
		cs_mdio_write(phy->dev_mdio, phy->space_mdio, phy->link,
			MDIO_PMA, PMA_CTRL1, utemp ^ PMA_CTRL1_LW);
		break;

	case CS_PHY_STXDISABLE:
	case CS_PHY_SPOWERDOWN:
		cs_i2c_set_data_bytes(1);

        /* Check, if feature is supported */
		bit = (command == CS_PHY_SPOWERDOWN ? XFP_EOB_SPOWERDOWN : XFP_EOB_STXDISABLE);
		cs_i2c_read(phy->dev_i2c, phy->space_i2c, phy->link, XFP_ENHANCED_OPTS_REG, &utemp);
        if(!(bit & utemp))
            return -1;

        /* Read, modify and write back to register */
		bit = (command == CS_PHY_SPOWERDOWN ? XFP_CRB_SPOWERDOWN : XFP_CRB_STXDISABLE);
		if (param1 == NULL)
			return -2;
		cs_i2c_read(phy->dev_i2c, phy->space_i2c, phy->link, XFP_CONTROL_REG, &utemp);
		utemp = *param1 ? (utemp | bit) : (utemp & ~bit);
		cs_i2c_write(phy->dev_i2c, phy->space_i2c, phy->link, XFP_CONTROL_REG, utemp);

        /* Check if change was applied */
		cs_i2c_read(phy->dev_i2c, phy->space_i2c, phy->link, XFP_CONTROL_REG, &utemp);
		if ((*param1 && !(utemp & bit)) || (!*param1 && (utemp & bit)))
			return 1;
		break;

	case CS_PHY_GETSPEED:
		return 10000;

	default:
		return -1;
	}

	return 0;
}

/*!
 * \brief Generic phyter services for 10gb AEL2005 phyter
 *        mostly identical to VITESSE phyter
 *
 * \param phy Pointer to mapped phyter combo space.
 * \param command Requested operation.
 * \param param1 Depends on command parameter
 * \param param2 Depends on command parameter
 * \param param3 Depends on command parameter
 *
 * \retval number of registers read on success, -1 value if not
 *         supported operation or error occured.
 */
int
phy_ael_service(cs_phy_t *phy, int command, u_int32_t *param1,
	u_int32_t *param2, u_int32_t *param3)
{
	u_int16_t offset;
	u_int8_t data;

	switch (command) {
	case CS_PHY_RESET:
	case CS_PHY_MDIOWR:
	case CS_PHY_MDIORD:
	case CS_PHY_GETLINK:
	case CS_PHY_TXENABLE:
		/* Identical to VSC phyter functionality */
		return phy_vsc_service(phy, command, param1, param2, param3);
		break;

	/* Here's the only difference */
	case CS_PHY_TRANSREGS:
		offset = AEL_PHY_PER_LINK_OFFSET * *param2;
		cs_i2cms_2_init(I2CMS_DEVICE_ADDRESS, AEL_PHY_CONTROL_REG +
			offset, AEL_PHY_TRANS_REG + offset);
		cs_i2cms_set_params(I2CMS_2_PARAM_ADDR_LEN, 8);
		if (cs_i2cms_2_read((* param3) ? I2CMS_2_DEVICE_ADDRESS2 :
			I2CMS_2_DEVICE_ADDRESS1, 0, (u_int8_t *)param1,
			MAX_10GB_TRANS_REGS)) {
				return -1;
		} else {
			return MAX_10GB_TRANS_REGS;
		}
		break;

	case CS_PHY_GETSPEED:
		return 10000;
		break;

	case CS_PHY_STXDISABLE:
		offset = AEL_PHY_PER_LINK_OFFSET * *param3;
		cs_i2cms_2_init(I2CMS_DEVICE_ADDRESS, AEL_PHY_CONTROL_REG +
			offset, AEL_PHY_TRANS_REG + offset);
		cs_i2cms_set_params(I2CMS_2_PARAM_ADDR_LEN, 8);

		/* Check, if feature is supported */
		cs_i2cms_2_read(I2CMS_2_DEVICE_ADDRESS1, SFP_ENHANCED_OPTS_REG, &data, 1);
		if(!(data & SFP_EOB_STXDISABLE))
			return -1;

		/* Read, modify and write back to register */
		if (param1 == NULL)
			return -2;
		cs_i2cms_2_read(I2CMS_2_DEVICE_ADDRESS2, SFP_CONTROL_REG, &data, 1);
		data = *param1 ? (data | XFP_CRB_STXDISABLE) : (data & ~XFP_CRB_STXDISABLE);

		cs_i2cms_2_write(I2CMS_2_DEVICE_ADDRESS2, SFP_CONTROL_REG, &data, 1);

		/* Check if change was applied */
		cs_i2cms_2_read(I2CMS_2_DEVICE_ADDRESS2, SFP_CONTROL_REG, &data, 1);
		if ((*param1 && !(data & XFP_CRB_STXDISABLE)) || (!*param1 && (data & XFP_CRB_STXDISABLE)))
			return 1;
		break;

	default:
		return -1;
		break;
	}

	return 0;
}

/**
 * \brief Initialize LEDs, should be used only with COMBO-2XFP2.2 addon card.
 *
 * \param dev		Combosix lowlevel cs_device_t component
 *
 * \return 		0 sucess, -1 error
 */
int
cs_phy_init_leds(cs_device_t * dev)
{
	int k;
	char *board, *card, *chip;
	cs_phy_t *phy;
	u_int32_t mdio_dev = 1;
	u_int32_t reg_addr = 0xe901;
	u_int32_t value = 0x2836;

	if (cs_identify(dev, &board, &card, &chip) != 0)
		return -1;

	if (card != NULL){
		if (!strncmp(card, COMBO6_2XFP2, sizeof((COMBO6_2XFP2) - 1)) ||
			!strcmp(card, COMBOI_10G2)) {

			/* card type is xfp2.2 */
			/* map phys for both interfaces */
			VERBOSE(2, "Attaching MDIO PHY 1");
			phy = cs_phy_attach(dev, dev, NULL, NULL, 1);

			if (phy == NULL)
				return -1;

			/* phyterctl -i1 -c mdiowr 1 0xe901 0x2836 */
			k = phy->phy_service(phy, CS_PHY_MDIOWR, &mdio_dev, &reg_addr, &value);
			if (k < 0)
				return -1;

			/* phyterctl -i1 -c mdiord 1 0xe901 */
			phy->phy_service(phy, CS_PHY_MDIORD, &mdio_dev, &reg_addr, NULL);

			/* map phys for both interfaces */
			VERBOSE(2, "Attaching MDIO PHY 2");
			phy = cs_phy_attach(dev, dev, NULL, NULL, 2);
			if (phy == NULL)
				return -1;

			/* phyterctl -i2 -c mdiowr 1 0xe901 0x2836 */
			k = phy->phy_service(phy, CS_PHY_MDIOWR, &mdio_dev, &reg_addr, &value);
			if (k < 0)
				return -1;

			/* phyterctl -i1 -c mdiord 1 0xe901 */
			phy->phy_service(phy, CS_PHY_MDIORD, &mdio_dev, &reg_addr, NULL);
			return 0;
		}
	}
	return 0;
}

/**
 * \brief Initialize AEL2005 phyter. Supported are only COMBOI-10G4 cards.
 *
 * \param phy		phyter structure pointer
 *
 * \return 		none
 */
void
phy_init_ael2005(cs_phy_t *phy)
{
	int off;
	int i = 0;

	while (init_sequence_AEL2005[i].flag != INIT_SEQ_FLAG_END) {
		if (init_sequence_AEL2005[i].flag == INIT_SEQ_FLAG_U_PROGRAM) {
			for (off = 0; off < 280; ++off) {
				cs_mdio_write(phy->dev_mdio, phy->space_mdio,
					(phy->link & 1) << 1, MDIO_PMA,
					PHY_INIT_AEL2005_ADDR + off,
					init_uprog_AEL2005[off]);
			}
		} else {
			cs_mdio_write(phy->dev_mdio, phy->space_mdio, (phy->link & 1) << 1,
				MDIO_PMA, init_sequence_AEL2005[i].address,
				init_sequence_AEL2005[i].data);
		}
		if (init_sequence_AEL2005[i].wait_time)
			sleep(init_sequence_AEL2005[i].wait_time / 1000);
			usleep((init_sequence_AEL2005[i].wait_time % 1000) * 1000);
		++i;
	}
}

/**
 * \brief Initialize AEL2005 phyter. Supported are only NetFPGA 10G4 cards.
 *
 * \param phy		phyter structure pointer
 *
 * \return			none
 */
void
phy_init_netfpga_ael2005(cs_phy_t *phy)
{
	int off;
	int i = 0;

	while (init_sequence_netfpga_AEL2005[i].flag != INIT_SEQ_FLAG_END) {
		if (init_sequence_netfpga_AEL2005[i].flag == INIT_SEQ_FLAG_U_PROGRAM) {
			for (off = 0; off < 280; ++off) {
				cs_mdio_write(phy->dev_mdio, phy->space_mdio,
					phy->link, MDIO_PMA,
					PHY_INIT_AEL2005_ADDR + off,
					init_uprog_AEL2005[off]);
			}
		} else {
			cs_mdio_write(phy->dev_mdio, phy->space_mdio, phy->link,
				MDIO_PMA, init_sequence_netfpga_AEL2005[i].address,
				init_sequence_netfpga_AEL2005[i].data);
		}
		if (init_sequence_netfpga_AEL2005[i].wait_time)
			sleep(init_sequence_netfpga_AEL2005[i].wait_time / 1000);
			usleep((init_sequence_netfpga_AEL2005[i].wait_time % 1000) * 1000);
		++i;
	}
}

/**
 * \brief Initialize VSC8486 phyter. Supported are only COMBOI-10G4 cards.
 *
 * \param phy		phyter structure pointer
 *
 * \return 		none
 */
void
phy_init_vsc8486(cs_phy_t *phy)
{
   /* Inverts polarity of XFI data: transciever -> phyter */
   cs_mdio_write(phy->dev_mdio, phy->space_mdio,
                 phy->link, MDIO_PMA,
                 VSC8486_PMA_CONFIG1_ADDR, VSC8486_PMA_CONFIG1_INIT);
}
