/*!
 * \file lup.h
 * \author Juraj Blaho
 *
 * \brief Provides control over the LUP component.
 */

/*
 * Copyright (C) 2007  CESNET
 *
 * LICENSE TERMS
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *        notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *        notice, this list of conditions and the following disclaimer in
 *        the documentation and/or other materials provided with the
 *        distribution.
 * 3. Neither the name of the Company nor the names of its contributors
 *        may be used to endorse or promote products derived from this
 *        software without specific prior written permission.
 *
 * This software is provided ``as is'', and any express or implied
 * warranties, including, but not limited to, the implied warranties of
 * merchantability and fitness for a particular purpose are disclaimed.
 * In no event shall the company or contributors be liable for any
 * direct, indirect, incidental, special, exemplary, or consequential
 * damages (including, but not limited to, procurement of substitute
 * goods or services; loss of use, data, or profits; or business
 * interruption) however caused and on any theory of liability, whether
 * in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even
 * if advised of the possibility of such damage.
 *
 * $Id$
 *
 */

#ifndef _LIBEROUTER_LUP_H_
#define _LIBEROUTER_LUP_H_

#include <stdbool.h>

#include "../combo.h"

/*!
 * \defgroup lup_component LUP component
 * \ingroup components
 * 
 * \brief Provides LUP control.  
 * \{ */

/*
 * LUP address space
 */
#define CS_LUP_CAM_REG_ADDR_IFC02_OFFSET 0x0000
#define CS_LUP_CAM_REG_ADDR_IFC13_OFFSET 0x2000

#define CS_LUP_CAM_CTRL_REG 0x4000
#define CS_LUP_CAM_STATE_REG 0x4004
#define CS_LUP_CAM_STATE_REG_ALL_EMPTY_FLAG (1)
#define CS_LUP_FIRST_INSTRUCTION_ADDRESS_REG 0x4008

#define CS_LUP_INPUT_INFO_REG 0x4010
#define CS_LUP_OUTPUT_INFO_REG 0x4014
#define CS_LUP_NEW_ITEMS_INFO_REG 0x4018

#define CS_LUP_CAM_CTRL_MEMORY_OFFSET 0x4040

#define CS_LUP_CAM_INDEX_IFC02_OFFSET 0x4080

#define CS_LUP_CAM_INDEX_IFC13_OFFSET 0x40C0

#define CS_LUP_PU_STATE_REG 0x6000
#define CS_LUP_PU_STATE_REG_PU_EMPTY_FLAG (1<<4)

#define CS_LUP_FIRST_PIPELINE_STATE_REG 0x6010
#define CS_LUP_PACKET_ID_ADDRESS_REG 0x6040

#define CS_LUP_RESULTS_MEMORY_OFFSET 0x7000
#define CS_LUP_RESULTS_MEMORY_ITEM_COUNT 512

#define CS_LUP_TCAM_BRIDGE 0x8000
#define CS_LUP_TCAM_BRIDGE_CTRL_REG 0x8000
#define CS_LUP_TCAM_BRIDGE_STATE_REG 0x8004

#define CS_LUP_SSRAM_BRIDGE 0xC000
#define CS_LUP_SSRAM_BRIDGE_CTRL_REG 0xC000
#define CS_LUP_SSRAM_BRIDGE_STATE_REG 0xC004

/*
 * LUP TCAM address space
 */
#define CS_LUP_CAM_DATA_ADDRESS 0x00000	/*!< address of TCAM row where data are stored */
#define CS_LUP_CAM_MASK_ADDRESS 0x80000	/*!< address of SSRAM row where masks are stored */

/*
 * LUP constants
 */
#define CS_LUP_REGISTERS_PER_CONTEXT 8	/*!<how much 32b registers is in one context */

#define CS_LUP_CAM_CTRL_MEMORY_ITEM_COUNT 16
#define CS_LUP_CAM_INDEX_IFC02_ITEM_COUNT 16
#define CS_LUP_CAM_INDEX_IFC13_ITEM_COUNT 16

#define CS_LUP_PIPELINE_STATE_REG_COUNT 8	/*!< number of pipeline states */

/*!
 * \brief Contains information about one CAM context.
 */
typedef struct cs_lup_context
{
	u_int32_t id; /*!< context ID (number 0-255)*/

	u_int32_t regs[CS_LUP_REGISTERS_PER_CONTEXT];	/*!< UH registers numbers. */
} cs_lup_context_t;

/*!
 * \brief One row of the debug memory.
 */
typedef struct cs_lup_debug_row
{
	u_int32_t edit_parameters;	/*!< Record returned by LUP. */
	u_int16_t interface_number;	/*!< Input interface number. */
	u_int16_t packet_id;	/*!< Packet ID. */
} cs_lup_debug_row_t;

/*!
 * \brief LUP status.
 */
typedef struct cs_lup_status
{
	bool pu_running;	/*!< True if PU is running. */
	bool cam_block_running;	/*!< True if CAM block is running. */
	bool uh_input_enabled;	/*!< True if UH input is enabled. */

	u_int32_t input_info;	/*!< Info about input. */
	u_int32_t output_info;	/*!< Info about output. */
	u_int32_t new_items_info;	/*!< Info about new items */

	u_int32_t pipeline_status_regs[CS_LUP_PIPELINE_STATE_REG_COUNT];	/*!< Pipeline status registers */
} cs_lup_status_t;

/*!
 * \brief Instruction for processing unit with address.
 */
typedef struct cs_lup_ssram_instruction
{
	u_int32_t address;	/*!< 20b SSRAM address */

	u_int32_t code_lo;	/*!< Instruction code - low 32 bits. */
	u_int32_t code_hi;	/*!< Instruction code - high 4 bits. */
} cs_lup_ssram_instruction_t;

/*!
\brief Complete LUP configuration.
*/
typedef struct cs_lup_config{
	/* sequential program */
	cs_lup_ssram_instruction_t *instructions;/*!< Instructions - SSRAM memory content */
	u_int32_t instruction_count; /*!< Instruction count*/
	
	u_int32_t pu_first_addr; /*!< address of the first performed instruction */
	
	/* TCAM */
	cs_tcam_ctrl_row_t * data_rows; /*!< TCAM data rows. Note: One LUP row consists of 4 TCAM rows*/
	cs_tcam_ctrl_row_t * mask_rows; /*!< TCAM mask rows*/
	u_int32_t tcam_row_count; /*!< TCAM row count*/
	
	cs_lup_context_t * contexts; /*!< array of CAM contexts */
	u_int32_t context_count; /*!< Context count */
	
	/* other */
	u_int32_t packet_id_addr; /*!< address in the UH where packet ID is stored */
}cs_lup_config_t;

/*function prototypes */

void cs_lup_stop_pu(cs_device_t * dev, cs_space_t * space);

void cs_lup_stop_cam(cs_device_t * dev, cs_space_t * space);

void cs_lup_stop_input(cs_device_t * dev, cs_space_t * space);

void cs_lup_empty(cs_device_t * dev, cs_space_t * space);

void cs_lup_start_pu(cs_device_t * dev, cs_space_t * space);

void cs_lup_start_cam(cs_device_t * dev, cs_space_t * space);

void cs_lup_start_input(cs_device_t * dev, cs_space_t * space);

void cs_lup_erase_debug_memory(cs_device_t * dev, cs_space_t * space);

void cs_lup_write_cam_control_memory(cs_device_t * dev, cs_space_t * space);

void cs_lup_write_cam_index_ifc02(cs_device_t * dev, cs_space_t * space);

void cs_lup_write_cam_index_ifc13(cs_device_t * dev, cs_space_t * space);

void cs_lup_write_cam_context(cs_device_t * dev, cs_space_t * space,
    cs_lup_context_t * context);

void cs_lup_write_packet_id_address(cs_device_t * dev, cs_space_t * space,
    u_int32_t address);

void cs_lup_write_pu_first_address(cs_device_t * dev, cs_space_t * space,
    u_int32_t address);

void cs_lup_read_debug_memory(cs_device_t * dev, cs_space_t * space,
    u_int32_t row_id, cs_lup_debug_row_t * row_data);

void cs_lup_read_cam_control_memory(cs_device_t * dev, cs_space_t * space,
    u_int32_t data[]);

void cs_lup_read_cam_index_ifc02(cs_device_t * dev, cs_space_t * space,
    u_int32_t data[]);

void cs_lup_read_cam_index_ifc13(cs_device_t * dev, cs_space_t * space,
    u_int32_t data[]);

void cs_lup_read_cam_context(cs_device_t * dev, cs_space_t * space,
    u_int32_t context_id, cs_lup_context_t * context_data);

u_int32_t cs_lup_read_packet_id_address(cs_device_t * dev, cs_space_t * space);

u_int32_t cs_lup_read_pu_first_address(cs_device_t * dev, cs_space_t * space);

void cs_lup_read_status(cs_device_t * dev, cs_space_t * space,
    cs_lup_status_t * status);

void cs_lup_init_ssram(cs_device_t * ssram_dev, cs_space_t * ssram_space,
    cs_lup_ssram_instruction_t * instructions, u_int32_t instruction_count);

int cs_lup_init_tcam(cs_device_t * tcam_dev, cs_space_t * tcam_space,
    cs_tcam_ctrl_row_t * data, cs_tcam_ctrl_row_t * mask, u_int32_t row_count);

int cs_lup_init(cs_device_t * dev, cs_space_t * space, cs_space_t * ssram_space,
	cs_space_t * tcam_space, cs_lup_config_t * config);

/*! \} */

#endif
