/*!

 * \file 	ibuf.h
 * \brief 	Function prototypes for controlling IBUF component
 * \author 	Andrej Hank <xhanka00@liberouter.org>
 * \author 	Miroslav Vadkerti <xvadke010@liberouter.org>
 * \date 	2006-2008
 */

/* Copyright (C) 2006, 2007 CESNET
 *
 * LICENSE TERMS
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 * 3. Neither the name of the Company nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * ALTERNATIVELY, provided that this notice is retained in full, this
 * product may be distributed under the terms of the GNU General Public
 * License (GPL), in which case the provisions of the GPL apply INSTEAD
 * OF those given above.
 *
 * This software is provided ``as is'', and any express or implied
 * warranties, including, but not limited to, the implied warranties of
 * merchantability and fitness for a particular purpose are disclaimed.
 * In no event shall the company or contributors be liable for any
 * direct, indirect, incidental, special, exemplary, or consequential
 * damages (including, but not limited to, procurement of substitute
 * goods or services; loss of use, data, or profits; or business
 * interruption) however caused and on any theory of liability, whether
 * in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even
 * if advised of the possibility of such damage.
 *
 * $Id$
 *
 */

#ifndef _LIBEROUTER_IBUF_H_
#define _LIBEROUTER_IBUF_H_

#include "../combosix.h"
#include <stdbool.h>

/*!
 * \defgroup ibuf_component IBUF component.
 * \ingroup components
 * \{
 *
 * \brief This module implements functions for controlling IBUF component.
 */

/*!
 * \struct 	cs_ibuf_t
 * \brief 	Ibuf data structure
 */
typedef struct {
	/*! All received frames */
	uint64_t	cnt_packets;
	/*! Correct frames */
	uint64_t	cnt_recv;
	/*! Discarded frames */
	uint64_t	cnt_error;
	/*! Discarded frames due to buffer overflow */
	uint64_t	cnt_overflow;
	/*! IBUF enable register */
	uint32_t	reg_en;
	/*! Error mask register */
	uint32_t	err_mask;
	/*! IBUF status register */
	uint32_t	ibuf_stat;
	/*! IBUF accepted minmum frame length */
	uint32_t	ibuf_frame_min_len;
	/*! Frame length check */
	uint32_t	ibuf_frame_mtu;
	/*! MAC addressed check */
	uint32_t	ibuf_mac_check;
	/*! Frame buffer size */
	uint32_t	ibuf_frame_buffer_size;
} cs_ibuf_t;

/*
 * \def 	CS_IBUF_MACLIST_SIZE
 * \brief 	Number of slots for filtered MAC addresses in IBUF
 */
#define CS_IBUF_MACLIST_SIZE	16

/*!
 * \struct 	cs_ibuf_mac_t
 * \brief 	Higher and lower part of MAC address in IBUF
 */
typedef struct {
	/*! Lower 32 bits of MAC */
	u_int32_t	lo;
	/*! Higher 16 bits of MAC + 1 validity bit*/
	u_int32_t	hi;
} cs_ibuf_mac_t;

/*
 *  Read IBUF counters
 */
cs_ibuf_t
cs_ibuf_read (cs_device_t *dev, cs_space_t *space, bool xgmii);

/*
 * Enable ibuf
 */
void
cs_ibuf_enable(cs_device_t *dev, cs_space_t *space);

/*
 * Disable ibuf
 */
void
cs_ibuf_disable(cs_device_t *dev, cs_space_t *space);

/*
 * Enable or disable MAC check mode
 */
void
cs_ibuf_enable_mac_check (cs_device_t *dev, cs_space_t *space);

/*
 * Set mac check mode register
 */
void
cs_ibuf_mac_check (cs_device_t *dev, cs_space_t *space, u_int32_t mode, int enabled);

/*
 * Get list of filtered MAC addresses
 */
void
cs_ibuf_mac_get_list (cs_device_t *dev, cs_space_t *space, cs_ibuf_mac_t *list, int enabled, int mac_count);

/*
 * Set list of filtered MAC addresses
 */
void
cs_ibuf_mac_set_list (cs_device_t *dev, cs_space_t *space, cs_ibuf_mac_t *list, int enabled, int mac_count);

/*
 * Set error mask
 */
void
cs_ibuf_err_mask (cs_device_t *dev, cs_space_t *space, u_int32_t mask, int enabled);

/*
 * Reset IBUF frame counters
 */
void
cs_ibuf_reset(cs_device_t *dev, cs_space_t *space);

/*
 * IBUF set speed
 */
void
cs_ibuf_set_speed(cs_device_t *dev, cs_space_t *space, int speed);

/*
 * IBUF minimal/maximal frame length
 */
void
cs_ibuf_frame_length (cs_device_t *dev, cs_space_t *space, u_int32_t mask, int max, int enabled);

/*! \} */

#endif	/* _LIBEROUTER_IBUF_H_ */
