/*!
 * \file hfex.c
 * \brief Functions for configuring hfex
 * \author Daniel Ivanco <xivanc01@stud.fit.vutbr.cz>
 * \date 2008
 */
/*
 * hfexctl.c: Tool for controlling hfe
 * Copyright (C) 2008 CESNET
 * Author: Daniel Ivanco <xivanc01@stud.fit.vutbr.cz>
 *
 * LICENSE TERMS
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 * 3. Neither the name of the Company nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * ALTERNATIVELY, provided that this notice is retained in full, this
 * product may be distributed under the terms of the GNU General Public
 * License (GPL), in which case the provisions of the GPL apply INSTEAD
 * OF those given above.
 *
 * This software is provided ``as is'', and any express or implied
 * warranties, including, but not limited to, the implied warranties of
 * merchantability and fitness for a particular purpose are disclaimed.
 * In no event shall the company or contributors be liable for any
 * direct, indirect, incidental, special, exemplary, or consequential
 * damages (including, but not limited to, procurement of substitute
 * goods or services; loss of use, data, or profits; or business
 * interruption) however caused and on any theory of liability, whether
 * in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even
 * if advised of the possibility of such damage.
 *
 * $Id$
 *
 */

#include <err.h>
#include <commlbr.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#include "hfex.h"

__RCSID ("$Id$");


/*!
 * \brief init hfex from configuration files
 *
 * \param dev Device
 * \param space Space
 * \param bram_file_name filename for BRAM register configuration
 * \param registers_file_name  filename for registers initialization
 * \return      0 if success, <0 otherwise
 */
int
cs_hfex_init_from_file(cs_device_t * dev, cs_space_t * space, char *bram_file_name, char *registers_file_name)
{

	int count = 0;
	hfex_row_t *rows = NULL;

	VERBOSE(CL_VERBOSE_LIBRARY, "BRAM file: %s", bram_file_name);
	VERBOSE(CL_VERBOSE_LIBRARY, "REGS file: %s", registers_file_name);

	/* first stop HFEX core */
	cs_hfex_stop(dev, space);

	/* read rows from config file */
	count = cs_hfex_read_rows_from_file(bram_file_name, &rows);
	if (count < 0) {
		return -1;
	}
	/* init HFEX */
	cs_hfex_init_bram(dev, space, rows, count);
	free(rows);
	rows = NULL;

	/* read rows from register file */
	count = cs_hfex_read_rows_from_file(registers_file_name, &rows);
	if (count < 0) {
		return -1;
	}
	cs_hfex_init_registers(dev, space, rows, count);
	free(rows);

	/* SET CORE RUNNING */
	cs_hfex_set_running(dev, space);

	return 0;
}

/*!
 * \brief Read 32bit number rows from file
 *
 * \param src_file_name name of file
 * \param rows Pointer to array of hfex_row_t
 * \return      number of read rows if succes, <0 otherwise
 */
int
cs_hfex_read_rows_from_file(char *src_file_name, hfex_row_t ** rows)
{

#define ROW_BUFFER 100

	uint32_t my_value;
	int count = 0;
	hfex_row_t *tmp_rows;

	FILE *f = fopen(src_file_name, "r");
	if (f == NULL) {
		VERBOSE(CL_VERBOSE_LIBRARY, "failed to open file");
		return -1;
	}
	/* allocate array for 100 items */
	*rows = (hfex_row_t*)malloc(sizeof(hfex_row_t) * ROW_BUFFER);
	if (*rows == NULL) {
		VERBOSE(CL_VERBOSE_LIBRARY, "failed to allocate memory");
		fclose(f);
		return -1;
	}

	while (fscanf(f, "%08x", &my_value) != EOF) {
		(*rows)[count] = my_value;

		count += 1;
		/* if needed extend array */
		if (count >= ROW_BUFFER-1) {
			tmp_rows = (hfex_row_t*)realloc(*rows, sizeof(hfex_row_t) * (count + 5*ROW_BUFFER));
			if (tmp_rows == NULL) {
				VERBOSE(CL_VERBOSE_LIBRARY, "failed to reallocate memory");
				free(*rows);
				fclose(f);
				return -1;
			}
			*rows = tmp_rows;
		}
	}
	fclose(f);

	return count;
}

/*!
 * \brief Set hfex running
 *
 * \param dev Device
 * \param space Space
 */
void
cs_hfex_set_running(cs_device_t * dev, cs_space_t * space)
{
	VERBOSE(CL_VERBOSE_LIBRARY, "Setting core running");
	MSG(CL_VERBOSE_LIBRARY, "STOP_REG 0x%08x -> 0x%08x", 0, CS_HFEX_STOP_REG);
	cs_space_write_4(dev, space, CS_HFEX_STOP_REG, 0);
}

/*!
 * \brief Set hfex stopped
 *
 * \param dev Device
 * \param space Space
 */
void
cs_hfex_stop(cs_device_t * dev, cs_space_t * space)
{

	VERBOSE(CL_VERBOSE_LIBRARY, "Stopping core");
	MSG(CL_VERBOSE_LIBRARY, "STOP_REG 0x%08x -> 0x%08x", 1, CS_HFEX_STOP_REG);
	cs_space_write_4(dev, space, CS_HFEX_STOP_REG, 1);

	VERBOSE(CL_VERBOSE_LIBRARY, "Waiting for core to stop");
	while ((cs_space_read_4(dev, space, CS_HFEX_WORKING_REG) & 1) != 0);	/* waiting for core to stop */
	VERBOSE(CL_VERBOSE_LIBRARY, "Core stopped");
}

/*!
 * \brief writes BRAM configuration to HFEX from hfex_row_t array
 *
 * \param dev Device
 * \param space Space
 * \param rows int array to be written
 * \param count array size
 * \return      number of written rows
 */
int
cs_hfex_init_bram(cs_device_t * dev, cs_space_t * space, hfex_row_t * rows, int count)
{

	VERBOSE(CL_VERBOSE_LIBRARY, "Writing BRAM config");
	int i = 0;
	for (i = 0; i < count; i++) {
		cs_space_write_4(dev, space, CS_HFEX_BRAM_CONF_REG + (i * 4), rows[i]);
		MSG(1, "BRAM_CONF_REG 0x%08x -> 0x%08x", rows[i], CS_HFEX_BRAM_CONF_REG + (i * 4));
	}

	return i;
}

/*!
 * \brief init FrameLink Rem and Output Words registers
 *
 * \param dev Device
 * \param space Space
 * \param rows int array to be written
 * \param count array size
 * \return 0 if success , <0 otherwise
 */
int
cs_hfex_init_registers(cs_device_t * dev, cs_space_t * space, hfex_row_t * rows, int count)
{

	VERBOSE(CL_VERBOSE_LIBRARY, "Writing registers");

        int i = 0;
	for (i = 0; i < count; i++) {
		cs_space_write_4(dev, space, CS_HFEX_FRAME_LINK_REM_REG + (i * 4), rows[i]);
		MSG(1, "BRAM_CONF_REG 0x%08x -> 0x%08x", rows[i], CS_HFEX_FRAME_LINK_REM_REG + (i * 4));
	}

	return 0;
}
