/*!
 * \file bus.c
 * \brief Basic device access primitives.
 * \author Jachym Holecek <freza@liberouter.org>
 * \author Jaroslav Kysela <perex@perex.cz>
 * \author Juraj Blaho
 * \date 2003,2004,2005,2006,2007
 *
 * Copyright (C) 2003,2004,2005,2006 CESNET
 *
 * The basic device access functions allow raw access to hardware.
 * For further information, please look to \ref bus page.
 */
/*
 *
 * LICENSE TERMS
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 * 3. Neither the name of the Company nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * ALTERNATIVELY, provided that this notice is retained in full, this
 * product may be distributed under the terms of the GNU General Public
 * License (GPL) version 2 or later, in which case the provisions
 * of the GPL apply INSTEAD OF those given above.
 *
 * This software is provided ``as is'', and any express or implied
 * warranties, including, but not limited to, the implied warranties of
 * merchantability and fitness for a particular purpose are disclaimed.
 * In no event shall the company or contributors be liable for any
 * direct, indirect, incidental, special, exemplary, or consequential
 * damages (including, but not limited to, procurement of substitute
 * goods or services; loss of use, data, or profits; or business
 * interruption) however caused and on any theory of liability, whether
 * in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even
 * if advised of the possibility of such damage.
 *
 * $Id$
 *
 */
/*! \page bus Basic device access
 *
 * Functions #cs_attach and #cs_detach open and close specified combo device
 * using #CS_PATH_DEV macro.
 *
 * Functions #cs_children_attach and #cs_children_detach manage scanning for
 * new design and attaching and detaching appropriate driver in OS kernel.
 *
 * Functions #cs_space_map and #cs_space_unmap manage mapping of specified
 * memory area to the user space (application).
 *
 * Functions #cs_space_read_4, #cs_space_read_multi_4, #cs_space_write_4,
 * #cs_space_write_multi_4 allow raw access to given address space.
 */


#include <sys/ioctl.h>
#include <sys/mman.h>

#include <err.h>
#include <errno.h>
#include <fcntl.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#ifdef __linux__
#include <net/if.h>
#endif

#include <commlbr.h>
#include <combo6.h>
//#ifndef __linux__
//#include <plx9060reg.h>
//#else
//#include <combo6plx.h>
//#endif

#include "combosix.h"
#include "cs_local.h"

__RCSID("$Id$");

/**
 * \def OPENFLAG
 * \brief Flag for opening device in exclusive mode
 */

/**
 * \def OPEN2FLAG
 * \brief Flag for opening device in non-exclusive mode
 */
#if defined(__NetBSD__) || defined(__FreeBSD__)
#define OPENFLAG	O_EXLOCK
#define OPEN2FLAG	0
#endif

#ifdef __linux__
#define OPENFLAG	0
#define OPEN2FLAG	O_APPEND
#endif

/**
 * \def MAP_FILE
 * \brief Compatibility macro for mmap
 */
#ifndef MAP_FILE
#define MAP_FILE	0
#endif

/**
 * Runtime settable debugging.
 */
#define COND_VERBOSE		(getenv("CS_VERBOSE") != NULL)

/**
 * Decide if we can use mmap and if yes, setup local bus switch, and fill in
 * target number for ioctl.
 */
#define INDIRECTSW(dv, sp, offs, target, indirect) \
	do {								\
		(indirect) = 1;						\
									\
		switch ((sp)->sp_type) {				\
		case CS_SPACE_CPLD:					\
			(target) = COMBO6_CPLD_BUS;			\
			break;						\
		case CS_SPACE_FPGA:					\
			(target) = COMBO6_FPGA_BUS;			\
			if (!((sp)->sp_flag & CS_MAP_INDIRECT) &&	\
				((dv)->dv_mmap != NULL))		\
				(indirect) = 0;				\
			break;						\
		case CS_SPACE_BRIDGE:					\
			(target) = COMBO6_BRIDGE_REGS;			\
			break;						\
		case CS_SPACE_PLX_EEPROM:				\
			(target) = COMBO6_PLX_EEPROM;			\
			break;						\
		case CS_SPACE_PCI_CONF:					\
			(target) = COMBO6_PCI_REGS;			\
			break;						\
		default:						\
			errx(1, "%s: type = %d", __func__,		\
				(sp)->sp_type);				\
		}							\
	} while (/*CONSTCOND*/ 0)


/*
 * Internal utils.
 */

static __inline void
ioc_write_multi(cs_device_t *d, u_int32_t offs, u_int32_t *vals,
    int target, int count)
{
	struct combo6_ioc_arg	c;

	c.size   = 4;
	c.offset = (u_int64_t) offs;
	c.count  = count;
	c.data   = vals;
	c.target = target;

	if (-1 == ioctl(d->dv_file, COMBO6_WRITE, &c))
		err(1, "ioctl()");
}

static __inline void
ioc_read_multi(cs_device_t *d, u_int32_t offs, u_int32_t *vals,
    int target, int count)
{
	struct combo6_ioc_arg	c;

	c.size   = 4;
	c.offset = (u_int64_t) offs;
	c.count  = count;
	c.data   = vals;
	c.target = target;

	if (-1 == ioctl(d->dv_file, COMBO6_READ, &c))
		err(1, "ioctl()");
}

#if 0
static void
bus_arbiter(cs_device_t *d, cs_target_t sel)
{
	u_int32_t	val;

	ioc_read_multi(d, PLX_CONTROL, &val, CS_SPACE_BRIDGE, 1);
	switch (sel) {
	case CS_SPACE_FPGA:
		val &= ~CONTROL_GPO;
		break;
	case CS_SPACE_CPLD:
		val |= CONTROL_GPO;
		break;
	default:
		errx(1, "cs: bus_arbiter sel = %d", sel);
	}

	ioc_write_multi(d, PLX_CONTROL, &val, CS_SPACE_BRIDGE, 1);
}
#endif

/**
 * \brief Open combo6 device
 * \param d Returned device structure
 * \param path Path for device (see #CS_PATH_DEV)
 * \return zero on success or a negative error code
 *
 * Initialize device abstraction structure, opening device file
 * given by \a path for (non-)exclusive access.
 */
static int
cs_attach_real(cs_device_t **d, const char *path, int mode)
{
	struct combo6_info	cfg;
	int			fd;

	*d = (cs_device_t*)malloc(sizeof(**d));
	if (*d == NULL)
		return -ENOMEM;

	fd = open(path, mode, 0);
	if (fd == -1) {
		free(*d);
		*d = NULL;
		return -errno;
	}

	memset(&cfg, 0, sizeof(cfg));
	if (ioctl(fd, COMBO6_IOC_INFO, &cfg) != 0)
		if (COND_VERBOSE)
			warn("%s: can't get LAS size", __func__);

	(*d)->dv_mmap = NULL;
	(*d)->dv_flag = 0;
	(*d)->dv_file = fd;
	(*d)->dv_bussel = CS_SPACE_NONE;
	(*d)->dv_msiz = cfg.ci_win_size;
	(*d)->first = NULL;

	/* Hardware identification */
	(*d)->dv_board = strdup(cfg.ci_board_name);
	(*d)->dv_board_subtype = strdup(cfg.ci_board_subtype);
	(*d)->dv_if0_card = strdup(cfg.ci_addon[0].card);
	(*d)->dv_if0_chip = strdup(cfg.ci_addon[0].chip);
	(*d)->dv_if1_card = strdup(cfg.ci_addon[1].card);
	(*d)->dv_if1_chip = strdup(cfg.ci_addon[1].chip);

	(*d)->name = NULL;
	(*d)->dv_id = cfg.ci_id_sw;
	(*d)->dv_id_hw = cfg.ci_id_hw;
	strncpy((char *)(*d)->dv_strid, cfg.ci_id_txt, CS_DESIGN_STRID_SIZE);
	(*d)->dv_strid[CS_DESIGN_STRID_SIZE - 1] = '\0';

#if !defined(__NetBSD__)
	/*
	 * Map whole address window. More logical spaces may reside here.
	 * If we fail, we're left with nil dv_mmap and fallback to ioctl.
	 *
	 * Note that this will fail for degraded device.
	 */
	(*d)->dv_mmap = (u_int32_t *) mmap(NULL, (size_t)(*d)->dv_msiz,
		PROT_READ | PROT_WRITE, MAP_FILE | MAP_SHARED, (*d)->dv_file,
		(off_t)0);
	if ((*d)->dv_mmap == MAP_FAILED) {
		if (COND_VERBOSE)
			warn("%s: mmap", __func__);
		(*d)->dv_mmap = NULL;
	}
#endif /* NetBSD */

	return 0;
}

/**
 * \brief Open combo6 device (exclusive)
 * \param d Returned device structure
 * \param path Path for device (see #CS_PATH_DEV)
 * \return zero on success or a negative error code
 *
 * Initialize device abstraction structure, opening device file
 * given by \a path for exclusive access.
 */
int
cs_attach(cs_device_t **d, const char *path)
{
	return cs_attach_real(d, path, O_RDWR | OPENFLAG);
}

/**
 * \brief Open combo6 device (non-exclusive)
 * \param d Returned device structure
 * \param path Path for device (see #CS_PATH_DEV)
 * \return zero on success or a negative error code
 *
 * Initialize device abstraction structure, opening device file
 * given by \a path for non-exclusive access.
 */
int
cs_attach_noex(cs_device_t **d, const char *path)
{
	return cs_attach_real(d, path, O_RDWR | OPEN2FLAG);
}

/**
 * \brief Open combo6 device (and wait for device to appear)
 * \param d Returned device structure
 * \param path Path for device (see #CS_PATH_DEV)
 * \param timeout in ms, 0=no timeout
 * \return zero on success or a negative error code
 *
 * Initialize device abstraction structure, opening device file
 * given by \a path for exclusive access. But wait if it doesn't
 * exist yet.
 */
int
cs_attach_wait(cs_device_t **d, const char *path, unsigned int timeout)
{
	int ret;

	while (1) {
		ret = cs_attach(d, path);
		if (ret != -ENOENT)
			break;
		if (timeout) {
			if (timeout <= 50) {
				break;
			}
			timeout -= 50;
		}
		usleep(50000);
	}

	return ret;
}


/**
 * \brief Close the combo6 device
 * \param d Combo6 device structure
 *
 * Close the device and invalidate any resources allocated. If there is a design
 * loaded, free it as well.
 */
void
cs_detach(cs_device_t **d)
{
	if ((*d)->dv_mmap != NULL)
		munmap((void*)(*d)->dv_mmap, (*d)->dv_msiz);

	free((*d)->dv_board);
	free((*d)->dv_board_subtype);
	free((*d)->dv_if0_card);
	free((*d)->dv_if0_chip);
	free((*d)->dv_if1_card);
	free((*d)->dv_if1_chip);

	close((*d)->dv_file);
	cs_design_free(*d);
	free(*d);
	*d = NULL;
}


/**
 * \brief Attach children on the bus
 * \param d Combo6 device structure
 *
 * Scan for active design and attach the appropriate driver code.
 * Force the driver to scan for known design (such as io_buf_v2)
 * that has a driver itself. Currently, the driver never scans
 * for 'subdevices' on its own to preserve clean testing environment.
 */
int
cs_children_attach(cs_device_t *d)
{
	if (ioctl(d->dv_file, COMBO6_IOC_BUS_ATTACH, 0) != 0)
		return -errno;
	return 0;
}

/**
 * \brief Detach children on the bus
 * \param d Combo6 device structure
 *
 * Force the driver to detach all 'subdevice' drivers.
 */
int
cs_children_detach(cs_device_t *d)
{
	if (ioctl(d->dv_file, COMBO6_IOC_BUS_DETACH, 0) != 0)
		return -errno;
	return 0;
}

/**
 * \brief Map an address region
 * \param d Combo6 device structure
 * \param s Returned allocated space structure
 * \param type Target type
 * \param size Region size in bytes
 * \param offs Absolute region offset in bytes
 * \param flag Mapping options
 * \return zero on success or a negative error code
 *
 * Map portion of device address space denoted by \a type beginning at offset
 * \a offs covering \a size bytes. If the \a size argument equals #CS_MAP_ALL,
 * whole address space beginning at given offset is mapped. Returns zero on
 * success. The \a flag argument may be a bitwise OR of:
 *
 * - #CS_MAP_INDIRECT
 *	- The region will be accessed using ioctl() syscall. By default,
 *	  regions are accessed using direct memory mapping, if possible,
 *	  which is significantly faster.
 *
 * If \a flag is zero, default options are used.
 */
int
cs_space_map(cs_device_t *d, cs_space_t **s, cs_target_t type,
    cs_size_t size, cs_addr_t offs, u_int32_t flag)
{
	if (type == CS_SPACE_FPGA && \
	    ((offs + size) >= d->dv_msiz || offs >= d->dv_msiz || \
	     size > d->dv_msiz)) {
		if (COND_VERBOSE)
			warnx("cs_space_map: address out of range");

		*s = NULL;
		return -EINVAL;
	}

	*s = (cs_space_t*)malloc(sizeof(**s));
	if (*s == NULL)
		return -ENOMEM;

	(*s)->sp_base = offs;
	(*s)->sp_size = (size == CS_MAP_ALL) ? (d->dv_msiz - offs) : size;
	(*s)->sp_type = type;
	(*s)->sp_flag = flag;

	return 0;
}

/**
 * \brief Unmap an address space region
 * \param d Combo6 device structure
 * \param s Space structure
 *
 * Invalidate the space abstraction object and release any associated
 * resources.
 */
void
cs_space_unmap(cs_device_t *d ATTRIBUTE_UNUSED, cs_space_t **s)
{
	free(*s);
	*s = NULL;
}

/**
 * \brief Get the base address offset of the space
 * \param s The space
 * \return The base address
 */
cs_addr_t cs_space_base_addr(cs_space_t *s)
{
	return s->sp_base;
}

/**
 * \brief Read four bytes (32-bit word) from bus
 * \param d Combo6 device structure
 * \param s Space structure
 * \param offs Relative offset into space structure in bytes
 * \retval Read 32-bit word
 *
 * Read a single 32-bit value from offset \a offs relative to space region \a s.
 */
u_int32_t
cs_space_read_4(cs_device_t *d, cs_space_t *s, cs_addr_t offs)
{
	u_int32_t	val = 0;
	int		indirect;
	int		target;

	if (offs % 4 || offs >= s->sp_size)
		errx(1, "%s: invalid argument", __func__ );

	INDIRECTSW(d, s, offs, target, indirect);

	if (indirect)
		ioc_read_multi(d, s->sp_base + offs, &val, target, 1);
	else
		val = d->dv_mmap[(s->sp_base + offs) / 4];

	return val;
}

/**
 * \brief Read an array of four byte (32-bit word) units from bus
 * \param d Combo6 device structure
 * \param s Space structure
 * \param offs Relative offset into space structure in bytes
 * \param count Number of 32-bit words to read
 * \param buf Array for read 32-bit words
 *
 * Read \a count 32-bit values from space region \a s beginning at offset
 * \a offs and store them to \a buf. Offset must be aligned at 32 bits.
 */
void
cs_space_read_multi_4(cs_device_t *d, cs_space_t *s, cs_addr_t offs,
    cs_size_t count, u_int32_t *buf)
{
	cs_size_t	i;
	cs_addr_t	addr;
	int		indirect;
	int		target;

	if (offs % 4 || (offs + count * 4) >= s->sp_size)
		errx(1, "%s: invalid argument", __func__);

	INDIRECTSW(d, s, offs, target, indirect);

	addr = s->sp_base + offs;

	if (indirect)
		ioc_read_multi(d, addr, buf, target, count);
	else
		for (i = 0; i < count; i++)
			buf[i] = d->dv_mmap[addr / 4 + i];
}

/**
 * \brief Write four bytes (32-bit word) to bus
 * \param d Combo6 device structure
 * \param s Space structure
 * \param offs Relative offset into space structure in bytes
 * \param val 32-bit word to write
 *
 * Write value \a val at offset \a offs relative to beginning of space region
 * \a s with a single 32-bit bus cycle. Offset must be aligned at 32 bits.
 */
void
cs_space_write_4(cs_device_t *d, cs_space_t *s, cs_addr_t offs, u_int32_t val)
{
	int	indirect;
	int	target;

	if (offs % 4 || offs >= s->sp_size)
		errx(1, "%s: invalid argument", __func__);

	INDIRECTSW(d, s, offs, target, indirect);

	if (indirect)
		ioc_write_multi(d, s->sp_base + offs, &val, target, 1);
	else
		d->dv_mmap[(s->sp_base + offs) / 4] = val;

	#ifdef SLOW_WRITE
	cs_space_read_4(d, s, offs);
	#endif

	#ifdef LOGGED_WRITE
	printf("write: address=%08X=%08X+%08X value=%08X\n", s->sp_base+ offs,
		s->sp_base, offs, val);
	#endif
}

/**
 * \brief Write an array of four byte (32-bit word) units to bus
 * \param d Combo6 device structure
 * \param s Space structure
 * \param offs Relative offset into space structure in bytes
 * \param count Number of 32-bit words to write
 * \param buf Array of 32-bit words to write
 *
 * Write \a count 32-bit values stored in \a buf to space region \a s beginning
 * at offset \a offs. Offset must be aligned at 32 bits.
 */
void
cs_space_write_multi_4(cs_device_t *d, cs_space_t *s, cs_addr_t offs,
    cs_size_t count, u_int32_t *buf)
{
	cs_size_t	i;
	cs_addr_t	addr;
	int		indirect;
	int		target;

	if (offs % 4 || (offs + count * 4) >= s->sp_size)
		errx(1, "%s: invalid argument", __func__);

	INDIRECTSW(d, s, offs, target, indirect);

	addr = s->sp_base + offs;

	if (indirect)
		ioc_write_multi(d, addr, buf, target, count);
	else
		for (i = 0; i < count; i++)
			d->dv_mmap[addr / 4 + i] = buf[i];
}

/**
 * \brief Return the name of given target type
 * \param type Target type
 * \retval Static ASCII string
 *
 * Return textual description of selected address space type. Returned pointer
 * is allocated statically and must not be freed.
 */
const char *
cs_space_name(cs_target_t type)
{
	switch (type) {
	case CS_SPACE_CPLD:
		return "cpld";
	case CS_SPACE_FPGA:
		return "fpga";
	case CS_SPACE_BRIDGE:
		return "br";
	case CS_SPACE_PLX_EEPROM:
		return "plxee";
	case CS_SPACE_NONE:
		return "<none>";
	default:
		return "<unknown>";
	}
}

/**
 * \brief Return whether the interface name (ASCII) is combo6 ethernet device
 * \param if_name Interface name
 * \param cs_card Returned card number
 * \param cs_intf Returned interface (port) number
 * \retval zero if \a if_name does not represent combo6 ethernet device,
 * \retval one if \a if_name is combo6 ethernet device,
 * \retval negative error code
 */
#ifdef __linux__
int
cs_netdev(const char *if_name, int *cs_card, int *cs_intf)
{
	int s, err;
	struct ifreq ifr;
	struct linux_priv_info {
		u_int32_t magic;
		u_int32_t card;
		u_int32_t intf;
	} info;

	if ((s = socket(AF_INET, SOCK_DGRAM, 0)) < 0)
		return -errno;
	info.magic = 0x434d4236U;
	strncpy(ifr.ifr_name, if_name, IFNAMSIZ);
	ifr.ifr_data = (caddr_t) &info;
	if ((err = ioctl(s, SIOCDEVPRIVATE, &ifr)) < 0) {
		close(s);
		return err;
	}
	if (info.magic != ~0x434d4236U) {
		close(s);
		return 0;
	}
	if (cs_card)
		*cs_card = info.card;
	if (cs_intf)
		*cs_intf = info.intf;
	close(s);
	return 1;
}
#else
int
cs_netdev(const char *if_name ATTRIBUTE_UNUSED, int *cs_card ATTRIBUTE_UNUSED,
    int *cs_intf ATTRIBUTE_UNUSED)
{
	return -ENXIO;
}
#endif

/**
 * Check if \a parm and \a name are not NULL and then assign \a name to \a parm.
 */
#define frob(parm, name)			\
do {						\
	if ((name) == NULL) {			\
		ret = -ENXIO;			\
	} else {				\
		if ((parm) != NULL)		\
			*(parm) = (name);	\
	}					\
} while (/*CONSTCOND*/ 0)

/**
 * \brief Describe the main board, addon card & chip of given device instance
 * \param dev Device handle
 * \param board Filled with static name of main board if not NULL
 * \param card Filled with static name of addon card if not NULL
 * \param chip Filled with static name of addon chip if not NULL
 * \retval zero on success,
 * \retval negative error code, as defined by errno(2)
 *
 * \note
 * This function is obsolete - do not use it anymore. Here is still only
 * because of backward compatibility.
 */
int
cs_identify(cs_device_t *dev, char **board, char **card, char **chip)
{
	int ret = 0;

	frob(board, dev->dv_board);
	frob(card, dev->dv_if0_card);
	frob(chip, dev->dv_if0_chip);

	return ret;
}

/**
 * \brief Describe the main board & subtype, addon cards & chips of given device
 * instance
 * \param dev Device handle
 * \param board Filled with static name of main board if not NULL
 * \param board_subtype Filled with static name of main board subtype if not NULL
 * \param if0_card Filled with static name of addon card in iface 0 if not NULL
 * \param if0_chip Filled with static name of addon chip in iface 0 if not NULL
 * \param if1_card Filled with static name of addon card in iface 1 if not NULL
 * \param if1_chip Filled with static name of addon chip in iface 1 if not NULL
 * \retval zero on success,
 * \retval negative error code, as defined by errno(2)
 *
 * \note
 * This function is usable for cards of CV2 family and for older too
 * (but needs combo drivers version 0.5.1 or higher).
 */
int
cs_identify_cv2(cs_device_t *dev, char **board, char **board_subtype,
	char **if0_card, char **if0_chip, char **if1_card, char **if1_chip)
{
	int ret = 0;

	frob(board, dev->dv_board);
	frob(board_subtype, dev->dv_board_subtype);
	frob(if0_card, dev->dv_if0_card);
	frob(if0_chip, dev->dv_if0_chip);
	frob(if1_card, dev->dv_if1_card);
	frob(if1_chip, dev->dv_if1_chip);

	return ret;
}

#undef frob

/**
 * \brief Switch firmware in ComboV2 card
 * \param path Combo6 device path
 * \param fwnum Firmware number (0-3)
 * \retval zero on success,
 * \retval negative error code, as defined by errno(2)
 */
int
cs_firmware_switch(const char *path, int fwnum)
{
	cs_device_t *dev;
	int err;

	if ((err = cs_attach(&dev, path)) < 0)
		return err;
	if (ioctl(dev->dv_file, COMBO6_IOC_FWSEL_WRITE, &fwnum) != 0) {
		err = -errno;
		if (COND_VERBOSE)
			warn("%s: can't switch to new firmware", __func__);
	}
	cs_detach(&dev);
	return err;
}

/**
 * \brief Get firmware number from ComboV2 card
 * \param dev Device handle
 * \retval positive firmware number on success,
 * \retval negative error code, as defined by errno(2)
 */
int
cs_firmware_get(cs_device_t *dev)
{
	int fwnum;

	if (ioctl(dev->dv_file, COMBO6_IOC_FWSEL_READ, &fwnum) != 0) {
		fwnum = -errno;
		if (COND_VERBOSE)
			warn("%s: can't get firmware number", __func__);
	}
	return fwnum;
}
