/*!
 * \file boot.c
 * \brief Combo6 boot / FPGA programming
 * \author Jachym Holecek <freza@liberouter.org>
 * \author Jaroslav Kysela <perex@perex.cz>
 * \date 2003,2004
 *
 * Copyright (C) 2003,2004  CESNET
 *
 * The design and component functions for work with the FPGA designs.
 * For more details, please, look to \ref design page.
 */
/*
 * LICENSE TERMS
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 * 3. Neither the name of the Company nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * ALTERNATIVELY, provided that this notice is retained in full, this
 * product may be distributed under the terms of the GNU General Public
 * License (GPL) version 2 or later, in which case the provisions
 * of the GPL apply INSTEAD OF those given above.
 *
 * This software is provided ``as is'', and any express or implied
 * warranties, including, but not limited to, the implied warranties of
 * merchantability and fitness for a particular purpose are disclaimed.
 * In no event shall the company or contributors be liable for any
 * direct, indirect, incidental, special, exemplary, or consequential
 * damages (including, but not limited to, procurement of substitute
 * goods or services; loss of use, data, or profits; or business
 * interruption) however caused and on any theory of liability, whether
 * in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even
 * if advised of the possibility of such damage.
 *
 * $Id$
 *
 */
/*! \page boot Combo6 boot / FPGA programming
 *
 * See #cs_boot.
 */

#include <errno.h>
#include <unistd.h>
#include <string.h>
#include <sys/ioctl.h>
#include "combosix.h"
#include <combo6.h>
#include <commlbr.h>
#include "cs_local.h"

__RCSID("$Id$");

int
cs_boot_get_id(cs_device_t *d);

/**
 * \brief Boot Combo6 hardware
 * \param d Combo6 device
 * \param type Boot type - see #CS_BOOT_V1_STANDARD
 * \param boot Structure chain with boot data
 * \return zero on success otherwise a negative error code
 *
 * This function will program FPGA chips on Combo6.
 */
int
cs_boot_v1(cs_device_t *d, enum cs_boot_v1_type type, cs_boot_v1_t *boot)
{
	int hasbase = 0, idx, xerr;
	cs_boot_v1_t *boot1;
	struct combo6_boot xboot;
	memset(&xboot, 0, sizeof(xboot));

	if (type != CS_BOOT_V1_STANDARD)
		return -EINVAL;
	if (boot == NULL)
		return -EINVAL;

	for (boot1 = boot, idx = 0; (boot1 != NULL) &&
		(idx < COMBO6_BOOT_IMAGES); boot1 = boot1->next, idx++) {
		switch (boot1->addr) {
		case 0:
			hasbase = 1;
			/* fall thru */
		case 8:
		case 9:
		case 100 ... 1023:
		case 1024 ... 2047:
			xboot.d[idx].unit = boot1->addr;
			xboot.d[idx].size = boot1->size;
			xboot.d[idx].data = boot1->data;
			break;
		default:
			return -EINVAL;
		}
	}

	/* boot of 8/9 is fine without detaches on CV2 */
	if (hasbase || !strcmp(d->dv_board, "combov2")) {
		xerr = cs_children_detach(d);
		if (xerr < 0 && xerr != -EBADFD)
			return xerr;
	}

	if (ioctl(d->dv_file, COMBO6_IOC_BOOT, &xboot) == -1) {
		if (hasbase) {
			d->dv_id = 0;
			d->dv_id_hw = 0;
			d->dv_strid[0] = '\0';
		}
		return -errno;
	}
	if (hasbase) {
		xerr = cs_boot_get_id(d);
		if (xerr < 0)
			return xerr;
	}

	return 0;
}

#ifndef DOC_HIDDEN
/**
 * \brief Update the firmware id from driver
 * \param d Combo6 device
 * \return zero on success otherwise a negative error code
 *
 * This functions is intended only for internal purpose.
 */
int
cs_boot_get_id(cs_device_t *d)
{
	struct combo6_info cfg;
	memset(&cfg, 0, sizeof(cfg));
	if (ioctl(d->dv_file, COMBO6_IOC_INFO, &cfg) != 0)
		return -errno;
	d->dv_id = cfg.ci_id_sw;
	d->dv_id_hw = cfg.ci_id_hw;
	strncpy((char *)d->dv_strid, cfg.ci_id_txt, CS_DESIGN_STRID_SIZE);
	d->dv_strid[CS_DESIGN_STRID_SIZE - 1] = '\0';
	return 0;
}
#endif
