/*
 *  szedata_driver.h: szedata driver <-> szedata midlevel interface
 *  Copyright (c) 2003-2006 CESNET
 *  Author(s): Jaroslav Kysela <perex@perex.cz>
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 */

#ifndef __SZEDATA_DRIVER_H
#define __SZEDATA_DRIVER_H

#include <linux/pci.h>

#include "szedataproc.h"

struct szedata_block {
	char *virt;		/* block contents (virtual address) */
	dma_addr_t phys;	/* physical PCI address */
	struct szedata_mmap_block_descriptor *desc; /* user block descriptor */
	struct list_head list;
	unsigned int app_count;
	unsigned int locked_count;
	struct list_head apps[0];
};

struct szedata_instance {
	unsigned int inum;
	struct szedata_device *dev;
	wait_queue_head_t dma_sleep;
	struct timer_list timer;
	struct list_head apps;
	u32 active_ports;	/* bitmap */
	unsigned int apps_count;
	unsigned int apps_peak_count;
	unsigned int poll_threshold;
	unsigned int poll_timeout;
	unsigned int poll_timeouted;
	struct list_head apps_locked;
	unsigned long long apps_slocked;
	unsigned long long apps_sunlocked;
	struct szedata_mmap_app_lock *app_lock;
	struct szedata_mmap_block_istats *istats;
	struct szedata_mmap_block_status *status;
};

struct szedata_device {
	struct list_head list;
	struct module *owner;
	int (*open)(struct szedata_device *dev);
	int (*close)(struct szedata_device *dev);
	int (*start)(struct szedata_device *dev);
	int (*stop)(struct szedata_device *dev);
	int (*port_info)(struct szedata_device *dev,
			struct szedata_ioctl_get_interface *info);
	int (*get_timestamp)(u_int64_t *val);
	void *private_data;	/* driver private data */
	spinlock_t lock;
	u32 max_apps;
	u32 running;
	u32 alloc_block_size;
	u32 alloc_blocks;
	u32 alloc_mmap_pages;
	u32 alloc_failed;
	u32 alloc_over;
	struct list_head free_pool;	/* pool of free blocks */
	struct list_head used_pool;	/* pool of used blocks */
	struct list_head locked_pool;	/* pool of locked blocks */
	struct list_head driver_pool;	/* pool of currently DMAed blocks */
	struct list_head alloc_pages;	/* internal bus usage */
	struct szedata_mmap_block_descriptor *desc;
	void **mmap_page_table;
	int desc_init;
	int device;
	int used_ports;
	u32 active_apps;
	struct szedata_info_entry *proc_root;
	struct szedata_info_entry *proc_stats;
	struct szedata_info_entry *proc_debug;
	struct szedata_info_entry *proc_pktpages;
	/* file related block */
	int instances;
	int registered;
	/* and last - application instances */
	struct szedata_instance appinst[0];
};

int szedata_device_alloc(struct szedata_device **dev, int ports, long blocks,
		long block_size, long private_size);
int szedata_device_free(struct szedata_device *dev);
int szedata_device_register(struct szedata_device *dev, struct module *module);
int szedata_device_unregister(struct szedata_device *dev);
int szedata_device_alloc_pci(struct szedata_device *dev, struct pci_dev *pci,
		long blocks, long block_size);
void szedata_device_free_pci(struct szedata_device *dev, struct pci_dev *pci);
static inline void szedata_device_carrier_off(struct szedata_device *dev) { }
static inline void szedata_device_carrier_on(struct szedata_device *dev) { }

static inline void szedata_device_lock(struct szedata_device *dev)
{
	spin_lock(&dev->lock);
}

static inline void szedata_device_unlock(struct szedata_device *dev)
{
	spin_unlock(&dev->lock);
}

static inline void szedata_device_lock_irq(struct szedata_device *dev)
{
	spin_lock_irq(&dev->lock);
}

static inline void szedata_device_unlock_irq(struct szedata_device *dev)
{
	spin_unlock_irq(&dev->lock);
}

struct szedata_block *szedata_device_get_block(struct szedata_device *dev,
		u_int32_t app_list, u_int32_t port);
void szedata_device_put_block(struct szedata_device *dev,
		struct szedata_block *block, u_int32_t app_list);
void szedata_device_detach_block(struct szedata_device *dev,
		struct szedata_block *block);
void szedata_length_error(struct szedata_device *dev, u_int32_t app_list,
		u_int32_t ifc, u_int32_t len);
void szedata_block_dropped(struct szedata_device *dev, u_int32_t app_list,
		u_int32_t ifc, u_int32_t len);

#endif /* __SZEDATA_DRIVER_H */
