/*
 * Copyright (c) 2007-2008 Jiri Slaby <jirislaby@gmail.com>
 *
 * Licensed under GPLv2
 */

#ifndef SZEDATA2_H_FILE
#define SZEDATA2_H_FILE

#include <linux/ioctl.h>
#include <linux/types.h>

#define SZE2_MMIO_RX		0
#define SZE2_MMIO_TX		1
#define SZE2_MMIO_MAX		10

#define SZE2_ADESC_FRAG		1
#define SZE2_ADESC_NEXT_S	24
#define SZE2_ADESC_NEXT(flags)	((flags) >> SZE2_ADESC_NEXT_S)

/**
 * struct sze2_instance_info - info mapped at offset 0
 *
 * @magic: 0xde4dAAII (mAjor=0x02, mInor)
 * @reserved1: for future/testing purposes
 * @offsets: offset of each mmap space
 * @sizes: size of each mmap space
 * @areas: subscribed areas (rx and tx)
 * @areas_available: available areas count (rx and tx)
 * @reserved2: reserved for future purposes
 * @reserved3: reserved for future purposes
 * @reserved4: reserved for future purposes
 * @offset: offset of locked data
 * @size: size of locked data
 * @flags: flags (see SZE2_ADESC_*)
 * @adesc: desc for each area and direction
 */
struct sze2_instance_info {
	__u32 magic;
	__u32 reserved1;
	__u64 offsets[SZE2_MMIO_MAX];
	__u64 sizes[SZE2_MMIO_MAX];
	__u32 areas[2];
	__u8 areas_available[2];
	__u16 reserved2;
	__u32 reserved3;
	__u64 reserved4[4];
	struct sze2_adesc {
		__u64 offset;
		__u32 size;
		__u32 flags;
	} adesc[0][2];
};

/**
 * struct sze2_subscribe_area - SUBSCRIBE_AREA ioctl parameter
 *
 * @areas: bitmap of areas to be subscribed
 *
 * if areas are set to ~0U, it means "all available", corrected value
 * is in mmap space after the ioctl
 */
struct sze2_subscribe_area {
	__u32 areas[2];
};

/**
 * struct sze2_tx_lock - TXLOCKDATA ioctl parameter
 *
 * @size: how many bytes should be reserved
 * @area: which area to involve
 */
struct sze2_tx_lock {
	__u32 size;
	__u8 area;
};

/**
 * struct sze2_tx_unlock - TXUNLOCKDATA ioctl parameter
 *
 * @size: how many bytes were really written
 * @area: to which area; it must be locked earlier
 */
struct sze2_tx_unlock {
	__u32 size;
	__u8 area;
};

/**
 * struct sze2_areasize - AREASIZE ioctl parameter
 *
 * @space: mmio space (one of SZE2_MMIO_*)
 * @area: of which area
 * @size: return value
 */
struct sze2_areasize {
	union {
		struct {
			__u8 space;
			__u8 area;
		};
		__u32 size;
	};
};

#define SZE2_IOC_BASE			0xc0

#define SZE2_IOC_SUBSCRIBE_AREA 	_IOW (SZE2_IOC_BASE, 0, struct sze2_subscribe_area)
#define SZE2_IOC_START			_IO  (SZE2_IOC_BASE, 1)
#define SZE2_IOC_RXLOCKDATA		_IOWR(SZE2_IOC_BASE, 2, __u32)
#define SZE2_IOC_RXUNLOCKDATA		_IO  (SZE2_IOC_BASE, 3)
#define SZE2_IOC_TXLOCKDATA		_IOW (SZE2_IOC_BASE, 4, struct sze2_tx_lock)
#define SZE2_IOC_TXUNLOCKDATA		_IOW (SZE2_IOC_BASE, 5, struct sze2_tx_unlock)
#define SZE2_IOC_STOP			_IO  (SZE2_IOC_BASE, 6)
#define SZE2_IOC_TXLOCKUNLOCKDATA	_IOW (SZE2_IOC_BASE, 7, struct sze2_tx_lock)
#define SZE2_IOC_AREASIZE		_IOWR(SZE2_IOC_BASE, 8, struct sze2_areasize)
#define SZE2_IOC_GETNUMANODE 		_IOR (SZE2_IOC_BASE, 9, int)

#endif
