/*
 * combo6k.h: Interface to the Combo6 Linux core.
 * Copyright (C) 2003,2004,2005,2006,2007,2008 CESNET
 * Author(s): Jaroslav Kysela <perex@perex.cz>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 */

#ifndef __COMBO6K_H
#define __COMBO6K_H

#ifndef PCI_VENDOR_ID_CESNET
#define PCI_VENDOR_ID_CESNET	0x18ec
#endif

#define COMBO6_CARDS	8

#define USEMODE_SKIP_CPLD	(1<<0)
#define USEMODE_SKIP_LOCALBUS	(1<<1)
#define USEMODE_SKIP_BAR12	(1<<8)
#define USEMODE_SKIP_BRVER	(1<<9)

struct combo6;

#include <linux/device.h>
#include <linux/interrupt.h>
#include <linux/io.h>
#include <linux/module.h>
#include <linux/mutex.h>
#include <linux/types.h>

#include <asm/atomic.h>

#include "combo6.h"
#include "combo6proc.h"

enum combo6_info_type {
	COMBO6_INFO_TYPE,
	COMBO6_INFO_SUBTYPE,
	COMBO6_INFO_ADDON_CARD,
	COMBO6_INFO_ADDON_CHIP,
	COMBO6_INFO_PORTS,
	COMBO6_INFO_SERNO,
	COMBO6_INFO_SPGRADE,
};

/**
 * struct combo6_ops - structure with event handlers
 *
 * @proc_info: returns additional info shown in /proc
 */
struct combo6_ops {
	int (*ioctl)(struct combo6 *combo6,
			struct inode *inode, struct file *file,
			unsigned int cmd, unsigned long arg);
	void (*go_attach)(struct combo6 *combo6);
	void (*attached)(struct combo6 *combo6);
	void (*go_detach)(struct combo6 *combo6);
	void (*detached)(struct combo6 *combo6);
	void (*go_release)(struct combo6 *combo6);
	const char *(*get_info)(struct combo6 *combo6, char *buf,
			enum combo6_info_type type, unsigned long data);
	void (*proc_info)(struct combo6 *combo6,
			struct combo6_info_buffer *buffer);
};

#define COMBO_GOING_DOWN	1
#define COMBO_BOOT_OK		2
#define COMBO_BOOTPROTECT	3
#define COMBO_MSI		4
#define COMBO_BOOTING		5

/*
 * struct combo6_i2c_adapter - combo6 i2c adapter structure
 */

struct combo6_i2c_adapter {
	struct i2c_adapter adap;
	struct combo6 *combo6;
	u32 i2c_ctls;
	u32 i2c_cmds;
};

/**
 * struct combo6 - basic combo structure
 *
 * @dev: generic device for this combo
 * @index: card index number
 * @module: associated kernel module
 * @pci: associated pci device
 * @is_v2: is ComboV2 card?
 * @is_6e: is Combo6E card?
 * @is_6x: is Combo6X card?
 * @irq: interrupt number
 * @mem_phys: physical address of the PLX chip / PCI bridge
 * @mem_virt: virtual address of the PLX chip / PCI bridge
 * @mem_len: size of mem resource
 * @lmem_phys: physical address of the local bus
 * @lmem_virt: virtual address of the local bus
 * @lmem_len: size of lmem resource
 * @cmem_phys: physical address of the cpld space
 * @cmem_virt: virtual address of the cpld space
 * @cmem_len: size of lmem resource
 * @addon_card: addon card type see COMBO6_ADDON_CARD_*
 * @addon_chip: addon chip type see COMBO6_ADDON_CHIP_*
 * @addon_interfaces: addon interfaces (LAN ports) count
 * @msi: does the device use msi interrupts?
 * @driver: devices attached to this card
 * @openers: how many threads have the device opened
 * @ref_count: to prevent device detach
 * @u: per card specific variables
 */
struct combo6 {
	struct device dev;
	unsigned int index;
	struct module *module;
	struct pci_dev *pci;
	unsigned int is_v2:1,
		     is_6e:1,
		     is_6x:1;
	int irq;
	int numa_node; 	/* Inherited from PCI device or -1 if unsupported. */
	resource_size_t mem_phys;
	void __iomem *mem_virt;
	resource_size_t mem_len;
	resource_size_t lmem_phys;
	void __iomem *lmem_virt;
	resource_size_t lmem_len;
	resource_size_t cmem_phys;
	void __iomem *cmem_virt;
	resource_size_t cmem_len;
	const struct combo6_ops *ops;
	unsigned int addon_card[COMBO6_ADDON_CARDS_MAX];
	unsigned int addon_chip[COMBO6_ADDON_CARDS_MAX];
	int addon_interfaces;
	struct combo6_info_entry *proc_root;
	struct combo6_info_entry *proc_id;
	spinlock_t reg_lock;
	DECLARE_BITMAP(flags, 32);
	unsigned usemode;
	unsigned bootdelay;
	u32 id_sw;
	u32 id_hw;
	char id_txt[64];
	u32 pcibr_version;
	u32 pcibr_date;
	struct mutex drv_mutex;
	struct combo_driver *driver;
	atomic_t openers;
	atomic_t ref_count;
	atomic_t detaching;
	union {
		struct {
			unsigned pcippc_ok:1,
				 pcippc_run:1;
			struct combo6_eppc_id pcippc;
		} two;
		struct {
			u16 id_ver;
			u16 netcope_ver;
			int fwnum;
			/* i2c adapters (LXT, IF1, IF2) */
			struct combo6_i2c_adapter *i2c_adap[3];
			unsigned char ctype, cstype, spgrade;
			char serno[3][17];
			/* capabilities */
			u32 caps;

			struct combo_design_info *info;
			unsigned int info_count;
			struct mutex info_lock;
		} three;
		struct {
			u16 id_ver;
			u16 netcope_ver;
			int fwnum;
			unsigned char ctype, cstype, spgrade;
			char serno[0][17];
			u32 caps;

			struct map_info* map;
			struct mtd_info* mtd;
			struct mtd_partition *part;
		} cv3;
	} u;
	void *private_data;
};

/*
 * device on local bus
 */

/**
 * struct combo_device_id - single device id to be matched
 *
 * @id_lsw: demanded sw_id, bottom of the range
 * @id_hsw: demanded sw_id, top of the range
 * @id_text: text to match (if not empty)
 * @intfs: restricted count of interfaces, <= 0 means read from HW
 * @driver_data: may be used arbitrarily
 */
struct combo_device_id {
	u32 id_lsw;
	u32 id_hsw;
	u8 id_text[36];
	int intfs;
	kernel_ulong_t driver_data;
};

/**
 * struct combo_driver - ops registered with combo6_device_register
 *
 * @list: entry in combo_drivers
 * @name: device driver string identification
 * @dhw: combo type
 * @interrupt: non-dma interrupt callback
 * @dma_interrupt: dma interrupt callback
 */
struct combo_driver {
	enum { DHW_COMBO6X = 1, DHW_COMBOV2 } dhw;
	const struct combo_device_id *id_table;
	irqreturn_t (*interrupt)(struct combo6 *combo, unsigned int mask);
	int (*attach)(struct combo6 *combo, const struct combo_device_id *id,
			int interfaces);
	int (*detach)(struct combo6 *combo);
	struct device_driver drv;
};

/*
 * h/w access primitives
 */
static inline void combo6_fpga_writel(struct combo6 *combo6, u32 reg, u32 val)
{
	writel(val, combo6->lmem_virt + reg);
}

static inline u32 combo6_fpga_readl(struct combo6 *combo6, u32 reg)
{
	return readl(combo6->lmem_virt + reg);
}

static inline u64 combo6_fpga_readq(struct combo6 *combo, u32 reg)
{
	return readq(combo->lmem_virt + reg);
}
/*
 * initialization
 */
struct combo6 *__combo6_alloc(size_t priv_size, struct module *mod);
void combo6_free(struct combo6 *combo6);
int combo6_add(struct combo6 *combo6);
void combo6_remove(struct combo6 *combo6);

#define combo6_alloc(sz) __combo6_alloc((sz), THIS_MODULE)

/**
 * combo6_device_get - atomically get combo6 device reference
 * @combo6: which device to get reference for
 *
 * Returns 0 on success, otherwise negative errno.
 */
static inline int combo6_device_get(struct combo6 *combo6)
{
	atomic_inc(&combo6->ref_count);
	if (atomic_read(&combo6->detaching) > 0) {
		atomic_dec(&combo6->ref_count);
		return -EBUSY;
	}
	return 0;
}

/**
 * combo6_device_put - atomically put combo6 device reference
 * @combo6: which device to put reference of
 */
static inline void combo6_device_put(struct combo6 *combo6)
{
	atomic_dec(&combo6->ref_count);
}

static inline const char *combo6_get_info(struct combo6 *combo6, char *buf,
		enum combo6_info_type type, unsigned long data)
{
	return combo6->ops->get_info(combo6, buf, type, data) ? : "n/a";
}

/*
 * misc
 */
int combo6_ioctl_info(struct combo6 *combo6, struct combo6_info __user *_info);
int combo6_ioctl_bus_attach(struct combo6 *combo6);
int combo6_ioctl_bus_detach(struct combo6 *combo6);

/*
 * management for devices on local bus
 */
extern int combo_register_driver(struct combo_driver *ops);
extern void combo_unregister_driver(struct combo_driver *ops);

void combo6x_module_ref(void);

extern int combo_queue_message(int card, int message);

/**
 * COMBOV2_RX_IFACES - get rx interface count on combov2 cards
 * @ifaces: interfaces count passed to attach
 *
 * It is necessary to use this macro on cv2 cards, interfaces passed to
 * attach function are encoded this way.
 */
#define COMBOV2_RX_IFACES(ifaces)	(((ifaces) < 0) ? (ifaces) : \
						(ifaces) & 0xff)
/**
 * COMBOV2_TX_IFACES - get tx interface count on combov2 cards
 * @ifaces: interfaces count passed to attach
 *
 * See COMBOV2_RX_IFACES.
 */
#define COMBOV2_TX_IFACES(ifaces)	(((ifaces) < 0) ? (ifaces) : \
						((ifaces) >> 8) & 0xff)

#endif /* __COMBO6K_H */
