/*
 * Support for third generation card I2C controllers
 *
 * Copyright (c) 2009 CESNET
 * Author(s): Jiri Slaby <jirislaby@gmail.com>
 *
 * Licensed under GPLv2
 */

#include <linux/delay.h>
#include <linux/i2c.h>
#include <linux/kernel.h>
#include <linux/pci.h>

#include "combo6k.h"
#include "combov2.h"
#include "combov2_i2c.h"

static int combov2_is_crypto_chip(u16 addr)
{
	switch (addr) {
	case AT88SC_CMD_WRITE_UZ:
	case AT88SC_CMD_READ_UZ:
	case AT88SC_CMD_SYS_WRITE:
	case AT88SC_CMD_SYS_READ:
	case AT88SC_CMD_VRFY_CRYPTO:
	case AT88SC_CMD_VRFY_PASSWD:
		return 1;
	}
	return 0;
}

static int combov2_i2c_transfer(struct combo6 *combo6, unsigned int adapi,
		struct i2c_msg *msgs, int nr)
{
	struct i2c_adapter *adap = &combo6->u.three.i2c_adap[adapi]->adap;
	int ret;

	if (!adap)
		return -ENODEV;

	ret = i2c_transfer(adap, msgs, nr);
	if (ret < 0)
		return ret;
	if (ret == 0)
		return -EINVAL;

	 return 0;
}

/**
 * combov2_i2c_read_zone - read cryptochip user zone
 *
 * @combo: combo device to operate with
 * @adapi: adapter index (0=base, 1+2=addons)
 * @zone: user zone to read from
 * @buf: where to store the data (has to be at least 3B long)
 * @len: how much to read
 */
int combov2_i2c_read_zone(struct combo6 *combo, unsigned int adapi, u8 zone,
		void *buf, unsigned int len)
{
	u8 set_uz[] = { AT88SC_SYSWR_SET_UZ, zone, 0 };
	struct i2c_msg msgs[] = {
		{
			.addr = AT88SC_CMD_SYS_WRITE,
			.flags = 0,
			.buf = set_uz,
			.len = 3,
		}, {
			.addr = AT88SC_CMD_READ_UZ,
			.flags = I2C_M_RD,
			.buf = buf,
			.len = 32,
		}
	};
	u8 *cmd = buf;

	cmd[0] = 0; /* addr HI */
	cmd[1] = 0; /* addr LO */
	cmd[2] = len; /* length */

	return combov2_i2c_transfer(combo, adapi, msgs, ARRAY_SIZE(msgs));
}
EXPORT_SYMBOL_GPL(combov2_i2c_read_zone);

int combov2_i2c_read_id(struct combo6 *combo6)
{
	struct combov2_i2c_0 i2c0;
	unsigned int a;
	int ret;

	ret = combov2_i2c_read_zone(combo6, 0, 0, &i2c0, sizeof(i2c0));
	if (ret)
		return ret;

	if (i2c0.ctype != 0xc2) {
		dev_warn(&combo6->pci->dev, "invalid card type (%.2x) in user "
				"zone, skipping i2c reading\n", i2c0.ctype);
		return -EIO;
	}

	memcpy(combo6->u.three.serno[0], i2c0.sn, sizeof(i2c0.sn));
	combo6->u.three.serno[0][16] = 0;
	combo6->u.three.spgrade = i2c0.spgrade;
	combo6->u.three.ctype = i2c0.ctype;
	combo6->u.three.cstype = i2c0.cstype;

	for (a = 0; a < 2; a++) {
		ret = combov2_i2c_read_zone(combo6, a + 1, 0, &i2c0,
				sizeof(i2c0));
		if (ret == -ENODEV)
			continue;
		if (ret)
			return ret;

		combo6->addon_card[a] = i2c0.ctype;
		memcpy(combo6->u.three.serno[a + 1], i2c0.sn,
				sizeof(i2c0.sn));
		combo6->u.three.serno[a + 1][16] = 0;
	}

	return 0;
}

int combov2_i2c_do_op(struct combo6 *combo6, struct combo6_i2c __user *op)
{
	u8 * __user *old_ptrs;
	struct i2c_msg *msgs;
	struct combo6_i2c my_op;
	unsigned int a;
	int ret;

	if (copy_from_user(&my_op, op, sizeof(my_op)))
		return -EFAULT;

	if (my_op.nmsg > 42) /* why 42? (it's I2C_RDRW_IOCTL_MAX_MSGS btw.) */
		return -EINVAL;

	switch (my_op.controller) {
	case COMBO6_I2C_ADDON(0) ... COMBO6_I2C_ADDON(1):
		/* i2c_adap idx 0: card, 1: first addon, 2: second addon */
		my_op.controller -= COMBO6_I2C_ADDON(0) - 1;
	case COMBO6_I2C_CARD:
		break;
	default:
		return -EINVAL;
	}

	if (WARN_ON(my_op.controller >= ARRAY_SIZE(combo6->u.three.i2c_adap)))
		return -EINVAL;

	msgs = kmalloc((sizeof(struct i2c_msg) + sizeof(void *)) * my_op.nmsg,
			GFP_KERNEL);
	if (msgs == NULL)
		return -ENOMEM;

	old_ptrs = (u8 * __force *)(msgs + my_op.nmsg);

	if (copy_from_user(msgs, my_op.msg,
				sizeof(struct i2c_msg) * my_op.nmsg)) {
		ret = -EFAULT;
		goto free;
	}

	for (a = 0; a < my_op.nmsg; a++) {
		struct i2c_msg *msg = &msgs[a];
		u8 __user *uaddr = (u8 __force *)msg->buf;
		old_ptrs[a] = uaddr;
		if (msg->len > 8192) {
			ret = -EINVAL;
			goto free;
		}
		msg->buf = kmalloc(msg->len, GFP_KERNEL);
		if (msg->buf == NULL) {
			ret = -ENOMEM;
			goto free_msgs;
		}
		if (copy_from_user(msg->buf, uaddr, msg->len)) {
			ret = -EFAULT;
			goto free_msgs1;
		}
	}

	ret = combov2_i2c_transfer(combo6, my_op.controller, msgs, my_op.nmsg);

	for (a = 0; a < my_op.nmsg; a++) {
		struct i2c_msg *msg = &msgs[a];
		if (!ret && (msg->flags & I2C_M_RD))
			if (copy_to_user(old_ptrs[a], msg->buf, msg->len))
				ret = -EFAULT;
		kfree(msg->buf);
	}

free:
	kfree(msgs);
	return ret;
free_msgs1:
	a++; /* free also this one */
free_msgs:
	for (; a > 0; a--)
		kfree(msgs[a - 1].buf);
	goto free;
}

static inline void combov2_wait_for_tip(struct combo6 *combo6, u32 cmd)
{
	unsigned int counter = 1000;

	while (--counter > 0 && PCR_I2C_CMD_STAT(combo6_br_readl(combo6, cmd)) &
			PCR_I2C_STAT_TIP) {
		if (!(counter % 100))
			msleep(10);
		else
			cpu_relax();
	}
	if (!counter && PCR_I2C_CMD_STAT(combo6_br_readl(combo6, cmd)) &
			PCR_I2C_STAT_TIP)
		printk(KERN_WARNING "%s: failed to wait for TIP\n", __func__);
}

static inline int combov2_check_ack(struct combo6 *combo6, u32 cmd, u8 addr)
{
	int ack = PCR_I2C_CMD_STAT(combo6_br_readl(combo6, cmd)) &
		PCR_I2C_STAT_RXACK;

	/* 0 is we got an ACK */
	if (ack)
		printk(KERN_WARNING "%s: we didn't get an ACK from I2C "
				"controller at %.8x, i2c address %.2x\n",
				__func__, cmd, addr);
	return !ack;
}

static void combov2_handle_msg(struct i2c_adapter *adap, struct i2c_msg *msg)
{
	struct combo6_i2c_adapter *c6adap = i2c_get_adapdata(adap);
	struct combo6 *combo6 = c6adap->combo6;
	unsigned int a, len;
	u32 addr = c6adap->i2c_cmds;
	int read = msg->flags & I2C_M_RD;
	int crypto = combov2_is_crypto_chip(msg->addr);

	combo6_br_writel(combo6, addr,
			PCR_I2C_CMD_TX(PCR_I2C_CMD_STA | PCR_I2C_CMD_WR,
			msg->addr | (crypto ? 0 : !!read)));
	combov2_wait_for_tip(combo6, addr);
	combov2_check_ack(combo6, addr, msg->addr);

	if (crypto || !read) {
		len = (read && crypto) ? 3 : msg->len;
		for (a = 0; a < len; a++) {
			u8 cmd = PCR_I2C_CMD_WR;
			if (a == msg->len - 1)
				cmd |= PCR_I2C_CMD_STO;
			combo6_br_writel(combo6, addr, PCR_I2C_CMD_TX(cmd,
						msg->buf[a]));
			combov2_wait_for_tip(combo6, addr);
			combov2_check_ack(combo6, addr, msg->addr);
		}

		if (!read)
			return;
	}

	len = crypto ? msg->buf[2] : msg->len;
	for (a = 0; a < len; a++) {
		u8 cmd = PCR_I2C_CMD_RD;
		if (a == len - 1)
			cmd |= PCR_I2C_CMD_STO | PCR_I2C_CMD_ACK;
		combo6_br_writel(combo6, addr, PCR_I2C_CMD_TX(cmd,
					msg->addr | 1));
		combov2_wait_for_tip(combo6, addr);
		msg->buf[a] = PCR_I2C_CMD_RX(combo6_br_readl(combo6, addr));
	}
}

static int combov2_xfer(struct i2c_adapter *adap, struct i2c_msg *msgs,
		int num)
{
	unsigned int a;

	for (a = 0; a < num; a++) {
		struct i2c_msg *msg = &msgs[a];

		if (msg->flags & I2C_M_TEN)
			return -EINVAL;
		/* bit 0 is R/W */
		if (msg->addr & 1)
			return -EINVAL;
		if (combov2_is_crypto_chip(msg->addr)) {
			if (msg->len < 3)
				return -EINVAL;
			if ((msg->flags & I2C_M_RD) && msg->len < msg->buf[2])
				return -EINVAL;
		} else {
			if (msg->len == 0)
				return -EINVAL;
		}
	}

	for (a = 0; a < num; a++)
		combov2_handle_msg(adap, &msgs[a]);

	return a;
}

static u32 combov2_functionality(struct i2c_adapter *adap)
{
	return 0;
}

static struct i2c_algorithm combov2_algo = {
	.master_xfer = combov2_xfer,
	.functionality = combov2_functionality,
};

static const u32 combov2_i2c_ctls[] = {
	PCR_I2C_LXT_CTL, PCR_I2C_IF1_CTL, PCR_I2C_IF2_CTL
};

static const u32 combov2_i2c_cmds[] = {
	PCR_I2C_LXT_CMD, PCR_I2C_IF1_CMD, PCR_I2C_IF2_CMD
};

static int combov2_i2c_start_adap(struct combo6 *combo6,
		struct combo6_i2c_adapter *c6adap,
		unsigned int idx)
{
	struct i2c_adapter *adap = &c6adap->adap;
	struct pci_dev *pdev = combo6->pci;
	int ret;

	c6adap->combo6 = combo6;
	c6adap->i2c_ctls = combov2_i2c_ctls[idx];
	c6adap->i2c_cmds = combov2_i2c_cmds[idx];

	dev_dbg(&pdev->dev, "starting I2C[%u]\n", idx);
	combo6_br_writel(combo6, c6adap->i2c_ctls, 0x8000f9);

	/* Start-up Sequence (Atmel CryptoMemory Specification, page 31) */
	/* Use some cryptochip address, there are other devices on the bus 
	 * aswell, so do not mess them up */
	combo6_br_writel(combo6, c6adap->i2c_cmds,
			PCR_I2C_CMD_TX(PCR_I2C_CMD_STA | PCR_I2C_CMD_STO |
				PCR_I2C_CMD_WR, AT88SC_CMD_READ_UZ));
	combov2_wait_for_tip(combo6, c6adap->i2c_cmds);

	i2c_set_adapdata(adap, c6adap);
	snprintf(adap->name, sizeof(adap->name), "combov2-i2c: %s",
			dev_name(&pdev->dev));
	adap->dev.parent = &pdev->dev;
	adap->owner = THIS_MODULE;
	adap->algo = &combov2_algo;
	ret = i2c_add_adapter(adap);
	if (ret) {
		dev_err(&pdev->dev, "can't add i2c adapter %u\n", idx);
		goto err;
	}

	combo6->u.three.i2c_adap[idx] = c6adap;
err:
	return ret;
}

static void combov2_i2c_stop_adaps(struct combo6 *combo6)
{
	unsigned int a;

	for (a = 0; a < 3; a++) {
		struct combo6_i2c_adapter *adap = combo6->u.three.i2c_adap[a];

		if (!adap)
			continue;
		i2c_del_adapter(&adap->adap);
		combo6_br_writel(combo6, adap->i2c_ctls, 0x0000f9);
		combo6->u.three.i2c_adap[a] = NULL;
	}
}

int combov2_i2c_init(struct combo6 *combo6)
{
	struct pci_dev *pdev = combo6->pci;
	struct combo6_i2c_adapter *c6adap;
	unsigned int a;
	int ret = 0;
	u32 stat;

	c6adap = kcalloc(3, sizeof(*c6adap), GFP_KERNEL);
	if (c6adap == NULL) {
		dev_err(&pdev->dev, "can't alloc i2c adapter\n");
		ret = -ENOMEM;
		goto err;
	}

	stat = combo6_br_readl(combo6, PCR_FWSTAT);

	ret = combov2_i2c_start_adap(combo6, &c6adap[0], 0);
	if (ret)
		goto err_stop;
	for (a = 0; a < 2; a++)
		if (!(stat & PCR_FWS_IF_MISSING(a))) {
			ret = combov2_i2c_start_adap(combo6, &c6adap[a+1], a+1);
			if (ret)
				goto err_stop;
		}

	return 0;
err_stop:
	combov2_i2c_stop_adaps(combo6);
	kfree(c6adap);
err:
	return ret;
}

void combov2_i2c_exit(struct combo6 *combo6)
{
	struct combo6_i2c_adapter *c6adap = combo6->u.three.i2c_adap[0];

	if (!c6adap)
		return;
	combov2_i2c_stop_adaps(combo6);
	kfree(c6adap);
}
