/*
 *  ccombov2.c: driver for this generation combo hardware
 *  Copyright (c) 2008 CESNET
 *  Author(s): Jaroslav Kysela <perex@perex.cz>
 *	       Jiri Slaby <jirislaby@gmail.com>
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 */

#include <linux/module.h>
#include <asm/io.h>
#include <asm/uaccess.h>
#include <linux/time.h>
#include <linux/delay.h>
#include <linux/interrupt.h>
#include <linux/init.h>
#include <linux/pci.h>
#include <linux/slab.h>
#include <linux/mm.h>
#include <linux/mutex.h>
#include <linux/nmi.h>
#include <linux/types.h>

#include "combo6.h"
#include "combo6k.h"
#include "combov2.h"
#include "combo6cpld.h"

MODULE_AUTHOR("CESNET; Jaroslav Kysela <perex@perex.cz>");
MODULE_AUTHOR("CESNET; Jiri Slaby <jirislaby@gmail.com>");
MODULE_DESCRIPTION("Combo6 version 2 core Linux Driver");
MODULE_LICENSE("GPL");

static irqreturn_t combov2_interrupt(int irq, void *dev_id);
static int combov2_ioctl(struct combo6 *combo6,
			 struct inode *inode, struct file *file,
			 unsigned int cmd, unsigned long arg);
static void combov2_read_id(struct combo6 *combo6);

static void combov2_fwnum(struct combo6 *combo6)
{
	struct pci_dev *pci = combo6->pci;
	struct pci_bus *bus = pci->bus;
	unsigned short devfn = pci->devfn;
	int cap;
	u32 ctl = PCR_FWC_BOOT_DELAYED;
	u16 lctl;

	ctl |= (combo6->u.three.fwnum % 100) << PCR_FWC_BOOT_DESIGN_S;
	/* boot from flash? */
	if (combo6->u.three.fwnum >= 100)
		ctl |= PCR_FWC_BOOT_SRC;

	dev_info(&pci->dev, "switching to firmware %i\n",
			combo6->u.three.fwnum);
	cap = pci_find_capability(bus->self, PCI_CAP_ID_EXP);
	if (!cap) {
		dev_err(&pci->dev, "can't find PCI express capability on the "
				"parent bus\n");
		return;
	}

	combo6_br_writel(combo6, PCR_FWCONTROL, ctl);
	pci_remove_bus_device(pci);
	/* turn off PCI-E downstream port */
	pci_read_config_word(bus->self, cap + PCI_EXP_LNKCTL, &lctl);
	pci_write_config_word(bus->self, cap + PCI_EXP_LNKCTL,
			lctl | PCI_EXP_LNKCTL_LD);
	/*
	 * 500ms for booting is experimentally obtained value (min. is approx.
	 * 400ms). Including fallback design boot. So wait COMBOV2_BOOTDELAY
	 * before the card starts the boot process (since PCR_FWCONTROL write)
	 * plus 500ms for booting itself.
	 */
	msleep(500 + COMBOV2_BOOTDELAY);
	/* turn on PCI-E downstream port */
	pci_write_config_word(bus->self, cap + PCI_EXP_LNKCTL, lctl);
	/*
	 * We should wait here for some time until the link wakes up (to train
	 * up or so). It is possible that it's not needed at all, but we stay
	 * on the safe side.
	 */
	msleep(100);
	pci_scan_slot(bus, devfn);
	pci = pci_get_slot(bus, devfn);
	if (pci == NULL) {
		dev_err(&bus->self->dev, "unable to find new PCI device!\n");
		return;
	}
	bus = pci->bus;
	pci_bus_assign_resources(bus);
	pci_bus_add_devices(bus);
	dev_info(&pci->dev, "firmware switch done\n");
	pci_dev_put(pci);
}

static void combov2_go_release(struct combo6 *combo6)
{
	if (combo6->u.three.fwnum >= 0) {
		combov2_fwnum(combo6);
		combo6->u.three.fwnum = -1;
		clear_bit(COMBO_BOOTING, combo6->flags);
	}
}

struct combov2_nr_name {
	unsigned char nr;
	const char *name;
};

static const struct combov2_nr_name *combov2_find_name(unsigned char nr,
		const struct combov2_nr_name *nr_name)
{
	while (nr_name->name) {
		if (nr_name->nr == nr)
			break;
		nr_name++;
	}
	return nr_name;
}

static const char *combov2_get_info(struct combo6 *combo6, char *buf,
		enum combo6_info_type info, unsigned long data)
{
	static const struct combov2_nr_name ctype[] = {
		{ 0xc2, "combo" },
		{ 0x01, "1G4" },
		{ 0x02, "10G1" },
		{ 0x03, "10G2" },
		{ 0x04, "Test" },
		{ 0x05, "40G2" },
		{ }
	};
	static const struct combov2_nr_name cstype[] = {
		{ 0x00, "NONE" },
		{ 0x01, "LXT110" },
		{ 0x02, "LXT155" },
		{ 0x03, "FXT100" },
 		{ 0x04, "LXT240" },
 		{ 0x05, "HXT365" },
		{ }
	};

	switch (info) {
	case COMBO6_INFO_TYPE:
		return combov2_find_name(combo6->u.three.ctype,
				ctype)->name;
	case COMBO6_INFO_SUBTYPE:
		return combov2_find_name(combo6->u.three.cstype,
				cstype)->name;
	case COMBO6_INFO_ADDON_CARD:
		return combov2_find_name(combo6->addon_card[data],
				ctype)->name;
	case COMBO6_INFO_PORTS:
		sprintf(buf, "%d", data ?
				COMBOV2_TX_IFACES(combo6->addon_interfaces) :
				COMBOV2_RX_IFACES(combo6->addon_interfaces));
		return buf;
	case COMBO6_INFO_SERNO:
		return combo6->u.three.serno[data];
	case COMBO6_INFO_SPGRADE:
		sprintf(buf, "%u", combo6->u.three.spgrade);
		return buf;
	default:
		return NULL;
	}
}

static void combov2_proc_info(struct combo6 *combo6,
		struct combo6_info_buffer *buffer)
{
	u32 ver, date;

	ver = combo6->pcibr_version;
	date = combo6->pcibr_date;
	combo6_iprintf(buffer, "Caps     : 0x%08x\n"
		"ID ver.  : 0x%04x\n"
		"NetCope  : 0x%04x\n"
		"PCI brver: %04x.%02x.%02x (%04i/%02i/%02i %02i:%02i)\n",
		combo6->u.three.caps,
		combo6->u.three.id_ver,
		combo6->u.three.netcope_ver,
		PCR_ID_IDENT(ver),
		PCR_ID_MAJOR(ver),
		PCR_ID_MINOR(ver),
		BIRTH_YEAR(date) + 2000,
		BIRTH_MONTH(date),
		BIRTH_DAY(date),
		BIRTH_HOUR(date),
		BIRTH_MINUTE(date));
}

/**
 * combov2_set_capabilities - reads and caches HW capabilities
 *
 * @combo: device to use
 *
 * The cached capabilities are stored into @combo->u.three.caps
 */
static void combov2_set_capabilities(struct combo6 *combo)
{
	static const struct combo_conf {
		u32 sw_low;
		u32 sw_high;
		u32 caps;
	} combo_conf[] = {
		{ 0x41c10507, 0x41c105ff, COMBOV2_CAP_MRRS4096 }, /* NIC */
		{ 0xf1010300, 0xf1010fff, COMBOV2_CAP_MRRS4096 }, /* FlowMon */
		{          0,         ~0, 0 } /* fallback, must be last */
	};
	const struct combo_conf *conf = combo_conf;
	u32 caps;
	u16 ncver = combo->u.three.netcope_ver;

	if (!test_bit(COMBO_BOOT_OK, combo->flags))
		return;

	for ( ; ; conf++)
		if (combo->id_sw >= conf->sw_low &&
				combo->id_sw <= conf->sw_high) {
			combo->u.three.caps = conf->caps;
			break;
		}

	if (!combo->u.three.id_ver || !ncver)
		return;

	caps = COMBOV2_CAP_NCVER;

	if (COMBOV2_NC_VER_MAJ(ncver) >= 1)
		caps |= COMBOV2_CAP_IRQSTAT;

	if (COMBOV2_NC_VER_MAJ(ncver) >= 2)
		caps |= COMBOV2_CAP_GICS;

	combo->u.three.caps |= caps;
}

/**
 * combov2_tuneup - tune up PCI parameters for higher performance
 *
 * For now we just increase read request and payload sizes (if possible).
 * See PCI specs for details on this topic.
 *
 * @combo: device to tune up
 * @exp_cap: PCI express capability pointer
 */
static void combov2_tuneup(struct combo6 *combo, int exp_cap)
{
	struct pci_dev *pdev = combo->pci;
	struct pci_dev *bus = pdev->bus->self;
	int ret, bus_ecap;
	u16 bus_payload, devctl, dev_allows;

	/* the card goes crazy when set to 4096 */
	ret = pcie_set_readrq(pdev, combo->u.three.caps & COMBOV2_CAP_MRRS4096 ?
			4096 : 2048);
	if (ret)
		dev_err(&pdev->dev, "can't set read request size\n");

	if (!exp_cap)
		return;
	/*
	 * first we check bus payload size, then device capabilities and
	 * choose the lower
	 */
	bus_ecap = pci_find_capability(bus, PCI_CAP_ID_EXP);
	if (!bus_ecap) {
		dev_err(&pdev->dev, "can't find PCI express capability on the "
				"parent bus\n");
		return;
	}
	pci_read_config_word(bus, bus_ecap + PCI_EXP_DEVCTL, &bus_payload);
	bus_payload &= PCI_EXP_DEVCTL_PAYLOAD;

	pci_read_config_word(pdev, exp_cap + PCI_EXP_DEVCTL, &devctl);
	pci_read_config_word(pdev, exp_cap + PCI_EXP_DEVCAP, &dev_allows);
	dev_allows &= PCI_EXP_DEVCAP_PAYLOAD;
	dev_allows <<= 12;
	dev_allows = min_t(u16, bus_payload, dev_allows);

	/* it's already there, bail out */
	if (dev_allows == (devctl & PCI_EXP_DEVCTL_PAYLOAD))
		return;
	devctl &= ~PCI_EXP_DEVCTL_PAYLOAD;
	devctl |= dev_allows;
	pci_write_config_word(pdev, exp_cap + PCI_EXP_DEVCTL, devctl);
}

/**
 * combov2_check_performance - check that the PCI device is set up correctly
 *
 * @combo: device to check
 * @exp_cap: PCI express capability pointer
 */
static void combov2_check_performance(struct combo6 *combo, int exp_cap)
{
#define CV2_WARN(x, ...) dev_warn(&pdev->dev, "WARNING: " x \
		" This will have impact on performance.\n", ##__VA_ARGS__);
	struct pci_dev *pdev = combo->pci;
	u16 tmp;

	if (!exp_cap)
		return;

	/* PCI-e specs, section 7.8.8 */
	pci_read_config_word(pdev, exp_cap + PCI_EXP_LNKSTA, &tmp);
	tmp &= PCI_EXP_LNKSTA_NLW;
	tmp >>= 4;
	if (tmp < 8)
		CV2_WARN("the device is running on link width x%u.", tmp);

	/* PCI-e specs, section 7.8.4 */
	pci_read_config_word(pdev, exp_cap + PCI_EXP_DEVCTL, &tmp);
	tmp &= PCI_EXP_DEVCTL_PAYLOAD;
	tmp >>= 5;
	tmp = 128 << tmp;
	if (tmp < 256)
		CV2_WARN("PCI-E max payload size is %u B.", tmp);
#undef CV2_WARN
}

int combov2_probe(struct pci_dev *pci,
		  const struct pci_device_id *id,
		  struct combov2_init *init,
		  struct module *this_module)
{
	static const struct combo6_ops ops = {
		.ioctl = combov2_ioctl,
		.go_release = combov2_go_release,
		.get_info = combov2_get_info,
		.proc_info = combov2_proc_info,
	};
	struct combo6 *combo6;
	int ret, exp_cap;

	ret = pci_enable_device(pci);
	if (ret) {
		dev_err(&pci->dev, "can't enable pci device\n");
		goto err;
	}

	combo6 = __combo6_alloc(0, this_module);
	if (combo6 == NULL) {
		dev_err(&pci->dev, "can't allocate combo6\n");
		ret = -ENOMEM;
		goto err_dis;
	}
	combo6->pci = pci;
	combo6->ops = &ops;
	combo6->usemode = init->usemode;
	if (init->bootprotect)
		set_bit(COMBO_BOOTPROTECT, combo6->flags);
	combo6->bootdelay = init->bootdelay;

	combo6->is_v2 = 1;
	combo6->u.three.fwnum = -1;
	mutex_init(&combo6->u.three.info_lock);
	set_bit(COMBO_BOOT_OK, combo6->flags);

	ret = pci_enable_msi(pci);
	if (ret)
		dev_err(&pci->dev, "can't enable msi\n");
	else
		set_bit(COMBO_MSI, combo6->flags);

	ret = pci_set_dma_mask(pci, DMA_BIT_MASK(64));
	if (ret)
		goto err_free;

	pci_set_master(pci);

	ret = -ENOMEM;
	combo6->mem_phys = pci_resource_start(pci, 0);
	combo6->mem_len = pci_resource_len(pci, 0);
	if (!request_mem_region(combo6->mem_phys, combo6->mem_len,
			"ComboV2 PLX")) {
		dev_err(&pci->dev, "unable to grab memory region "
			"0x%llx-0x%llx\n", (u64)combo6->mem_phys,
			(u64)(combo6->mem_phys + combo6->mem_len - 1));
		goto err_free;
	}
	combo6->mem_virt = ioremap_nocache(combo6->mem_phys, combo6->mem_len);
	if (combo6->mem_virt == NULL) {
		dev_err(&pci->dev, "unable to remap memory region "
			"0x%llx-0x%llx\n", (u64)combo6->mem_phys,
			(u64)(combo6->mem_phys + combo6->mem_len - 1));
		release_mem_region(combo6->mem_phys, combo6->mem_len);
		goto err_free;
	}
	if (combo6->usemode & USEMODE_SKIP_BAR12)
		goto __skip_bar12;
	combo6->lmem_phys = pci_resource_start(pci, 2);
	combo6->lmem_len = pci_resource_len(pci, 2);
	combo6->cmem_phys = pci_resource_start(pci, 4);
	combo6->cmem_len = pci_resource_len(pci, 4);
	if (init->invalidmemsize) {
		if (combo6->mem_len > 64 * 1024 * 1024)
			combo6->mem_len = 64 * 1024 * 1024;
		if (combo6->lmem_len > 64 * 1024 * 1024)
			combo6->lmem_len = 64 * 1024 * 1024;
		if (combo6->cmem_len > 64 * 1024 * 1024)
			combo6->cmem_len = 64 * 1024 * 1024;
	}
	if (!request_mem_region(combo6->lmem_phys,
			combo6->lmem_len, "ComboV2 LB")) {
		dev_err(&pci->dev, "unable to grab memory region "
			"0x%llx-0x%llx\n", (u64)combo6->lmem_phys,
			(u64)(combo6->lmem_phys + combo6->lmem_len - 1));
		goto err_free;
	}
	combo6->lmem_virt = ioremap_nocache(combo6->lmem_phys,
			combo6->lmem_len);
	if (combo6->lmem_virt == NULL) {
		dev_err(&pci->dev, "unable to remap memory region "
			"0x%llx-0x%llx\n", (u64)combo6->lmem_phys,
			(u64)(combo6->lmem_phys + combo6->lmem_len - 1));
		release_mem_region(combo6->lmem_phys, combo6->lmem_len);
		goto err_free;
	}
	if (!request_mem_region(combo6->cmem_phys, combo6->cmem_len,
				"ComboV2 CPLD")) {
		dev_err(&pci->dev, "unable to grab memory region "
			"0x%llx-0x%llx\n", (u64)combo6->cmem_phys,
			(u64)(combo6->cmem_phys + combo6->cmem_len - 1));
		goto err_free;
	}
	combo6->cmem_virt = ioremap_nocache(combo6->cmem_phys,
			combo6->cmem_len);
	if (combo6->cmem_virt == NULL) {
		dev_err(&pci->dev, "unable to remap memory region "
			"0x%llx-0x%llx\n", (u64)combo6->cmem_phys,
			(u64)(combo6->cmem_phys + combo6->cmem_len - 1));
		release_mem_region(combo6->cmem_phys, combo6->cmem_len);
		goto err_free;
	}
__skip_bar12:
	ret = request_irq(pci->irq, combov2_interrupt, IRQF_SHARED, "ComboV2",
			combo6);
	if (ret) {
		dev_err(&pci->dev, "unable to allocate interrupt %d\n",
				pci->irq);
		goto err_free;
	}
	combo6->irq = pci->irq;

	pci_set_drvdata(pci, combo6);

	if (combo6->usemode & USEMODE_SKIP_BRVER)
		goto __skip_brver;
	combo6->pcibr_version = combo6_br_readl(combo6, PCR_IDENT);
	if (PCR_ID_IDENT(combo6->pcibr_version) != 0x6d05) {
		dev_err(&pci->dev, "bad PCI bridge version: 0x%x\n",
				combo6->pcibr_version);
		ret = -EINVAL;
		goto err_free;
	}
	if (combo6_br_readl(combo6, PCR_FWCONTROL) & PCR_FWC_REC_ERR)
		dev_warn(&pci->dev, "firmware switch failed, reconfiguration "
				"error bit raised, fell back to design 0\n");

	combo6_br_writel(combo6, PCR_FWDELAY, PCR_FWD_MS(COMBOV2_BOOTDELAY));

	combo6->pcibr_date = combo6_br_readl(combo6, PCR_BIRTH);
__skip_brver:
	combo6->addon_interfaces = -1;

	/* ML555 and HTG have no I2C */
	if (pci->device != 0x6d05 && pci->device != 0x6106) {
		ret = combov2_i2c_init(combo6);
		if (ret) {
			dev_err(&pci->dev, "can't initialize i2c");
			goto err_free;
		}

		ret = combov2_i2c_read_id(combo6);
		if (ret)
			dev_err(&pci->dev, "can't read i2c info: %d\n", ret);
	}
	/* Card used olny for testing */
	else if(pci->device == 0x6106) {
		combo6->u.three.serno[0][16] = 0;
		combo6->u.three.spgrade = 1;
		combo6->u.three.ctype = 0xc2;
		combo6->u.three.cstype = 0x04;
	}

	if ((combo6->usemode & USEMODE_SKIP_BAR12) == 0)
		combov2_read_id(combo6);

	/* From ID Comp >=1.04 all interrupt except DMA are masked by default */
	if(combo6->u.three.id_ver >= 0x0104) {
		/* Enable all known interrupts, mask is inverted */
		ret = ~(COMBOV2_IRQS_RX | COMBOV2_IRQS_TX | COMBOV2_IRQS_SYSMON | COMBOV2_IRQS_LINK_CHG);
		combo6_fpga_writel(combo6, COMBOV2_ID_IRQMASKAND, ret);
	}

	combov2_set_capabilities(combo6);

	exp_cap = pci_find_capability(pci, PCI_CAP_ID_EXP);
	if (!exp_cap)
		dev_err(&pci->dev, "can't find PCI express capability on the "
				"card\n");

	combov2_tuneup(combo6, exp_cap);
	combov2_check_performance(combo6, exp_cap);

	ret = combo6_add(combo6);
	if (ret) {
		dev_err(&pci->dev, "too many cards in system\n");
		goto err_free;
	}
	return 0;
err_free:
	combo6_free(combo6);
err_dis:
	pci_disable_device(pci);
err:
	return ret;
}
EXPORT_SYMBOL(combov2_probe);

void combov2_remove(struct pci_dev *pci)
{
	struct combo6 *combo6 = (struct combo6 *)pci_get_drvdata(pci);
	int tmp;

	/* From ID Comp >=1.04 all interrupt except DMA are masked by default */
	if(combo6->u.three.id_ver >= 0x0104) {
		/* So disable all known interrupts, except DMA. Mask is inverted */
		tmp = ~(COMBOV2_IRQS_RX	| COMBOV2_IRQS_TX);
		combo6_fpga_writel(combo6, COMBOV2_ID_IRQMASKOR, tmp);
	}

	set_bit(COMBO_GOING_DOWN, combo6->flags);
	combov2_i2c_exit(combo6);
	combo6_remove(combo6);
	combo6_ioctl_bus_detach(combo6);
	vfree(combo6->u.three.info);
	combo6_free(combo6);
	pci_set_drvdata(pci, NULL);
}
EXPORT_SYMBOL(combov2_remove);

static int combov2_ioctl_read(struct combo6 *combo6,
		struct combo6_ioc_arg __user *_r)
{
	struct combo6_ioc_arg r;

	if (copy_from_user(&r, _r, sizeof(r)))
		return -EFAULT;
	switch (r.target) {
	case COMBO6_CPLD_BUS:
		if (r.offset >= combo6->cmem_len || (r.offset % 4))
			return -EINVAL;
		if (r.size != 4)
			return -EINVAL;
		if (r.count != 1)
			return -EINVAL;
		if (put_user(readl(combo6->cmem_virt + r.offset),
				(u32 __user __force *)r.data))
			return -EFAULT;
		break;
	case COMBO6_BRIDGE_REGS:
		if (r.offset >= combo6->mem_len || (r.offset % 4))
			return -EINVAL;
		if (r.size != 4)
			return -EINVAL;
		if (r.count != 1)
			return -EINVAL;
		if (put_user(readl(combo6->mem_virt + r.offset),
				(u32 __user __force *)r.data))
			return -EFAULT;
		break;
	case COMBO6_FPGA_BUS:
		if (!test_bit(COMBO_BOOT_OK, combo6->flags))
			return -EBADFD;
		if (r.offset >= combo6->lmem_len || (r.offset % 4))
			return -EINVAL;
		if (r.size != 4)
			return -EINVAL;
		if (r.count != 1)
			return -EINVAL;
		if (put_user(readl(combo6->lmem_virt + r.offset),
				(u32 __user __force *)r.data))
			return -EFAULT;
		break;
	default:
		return -EINVAL;
	}
	return 0;
}

static int combov2_ioctl_write(struct combo6 *combo6,
		struct combo6_ioc_arg __user *_r)
{
	struct combo6_ioc_arg r;
	u32 val;

	if (copy_from_user(&r, _r, sizeof(r)))
		return -EFAULT;
	switch (r.target) {
	case COMBO6_CPLD_BUS:
		if (r.offset >= combo6->cmem_len || (r.offset % 4))
			return -EINVAL;
		if (r.size != 4)
			return -EINVAL;
		if (r.count != 1)
			return -EINVAL;
		if (get_user(val, (u32 __user __force *)r.data))
			return -EFAULT;
		writel(val, combo6->cmem_virt + r.offset);
		break;
	case COMBO6_BRIDGE_REGS:
		if (r.offset >= combo6->mem_len || (r.offset % 4))
			return -EINVAL;
		if (r.size != 4)
			return -EINVAL;
		if (r.count != 1)
			return -EINVAL;
		if (get_user(val, (u32 __user __force *)r.data))
			return -EFAULT;
		writel(val, combo6->mem_virt + r.offset);
		break;
	case COMBO6_FPGA_BUS:
		if (r.offset >= combo6->lmem_len || (r.offset % 4))
			return -EINVAL;
		if (r.size != 4)
			return -EINVAL;
		if (r.count != 1)
			return -EINVAL;
		if (get_user(val, (u32 __user __force *)r.data))
			return -EFAULT;
		writel(val, combo6->lmem_virt + r.offset);
		break;
	default:
		return -EINVAL;
	}
	return 0;
}

/*
 *  combo6 device management
 */

#define ID32_TO_STR(strptr, val32) \
	*((strptr)+1) = (val32) & 0xff; \
	*((strptr)+0) = ((val32) >> 8) & 0xff; \
	*((strptr)+3) = ((val32) >> 16) & 0xff; \
	*((strptr)+2) = ((val32) >> 24) & 0xff;

static void combov2_read_id(struct combo6 *combo6)
{
	u32 tmp;
	int i;

	if (combo6->usemode & USEMODE_SKIP_LOCALBUS)
		return;
	if (!test_bit(COMBO_BOOT_OK, combo6->flags))
		return;
	if (combo6->bootdelay > 0 && combo6->bootdelay <= 100)
		msleep(combo6->bootdelay);
	tmp = combo6_fpga_readl(combo6, COMBOV2_ID_NEG);
	combo6_fpga_writel(combo6, COMBOV2_ID_NEG, tmp);
	if (combo6_fpga_readl(combo6, COMBOV2_ID_NEG) == ~tmp) {
		combo6->id_sw = combo6_fpga_readl(combo6, COMBOV2_ID_SW);
		combo6->id_hw = combo6_fpga_readl(combo6, COMBOV2_ID_HW);
		for (i = 0; i < 32; i += 4) {
			tmp = combo6_fpga_readl(combo6, COMBOV2_ID_TEXT + i);
			ID32_TO_STR(combo6->id_txt + i, tmp);
		}
		combo6->id_txt[32] = '\0';
		combo6->addon_interfaces = combo6_fpga_readl(combo6,
				COMBOV2_ID_CHANNELS) & 0xffff;
		tmp = combo6_fpga_readl(combo6, COMBOV2_ID_ID_VER);
		combo6->u.three.id_ver = tmp & 0xffff;
		combo6->u.three.netcope_ver = tmp >> 16;
	} else {
		combo6->id_sw = 0;
		combo6->id_hw = 0;
		combo6->id_txt[0] = '\0';
	}
	if ((combo6->id_sw == 0 || combo6->id_sw == ~0) &&
			test_bit(COMBO_BOOTPROTECT, combo6->flags))
		clear_bit(COMBO_BOOT_OK, combo6->flags);
}

#define FLASH_AREA_SIZE		(8 << 20) /* 64M split into eight 8M pcs. */
#define FLASH_SECTOR_SIZE	(64 << 10)

static int combov2_wait_for_rdy(struct combo6 *combo6, int write)
{
	unsigned int counter = 2000;
	int ready;

	while (--counter > 0) {
		ready = combo6_br_readl(combo6, PCR_MEM_CMDSTAT) &
				PCR_MEM_STAT_READY;
		if (ready)
			break;
		/* typically 10-20 us for writing, 500 ms for flashing */
		if (write)
			udelay(2);
		else {
			msleep(60);

			/* this loop may last looong */
			touch_nmi_watchdog();
		}
	}

	return !ready;
}

static int combov2_erase_flash(struct combo6 *combo6, u32 addr, u32 size)
{
	unsigned int a;

	size = DIV_ROUND_UP(size, FLASH_SECTOR_SIZE);

	for (a = 0; a < size; a++) {
		if (combov2_wait_for_rdy(combo6, 0))
			return -EIO;
		combo6_br_writel(combo6, PCR_MEM_CMDSTAT,
				PCR_MEM_CMD(PCR_MEM_CMD_FLASH,
					PCR_MEM_CMD_MEM_FLASH, addr >> 1));
		addr += FLASH_SECTOR_SIZE;
	}

	return combov2_wait_for_rdy(combo6, 0) ? -EIO : 0;
}

static int combov2_wait_for_iface_rdy(struct combo6 *combo, u32 iface)
{
	unsigned int counter = 2000;
	int ready;

	while (--counter > 0) {
		ready = combo6_br_readl(combo, PCR_IF_CTL(iface)) &
				PCR_IF_CTL_DRDY;
		if (ready)
			break;
		udelay(1);
	}

	return !ready;
}

static int combov2_boot_iface_fpga(struct combo6 *combo, u32 iface, u32 *fw,
		u32 count)
{
	unsigned int a, tmp;
	u32 stat;

	stat = combo6_br_readl(combo, PCR_FWSTAT);

	if (stat & PCR_FWS_IF_MISSING(iface))
		return -EINVAL;

	combo6_br_writel(combo, PCR_IF_CTL(iface), PCR_IF_CTL_BOOT);
	for (a = tmp = 0; a < count / 4; a++) {
		if (a % 16 == 0 && combov2_wait_for_iface_rdy(combo, iface))
			return -EIO;

		combo6_br_writel(combo, PCR_IF_DATA(iface), *fw++);
		if (tmp++ > 0x2000) {
			tmp = 0;
			cond_resched();
		}
	}
	combo6_br_writel(combo, PCR_IF_CTL(iface), 0);

	stat = combo6_br_readl(combo, PCR_FWSTAT);
	if (!(stat & PCR_FWS_IF_BOOTOK(iface))) {
		dev_err(&combo->pci->dev, "booting interface %u failed!\n",
				iface);
		return -EIO;
	}

	return 0;
}

static int combov2_boot_main_fpga(struct combo6 *combo6, u32 unit, u32 *data,
		u32 count)
{
	unsigned int a, tmp;
	u32 addr = 0, mem;
	int flash = 0;

	switch (unit) {
	case 0:
		mem = PCR_MEM_CMD_MEM_PSRAM;
		/* switch to the fw 0 when overwriting PSRAM */
		combo6->u.three.fwnum = 0;
		break;
	case 200 ... 215:
		addr = FLASH_AREA_SIZE - count;
	case 100 ... 115:
		flash = 1;
		mem = PCR_MEM_CMD_MEM_FLASH;
		addr += (unit % 100) * FLASH_AREA_SIZE;
		combov2_erase_flash(combo6, addr & ~(FLASH_SECTOR_SIZE - 1),
				count);
		/* flash is addressed by 16-bit words, divide by 2 */
		addr /= 2;
		break;
	default:
		return -EINVAL;
	}

	combo6_br_writel(combo6, PCR_MEM_CMDSTAT, PCR_MEM_CMD(PCR_MEM_CMD_WRITE,
				mem, addr));
	for (a = tmp = 0; a < count / 4; a++) {
		if (flash && combov2_wait_for_rdy(combo6, 1))
			return -EIO;
		combo6_br_writel(combo6, PCR_MEM_WR, *data++);
		if (tmp++ > 0x2000) {
			tmp = 0;
			cond_resched();
		}
	}
	/* Stop writing to the PSRAM/FLASH */
	combo6_br_writel(combo6, PCR_MEM_CMDSTAT, PCR_MEM_CMD(PCR_MEM_CMD_READ,
				mem, 0x0));
	
	return 0;
}

static int combov2_store_address(struct combo6 *combo,
		struct combo_design_info *info, unsigned int count)
{
	struct combo_design_info *dup;
	unsigned int a;

	for (a = 0; a < count; a++)
		if (info[a].id > COMBO_DI_MAX)
			return -EINVAL;

	dup = vmalloc(sizeof(*info) * count);
	if (!dup)
		return -ENOMEM;

	memcpy(dup, info, sizeof(*info) * count);

	mutex_lock(&combo->u.three.info_lock);
	if (combo->u.three.info)
		vfree(combo->u.three.info);
	combo->u.three.info = dup;
	combo->u.three.info_count = count;
	mutex_unlock(&combo->u.three.info_lock);

	return 0;
}

static int combov2_boot_unit(struct combo6 *combo, u32 unit, void *data,
		u32 count)
{
	switch (unit) {
	case 0:
	case 100 ... 115:
	case 200 ... 215:
		return combov2_boot_main_fpga(combo, unit, data, count);
	case 8:
	case 9:
		return combov2_boot_iface_fpga(combo, unit - 8, data, count);
	case 1025:
		if (count % sizeof(struct combo_design_info))
			return -EINVAL;
		return combov2_store_address(combo, data,
				count / sizeof(struct combo_design_info));
	default:
		return -EINVAL;
	}
}

static int combov2_ioctl_boot(struct combo6 *combo6,
		struct combo6_boot __user *_boot)
{
	struct combo6_boot boot;
	unsigned int boot_base = 0;
	int i, res = 0;
	void *data;
	u32 size;

	if (combo6->usemode & USEMODE_SKIP_BAR12)
		return -ENXIO;
	if (copy_from_user(&boot, _boot, sizeof(boot)))
		return -EFAULT;

	for (i = 0; i < COMBO6_BOOT_IMAGES && boot.d[i].size; i++)
		if (boot.d[i].unit == 0) {
			boot_base = 1;
			break;
		}

	mutex_lock(&combo6->drv_mutex);
	if (boot_base && combo6->driver) {
		mutex_unlock(&combo6->drv_mutex);
		return -EBUSY;
	}
	set_bit(COMBO_BOOTING, combo6->flags);
	mutex_unlock(&combo6->drv_mutex);
	for (i = 0; i < COMBO6_BOOT_IMAGES; i++) {
		size = boot.d[i].size;
		if (size == 0) {
			if (i == 0)
				res = -EINVAL;
			break;
		}
		data = vmalloc(size);
		if (data == NULL) {
			res = -ENOMEM;
			break;
		}
		if (copy_from_user(data, (void __user __force *)boot.d[i].data,
					size)) {
			vfree(data);
			res = -EFAULT;
			break;
		}
		res = combov2_boot_unit(combo6, boot.d[i].unit, data, size);
		vfree(data);
		if (res < 0)
			break;
	}
	if (boot_base) {
		if (res >= 0) {
			combov2_read_id(combo6);
		} else {
			clear_bit(COMBO_BOOT_OK, combo6->flags);
		}
	}
	if (combo6->u.three.fwnum < 0)
		clear_bit(COMBO_BOOTING, combo6->flags);
	return res;
}

static int combov2_design_info(struct combo6 *combo,
		struct combo_design_info *info)
{
	struct combo_design_info *i;
	unsigned int a;
	int ret = -ENODEV;

	mutex_lock(&combo->u.three.info_lock);
	i = combo->u.three.info;
	for (a = 0; a < combo->u.three.info_count; a++, i++)
		if (info->id == i->id && info->idx == i->idx) {
			info->addr = i->addr;
			info->size = i->size;
			ret = 0;
			break;
		}
	mutex_unlock(&combo->u.three.info_lock);

	return ret;
}

static int combov2_ioctl_fwsel_read(struct combo6 *combo6, int __user *_fwnum)
{
	u32 ctl = combo6_br_readl(combo6, PCR_FWCONTROL);
	int fwnum = (ctl & PCR_FWC_BOOT_DESIGN_M) >> PCR_FWC_BOOT_DESIGN_S;

	if (ctl & PCR_FWC_BOOT_SRC)
		fwnum += 100;

	return put_user(fwnum, _fwnum);
}

static int combov2_ioctl_fwsel_write(struct combo6 *combo6, int __user *_fwnum)
{
	int fwnum;

	if (get_user(fwnum, _fwnum))
		return -EFAULT;
	switch (fwnum) {
	case 0:
	case 100 ... 131:
		break;
	default:
		return -EINVAL;
	}
	combo6->u.three.fwnum = fwnum;
	return 0;
}

static int combov2_ioctl(struct combo6 *combo6,
			 struct inode *inode, struct file *file,
			 unsigned int cmd, unsigned long arg)
{
	void __user *argp = (void __user *)arg;
	int ret;

	switch (cmd) {
	case COMBO6_READ:
		return combov2_ioctl_read(combo6, argp);
	case COMBO6_WRITE:
		return combov2_ioctl_write(combo6, argp);
	case COMBO6_IOC_BOOT:
		return combov2_ioctl_boot(combo6, argp);
	case COMBO6_IOC_BUS_ATTACH:
		return combo6_ioctl_bus_attach(combo6);
	case COMBO6_IOC_BUS_DETACH:
		return combo6_ioctl_bus_detach(combo6);
	case COMBO6_IOC_INFO:
		return combo6_ioctl_info(combo6, argp);
	case COMBO6_IOC_DINFO: {
		struct combo_design_info di;
		if (copy_from_user(&di, argp, sizeof(di)))
			return -EFAULT;

		ret = combov2_design_info(combo6, &di);
		if (ret)
			return ret;

		if (copy_to_user(argp, &di, sizeof(di)))
			return -EFAULT;
		return 0;
	}
	case COMBO6_IOC_FWSEL_READ:
		return combov2_ioctl_fwsel_read(combo6, argp);
	case COMBO6_IOC_FWSEL_WRITE:
		if (file->f_flags & O_APPEND)
			return -EBUSY;
		return combov2_ioctl_fwsel_write(combo6, argp);
	case COMBO6_IOC_I2C_OP:
		return combov2_i2c_do_op(combo6, argp);
	default:
		return -ENOTTY;
	}
}

static irqreturn_t combov2_interrupt(int irq, void *dev_id)
{
	struct combo6 *combo6 = dev_id;
	u32 source = 0;

	if (test_bit(COMBO_GOING_DOWN, combo6->flags))
		return IRQ_NONE;

	if (combo6->u.three.caps & COMBOV2_CAP_IRQSTAT) {
		irqreturn_t ret = IRQ_NONE;

		source = combo6_fpga_readl(combo6, COMBOV2_ID_IRQSTAT);

		if (WARN_ON_ONCE(!source))
			return IRQ_NONE;

		if (source & COMBOV2_IRQS_SYSMON) {
			if (printk_ratelimit())
				dev_info(&combo6->pci->dev, "SYSMON alert\n");
			combo_queue_message(combo6->index, COMBOCTL_MSG_SYSMON);
			ret = IRQ_HANDLED;
		}

		if (source & COMBOV2_IRQS_LINK_CHG) {
			if (printk_ratelimit())
				dev_info(&combo6->pci->dev, "link changed\n");
			combo_queue_message(combo6->index, COMBOCTL_MSG_LINK);
			ret = IRQ_HANDLED;
		}

		source &= COMBOV2_IRQS_RX | COMBOV2_IRQS_TX;
		if (!source)
			return ret;
	}

	spin_lock(&combo6->reg_lock);
	if (combo6->driver && combo6->driver->interrupt)
		combo6->driver->interrupt(combo6, source);
	spin_unlock(&combo6->reg_lock);

	return IRQ_HANDLED;
}

/*
 * real initialization
 */

static int __init combov2_init(void)
{
	return 0;
}

static void __exit combov2_exit(void)
{
}

module_init(combov2_init)
module_exit(combov2_exit)
