/*
 *  ccombov2.c: driver for third generation of combo hardware
 *  Copyright (c) 2008 CESNET
 *  Author(s): Jaroslav Kysela <perex@perex.cz>
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 */

#include <linux/version.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/pci.h>

#include "combo6.h"
#include "combo6k.h"
#include "combov2.h"

/*
 *  PCI initialization and detection part
 */

static int enable[COMBO6_CARDS] = { [0 ... (COMBO6_CARDS-1)] = 1 };      /* Enable this card */
static int invalidmemsize[COMBO6_CARDS] = { [0 ... (COMBO6_CARDS-1)] = 0 }; /* Invalid memory region, fix to 64MB */
static int bootprotect[COMBO6_CARDS] = { [0 ... (COMBO6_CARDS-1)] = 1 }; /* Check design boot */
static int bootdelay[COMBO6_CARDS] = { [0 ... (COMBO6_CARDS-1)] = 0 }; /* Boot delay in ms before fpga access */
static int usemode[COMBO6_CARDS] = { [0 ... (COMBO6_CARDS-1)] = 0 }; /* Use mode (see README) */

MODULE_AUTHOR("CESNET; Jaroslav Kysela <perex@perex.cz>");
MODULE_DESCRIPTION("Combo6 version 2 core Linux Driver");
MODULE_LICENSE("GPL");

module_param_array(enable, int, NULL, 0);
MODULE_PARM_DESC(enable, "Enable ComboV2 card.");
module_param_array(invalidmemsize, int, NULL, 0);
MODULE_PARM_DESC(invalidmemsize, "Invalid PCI memory area size - fix to 64MB.");
module_param_array(bootprotect, int, NULL, 0);
MODULE_PARM_DESC(bootprotect, "Enable(default)=1/Disable=0 boot protection checks.");
module_param_array(bootdelay, int, NULL, 0);
MODULE_PARM_DESC(bootdelay, "Boot delay in ms before FPGA access.");
module_param_array(usemode, int, NULL, 0);
MODULE_PARM_DESC(usemode, "Use mode (for debugging only). See README.");

static int __devinit combov2_boot_probe(struct pci_dev *pci,
					const struct pci_device_id *id)
{
	struct combov2_init init;
	static int dev;

	if (!enable[dev]) {
		dev++;
		return -ENOENT;
	}
	memset(&init, 0, sizeof(init));
	init.invalidmemsize = invalidmemsize[dev];
	init.bootprotect = bootprotect[dev];
	init.bootdelay = bootdelay[dev];
	init.usemode = usemode[dev];
	return combov2_probe(pci, id, &init, THIS_MODULE);
}

static void __devexit combov2_boot_remove(struct pci_dev *pci)
{
        combov2_remove(pci);
}

static struct pci_device_id combov2_ids[] __devinitdata = {
	{ .vendor = PCI_VENDOR_ID_CESNET, .device = 0x6d05, /* ML555 */
		.subvendor = PCI_VENDOR_ID_CESNET, .subdevice = 0xff01 },
	{ .vendor = PCI_VENDOR_ID_CESNET, .device = 0xc032, /* LXT110 */
		.subvendor = PCI_VENDOR_ID_CESNET, .subdevice = 0xff01 },
	{ .vendor = PCI_VENDOR_ID_CESNET, .device = 0xc132, /* LXT155 */
		.subvendor = PCI_VENDOR_ID_CESNET, .subdevice = 0xff01 },
	{ .vendor = PCI_VENDOR_ID_CESNET, .device = 0xc232, /* FXT100 */
		.subvendor = PCI_VENDOR_ID_CESNET, .subdevice = 0xff01 },
	{ 0, }
};
MODULE_DEVICE_TABLE(pci, combov2_ids);

static struct pci_driver driver = {
	.name = "ComboV2 BOOT",
	.id_table = combov2_ids,
	.probe = combov2_boot_probe,
	.remove = __devexit_p(combov2_boot_remove),
};

/*
 * real initialization
 */

static int __init combov2_boot_init(void)
{
	int err;

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,22)
	if ((err = pci_register_driver(&driver)) < 0)
#else
	if ((err = pci_module_init(&driver)) < 0)
#endif
		return err;

	return 0;
}

static void __exit combov2_boot_exit(void)
{
	pci_unregister_driver(&driver);
}

module_init(combov2_boot_init)
module_exit(combov2_boot_exit)

#ifndef MODULE

/* format is: combov2boot=enable */

static int __init combov2_card_setup(char *str)
{
	static unsigned __initdata nr_dev = 0;

	if (nr_dev >= COMBO6_CARDS)
		return 0;
	(void)(get_option(&str,&enable[nr_dev]) == 2);
	nr_dev++;
	return 1;
}

__setup("combov2boot=", combov2_card_setup);

#endif
