/*
 * eth interface for szedata 2
 *
 * Copyright (c) 2009 CESNET
 * Copyright (c) 2009 Jiri Slaby <jirislaby@gmail.com>
 *
 * Licensed under GPLv2
 */

#include <linux/etherdevice.h>
#include <linux/kthread.h>
#include <linux/log2.h>
#include <linux/module.h>
#include <linux/netdevice.h>
#include <linux/poll.h>
#include <linux/skbuff.h>

#include "szedata2k.h"
#include "sze2.h"

struct szedata2_eth {
	struct szedata2 *sd;
	struct szedata2_app *app;
	struct task_struct *rx_task;
	unsigned int idx;
	unsigned int tx_idx_shift;
#ifdef CONFIG_NDEV_NO_STATS
	struct net_device_stats ndev_stats;
#endif
};

struct szedata2_packet {
	__u16   seg_size;	/* size of whole packet (header incl.) */
	__u16   hw_size;	/* size of hw data (optional) */
	__u8    data[0];	/* data themselves */
} __attribute((packed));

static int no_eth = 1;
module_param(no_eth, bool, S_IRUGO);
MODULE_PARM_DESC(no_eth, "disables standard eth szedata2 support");

static void szedata2_eth_copy(struct szedata2_eth *eth, unsigned int space,
		void *kptr, u32 off, unsigned int len, size_t asize)
{
	struct szedata2 *sd = eth->sd;
	void *rngptr;
	size_t s;
	unsigned int idx = eth->idx;
	int rx = space == SZE2_MMIO_RX;

	asize--; /* get mask */

	while (len) {
		s = szedata2_get_virt(sd, space, idx, &rngptr, off, len);
		if (kptr) {
			if (rx)
				memcpy(kptr, rngptr, s);
			else
				memcpy(rngptr, kptr, s);
			kptr += s;
		} else
			memset(rngptr, 0, s);
		len -= s;
		off += s;
		off &= asize;
	}
}

static struct sze2_adesc *szedata2_eth_rx_data(struct szedata2_eth *eth,
		struct szedata2_app *app)
{
	int a = szedata2_rxlock_data(app, BIT(eth->idx));

	if (WARN_ON(a < 0))
		a = 0;

	return &app->status_page->adesc[a][SZE2_MMIO_RX];
}

static int szedata2_eth_rx_handle(struct net_device *ndev,
		struct szedata2_eth *eth, struct sze2_adesc *adesc)
{
	struct szedata2 *sd = eth->sd;
	struct szedata2_ring *ring = &sd->ring[SZE2_MMIO_RX];
	size_t asize = ring->area[eth->idx].asize;
	unsigned int size = adesc->size;
	u64 offset = adesc->offset;
	u32 off = do_div(offset, asize);

	if (adesc->flags & SZE2_ADESC_FRAG)
		size += eth->app->status_page->adesc[
			SZE2_ADESC_NEXT(adesc->flags)][SZE2_MMIO_RX].size;

	while (size) {
		struct szedata2_packet pkt;
		struct sk_buff *skb;
		unsigned int plen, hlen;

		szedata2_eth_copy(eth, SZE2_MMIO_RX, &pkt, off, sizeof(pkt),
				asize);

		if (WARN_ON_ONCE(pkt.seg_size < 4 || pkt.seg_size >= 8192)) {
			const unsigned int buf_size = 2048;
			char *buf = kmalloc(buf_size, GFP_KERNEL);
			off += asize - buf_size / 2;
			off &= asize - 1;
			szedata2_eth_copy(eth, SZE2_MMIO_RX, buf, off, buf_size,
					asize);
			print_hex_dump_bytes("X ", DUMP_PREFIX_OFFSET, buf,
					buf_size / 2);
			print_hex_dump_bytes("Y ", DUMP_PREFIX_OFFSET,
					buf + buf_size / 2, buf_size / 2);
			kfree(buf);
			printk(KERN_WARNING "%s: won't receive anything else "
					"on interface %u. Segsize was %u.\n",
					__func__, eth->idx, pkt.seg_size);
			return -EIO;
		}

		hlen = ALIGN(4 + pkt.hw_size, 8);
		plen = pkt.seg_size - hlen;
		off += hlen;
		off &= asize - 1;

		skb = __netdev_alloc_skb(ndev, plen + NET_IP_ALIGN, GFP_KERNEL);
		if (!skb)
			break;
		skb_reserve(skb, NET_IP_ALIGN);
		szedata2_eth_copy(eth, SZE2_MMIO_RX, skb->data, off, plen,
				asize);
		skb_put(skb, plen);
		skb->protocol = eth_type_trans(skb, ndev);
		if (netif_rx_ni(skb) != NET_RX_DROP) {
			combo_get_ndev_stats(ndev)->rx_bytes += plen;
			combo_get_ndev_stats(ndev)->rx_packets++;
		} else
			combo_get_ndev_stats(ndev)->rx_dropped++;

		size -= ALIGN(pkt.seg_size, 8);
		off += ALIGN(plen, 8);
		off &= asize - 1;
	}

	return 0;
}

static int szedata2_eth_rx_thread(void *data)
{
	struct net_device *dev = data;
	struct szedata2_eth *eth = netdev_priv(dev);
	struct szedata2_app *app = eth->app;
	struct sze2_adesc *adesc;
	unsigned int poll = 0;
	int ret;

	while (!kthread_should_stop()) {
		adesc = szedata2_eth_rx_data(eth, app);
		if (!adesc->size) {
			if (wait_event_interruptible(app->poll_wait,
					kthread_should_stop() ||
					((poll = szedata2_poll(app)) &
					(POLLIN | POLLERR)))) {
				ret = -ERESTARTSYS;
				goto die;
			}
			if (poll & POLLERR) {
				printk(KERN_ERR "%s: poll failed\n", __func__);
				ret = 0;
				goto die;
			}
			continue;
		}

		if (szedata2_eth_rx_handle(dev, eth, adesc))
			break;
		szedata2_rxunlock_data(app);
	}

	return 0;
die:
	/*
	 * we want exit from kthread, but __put_task_struct is not exported for
	 * correct reference counting and we have to do this hack
	 */
	while (!kthread_should_stop()) {
		set_current_state(TASK_UNINTERRUPTIBLE);
		schedule();
	}
	return ret;
}

static int szedata2_eth_open(struct net_device *dev)
{
	struct szedata2_eth *eth = netdev_priv(dev);
	struct szedata2 *sd = eth->sd;
	unsigned int idx = eth->idx;
	struct sze2_subscribe_area sub = {
		.areas = { BIT(idx), BIT(idx) },
	};
	int ret;

	if (idx < sd->eth_ifaces[SZE2_MMIO_RX]) {
		eth->rx_task = kthread_create(szedata2_eth_rx_thread, dev,
				"szedata2_rx/%u", idx);
		if (IS_ERR(eth->rx_task)) {
			ret = PTR_ERR(eth->rx_task);
			goto err;
		}
	}

	eth->app = szedata2_open(sd, MINOR(sd->cdev.dev));
	if (IS_ERR(eth->app)) {
		ret = PTR_ERR(eth->app);
		goto err_kthread;
	}

	ret = szedata2_subscribe_area(eth->app, &sub);
	if (ret)
		goto err_close;

	ret = szedata2_start_devices(eth->app);
	if (ret)
		goto err_close;

	if (idx < sd->eth_ifaces[SZE2_MMIO_RX])
		wake_up_process(eth->rx_task);

	return 0;
err_close:
	szedata2_close(eth->app);
err_kthread:
	if (idx < sd->eth_ifaces[SZE2_MMIO_RX])
		kthread_stop(eth->rx_task);
err:
	return ret;
}

static int szedata2_eth_close(struct net_device *dev)
{
	struct szedata2_eth *eth = netdev_priv(dev);

	if (eth->idx < eth->sd->eth_ifaces[SZE2_MMIO_RX])
		kthread_stop(eth->rx_task);

	szedata2_close(eth->app);

	return 0;
}

static int szedata2_eth_xmit(struct sk_buff *skb, struct net_device *dev)
{
	struct szedata2_eth *eth = netdev_priv(dev);
	struct szedata2_app *app = eth->app;
	struct szedata2 *sd = eth->sd;
	struct szedata2_ring *ring = &sd->ring[SZE2_MMIO_TX];
	struct sze2_adesc *adesc;
	unsigned int lsize, size = 8 + max_t(unsigned int, skb->len, ETH_ZLEN);
	unsigned int idx = eth->idx >> eth->tx_idx_shift;
	struct sze2_tx_lock txl = { .size = size, .area = idx };
	struct sze2_tx_unlock txu = { .size = size, .area = idx };
	struct szedata2_packet pkt = { .seg_size = size };
	unsigned int len = skb->len;
	size_t asize;
	u64 off1;
	u32 off;
	int ret;

	if (idx >= sd->eth_ifaces[SZE2_MMIO_TX])
		goto free;

	adesc = &app->status_page->adesc[idx][SZE2_MMIO_TX];
	asize = ring->area[idx].asize;

#ifdef CONFIG_OLD_SKB_LINEARIZE
	ret = skb_linearize(skb, GFP_ATOMIC);
#else
	ret = skb_linearize(skb);
#endif
	if (ret) {
		if (net_ratelimit())
			printk(KERN_ERR "%s: can't linearize sk_buff: %d\n",
					__func__, ret);
		goto free;
	}

	ret = szedata2_txlock_data(app, &txl);
	if (ret)
		return NETDEV_TX_BUSY;

	lsize = adesc->size;
	if (adesc->flags & SZE2_ADESC_FRAG)
		lsize += app->status_page->adesc[SZE2_ADESC_NEXT(adesc->flags)][SZE2_MMIO_TX].size;
	if (lsize < size) {
		if (net_ratelimit())
			printk(KERN_DEBUG "%s: won't fit, requeueing\n",
					__func__);
		goto unlock;
	}

	off1 = app->status_page->adesc[idx][SZE2_MMIO_TX].offset;
	off = do_div(off1, asize);

	szedata2_eth_copy(eth, SZE2_MMIO_TX, &pkt, off, sizeof(pkt), asize);
	off = (off + 8) & (asize - 1);
	szedata2_eth_copy(eth, SZE2_MMIO_TX, skb->data, off, len, asize);
	off = (off + len) & (asize - 1);
	szedata2_eth_copy(eth, SZE2_MMIO_TX, NULL, off, size - 8 - len, asize);
	combo_get_ndev_stats(dev)->tx_packets++;
	combo_get_ndev_stats(dev)->tx_bytes += len;

	BUG_ON(szedata2_txunlock_data(app, &txu));
free:
	dev_kfree_skb(skb);
	return NETDEV_TX_OK;
unlock:
	txu.size = 0;
	BUG_ON(szedata2_txunlock_data(app, &txu));
	return NETDEV_TX_BUSY;
}

static struct net_device_stats *szedata2_eth_get_stats(struct net_device *dev)
{
	return combo_get_ndev_stats(dev);
}

/**
 * szedata2_eth_add - register ethernet interfaces
 *
 * @sd: related struct szedata2
 * @interfaces: count of interfaces (RX/TX)
 * @get_mac: function to obtain MAC address of @i (ret 0 = success; may be NULL)
 */
int szedata2_eth_add(struct szedata2 *sd, unsigned int interfaces[2],
		int (*get_mac)(struct szedata2 *sd, unsigned int i, u8 *mac))
{
	static const struct net_device_ops ops = {
		.ndo_open = szedata2_eth_open,
		.ndo_stop = szedata2_eth_close,
		.ndo_start_xmit = szedata2_eth_xmit,
		.ndo_get_stats = szedata2_eth_get_stats,
	};
	struct szedata2_eth *eth;
	struct net_device *ndev;
	unsigned int a, maxif = max(interfaces[0], interfaces[1]);
	int ret;

	if (no_eth)
		return 0;

	sd->ndev = kmalloc(sizeof(*sd->ndev) * maxif, GFP_KERNEL);
	if (sd->ndev == NULL)
		return -ENOMEM;

	for (a = 0; a < maxif; a++) {
		ndev = sd->ndev[a] = alloc_etherdev(sizeof(*eth));
		if (ndev == NULL) {
			printk(KERN_ERR "%s: failed to alloc etherdev\n",
					__func__);
			ret = -ENOMEM;
			goto err;
		}
		eth = netdev_priv(ndev);
		eth->idx = a;
		if (interfaces[1])
			eth->tx_idx_shift = ilog2(interfaces[0]/interfaces[1]);
		eth->sd = sd;
#ifdef CONFIG_HAVE_NET_DEVICE_OPS
		ndev->netdev_ops = &ops;
#else
		combo6_copy_ndev_ops(ndev, &ops);
#endif
		SET_NETDEV_DEV(ndev, sd->parent);
		sprintf(ndev->name, "ceth%u", a);
		if (!get_mac || get_mac(sd, a >> eth->tx_idx_shift,
					ndev->dev_addr)) {
			random_ether_addr(ndev->dev_addr);
			ndev->dev_addr[0] = 0x00;
			ndev->dev_addr[1] = 0x11;
			ndev->dev_addr[2] = 0x17;
		}
		ret = register_netdev(sd->ndev[a]);
		if (ret) {
			printk(KERN_ERR "%s: failed to register etherdev "
					"ceth%u\n", __func__, a);
			free_netdev(ndev);
			goto err;
		}
	}
	memcpy(sd->eth_ifaces, interfaces, sizeof(sd->eth_ifaces));

	return 0;
err:
	for (; a > 0; a--) {
		unregister_netdev(sd->ndev[a - 1]);
		free_netdev(sd->ndev[a - 1]);
	}
	kfree(sd->ndev);
	return ret;
}
EXPORT_SYMBOL_GPL(szedata2_eth_add);

void szedata2_eth_remove(struct szedata2 *sd)
{
	unsigned int a;

	if (no_eth)
		return;

	for (a = 0; a < max(sd->eth_ifaces[0], sd->eth_ifaces[1]); a++) {
		unregister_netdev(sd->ndev[a]);
		free_netdev(sd->ndev[a]);
	}

	kfree(sd->ndev);
}
EXPORT_SYMBOL_GPL(szedata2_eth_remove);
